##
## VMD Movie Generator version 1.1
##
## A script to generate movies through a simple Tk interface
## Supports a number of built-in movie types, can be easily modified
## or customized by the user to make new kinds of movies.
##
## Author: John E. Stone
##         johns@ks.uiuc.edu
##         vmd@ks.uiuc.edu
##
## $Id: vmdmovie.tcl,v 1.59 2003/05/23 16:05:46 johns Exp $
##
##
## Home Page
## ---------
##   http://www.ks.uiuc.edu/Research/vmd/script_library/scripts/vmdmovie/
##
## Instructions:
## -------------
##   Make sure you have the NetPBM 9.23 (or later) package in 
##   your command path if you're using Unix or MacOS X (required)
##
##   Make sure you have ImageMagick in your command path 
##   (required for "animated GIF" movies)
##
##   On Windows, you must install VideoMach 2.7.2 
##     http://www.gromada.com/VideoMach.html
##
##   On SGI, make sure you have "dmconvert" in your command path
##   (required for all SGI-specific compressor modes)
##
##   Make sure you have "ffmpeg" in your command path
##   (optional MPEG-1 video encoder)
##
##   After you have checked that these programs are avaiable,
##   source this script via "source movie.tcl".  Having sourced
##   the script, you can run it with "vmdmovie".
##    
##   Once the script loads up and you get your GUI window, you
##   can select one of the available movie types, configure the
##   working directory, etc.
##
## If you have problems:
## ---------------------
##   Several of the back-end video encoders are very touchy/flaky
##   about the input files they can handle, and many of them have 
##   little or no error handling.  Try changing the size of the 
##   VMD window you're rendering from, or changing the output resolution
##   if the video encoders give you trouble.
##

## Tell Tcl that we're a package and any dependencies we may have
package provide vmdmovie 1.1

namespace eval ::MovieMaker:: {
  namespace export moviemaker

  # window handles
  variable w                         ;# handle to main window

  # global settings for work directories etc
  variable workdir     "frames"      ;# area for temp and output files
  variable basename    "untitled"    ;# basename for animation files 

  # parameters for the movie generator routines
  variable movietype   "rockandroll" ;# what kind of movie we're making
  variable guiframes   180           ;# number of frames in animation GUI
  variable numframes     1           ;# number of real frames in animation
  variable anglestep     1           ;# change in angle per frame step 
  variable trjstart      0           ;# first trajectory frame in anim
  variable trjend        1           ;# last trajectory frame in anim
  variable trjstep       1           ;# step size for trajectory playback

  # parameters for renderers and video encoders
  variable framerate    24           ;# playback frame rate of animation
  variable renderer    "snapshot"    ;# which rendering method to use
  variable imgformat   "tga"         ;# which format generated images will use
  variable movieformat "ppmtompeg"   ;# which compress video format to use
  variable bitrate     "8000000"     ;# bitrate for movie encoders

  # post-processing, pre-compression settings and options
  variable presmooth    "0"           ;# smooth images prior to compression
  variable prescale     "0"           ;# post blur enlargement/decimation
  variable scalefactor  "0.5"         ;# post blur enlargement/decimation factor
  ;# default string for  image text labels
  variable prelabel     "0"
  variable labeltext    {Produced by VMD:
 http://www.ks.uiuc.edu/Research/vmd/}
  variable labelsize    "7"           ;# height of characters above baseline
  variable labelrow     "auto"        ;# baseline pixel row
  variable labelcolor   "gray"        ;# color to use, black, gray, or white
  variable labelbgcolor "transparent" ;# background color or transparent      

  # rendering/compression status information
  variable statusmsg   "Ready      " ;# most recent status message
  variable statusstep  "0"           ;# what step we're on in the process
  variable statussteps "0"           ;# how many steps to complete in process
  variable statusfrac  "0"           ;# fraction of total work to complete
  variable statustotal "0"           ;# total work units to complete for phase
  variable usercancel  "0"           ;# user cancellation of running anim

  # post-compression steps
  variable cleanfiles   "1"           ;# delete intermediate image frames etc

  # place to stuff exception info from catch
  variable foo                       ;# for catching exceptions
  variable viewpoints                ;# used by the rock+roll code
}
  
##
## Main routine
## Create the window and initialize data structures
##
proc ::MovieMaker::moviemaker {} {
  variable w
  variable workdir
  variable movietype
  variable movieformat
  variable statusmsg
  variable statusstep
  variable statussteps
  variable statusfrac
  variable statustotal
  global tcl_platform

  switch [vmdinfo arch] { 
    WIN32 {
      set workdir "c:/temp"
      set movieformat "videomachmpeg" 
    }
    MACOSX {
      set workdir "/"
    }
    default {
      set workdir "/tmp"
    } 
  }
 
  # If already initialized, just turn on
  if { [winfo exists .movie] } {
    wm deiconify $w
    return
  }

  set w [toplevel ".movie"]
  wm title $w "VMD Movie Generator" 
  wm resizable $w 0 0

  ##
  ## make the menu bar
  ## 
  frame $w.menubar -relief raised -bd 2 ;# frame for menubar
  pack $w.menubar -padx 1 -fill x

  menubutton $w.menubar.renderer -text "Renderer" -underline 0 -menu $w.menubar.renderer.menu
  menubutton $w.menubar.movietype -text "Movie Settings" -underline 0 -menu $w.menubar.movietype.menu
  menubutton $w.menubar.format -text Format -underline 0 -menu $w.menubar.format.menu
  menubutton $w.menubar.help -text Help -underline 0 -menu $w.menubar.help.menu

  ##
  ## movie type menu
  ##
  menu $w.menubar.movietype.menu -tearoff no
  $w.menubar.movietype.menu add radiobutton -label "Rock and Roll (XY lemniscate)" \
    -value "rockandroll" -variable "::MovieMaker::movietype"  -command ::MovieMaker::setmovietypedefaults
  $w.menubar.movietype.menu add radiobutton -label "Rotation about Y axis" \
    -value "rotation" -variable "::MovieMaker::movietype" -command ::MovieMaker::setmovietypedefaults
  $w.menubar.movietype.menu add radiobutton -label "Trajectory" \
    -value "trajectory" -variable "::MovieMaker::movietype" -command ::MovieMaker::setmovietypedefaults
  $w.menubar.movietype.menu add radiobutton -label "Trajectory Rock" \
    -value "trajectoryrock" -variable "::MovieMaker::movietype" -command ::MovieMaker::setmovietypedefaults


  ##
  ## pre/post-compression processing buttons
  ##
  $w.menubar.movietype.menu add separator 
  switch [vmdinfo arch] {
    WIN32 {
      $w.menubar.movietype.menu add checkbutton -label "Delete image files" \
        -variable "::MovieMaker::cleanfiles"
    }
    default {
      $w.menubar.movietype.menu add checkbutton -label "1: Image smoothing" \
        -variable "::MovieMaker::presmooth"
      $w.menubar.movietype.menu add checkbutton -label "2: Half-size rescaling"\
        -variable "::MovieMaker::prescale"
      $w.menubar.movietype.menu add checkbutton -label "3: Text labelling" \
        -variable "::MovieMaker::prelabel"
      $w.menubar.movietype.menu add checkbutton -label "4: Delete image files" \
        -variable "::MovieMaker::cleanfiles"
      $w.menubar.movietype.menu add command -label "Text label settings..." \
        -command "::MovieMaker::labelconfig"
    }
  }


  ##
  ## What renderer to use 
  ## 
  menu $w.menubar.renderer.menu -tearoff no

  # until program paths are fixed, Windows won't work for ray tracers
  switch $tcl_platform(platform) { 
    windows {
      $w.menubar.renderer.menu add radiobutton -label "Snapshot (Screen Capture)" \
        -value "snapshot" -variable "::MovieMaker::renderer" 
      $w.menubar.renderer.menu add radiobutton -label "Tachyon (Ray Tracer)" \
        -value "tachyon" -variable "::MovieMaker::renderer"  
    }

    default {
      $w.menubar.renderer.menu add radiobutton -label "Snapshot (Screen Capture)" \
        -value "snapshot" -variable "::MovieMaker::renderer" 
      $w.menubar.renderer.menu add radiobutton -label "Internal Tachyon (Ray Tracer)" \
        -value "libtachyon" -variable "::MovieMaker::renderer"  
      $w.menubar.renderer.menu add radiobutton -label "Tachyon (Ray Tracer)" \
        -value "tachyon" -variable "::MovieMaker::renderer"  
#      $w.menubar.renderer.menu add radiobutton -label "POV-Ray (Ray Tracer)" \
#        -value "povray" -variable "::MovieMaker::renderer"  
    }
  }


  ##
  ## compression format menu
  ##
  menu $w.menubar.format.menu -tearoff no
  switch [vmdinfo arch] {
    IRIX6 -
    IRIX6_64 {
      $w.menubar.format.menu add radiobutton -label "AVI (SGI dmconvert)" \
        -value "sgiavi" -variable "::MovieMaker::movieformat"
      $w.menubar.format.menu add radiobutton -label "MPEG-1 (SGI dmconvert)" \
        -value "sgimpeg1v" -variable "::MovieMaker::movieformat"
      $w.menubar.format.menu add radiobutton -label "SGI Movie (SGI dmconvert)" \
        -value "sgimv" -variable "::MovieMaker::movieformat"
      $w.menubar.format.menu add radiobutton -label "Quicktime (SGI dmconvert)" \
        -value "sgiqt"  -variable "::MovieMaker::movieformat"
      $w.menubar.format.menu add radiobutton -label "Animated GIF (ImageMagick)" \
        -value "imgif" -variable "::MovieMaker::movieformat"
      $w.menubar.format.menu add radiobutton -label "MPEG-1 (ppmtompeg)" \
        -value "ppmtompeg" -variable "::MovieMaker::movieformat"
#      $w.menubar.format.menu add radiobutton -label "MPEG-1 (ffmpeg)" \
#        -value "ffmpeg" -variable "::MovieMaker::movieformat"
    }

    WIN32 {
      $w.menubar.format.menu add radiobutton -label "AVI (VideoMach)" \
        -value "videomachavi" -variable "::MovieMaker::movieformat"
      $w.menubar.format.menu add radiobutton -label "MPEG-1 (VideoMach)" \
        -value "videomachmpeg" -variable "::MovieMaker::movieformat"
    }

    default {
      $w.menubar.format.menu add radiobutton -label "Animated GIF (ImageMagick)" \
        -value "imgif" -variable "::MovieMaker::movieformat"
      $w.menubar.format.menu add radiobutton -label "MPEG-1 (ppmtompeg)" \
        -value "ppmtompeg" -variable "::MovieMaker::movieformat"
#      $w.menubar.format.menu add radiobutton -label "MPEG-1 (ffmpeg)" \
#        -value "ffmpeg" -variable "::MovieMaker::movieformat"
    }
  }
  $w.menubar.format.menu add command -label "Change Compression Settings..." \
    -command "::MovieMaker::movieconfig"

   
  ##
  ## help menu
  ##
  menu $w.menubar.help.menu -tearoff no
  $w.menubar.help.menu add command -label "Help..." -command "vmd_open_url [string trimright [vmdinfo www] /]/plugins/vmdmovie"

  pack $w.menubar.renderer $w.menubar.movietype $w.menubar.format -side left 
  pack $w.menubar.help -side right

  frame $w.workdir    ;# frame for data entry areas
  button $w.workdir.button -text "Set working directory:" \
    -command "::MovieMaker::getworkdir"
  label $w.workdir.label -textvariable ::MovieMaker::workdir
  pack $w.workdir.button $w.workdir.label -side left -anchor w

  frame $w.basename   ;# frame for basename of animation files
  label $w.basename.label -text "Name of movie:"
  entry $w.basename.entry -width 15 -relief sunken -bd 2 \
    -textvariable ::MovieMaker::basename
  pack $w.basename.label $w.basename.entry -side left -anchor w


  ##
  ## Number of frames to generate
  ## 
  frame $w.numframes ;# frame for movie parameters
  label $w.numframes.label -text "Number of frames to render:"
  entry $w.numframes.entry -width 4 -relief sunken -bd 2 \
    -textvariable ::MovieMaker::guiframes
  pack $w.numframes.label $w.numframes.entry -side left -anchor w


  ## 
  ## Progress area
  ##
  frame $w.status ;# status frame
  frame $w.status.cond
  label $w.status.cond.label -text "  Status: "
  label $w.status.cond.msg   -textvariable ::MovieMaker::statusmsg
  pack  $w.status.cond.label $w.status.cond.msg -side left -anchor w

  frame $w.status.step
  label $w.status.step.label -text "   Stage: "
  label $w.status.step.step  -textvariable ::MovieMaker::statusstep
  label $w.status.step.slash -text " of "
  label $w.status.step.steps -textvariable ::MovieMaker::statussteps
  pack  $w.status.step.label $w.status.step.step $w.status.step.slash \
    $w.status.step.steps -side left -anchor w

  frame $w.status.frac
  label $w.status.frac.label -text "Progress: "
  label $w.status.frac.frac  -textvariable ::MovieMaker::statusfrac
  label $w.status.frac.slash -text " of "
  label $w.status.frac.total -textvariable ::MovieMaker::statustotal
  pack  $w.status.frac.label $w.status.frac.frac $w.status.frac.slash \
    $w.status.frac.total -side left -anchor w
 
  pack $w.status.cond $w.status.step $w.status.frac -side top -anchor w

  frame $w.goabort        ;# frame for Go/Abort buttons
  button $w.goabort.gobutton     -text "Make Movie" -command ::MovieMaker::buildmovie
  button $w.goabort.abortbutton  -text "Abort" -command ::MovieMaker::abort
  pack $w.goabort.gobutton $w.goabort.abortbutton -side left -anchor w

  ## 
  ## pack up the main frame
  ##
  pack $w.workdir $w.basename $w.numframes \
       $w.status \
       $w.goabort \
       -side top -pady 10 -fill x -anchor w
}


##
## Set default values when the movie type is changed
##
proc ::MovieMaker::setmovietypedefaults {} {
  variable guiframes
  variable movietype

  switch $movietype {
    trajectory {
      if {[llength [molinfo list]] > 0} {
        set guiframes [molinfo top get numframes]
      } else {
        set guiframes 1
      }
    }
    trajectoryrock {
      if {[llength [molinfo list]] > 0} {
        set guiframes [expr ([molinfo top get numframes] * 2)]
      } else {
        set guiframes 1
      }
    }
    default {
      set guiframes 180
    }
  }
}


##
## Test for file creation capability for work areas
##
proc ::MovieMaker::testfilesystem {} {
  variable workdir;
 
  # test access permissions on working directory
  if {[file isdirectory $workdir] != 1} {
    return 0; # failure 
  }    
  if {[file readable  $workdir] != 1} {
    return 0; # failure 
  }    
  if {[file writable  $workdir] != 1} {
    return 0; # failure 
  }    

  return 1; # success  
}

proc ::MovieMaker::abort { } {
  variable usercancel 
  set usercancel "1"
}

##
## Window for setting options for the selected renderer 
##
proc ::MovieMaker::renderconfig {} {
  variable sw

  set sw [toplevel ".renderconfig"]
  wm title $sw "Renderer Settings"  
  wm resizable $sw 0 0

  frame $sw.bottom        ;# frame for ok/cancel buttons
  button $sw.bottom.ok     -text "Ok"     -command "after idle destroy $sw"
  pack $sw.bottom.ok -side left -anchor w

  # pack up all of the frames
  pack $sw.bottom \
    -side top -pady 10 -fill x

}

##
## Get directory name
##
proc ::MovieMaker::getworkdir {} {
  variable workdir
  variable newdir

  set newdir [tk_chooseDirectory \
    -title "Choose working directory for temp files" \
    -initialdir $workdir -mustexist true]

  if {[string length $newdir] > 0} {
    set workdir $newdir 
  } 
}


##
## Window for setting options for the text label option
##
proc ::MovieMaker::labelconfig {} {
  variable sw

  set sw [toplevel ".labelconfig"]
  wm title $sw "Text Label Settings"  
  wm resizable $sw 0 0

  frame $sw.textlabel      ;# frame for text label settings

  label $sw.textlabel.textlabel -text "Text Label:"
  entry $sw.textlabel.entry -width 60 -relief sunken -bd 2 \
    -textvariable ::MovieMaker::labeltext

  label $sw.textlabel.size  -text "Text Size:"
  entry $sw.textlabel.labelsize -width 2 -relief sunken -bd 2 \
    -textvariable ::MovieMaker::labelsize

  label $sw.textlabel.label -text "Text Color:"
  radiobutton $sw.textlabel.white -text "White" -value "white" \
    -variable "::MovieMaker::labelcolor"  
  radiobutton $sw.textlabel.gray  -text "Gray"  -value "gray" \
    -variable "::MovieMaker::labelcolor"  
  radiobutton $sw.textlabel.black -text "Black" -value "black" \
    -variable "::MovieMaker::labelcolor"  

  label $sw.textlabel.bglabel -text "Text Background Color:"
  radiobutton $sw.textlabel.bgtrans -text "Transparent" -value "transparent" \
    -variable "::MovieMaker::labelbgcolor"  
  radiobutton $sw.textlabel.bgwhite -text "White" -value "white" \
    -variable "::MovieMaker::labelbgcolor"  
  radiobutton $sw.textlabel.bggray  -text "Gray"  -value "gray" \
    -variable "::MovieMaker::labelbgcolor"  
  radiobutton $sw.textlabel.bgblack -text "Black" -value "black" \
    -variable "::MovieMaker::labelbgcolor"  

  pack $sw.textlabel.textlabel $sw.textlabel.entry \
    $sw.textlabel.size $sw.textlabel.labelsize \
    $sw.textlabel.label \
    $sw.textlabel.white $sw.textlabel.gray $sw.textlabel.black \
    $sw.textlabel.bglabel $sw.textlabel.bgtrans \
    $sw.textlabel.bgwhite $sw.textlabel.bggray $sw.textlabel.bgblack \
    -side top -anchor w

  frame $sw.bottom        ;# frame for ok/cancel buttons
  button $sw.bottom.ok     -text "Ok"     -command "after idle destroy $sw"
  pack $sw.bottom.ok -side left -anchor w

  # pack up all of the frames
  pack $sw.textlabel $sw.bottom \
    -side top -pady 10 -fill x
}

##
## Window for setting options for the selected movie compression format
##
proc ::MovieMaker::movieconfig {} {
  variable sw

  set sw [toplevel ".movieconfig"]
  wm title $sw "Movie Settings"  
  wm resizable $sw 0 0

  # determine what configurable options this compressor has
  frame $sw.animrate      ;# frame for animation frame rate
  label $sw.animrate.label -text "Animation frame rate:"
  radiobutton $sw.animrate.24 -text "24 (Film)" -value "24" \
    -variable "::MovieMaker::framerate"  
  radiobutton $sw.animrate.25 -text "25 (PAL Video)" -value "25" \
    -variable "::MovieMaker::framerate"  
  radiobutton $sw.animrate.30 -text "30 (NTSC Video)" -value "30" \
    -variable "::MovieMaker::framerate" 
  pack $sw.animrate.label $sw.animrate.24 $sw.animrate.25 $sw.animrate.30 \
    -side top -anchor w

  frame $sw.bottom        ;# frame for ok/cancel buttons
  button $sw.bottom.ok     -text "Ok"     -command "after idle destroy $sw"
  pack $sw.bottom.ok -side left -anchor w

  # pack up all of the frames
  pack $sw.animrate $sw.bottom \
    -side top -pady 10 -fill x
}


proc ::MovieMaker::buildmovie {} {
  global tcl_platform
  variable imgformat 
  variable renderer
  variable statusmsg
  variable statusstep
  variable statussteps
  variable statusfrac
  variable statustotal
  variable usercancel
  variable foo
  variable movietype
  variable movieformat
  variable workdir
  variable guiframes
  variable numframes
  variable presmooth
  variable cleanfiles
  variable prescale
  variable prelabel
  variable scalefactor

  # copy GUI frame count into real frame count, prior to generation
  # this is done so that specific movie styles can override it on-the-fly.
  set numframes $guiframes

  # begin process
  set usercancel "0"
  set statusmsg "Preparing  " 
  set statusstep  "1"
  set statussteps "8"
  update    ;# update the Tk window, callbacks, etc

  # check to make sure the destination filesystem is writable
  # and that output files can be written.
  if {[::MovieMaker::testfilesystem] != 1} {
    puts "Temporary working directory $workdir is not usable."
    puts "Please double check file permissions and available"
    puts "space, and try again, or choose a different directory."
    return;
  }

  # set image format according to platform and renderer
  switch $renderer {
    snapshot {
      switch [vmdinfo arch] {
        WIN32 {
          set imgformat "bmp"
        }
        default {
          set imgformat "rgb"
        }
      }
    }
    libtachyon {
      set imgformat "tga"
    }
    tachyon {
      set imgformat "tga"
    }
    povray {
      set imgformat "tga"
    }
    default {
      set imgformat "rgb"    
    }
  }

  if {$usercancel == "0"} {
    set statusmsg "Rendering  " 
    set statusstep  "2"
    update    ;# update the Tk window, callbacks, etc

    switch $movietype {
      trajectory {
        genframes_trajectory       ;# Generate frames from VMD 
      }
      trajectoryrock {
        genframes_trajectory_rock  ;# Generate frames from VMD 
      }
      rotation {
        genframes_rotation         ;# Generate frames from VMD 
      }
      rockandroll {
        genframes_rockandroll      ;# Generate frames from VMD 
      }
      default {
        genframes_rockandroll      ;# Generate frames from VMD 
      }
    }
  }

  if {$usercancel == "0"} {
    set statusmsg "Converting  " 
    set statusstep  "3"
    update    ;# update the Tk window, callbacks, etc

    convertframes ppm    ;# Convert frames to PPM format
  }

  if {$usercancel == "0"} {
    set statusmsg "Smoothing   " 
    set statusstep  "4"
    update    ;# update the Tk window, callbacks, etc

    if {$presmooth == "1"} {
      smoothframes           ;# Smooth frames prior to compression
    }
  }

  if {$usercancel == "0"} {
    set statusmsg "Rescaling   " 
    set statusstep  "5"
    update    ;# update the Tk window, callbacks, etc

    if {$prescale == "1"} {
      rescaleframes          ;# Rescale frames prior to compression
    }
  }

  if {$usercancel == "0"} {
    set statusmsg "Text Labels " 
    set statusstep  "6"
    update    ;# update the Tk window, callbacks, etc

    if {$prelabel == "1"} {
      labelframes            ;# Add text labels to frames prior to compression
    }
  }

  if {$usercancel == "0"} {
    set statusmsg "Encoding    " 
    set statusstep  "7"
    update    ;# update the Tk window, callbacks, etc

    switch $movieformat {
      ffmpeg {
        ffmpeg           ;# Convert frame sequence to MPEG-1 video file
      }
      ppmtompeg {
        ppmtompeg        ;# Convert frame sequence to MPEG-1 video file
      }
      videomachavi {
        videomachavi     ;# Convert frame sequence to MPEG-1 video file
      }
      videomachmpeg {
        videomachmpeg    ;# Convert frame sequence to MPEG-1 video file
      }
      sgiavi {
        sgiavi           ;# Convert frame sequence to AVI video file
      }
      sgimv  {
        sgimv            ;# Convert frame sequence to SGI Movie video file
      }
      sgimpeg1v  {
        sgimpeg1v        ;# Convert frame sequence to SGI MPEG-1 video file
      }
      sgiqt {
        sgiqt            ;# Convert frame sequence to Quicktime video file
      }
      imgif -
      default {
        imgif            ;# Convert frame sequence to animated GIF file
      }
    }
  }

  set statusmsg "Cleaning    " 
  set statusstep  "8"
  if {$cleanfiles == "1"} {
    cleanframes          ;# delete temporary files and single frames
  }
  update    ;# update the Tk window, callbacks, etc

  if {$usercancel == "1"} {
    set statusmsg "Cancelled  "
    puts "Movie generation Cancelled."
  } else {
    set statusmsg "Completed   "
    puts "Movie generation complete."
  }
  update    ;# update the Tk window, callbacks, etc

  ##
  ## reset status area
  ##
  set statusmsg   "Ready      " ;# most recent status message
  set statusstep  "0"           ;# what step we're on in the process
  set statussteps "0"           ;# how many steps to complete in process
  set statusfrac  "0"           ;# fraction of total work to complete
  set statustotal "0"           ;# total work units to complete for phase
  update    ;# update the Tk window, callbacks, etc
}


##
## Support code for the rock and roll style
##
proc ::MovieMaker::roll_save_viewpoint {} {
   variable viewpoints
   if [info exists viewpoints] {unset viewpoints}
   # get the current matricies
   foreach mol [molinfo list] {
      set viewpoints($mol) [molinfo $mol get { center_matrix rotate_matrix scale_matrix global_matrix }]
   }
}

proc ::MovieMaker::roll_restore_viewpoint {} {
   variable viewpoints
   foreach mol [molinfo list] {
      if [info exists viewpoints($mol)] {
         molinfo $mol set { center_matrix rotate_matrix scale_matrix global_matrix } $viewpoints($mol)
      }
   }
}



##
## Generate all of the frames
##
proc ::MovieMaker::genframes_rockandroll {} {
  variable workdir
  variable numframes
  variable anglestep
  variable renderer
  variable basename 
  variable basefilename
  variable statusfrac
  variable statustotal
  variable usercancel

  variable yangle   15.0;
  variable xangle   10.0;
  variable x;
  variable y;
  variable i;

  puts "Generating image frames using renderer: $renderer"
  set statusfrac "0"
  set statustotal $numframes  
  update    ;# update the Tk window, callbacks, etc

  ::MovieMaker::roll_save_viewpoint; # save original viewpoint

  # Loop over all frames and generate images.
  set frame 0
  for {set i 0} {$i<$numframes} {incr i 1} {
    display resetview
    ::MovieMaker::roll_restore_viewpoint; # restore original viewpoint

    # now tweak relative to that view orientation
    set pipcnt [expr 6.283185 * ($i / ($numframes * 1.0))]
    rotate x by [expr $xangle * sin($pipcnt * 2.0)]
    rotate y by [expr $yangle * sin($pipcnt)]

    if {$usercancel == "1"} {
      set statusmsg "Cancelled  "
      display resetview
      ::MovieMaker::roll_restore_viewpoint; # restore original viewpoint
      return
    }

    set basefilename [format "%s/$basename.%04d" $workdir $frame]
    display update ui
    renderframe $basefilename

    incr frame
    set statusfrac $frame
  }
  display resetview
  ::MovieMaker::roll_restore_viewpoint; # restore original viewpoint
}


##
## Generate all of the frames
##
proc ::MovieMaker::genframes_rotation {} {
  variable workdir
  variable numframes
  variable anglestep
  variable renderer
  variable basename 
  variable basefilename
  variable statusfrac
  variable statustotal
  variable usercancel

  set statusfrac "0"
  set statustotal $numframes  
  update    ;# update the Tk window, callbacks, etc

  # Loop over all frames and generate images.
  set frame 0
  for {set i 0} {$i < $numframes} {incr i 1} {
    if {$usercancel == "1"} {
      set statusmsg "Cancelled  "
      return
    }

    set basefilename [format "%s/$basename.%04d" $workdir $frame]
    display update ui
    renderframe $basefilename

    rotate y by $anglestep

    incr frame
    set statusfrac $frame
  }
}


##
## Generate all of the frames
##
proc ::MovieMaker::genframes_trajectory_rock {} {
  variable workdir
  variable numframes
  variable renderer
  variable basename 
  variable basefilename
  variable statusfrac
  variable statustotal
  variable usercancel
  variable frame
  variable trjframe
  variable rockframes

  puts "Generating image frames using renderer: $renderer"
  set statusfrac "0"
  set statustotal $numframes  
  update    ;# update the Tk window, callbacks, etc

  # Loop over all frames and generate images.
  set frame 0
  set trjframe 0
  # calculate frames to run each way
  set rockframes [expr $numframes / 2]
 
  for {set i 0} {$i < $rockframes} {incr i 1} {
    if {$usercancel == "1"} {
      set statusmsg "Cancelled  "
      return
    }

    set basefilename [format "%s/$basename.%04d" $workdir $frame]

    puts "Trajectory frame: $trjframe"
    animate goto $trjframe

    display update ui
    renderframe $basefilename

    incr frame
    incr trjframe 1
    set statusfrac $frame
  }

  # fix up active frame for reverse direction
  incr trjframe -1
  for {set i 0} {$i < $rockframes} {incr i 1} {
    if {$usercancel == "1"} {
      set statusmsg "Cancelled  "
      return
    }

    set basefilename [format "%s/$basename.%04d" $workdir $frame]

    puts "Trajectory frame: $trjframe"
    animate goto $trjframe

    display update ui
    renderframe $basefilename

    incr frame
    incr trjframe -1
    set statusfrac $frame
  }

  # backpatch total frame count into variables for use
  # when making final movie..  Hopefully this works.
  set numframes $frame
}

##
## Generate all of the frames
##
proc ::MovieMaker::genframes_trajectory {} {
  variable workdir
  variable numframes
  variable trjstep
  variable renderer
  variable basename 
  variable basefilename
  variable statusfrac
  variable statustotal
  variable usercancel
  variable frame
  variable trjframe

  puts "Generating image frames using renderer: $renderer"
  set statusfrac "0"
  set statustotal $numframes  
  update    ;# update the Tk window, callbacks, etc

  # Loop over all frames and generate images.
  set frame 0
  set trjframe 0
  for {set i 0} {$i < $numframes} {incr i 1} {
    if {$usercancel == "1"} {
      set statusmsg "Cancelled  "
      return
    }

    set basefilename [format "%s/$basename.%04d" $workdir $frame]

    animate goto $trjframe

    display update ui
    renderframe $basefilename

    incr frame
    incr trjframe $trjstep
    set statusfrac $frame
  }
}

##
## Render one frame using existing settings
##
proc ::MovieMaker::renderframe { basefilename } {
  global env
  variable renderer
  variable rendercmd
  variable datfilename 
  variable imgfilename
  variable imgformat

  set datfilename $basefilename.dat
  set imgfilename $basefilename.$imgformat

  switch $renderer {
    snapshot {
      render snapshot $imgfilename
    } 
    libtachyon {
      render TachyonInternal $imgfilename
    } 
    tachyon {
      set rendercmd "\"$env(VMDDIR)/tachyon_[vmdinfo arch]\" -mediumshade $datfilename -format Targa -aasamples 4 -o $imgfilename"
      render Tachyon $datfilename $rendercmd
    } 
    povray {
      set rendercmd "povray  -I$datfilename -O$imgfilename +X +A +FT"
      render POV3 $datfilename $rendercmd
    }
    default {
      puts "Unsupported renderer"
    }
  }
}

##
## Convert to AVI or other formats using "videomach"
##
proc ::MovieMaker::videomachavi  {} {
  variable basename
  variable numframes
  variable framerate
  variable statusfrac
  variable statustotal
  variable mybasefilename 
  variable workdir

  set statusfrac  "0"
  set statustotal "1"
  update    ;# update the Tk window, callbacks, etc

  puts "Converting BMP sequence to AVI video format"
  puts "Attempting to convert using VideoMach 2.7.2"

  set mybasefilename [format "%s/$basename" $workdir] 
  file delete -force $mybasefilename.avi
  puts "c:/program files/videomach-2.7.2/vmach.exe /OpenSeq=$mybasefilename.0000.bmp /SaveVideo=$mybasefilename.avi /Start /Exit"

  # remap forward slashes to backslashes to make Windows happy.
  set mybasefilename [string map {/ \\} $mybasefilename]
  exec "c:/program files/videomach-2.7.2/vmach.exe" /OpenSeq=$mybasefilename.0000.bmp /SaveVideo=$mybasefilename.avi /Start /Exit
}


##
## Convert to MPEG or other formats using "videomach"
##
proc ::MovieMaker::videomachmpeg {} {
  variable basename
  variable numframes
  variable framerate
  variable statusfrac
  variable statustotal
  variable mybasefilename 
  variable workdir

  set statusfrac  "0"
  set statustotal "1"
  update    ;# update the Tk window, callbacks, etc

  puts "Converting BMP sequence to MPEG-1 video format"
  puts "Attempting to convert using VideoMach 2.7.2"

  set mybasefilename [format "%s/$basename" $workdir] 
  file delete -force $mybasefilename.mpg
  puts "c:/program files/videomach-2.7.2/vmach.exe /OpenSeq=$mybasefilename.0000.bmp /SaveVideo=$mybasefilename.mpg /Start /Exit"

  # remap forward slashes to backslashes to make Windows happy.
  set mybasefilename [string map {/ \\} $mybasefilename]
  exec "c:/program files/videomach-2.7.2/vmach.exe" /OpenSeq=$mybasefilename.0000.bmp /SaveVideo=$mybasefilename.mpg /Start /Exit
}

##
## Convert to AVI using SGI "dmconvert"
##
proc ::MovieMaker::sgiavi {} {
  variable numframes
  variable framerate
  variable basename
  variable basefilename
  variable workdir
  variable statusfrac
  variable statustotal
  variable mybasefilename 
  variable lastframe
  variable bitrate

  set statusfrac  "0"
  set statustotal "1"
  update    ;# update the Tk window, callbacks, etc
  set lastframe [expr $numframes - 1]

  puts "Converting frames to AVI video format"
  
  set mybasefilename [format "%s/$basename" $workdir] 

  file delete -force $mybasefilename.avi
  puts "dmconvert -v -f avi -p video,comp=jpeg,brate=$bitrate,rate=$framerate -n $mybasefilename.####.ppm,start=0,end=$lastframe,step=1 $mybasefilename.####.ppm $mybasefilename.avi"
  exec dmconvert -v -f avi -p video,comp=jpeg,brate=$bitrate,rate=$framerate \
   -n $mybasefilename.####.ppm,start=0,end=$lastframe,step=1 \
   $mybasefilename.####.ppm $mybasefilename.avi 
}

##
## Convert to SGI MPEG-1 video file using SGI "dmconvert"
##
proc ::MovieMaker::sgimpeg1v {} {
  variable numframes
  variable framerate
  variable basename
  variable basefilename
  variable workdir
  variable statusfrac
  variable statustotal
  variable lastframe 
  variable bitrate

  set statusfrac  "0"
  set statustotal "1"
  update    ;# update the Tk window, callbacks, etc
  set lastframe [expr $numframes - 1]

  puts "Converting frames to SGI MPEG-1 video format"
  
  set mybasefilename [format "%s/$basename" $workdir] 

  file delete -force $mybasefilename.mpg
  puts "dmconvert -v -f mpeg1v -p video,rate=$framerate -n $mybasefilename.####.ppm,start=0,end=$lastframe,step=1 $mybasefilename.####.ppm $mybasefilename.mpg" 
  exec dmconvert -v -f mpeg1v -p video,rate=$framerate \
   -n $mybasefilename.####.ppm,start=0,end=$lastframe,step=1 \
   $mybasefilename.####.ppm $mybasefilename.mpg 
}

##
## Convert to SGI Movie using SGI "dmconvert"
##
proc ::MovieMaker::sgimv {} {
  variable numframes
  variable framerate
  variable basename
  variable basefilename
  variable workdir
  variable statusfrac
  variable statustotal
  variable lastframe 
  variable bitrate

  set statusfrac  "0"
  set statustotal "1"
  update    ;# update the Tk window, callbacks, etc
  set lastframe [expr $numframes - 1]

  puts "Converting frames to SGI Movie video format"
  
  set mybasefilename [format "%s/$basename" $workdir] 

  file delete -force $mybasefilename.mv
  puts "dmconvert -v -f sgimv -p video,comp=jpeg,brate=$bitrate,rate=$framerate -n $mybasefilename.####.ppm,start=0,end=$lastframe,step=1 $mybasefilename.####.ppm $mybasefilename.mv"
  exec dmconvert -v -f sgimv -p video,comp=jpeg,brate=$bitrate,rate=$framerate \
   -n $mybasefilename.####.ppm,start=0,end=$lastframe,step=1 \
   $mybasefilename.####.ppm $mybasefilename.mv
}

##
## Convert to Quicktime using SGI "dmconvert"
##
proc ::MovieMaker::sgiqt {} {
  variable numframes
  variable framerate
  variable basename
  variable basefilename
  variable workdir
  variable statusfrac
  variable statustotal
  variable lastframe
  variable bitrate

  set statusfrac  "0"
  set statustotal "1"
  update    ;# update the Tk window, callbacks, etc
  set lastframe [expr $numframes - 1]

  puts "Converting frames to Quicktime video format"
  
  set mybasefilename [format "%s/$basename" $workdir] 

  file delete -force $mybasefilename.mov
  puts "dmconvert -v -f qt -p video,comp=qt_mjpega,brate=$bitrate,rate=$framerate -n $mybasefilename.####.ppm,start=0,end=$lastframe,step=1 $mybasefilename.####.ppm $mybasefilename.mov"
  exec dmconvert -v -f qt -p video,comp=qt_mjpega,brate=$bitrate,rate=$framerate \
   -n $mybasefilename.####.ppm,start=0,end=$lastframe,step=1 \
   $mybasefilename.####.ppm $mybasefilename.mov
}


##
## Convert to MPEG using "sampeg"
##
proc ::MovieMaker::sampeg2 {} {
  variable numframes
  variable framerate
  variable basename
  variable lastframe
  variable basefilename
  variable workdir
  variable statusfrac
  variable statustotal

  set statusfrac  "0"
  set statustotal "1"
  update    ;# update the Tk window, callbacks, etc

  puts "Converting Targa frames to MPEG-1 video format"
  
  set lastframe [expr $numframes - 1]
  set basefilename [format "%s/$basename" $workdir] 

  exec sampeg2/sampeg2_0.6.5.bin -1 --targa \
    --firstframe=0 --lastframe=$lastframe \
    $basefilename.%04d.tga $basefilename.mpg
}

##
## Convert to MPEG using "ffmpeg" 
##
proc ::MovieMaker::ffmpeg {} {
  variable numframes
  variable framerate
  variable basename
  variable workdir
  variable mybasefilename
  variable statusfrac
  variable statustotal
  variable foo

  set statusfrac  "0"
  set statustotal "1"
  update    ;# update the Tk window, callbacks, etc

  puts "Converting frames to MPEG-1 video format"
   
  set mybasefilename [format "%s/$basename" $workdir] 
 
  puts "ffmpeg -an -r $framerate -i $mybasefilename.%04d.ppm $mybasefilename.mpg"
  file delete -force $mybasefilename.mpg  
  set foo [catch { exec ffmpeg -an -r $framerate -i $mybasefilename.%04d.ppm $mybasefilename.mpg }]
#   exec ffmpeg -an -r $framerate -i $mybasefilename.%04d.ppm $mybasefilename.mpg
}

##
## Convert to MPEG using "ppmtompeg" 
##
proc ::MovieMaker::ppmtompeg {} {
  variable numframes
  variable framerate
  variable basename
  variable workdir
  variable mybasefilename
  variable statusfrac
  variable statustotal
  variable foo
  variable parfile
  variable frame
  variable framename
  variable lastframe

  set statusfrac  "0"
  set statustotal "1"
  update    ;# update the Tk window, callbacks, etc

  puts "Converting frames to MPEG-1 video format"
  
  set mybasefilename [format "%s/$basename" $workdir] 

  # generate MPEG-1 encoder parameter file
  set parfile [open "$mybasefilename.par" w]
  puts $parfile "PATTERN    IBBPBBPBBPBBPBB"
  puts $parfile "FORCE_ENCODE_LAST_FRAME"          ;# force anim loopable
  puts $parfile "OUTPUT     $mybasefilename.mpg"
  puts $parfile "INPUT_DIR  $workdir"
  puts $parfile "INPUT"

  set lastframe [format "%04d" [expr $numframes - 1]]
  puts $parfile "$basename.*.ppm \[0000-$lastframe\]"

#  for {set frame "0"} {$frame < $numframes} {incr frame} {
#    set framename [format "$basename.%04d.ppm" $frame]
#    puts $parfile "$framename"
#  } 

  puts $parfile "END_INPUT"
  puts $parfile "BASE_FILE_FORMAT PPM"
  puts $parfile "INPUT_CONVERT *"
  puts $parfile "GOP_SIZE 15"
  puts $parfile "SLICES_PER_FRAME 1"
  puts $parfile "PIXEL HALF"
  puts $parfile "RANGE 32"
  puts $parfile "PSEARCH_ALG LOGARITHMIC"
  puts $parfile "BSEARCH_ALG CROSS2"
  puts $parfile "IQSCALE 8"
  puts $parfile "PQSCALE 10"
  puts $parfile "BQSCALE 25"
  puts $parfile "REFERENCE_FRAME DECODED"
  close $parfile

  puts "ppmtompeg $mybasefilename.par"
  file delete -force $mybasefilename.mpg  
  set foo [catch { exec ppmtompeg $mybasefilename.par }]
}

##
## Convert to animated GIF using ImageMajick
##

proc ::MovieMaker::imgif {} {
  variable numframes
  variable framerate
  variable basename
  variable workdir
  variable mybasefilename
  variable statusfrac
  variable statustotal
  variable foo
  variable delay

  set statusfrac  "0"
  set statustotal "1"
  update    ;# update the Tk window, callbacks, etc

  puts "Converting frames to animated GIF format"
  set mybasefilename [format "%s/$basename" $workdir] 
  set delay [format "%5.2f" [expr 100.0 / $framerate]] 
 
  puts "convert -delay $delay -loop 4 $mybasefilename.*.ppm $mybasefilename.gif"
  file delete -force $mybasefilename.gif
  exec convert -delay $delay -loop 4 $mybasefilename.*.ppm $mybasefilename.gif

##
## A re-compress using gimp, yields a file 1/8th the size of convert's 
## original output.
##
## gimp --no-interface --no-data --batch \
##   '(let* ( (img (car (file-gif-load 1 "untitled.gif" "untitled.gif")))
##            (drawable (car (gimp-image-active-drawable img))))
##      (gimp-convert-indexed img 0 0 255 0 1 "mypalette")
##      (file-gif-save 1 img drawable
##                     "untitled2.gif" "untitled2.gif" 0 1 100 0)
##      (gimp-quit 0))'
##

}

##
## Convert frames to the format required by the selected video encoder
##
proc ::MovieMaker::convertframes { newformat } {
  variable imgformat
  variable numframes
  variable oldfilename
  variable newfilename
  variable basename
  variable workdir
  variable statusfrac
  variable statustotal
  variable usercancel
  variable foo

  set statusfrac  "0"
  set statustotal $numframes
  update    ;# update the Tk window, callbacks, etc

  if { $imgformat == $newformat } {
    puts "No frame format conversion necessary, continuing."
    return
  }

  puts "Converting frames from $imgformat format to $newformat..."

  # Loop over all frames and convert images.
  switch $newformat {
    ppm {
      switch $imgformat {
        tga {
          for {set i 0} {$i < $numframes} {incr i 1} {
            if {$usercancel == "1"} {
              set statusmsg "Cancelled  "
              return
            }
            set oldfilename [format "%s/$basename.%04d.tga" $workdir $i]
            set newfilename [format "%s/$basename.%04d.ppm" $workdir $i]
            file delete -force $newfilename
            exec tgatoppm $oldfilename > $newfilename
            set statusfrac $i
            update    ;# update the Tk window, callbacks, etc
          }
        }
        rgb {
          for {set i 0} {$i < $numframes} {incr i 1} {
            if {$usercancel == "1"} {
              set statusmsg "Cancelled  "
              return
            }
            set oldfilename [format "%s/$basename.%04d.rgb" $workdir $i]
            set newfilename [format "%s/$basename.%04d.ppm" $workdir $i]
            file delete -force $newfilename
            set foo [catch {exec sgitopnm $oldfilename > $newfilename}]
            set statusfrac $i
            update    ;# update the Tk window, callbacks, etc
          }
        }
        default { 
          puts "Conversion unsupported, exiting"
        } 
      }
    }
    default {
      puts "Conversion unsupported, exiting"
    }
  }
}


##
## Convert frames to the format required by the selected video encoder
##
proc ::MovieMaker::smoothframes { } {
  variable numframes
  variable oldfilename
  variable newfilename
  variable basename
  variable workdir
  variable statusfrac
  variable statustotal
  variable usercancel

  set statusfrac  "0"
  set statustotal $numframes
  update    ;# update the Tk window, callbacks, etc

  puts "Smoothing image frames."

  # Loop over all frames and process images.
  for {set i 0} {$i < $numframes} {incr i 1} {
    if {$usercancel == "1"} {
      set statusmsg "Cancelled  "
      return
    }
    set oldfilename [format "%s/$basename.%04d.ppm" $workdir $i]
    set newfilename [format "%s/orig.$basename.%04d.ppm" $workdir $i]
    file delete -force $newfilename
    file rename -force -- $oldfilename $newfilename
    exec pnmsmooth $newfilename > $oldfilename
    file delete -force $newfilename
    set statusfrac $i
    update    ;# update the Tk window, callbacks, etc
  }
}



##
## Convert frames to the format required by the selected video encoder
##
proc ::MovieMaker::rescaleframes { } {
  variable numframes
  variable oldfilename
  variable newfilename
  variable basename
  variable workdir
  variable statusfrac
  variable statustotal
  variable usercancel
  variable scalefactor

  set statusfrac  "0"
  set statustotal $numframes
  update    ;# update the Tk window, callbacks, etc

  puts "Rescaling image frames."

  # Loop over all frames and process images.
  for {set i 0} {$i < $numframes} {incr i 1} {
    if {$usercancel == "1"} {
      set statusmsg "Cancelled  "
      return
    }
    set oldfilename [format "%s/$basename.%04d.ppm" $workdir $i]
    set newfilename [format "%s/orig.$basename.%04d.ppm" $workdir $i]
    file delete -force $newfilename
    file rename -force -- $oldfilename $newfilename
    exec pnmscale $scalefactor  $newfilename > $oldfilename
    file delete -force $newfilename
    set statusfrac $i
    update    ;# update the Tk window, callbacks, etc
  }
}


##
## Tag frames with copyright or other text, prior to video encoding
##
proc ::MovieMaker::labelframes { } {
  variable numframes
  variable oldfilename
  variable newfilename
  variable basename
  variable workdir
  variable statusfrac
  variable statustotal
  variable usercancel
  variable labeltext
  variable labelcolor
  variable labelbgcolor
  variable labelsize
  variable labelrow
  variable localrow

  set statusfrac  "0"
  set statustotal $numframes
  update    ;# update the Tk window, callbacks, etc

  puts "Tagging image frames with user-specified text."

  if {$labelrow == "auto"} {
    set localrow [expr $labelsize + 3];
  } else {
    set localrow $labelrow
  }

  # Loop over all frames and process images.
  for {set i 0} {$i < $numframes} {incr i 1} {
    if {$usercancel == "1"} {
      set statusmsg "Cancelled  "
      return
    }
    set oldfilename [format "%s/$basename.%04d.ppm" $workdir $i]
    set newfilename [format "%s/orig.$basename.%04d.ppm" $workdir $i]
    file delete -force $newfilename
    file rename -force -- $oldfilename $newfilename

    ##
    ## image compositing method for image-based logos etc...
    ## exec  pnmcomp -invert -alpha stamp-30.pgm black-30.ppm $newfilename > $oldfilename
    ##

    exec ppmlabel -size $labelsize -y $localrow -color $labelcolor -background $labelbgcolor -text $labeltext $newfilename > $oldfilename
    file delete -force $newfilename
    set statusfrac $i
    update    ;# update the Tk window, callbacks, etc
  }
}


##
## Composite frames with copyright or other images, prior to video encoding
##
proc ::MovieMaker::compositeframes { } {
  variable numframes
  variable oldfilename
  variable newfilename
  variable basename
  variable workdir
  variable statusfrac
  variable statustotal
  variable usercancel

  set statusfrac  "0"
  set statustotal $numframes
  update    ;# update the Tk window, callbacks, etc

  puts "Compositing image frames with user-specified image."

  # Loop over all frames and process images.
  for {set i 0} {$i < $numframes} {incr i 1} {
    if {$usercancel == "1"} {
      set statusmsg "Cancelled  "
      return
    }
    set oldfilename [format "%s/$basename.%04d.ppm" $workdir $i]
    set newfilename [format "%s/orig.$basename.%04d.ppm" $workdir $i]
    file delete -force $newfilename
    file rename -force -- $oldfilename $newfilename
    
    # image compositing method for image-based logos etc...
    exec  pnmcomp -invert -alpha stamp-30.pgm black-30.ppm $newfilename > $oldfilename
    file delete -force $newfilename
    set statusfrac $i
    update    ;# update the Tk window, callbacks, etc
  }
}


##
## Clean up all of the temporary frames once the whole
## animation is done.
##
proc ::MovieMaker::cleanframes {} {
  variable numframes
  variable basename
  variable workdir 
  variable filespec
  variable fileset
  variable statusfrac
  variable statustotal

  set statusfrac  "0"
  set statustotal $numframes
  update    ;# update the Tk window, callbacks, etc

  puts "Cleaning generated data, frames, and encoder parameter files"

  file delete -force $workdir/$basename.par
  for {set i 0} {$i < $numframes} {incr i 1} {
    set filespec [format "$workdir/$basename.%04d" $i]
    set files $filespec 
    file delete -force $files.ppm $files.tga $files.jpg \
               $files.rgb $files.bmp $files.dat
    set statusfrac  $i
    update    ;# update the Tk window, callbacks, etc
  }  
}

# This gets called by VMD the first time the menu is opened.
proc vmdmovie_tk_cb {} {
  variable foobar
  # Don't destroy the main window, because we want to register the window
  # with VMD and keep reusing it.  The window gets iconified instead of
  # destroyed when closed for any reason.
  #set foobar [catch {destroy $::MovieMaker::w  }]  ;# destroy any old windows

  ::MovieMaker::moviemaker   ;# start the movie maker
  return $MovieMaker::w
}

