/***************************************************************************
 *cr
 *cr            (C) Copyright 1995-2003 The Board of Trustees of the
 *cr                        University of Illinois
 *cr                         All Rights Reserved
 *cr
 ***************************************************************************/

/***************************************************************************
 * RCS INFORMATION:
 *
 *      $RCSfile: xyzplugin.c,v $
 *      $Author: johns $       $Locker:  $             $State: Exp $
 *      $Revision: 1.16 $       $Date: 2003/08/26 22:05:01 $
 *
 ***************************************************************************/

/*
 *  XYZ molecule file format:
 *    XYZ files are a simple molecule file format suitable for output
 *    by homegrown software since they are very minimalistic.  They don't
 *    even include bonding information.
 *
 *  [ # optional comment line ] comment line (can be blank)
 *                              ^^^ note, this is not supported by the current
 *                                  version of this plugin.
 *  [ N                       ] # of atoms, required by this xyz reader plugin
 *  [ molecule name           ] name of molecule (can be blank)
 *  atom1 x y z [optional data] atom name followed by xyz coords 
 *  atom2 x y z [ ...         ] and and (optionally) other data.
 *  ...                         instead of atom name the atom number in 
 *  atomN x y z [ ...         ] the PTE can be given.
 *                      
 *  Note that this plugin currently ignores everything following the z 
 *  coordinate (the optional data fields).
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include "molfile_plugin.h"

/* periodic table of elements for translation of ordinal to atom type */
static const char *pte[] = { 
    "X",  "H",  "He", "Li", "Be", "B",  "C",  "N",  "O",  "F",  "Ne",
    "Na", "Mg", "Al", "Si", "P" , "S",  "Cl", "Ar", "K",  "Ca", "Sc",
    "Ti", "V",  "Cr", "Mn", "Fe", "Co", "Ni", "Cu", "Zn", "Ga", "Ge", 
    "As", "Se", "Br", "Kr", "Rb", "Sr", "Y",  "Zr", "Nb", "Mo", "Tc",
    "Ru", "Rh", "Pd", "Ag", "Cd", "In", "Sn", "Sb", "Te", "I",  "Xe",
    "Cs", "Ba", "La", "Ce", "Pr", "Nd", "Pm", "Sm", "Eu", "Gd", "Tb",
    "Dy", "Ho", "Er", "Tm", "Yb", "Lu", "Hf", "Ta", "W",  "Re", "Os",
    "Ir", "Pt", "Au", "Hg", "Tl", "Pb", "Bi", "Po", "At", "Rn", "Fr",
    "Ra", "Ac", "Th", "Pa", "U",  "Np", "Pu", "Am", "Cm", "Bk", "Cf",
    "Es", "Fm", "Md", "No", "Lr"
};
static const int nr_pte = sizeof(pte) / sizeof(char *);

typedef struct {
  FILE *file;
  int numatoms;
  char *file_name;
  molfile_atom_t *atomlist;
} xyzdata;
 
static void *open_xyz_read(const char *filename, const char *filetype, 
                           int *natoms) {
  FILE *fd;
  xyzdata *data;
  int i;

  fd = fopen(filename, "rb");
  if (!fd) return NULL;
  
  data = (xyzdata *)malloc(sizeof(xyzdata));
  data->file = fd;
  data->file_name = strdup(filename);

  /* First line is the number of atoms   */
  i = fscanf(data->file, "%d", natoms);
  if (i < 1) {
    fprintf(stderr, "\n\nread) ERROR: xyz file '%s' should have the number of atoms in the first line.\n", filename);
    return NULL;
  }
  data->numatoms=*natoms;

  while (getc(fd) != '\n');
  /* second line is a title or empty, so skip past it */
  while (getc(fd) != '\n');
  
  return data;
}

static int read_xyz_structure(void *mydata, int *optflags, 
                              molfile_atom_t *atoms) {
  int i, j;
  char *k;
  float coord;
  molfile_atom_t *atom;
  xyzdata *data = (xyzdata *)mydata;

  *optflags = MOLFILE_NOOPTIONS; /* no optional data */

  for(i=0;i<data->numatoms;i++) {
    char buffer[1024], fbuffer[1024];
    k = fgets(fbuffer, 1024, data->file);
    atom = atoms + i;
    j=sscanf(fbuffer, "%s %f %f %f", buffer, &coord, &coord, &coord);
    if (k == NULL) {
      fprintf(stderr, "xyz structure) missing atom(s) in file '%s'\n",data->file_name);
      fprintf(stderr, "xyz structure) expecting '%d' atoms, found only '%d'\n",data->numatoms,i+1);
      return MOLFILE_ERROR;
    } else if (j < 4) {
      fprintf(stderr, "xyz structure) missing type or coordinate(s) in file '%s' for atom '%d'\n",
          data->file_name, i+1);
      return MOLFILE_ERROR;
    }

    /* handle the case if the first item is an ordinal number 
     * from the PTE */
    if (isdigit(buffer[0])) {
      int idx;
      idx = atoi(buffer);
      if (idx < nr_pte) {
        strncpy(atom->name, pte[idx], sizeof(atom->name));
      } 
      else {
        strncpy(atom->name, pte[0], sizeof(atom->name));
      }
    } 
    else {
      strncpy(atom->name, buffer, sizeof(atom->name));
    }
    strncpy(atom->type, atom->name, sizeof(atom->type));
    strcpy(atom->resname, "");
    atom->resid = 1;
    strcpy(atom->chain, "");
    strcpy(atom->segid, "");
    /* skip to the end of line */
  }

  rewind(data->file);
  return MOLFILE_SUCCESS;
}

static int read_xyz_timestep(void *mydata, int natoms, molfile_timestep_t *ts) {
  int i, j;
  char atom_name[1024], fbuffer[1024], *k;
  float x, y, z;
  
  xyzdata *data = (xyzdata *)mydata;
  
  /* skip over the first two lines */
  if (NULL == fgets(fbuffer, 1024, data->file))  return MOLFILE_ERROR;
  if (NULL == fgets(fbuffer, 1024, data->file))  return MOLFILE_ERROR;

  /* read the coordinates */
  for (i=0; i<natoms; i++) {
    k = fgets(fbuffer, 1024, data->file);

    /* Read in atom type, X, Y, Z, skipping any remaining data fields */
    j = sscanf(fbuffer, "%s %f %f %f", atom_name, &x, &y, &z);
    if (k == NULL) {
      return MOLFILE_ERROR;
    } else if (j < 4) {
      fprintf(stderr, "xyz timestep) missing type or coordinate(s) in file '%s' for atom '%d'\n",data->file_name,i+1);
      return MOLFILE_ERROR;
    } else if (j>=4) {
      if (ts != NULL) { 
        /* only save coords if we're given a timestep pointer, */
        /* otherwise assume that VMD wants us to skip past it. */
        ts->coords[3*i  ] = x;
        ts->coords[3*i+1] = y;
        ts->coords[3*i+2] = z;
      }
    } else {
      break;
    }
  }
  
  return MOLFILE_SUCCESS;
}
    
static void close_xyz_read(void *mydata) {
  xyzdata *data = (xyzdata *)mydata;
  fclose(data->file);
  free(data->file_name);
  free(data);
}


static void *open_xyz_write(const char *filename, const char *filetype, 
                           int natoms) {
  FILE *fd;
  xyzdata *data;

  fd = fopen(filename, "w");
  if (!fd) { 
    fprintf(stderr, "Error) Unable to open xyz file %s for writing\n",
            filename);
    return NULL;
  }
  
  data = (xyzdata *)malloc(sizeof(xyzdata));
  data->numatoms = natoms;
  data->file = fd;
  data->file_name = strdup(filename);
  return data;
}

static int write_xyz_structure(void *mydata, int optflags, 
                               const molfile_atom_t *atoms) {
  xyzdata *data = (xyzdata *)mydata;
  data->atomlist = (molfile_atom_t *)malloc(data->numatoms*sizeof(molfile_atom_t));
  memcpy(data->atomlist, atoms, data->numatoms*sizeof(molfile_atom_t));
  return MOLFILE_SUCCESS;
}

static int write_xyz_timestep(void *mydata, const molfile_timestep_t *ts) {
  xyzdata *data = (xyzdata *)mydata; 
  const molfile_atom_t *atom;
  const float *pos;
  int i;

  fprintf(data->file, "%d\n", data->numatoms);
  fprintf(data->file, " generated by VMD\n");
  
  atom = data->atomlist;
  pos = ts->coords;
  for (i = 0; i < data->numatoms; ++i) {
    fprintf(data->file, " %-2s %15.6g %15.6g %15.6g\n", 
            atom->type, pos[0], pos[1], pos[2]);
    ++atom; 
    pos += 3;
  }
  return MOLFILE_SUCCESS;
}


static void close_xyz_write(void *mydata) {
  xyzdata *data = (xyzdata *)mydata;
  fclose(data->file);
  free(data->atomlist);
  free(data->file_name);
  free(data);
}

/* registration stuff */
static molfile_plugin_t xyzplugin = {
  vmdplugin_ABIVERSION,
  MOLFILE_PLUGIN_TYPE,                         /* type */
  "xyz",                                       /* name */
  "Mauricio Carrillo Tripp, John E. Stone, Axel Kohlmeyer",    /* author */
  0,                                           /* major version */
  3,                                           /* minor version */
  VMDPLUGIN_THREADSAFE,                        /* is reentrant */
  "xyz",
  open_xyz_read,
  read_xyz_structure,
  0,
  read_xyz_timestep,
  close_xyz_read,
  open_xyz_write,
  write_xyz_structure,
  write_xyz_timestep,
  close_xyz_write,
  0,                            /* read_volumetric_metadata */
  0,                            /* read_volumetric_data */
  0                             /* read_rawgraphics */
};

int VMDPLUGIN_init() {
  return VMDPLUGIN_SUCCESS;
}

int VMDPLUGIN_register(void *v, vmdplugin_register_cb cb) {
  (*cb)(v, (vmdplugin_t *)&xyzplugin);
  return VMDPLUGIN_SUCCESS;
}

int VMDPLUGIN_fini() {
  return VMDPLUGIN_SUCCESS;
}

