# String Method
# By Abhi Singharoy, UIUC
# March 2014

################################################
# User-Specified Parameters

# number of copies for each image
#set nc 20
# number of iterations
set runs 80
# simulation time for equilibration (biased)
set runE 2500
# simulation time for release (unbiased)
set runR 2500
# harmonic block used
set harm "rotate"
set force 100000
# all collective variables used
set cvs { q1i q1o q2i q2o q3i q3o q4i q4o q5i q5o q7i q7o q8i q8o q9i q9o q10i q10o q11i q11o }
# dimension of collective variables
set cvsd { 4 4 4 4 4 4 4 4 4 4 4 4 4 4 4 4 4 4 4 4 }
# metric used for the distance calculations
set cvsm { 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 }

# initial reparametrization (if 1 it will reparametrize/smoothen at the beginning, otherwise will use the average)
set initial_reparam 1

###############################################
# Some handy variables

# number of replicas
set nr [numReplicas]
# number of images
set ni [expr $nr/$nc]

set nci [expr 1.0/$nc]
set nii [expr 1.0/$ni]

# smoothing parameter (default: 1/ni; set 0 for no smoothing)
set smooth $nii

set oneminussmooth [expr 1-$smooth]
set halfsmooth [expr $smooth/2.0]
set ni1 [expr $ni-1]

###############################################
# Procedures

proc replica_centers { mycolvars } {
    global harm
    global cvsd
    set mycolvars_d 0
    set cnt "centers \{ "
    foreach cvd $cvsd {
	if { $cvd == 4 } {
	    append cnt " ( " [lindex $mycolvars $mycolvars_d] " , " [lindex $mycolvars [expr $mycolvars_d+1]] " , " [lindex $mycolvars [expr $mycolvars_d+2]] " , " [lindex $mycolvars [expr $mycolvars_d+3]] " ) "
	} elseif { $cvd == 3 } {
	    append cnt " ( " [lindex $mycolvars $mycolvars_d] " , " [lindex $mycolvars [expr $mycolvars_d+1]] " , " [lindex $mycolvars [expr $mycolvars_d+2]] " ) "
	} elseif { $cvd == 2 } {
	    append cnt " ( " [lindex $mycolvars $mycolvars_d] " , " [lindex $mycolvars [expr $mycolvars_d+1]] " ) "
	} else {
	    append cnt [lindex $mycolvars $mycolvars_d]
	}
	set mycolvars_d [expr $mycolvars_d+$cvd]
    }
    append cnt " \}"
    return [list $harm "$cnt"]
}

proc colvars_length { mycolvars } {
    global cvsm
    global cvsd
    set my_d 0
    set my_length 0.0
    for { set cv 0 } { $cv < [llength cvsd] } { incr cv } {
	for { set cvd 0 } { $cvd < [lindex $cvsd $cv] } { incr cvd } {
	    set my_length [expr "$my_length+([lindex $mycolvars $my_d])**2"]
	    incr my_d 1
	}
    }
    return [expr "sqrt($my_length)"]
}

proc normalize_colvars { mycolvars } {
    global cvsd
    set mycolvars_d 0
    foreach cvd $cvsd {
	if { $cvd == 4 } {
	    set q [list [lindex $mycolvars $mycolvars_d] [lindex $mycolvars [expr $mycolvars_d+1]] [lindex $mycolvars [expr $mycolvars_d+2]] [lindex $mycolvars [expr $mycolvars_d+3]]]
	    set norm [veclength $q]
	    for { set dd $mycolvars_d } { $dd < [expr $mycolvars_d+4] } { incr dd } {
		lset mycolvars $dd [expr [lindex $mycolvars $dd]/$norm]
	    }
	}
	set mycolvars_d [expr $mycolvars_d+$cvd]
    }
    return $mycolvars
}

###############################################
# Main Body

for { set r 0 } { $r < $runs } { incr r } {

# Drift
    if { $r == 0 } {
	run 0
    }
    set values ""
    foreach cv $cvs {
        set value [colvarvalue $cv]
        append values [regexp -all -inline {\S+} $value] " "
    }
    set values [string trimright $values]
    
    replicaBarrier

    if { $replica_id > 0 } {
	replicaSend $values 0
    } else {
	set colvars(0) [split $values]
	if { $nc == 1 } {
	    print "Drifted Centers" 0 $colvars(0)
	}
	if { $nr > 1 } {
	    for { set R 1 } { $R < $nr } { incr R } {
		set I [expr $R/$nc]
		set J [expr $R%$nc]
		if { $J == 0 } {
		    set colvars($I) [split [replicaRecv $R]]
		    if { $nc == 1 } {
			print "Drifted Centers" $I $colvars($I)
		    }
		} else {
		    set colvars($I) [vecadd $colvars($I) [split [replicaRecv $R]]]
		    if { $J == [expr $nc-1] } {
			set colvars($I) [vecscale $colvars($I) $nci]
			set colvars($I) [normalize_colvars $colvars($I)]
			print "Drifted Centers" $I $colvars($I)
		    }
		}
	    }
	}
    }

# Smoothing
    if { $replica_id == 0 } {
	if { ( $smooth > 0 ) && ( $r > 0 || $initial_reparam ) } {
	    set colvarsS(0) $colvars(0)
	    print "Smoothed Centers" 0 $colvarsS(0)
	    for { set I 1 } { $I < $ni1 } { incr I } {
		set Im [expr $I-1]
		set Ip [expr $I+1]
		set colvarsS($I) [vecadd [vecscale $colvars($I) $oneminussmooth] [vecscale [vecadd $colvars($Im) $colvars($Ip)] $halfsmooth]]
		set colvarsS($I) [normalize_colvars $colvarsS($I)]
		print "Smoothed Centers" $I $colvarsS($I)
	    }
	    set colvarsS($ni1) $colvars($ni1)
	    print "Smoothed Centers" $ni1 $colvarsS($ni1)
	} else {
	    array set colvarsS [array get colvars]
	}
    }

# Reparametrize
    
    if { $replica_id == 0 } {
	if { $r > 0 || $initial_reparam } {
	    set colvarsR(0) $colvarsS(0)
	    print "Reparametrized Centers" 0 $colvarsS(0)
	    set totlen(0) 0.0
	    for { set I 1 } { $I < $ni } { incr I } {
		set J [expr $I-1]
		set dx($I) [vecsub $colvars($I) $colvars($J)]
		set ds($I) [colvars_length $dx($I)]
		set dx($I) [vecscale $dx($I) [expr 1.0/$ds($I)]]
		set totlen($I) [expr $totlen($J)+$ds($I)]
	    }
	    set dS [expr $totlen($ni1)/$ni1]
	    set I 0
	    set J 0
	    while { $I < [expr $ni1-1] } {
		incr I 1
		set sm [expr $I*$dS]
		while { $J < $ni && $sm > $totlen($J) } {
		    incr J 1
		}
		set J1 [expr $J-1]
		set colvarsR($I) [vecadd $colvars($J1) [vecscale $dx($J) [expr $sm-$totlen($J1)]]]
		set colvarsR($I) [normalize_colvars $colvarsR($I)]
		print "Reparametrized Centers" $I $colvarsR($I)
	    }
	    set colvarsR($ni1) $colvarsS($ni1)
	    print "Reparametrized Centers" $ni1 $colvarsR($ni1)
	} else {
	    array set colvarsR [array get colvarsS]
	}
    }

# Update the centers

    replicaBarrier

    if { $replica_id > 0 } {
	set colvarsR($image_id) [split [replicaRecv 0]]
    } else {
	for { set R 1 } { $R < $nr } { incr R } {
	    set I [expr $R/$nc]
	    set values ""
	    foreach value $colvarsR($I) {
    		append values $value " "
	    }
	    set values [string trimright $values]
	    replicaSend $values $R
	}
    }
    set centers($image_id) [replica_centers $colvarsR($image_id)]
    
# Equilibration

    eval colvarbias [concat changeconfig $centers($image_id)]
    eval colvarbias [concat changeconfig [list $harm "forceConstant $force"]]
    run $runE

# Release
    eval colvarbias [concat changeconfig [list $harm "forceConstant 0"]]
    run $runR

}
