/***************************************************************************

  Wat2Ions: A program for placing ions inside a bath of water molecules
            surrounding a biological macromolecule, on the basis of a
            pre-computed electrostatic potential map.
  VERSION:  1.1.
  AUTHOR:   Alexander Balaeff.
 (C) Copyright 2002 The Theoretical Biophysics Group, Beckman Institute, and
                    The Board of Trustees of the University of Illinois

 ***************************************************************************
  DESCRIPTION:

  This program is complementary to DelPhi, the renowned solver of the 
 Poisson-Boltzmann equation (http://trantor.bioc.columbia.edu/delphi).  
 The electrostatic potential map, pre-computed by DelPhi, is used to 
 replace a certain number of water molecules of the bath, surrounding 
 a solvated macromolecule, with ions.  The potential on the oxygens of
 the water molecules is estimated using a 3D cubic spline interpolation 
 of the pre-computed potential.  The waters with the largest potential 
 are replaced with chlorides, and the waters with the smallest potential, 
 with sodii. No two ions are placed closer than a certain distance to
 each other.  To avoid placing ions too close to the macromolecule, 
 the potential map should be pre-computed taking into account the 
 ion-free Stern layer around the macromolecule (see DelPhi manual for
 further details). The coordinates of the macromolecule, the water bath 
 (except the replaced waters), and the placed ions are output to a new PDB file.

 ***************************************************************************
  SUGGESTED COMPILATION COMMAND LINE (FOR A DEC-ALPHA CC-COMPILER):

  cc -lm -fast -tune host -assume whole_program -arch host \
     -o wat2ions wat2ions.c

 ***************************************************************************
  COMMAND LINE FOR THE PROGRAM EXECUTION:

  wat2ions < config_file > log_file &

 ***************************************************************************
  A SAMPLE CONFIGURATION FILE:

  PDB_NAME  lac_dna_SOLVATE.pdb
  PHI_NAME  M_0.025.phi
  NUM_NA    64   
  NUM_CL    16
  R_ION_ION 11.0
  PDB_OUT   lac_dna_water_ions.pdb

 ***************************************************************************
  EXPLANATION OF THE CONFIGURATION PARAMETERS AND FILE FORMATS:

  PDB_NAME  -  the name of the file, containing the coordinates of the
               atoms of the macromolecule and the water bath in PDBformat.  
               The waters are recognized as residues named 'TIP3', 'HOH',
               or 'WAT'.  The water oxygens are recognized as atoms with
               the character 'O' in the name.  Only the coordinates of the
               oxygens are read, the other atoms are ignored.
  
  PHI_NAME  -  the name of the file, containing the pre-computed potential map.
               That should be an ASCII file in the following format:

                   LINE 1 : INT
                            Number of grid nodes in each dimension
                   LINE 2 : FLOAT
                            Scaling factor: the inverse size of the grid step,
                            in nodes/Angstrom
                   LINE 3 : FLOAT FLOAT FLOAT
                            The coordinates of the center of the grid box,
                            in Angstroms.
                   LINES from 4 to the end : INT INT INT FLOAT
                            The 3D index of a grid node, followed by the value
                            of the electrostatic potential in that point. 
		     
		   Such file can be obtained by either modifying the DelPhi
               output (in the module qdiff4o.f), or by using a separate
		   program to translate a potential map in the standard DelPhi
		   format.

  NUM_NA   -  the number of sodium ions to be placed inside the water bath.

  NUM_CL   -  the number of chloride ions to be placed inside the water bath.

  R_ION_ION - the closest distance allowed between any two ions (in Angstroms).

  PDB_OUT   - the name of the PDB file where the results will be output.

 ***************************************************************************/


/**********************************/
/*** The required C libraries:  ***/
/***                            ***/
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>


/************************************/
/*** The compilation parameters:  ***/
/***                              ***/
#define MAXWATS 200000   /*** the largest allowed number of molecules in the water bath ***/
#define MAXIONS 1000     /*** the largest allowed number of ions ***/

/*** If any of the above parameters is insufficient for your system, ***/
/*** recompile the program with that number accordingly changed.     ***/


/**********************************/
/*** The program return codes:  ***/
/***                            ***/
#define ERR_NO_ERROR             0
#define ERR_INCOMPLETE_CONFIG   -5
#define ERR_BAD_OUTPUT          -10
#define ERR_BAD_PDBFILE         -30
#define ERR_NO_WATER            -31
#define ERR_TOO_MUCH_WATER      -32
#define ERR_BAD_PHIFILE         -20
#define ERR_NOT_ENOUGH_WATERS  -100


/*******************************************/
/*** New data type for water molecules.  ***/
/***                                     ***/
  typedef struct {
    /*** position in the original PDB file ***/
    int id;
    /*** electrostatic potential ***/
    float phi;
    /*** 3D location ***/
    float x[3];
    /*** the low-bottom-left corner of the grid cell, containing the water molecule ***/
    int g[3];
    /*** the counter of the nodes of that grid cell during the computation of the potential ***/
    int node_count;
    /*** 3D spline coefficients ***/
    float a_spl[8];
    /*** availability flag ***/
    int out;
  } water;

  water wat[MAXWATS];

/*********************************************/
/***  The list of auxiliary subroutines    ***/
/*** (each is described below in the code) ***/
/***                                       ***/
int read_config_line_int(char *, const char *, long int *);
int read_config_line_float(char *, const char *, float *);
int read_config_line_string(char *, const char *, char *);
int wat_g_order(water *, water *);
int wat_phi_order(water *, water *);
int node_compare(int *, int *);


/**********************************************************/
/******************** THE MAIN MODULE  ********************/
/**********************************************************/

int main(int argc,
	   char *argv[])
{

/************************************/
/***  THE LIST OF THE VARIABLES:  ***/
/************************************/

/**********************************/
/*** I/O files and file names:  ***/

/*** The input PDB file containing the waters ***/
  FILE *pdb_file;
  char pdb_name[100]="\0";
/*** The potential map around the macromolecule ***/
  FILE *phi_file;
  char phi_name[100]="\0";
/*** The output PDB file containing all the remaining waters and ions ***/
  FILE *pdb_outfile;
  char pdb_out_name[100]="\0";


/****************************/
/*** Water and ions data: ***/

/*** the number of water molecules, sodium, and chloride ions: ***/
  long int N_wat, N_Na=0, N_Cl=0;
/*** the ionic coordinates: ***/
  float Na_coor[MAXIONS][3], Cl_coor[MAXIONS][3];
/*** the closest distance allowed between any two ions:  ***/
  float excl_ion_rad=0.0;
/*** the IDs of the waters being replaced with the ions: ***/
  long int Na_waters[MAXIONS], Cl_waters[MAXIONS];
/*** the loop counters for the ions: ***/
  long int i_Na, i_Cl;
/*** the index for the water molecules: ***/
  long int i_wat;
/*** the atom counter and the total number of atoms in the PDB file: ***/
  long int i_atom, N_atom=0;
/*** the blocking flag for the water printout: ***/
  int wat_on_hold=0;


/*******************************/
/*** The potential map data: ***/

/*** the size of the grid box: ***/
  long int N_grid;
/*** the box half-size, scaling factor, and origin coordinates: ***/
  float N_mid, scale, ori[3];
/*** the 3D index (coordinates) of the current grid node: ***/
  int i_phi[3];
/*** the value of potential in the current grid node: ***/
  float phi;


/**************************/
/*** The loop counters: ***/

/*** the waters counter: ***/
  long int i;
/*** the "other waters" counter: ***/
  long int ii;
/*** the dimension counter: ***/
  long int j;
/*** the counter for the 8 functions of the spline basis: ***/
  long int k;
/*** the odd/even flag for the functions of the spline basis: ***/
  long int k_sp;


/*************************************/
/***  Auxiliary (dummy) variables: ***/

  int i_dum;
  float f_dum, x_dum;
  char s[1000];


/********************************/
/*** THE BODY OF THE PROGRAM. ***/
/********************************/

/*******************************/
/*** The signature printout. ***/
/*******************************/

  printf("\n");
  printf("=========================================================================\n");
  printf(" Wat2Ions: A program for placing ions inside a bath of water molecules   \n");
  printf("           surrounding a biological macromolecule, on the basis of a     \n");
  printf("           pre-computed electrostatic potential map.                     \n");
  printf(" Version:  1.0.                                                          \n");
  printf(" Author:   Alexander Balaeff.                                            \n");
  printf(" (C) Copyright 2001 The Theoretical Biophysics Group, Beckman Institute, \n");
  printf("     and The Board of Trustees of the University of Illinois.            \n");
  printf("=========================================================================\n");
  printf("\n");

/************************************************/
/*** Readout of the configuration parameters. ***/
/************************************************/
  while( gets(s) ) {
    if( read_config_line_string(s,"PDB_NAME",pdb_name) ) continue;
    if( read_config_line_string(s,"PHI_NAME",phi_name) ) continue;
    if( read_config_line_int(s,"NUM_NA",&N_Na) ) continue;
    if( read_config_line_int(s,"NUM_CL",&N_Cl) ) continue;
    if( read_config_line_float(s,"R_ION_ION",&excl_ion_rad) ) continue;
    if( read_config_line_string(s,"PDB_OUT",pdb_out_name) ) continue;
  }  

  if( strlen(pdb_name) == 0 || 
	strlen(phi_name) == 0 || 
	strlen(pdb_out_name) == 0   ) {
    printf("Incomplete configuration\n");
    return ERR_INCOMPLETE_CONFIG;
  }
  
  printf("CONFIGURATION:\n\n");
  printf("  %ld sodium and %ld chloride ions will be placed\n", N_Na, N_Cl);
  printf("  around the molecule %s,\n", pdb_name);
  printf("  not closer than %f A to each other,\n", excl_ion_rad);
  printf("  the results being output to the file %s in the PDB format.\n\n", pdb_out_name);


/****************************************/
/*** Readout of the water coordinates ***/
/****************************************/

/** Open the input pdb file. **/
  if( (pdb_file=fopen(pdb_name,"rt")) == NULL ) {
    printf("File %s can not be opened for reading!\n", pdb_name);
    return ERR_BAD_PDBFILE;
  }

/** Scan the file for water molecules and read the coordinates of their oxygens. **/
  N_wat = 0;
  while( fgets(s,1000,pdb_file) ) { 
    if( strncmp(s,"ATOM",4) &&
	  strncmp(s,"HETATM",6) )  continue;
    N_atom++;
    if( strncmp(s+17,"HOH",3) &&
	  strncmp(s+17,"WAT",3) &&
	  strncmp(s+17,"TIP3",4)  )  continue;
    if( s[12] != 'O' && s[13] != 'O' &&
	  s[14] != 'O' && s[15] != 'O'   )  continue;
    if( N_wat >= MAXWATS ) {
	printf("ERROR: The allowed number (%d) of water molecules has been exceeded.\n",MAXWATS);
	printf("Please recompile the program with an increased parameter MAXWATS.\n");
	fclose(pdb_file);
	return ERR_TOO_MUCH_WATER;
    }
    /* sscanf(s+22,"%d\n", &wat[N_wat].id; */
    wat[N_wat].id = N_wat;
    sscanf(s+30,"%8f%8f%8f", &wat[N_wat].x[0], 
	                       &wat[N_wat].x[1],
	                       &wat[N_wat].x[2]);
    N_wat++;
  }

/** Check whether any water molecules were found. **/  
  if( N_wat != 0 )
    printf("%ld waters were read from the PDB file %s.\n", N_wat, pdb_name);
  else {
    printf("ERROR: No waters were found in the PDB file %s.\n", pdb_name);
    fclose(pdb_file);
    return ERR_NO_WATER;
  }

/***************************************************************/
/*** Readout of the potential map and computing the Poisson- ***/
/*** Boltzman electrostatic potential on the water oxygens.  ***/
/***************************************************************/

/** Open the potential map file. **/
  if( (phi_file=fopen(phi_name,"rt")) == NULL ) {
    printf("File %s can not be opened for reading!\n", phi_name);
    fclose(pdb_file);
    return ERR_BAD_PHIFILE;
  }

/** Read the map header: the grid size, the scaling factor, **/
/** and the grid origin in the real space.                  **/
  fscanf(phi_file, "%ld \n", &N_grid);
  fscanf(phi_file, "%f \n", &scale);
  fscanf(phi_file, "%f %f %f \n", &ori[0], &ori[1], &ori[2]);

  excl_ion_rad *= scale;
  excl_ion_rad *= excl_ion_rad;

  printf("\nThe potential map %s :\n", phi_name);
  printf("   - has %ld x %ld x %ld nodes,\n", N_grid, N_grid, N_grid);
  printf("   - is centered at x = %f A, y = %f A, z = %f A,\n", ori[0], ori[1], ori[2]);
  printf("   - scales as %f Angstroms per grid node.\n", 1/scale );

/** Determine the grid locations of the water molecules,   **/
/** compute the spline coefficients for approximating the  **/
/** electrostatic potential on each water molecule, and    **/
/** initialize the counters of the neighboring grid nodes. **/
  N_mid = ((float)N_grid+1.)/2.;

  for( i=0; i<N_wat; i++ ) {

    for( k=0; k<8; k++ )
	wat[i].a_spl[k] = 1;

    for( j=0; j<3; j++ ) {
	/** grid-space coordinates are computed **/
	wat[i].x[j] = (wat[i].x[j]-ori[j])*scale + N_mid;
	f_dum = floor(wat[i].x[j]);
	wat[i].g[j] = (int)f_dum;

	/** 3D spline coefficients are computed **/
	f_dum = wat[i].x[j] - f_dum;
	for( k=0; k<8; k++ ) {
	  k_sp = k & (1<<(2-j));
	  if( k_sp )  wat[i].a_spl[k] *= f_dum;
	  else  wat[i].a_spl[k] *= 1.0 - f_dum;
	}
    }
    
    /** grid node counter gets initialized **/
    wat[i].node_count = 0;
  }

  printf("\nThe water molecules have been placed within the grid.\n");

/** Sort the water molecules according to their grid coordinates. **/
  printf("\nThe waters are being sorted according to their grid coordinates...\n");
  qsort(wat,N_wat,sizeof(water),wat_g_order);
  printf("      ...sorting completed.\n");

/** Read the potential map and compute the electrostatic **/
/** potential on the water molecules.                    **/
  printf("\nNow the waters' electrostatic potential is computed in 8 passes through the potential map %s.\n\n", phi_name);

  for( k=0; k<8; k++ ) {

    printf(" ... pass %ld ... \n", k+1);

    if( k != 0 )
	for( j=0; j<3; j++ ) {
	  k_sp = k & (1<<(2-j));
	  for( i=0; i<N_wat; i++ )  wat[i].g[j] += k_sp;
	}

    i = 0;
    while( (i < N_wat) && (fgets(s,1000,phi_file) != NULL) ) {
	
	if( sscanf(s, "%d %d %d %f\n", 
		     &i_phi[0], &i_phi[1], &i_phi[2], &phi) != 4 ) {
	  printf("ERROR: Wrong format encountered in the following line \n");
	  printf("of the potential map %s : \n", phi_name);
	  printf("%s", s);
	  fclose(phi_file);
	  fclose(pdb_file);
	  return ERR_BAD_PHIFILE;
	}

	while( (i_dum=node_compare(i_phi,wat[i].g)) >= 0 ) {
	    
	  if( i_dum == 0 ) {
	    wat[i].phi += wat[i].a_spl[k]*phi;
	    wat[i].node_count++;
	  }

	  i++;
	  if( i == N_wat ) break;
	}
	
    }
    
    if( k < 7 ) {
	rewind(phi_file);
	fgets(s,1000,phi_file);
	fgets(s,1000,phi_file);
	fgets(s,1000,phi_file);
    }
  }

  fclose(phi_file);

  printf("\nThe electrostatic potential on the water molecules has been computed.\n");

/** Re-sort the waters according to the computed potential **/
  printf("\nThe waters are being sorted according to their electrostatic potential...\n");
  qsort(wat,N_wat,sizeof(water),wat_phi_order);
  printf("      ...sorting completed.\n");


/*********************************************************************/
/*** Replacing the waters with the lowest potential by sodium ions ***/
/*** and those with the highest potential by chloride ions.        ***/
/*********************************************************************/
  
  /** Initialize the exclusion index                     **/
  /** (all the waters with completely computed potential **/
  /**  are legal targets for being replaced by an ion).  **/
  for( i=0; i<N_wat; i++ ) {
    if( wat[i].node_count == 8 )
	wat[i].out = 0;
    else
	wat[i].out = 1;
  }

  /** Warn the user if some of the waters have been initially excluded. **/
  for( i=0; i<N_wat; i++ ) {
    if( wat[i].node_count != 8 ) {
	printf("\nWARNING: The potential map %s is insufficient to compute \n", phi_name); 
	printf("the potential on every water molecule. Perhaps, the size of the water system\n");
	printf("exceeds that of the potential grid, or some of the waters are located too close \n");
	printf("to the solvated biomolecule.\n");
	break;
    }
  }

  /** Place the sodium ions one-by-one. **/
  printf("\nAccommodating %d sodium ions...\n", N_Na);
	
  i=0;
  for( i_Na=0; i_Na<N_Na; i_Na++ ) {
    /** search for the water molecule with the lowest electrostatic potential, **/
    /** not yet excluded due to distance restraints or insufficient grid data. **/
    while( wat[i].out ) {
	i++;
	if( i == N_wat ) break;
    }
    if( i == N_wat ) {
	printf("ERROR: not enough water to accommodate more than %d sodium ions.\n", i_Na);
	printf("Reduce the number of ions, or increase the amount of water,\n");
	printf("or relax the ion-ion distance restraint (currently, %f A).\n", 
		 sqrt(excl_ion_rad)/scale );
	fclose(pdb_file);
	return ERR_NOT_ENOUGH_WATERS;
    }

    wat[i].out = 1;
    printf(" ... sodium %ld has replaced the %d-th water molecule ... \n", i_Na+1, wat[i].id+1); 

    /** assign the coordinates of the found water to the next sodium ion **/
    for( j=0; j<3; j++ )  Na_coor[i_Na][j] = wat[i].x[j];
    Na_waters[i_Na] = wat[i].id;

    /** exclude the water molecules located closer than the permitted **/
    /** distance to the placed ion from the list of available targets **/
    for( ii=i+1; ii<N_wat; ii++ ) {
	if( wat[ii].out ) continue;
	for( f_dum=0, j=0; j<3; j++ ) {
	  x_dum = wat[ii].x[j] - Na_coor[i_Na][j];
	  f_dum += x_dum*x_dum;
	}
	if( f_dum <= excl_ion_rad )  wat[ii].out=1;
    }
  }

  printf("Sodium ions have been succesfully placed.\n");


  /** Place the chloride ions one-by-one. **/
  printf("\nAccommodating %d chloride ions...\n", N_Cl);
	
  i=N_wat-1;
  for( i_Cl=0; i_Cl<N_Cl; i_Cl++ ) {
    /** search for the water molecule with the highest electrostatic potential, **/
    /** not yet excluded due to distance restraints or insufficient grid data.  **/
    while( wat[i].out ) {
	i--;
	if( i == -1 ) break;
    }
    if( i == -1 ) {
	printf("ERROR: not enough water to accommodate more than %d chloride ions.\n", i_Cl);
	printf("Reduce the number of ions, or increase the amount of water,\n");
	printf("or relax the ion-ion distance restraint (currently, %f A).\n", excl_ion_rad);
	fclose(pdb_file);
	return ERR_NOT_ENOUGH_WATERS;
    }

    wat[i].out = 1;
    printf(" ... chloride %ld has replaced the %d-th water molecule ... \n", i_Cl+1, wat[i].id+1); 

    /** assign the coordinates of the found water to the next sodium ion **/
    for( j=0; j<3; j++ )  Cl_coor[i_Cl][j] = wat[i].x[j];
    Cl_waters[i_Cl] = wat[i].id;

    /** exclude the water molecules located closer than the permitted **/
    /** distance to the placed ion from the list of available targets **/
    for( ii=i-1; ii>=0; ii-- ) {
	if( wat[ii].out ) continue;
	for( f_dum=0, j=0; j<3; j++ ) {
	  x_dum = wat[ii].x[j] - Cl_coor[i_Cl][j];
	  f_dum += x_dum*x_dum;
	}
	if( f_dum <= excl_ion_rad )  wat[ii].out=1;
    }
  }

  printf("Chloride ions have been succesfully placed.\n");

  /** Transform the ion coordinates from the grid space to the real space. **/
  for( j=0; j<3; j++ ) {

	for( i_Na=0; i_Na<N_Na; i_Na++ )
	  Na_coor[i_Na][j] = (Na_coor[i_Na][j] - N_mid)/scale + ori[j];

	for( i_Cl=0; i_Cl<N_Cl; i_Cl++ )
	  Cl_coor[i_Cl][j] = (Cl_coor[i_Cl][j] - N_mid)/scale + ori[j];

  }


/*********************************************************************/
/*** Printout of the PDB file excluding the waters to be replaced. ***/
/*********************************************************************/

  rewind(pdb_file);

/** Open the output pdb file. **/
  if( (pdb_outfile=fopen(pdb_out_name,"wt")) == NULL ) {
    printf("File %s can not be opened for writing!\n", pdb_out_name);
    fclose(pdb_file);
    return ERR_BAD_OUTPUT;
  }

/** Copy the original file to the output, except for  **/
/** those waters that should be replaced by the ions. **/
  i_wat = -1;
  wat_on_hold = 0;

  while( fgets(s,1000,pdb_file) ) { 

    /** the "END" record is not printed **/
    /** (the ions' coordinates have to be printed yet) **/
    if( !strncmp(s,"END",3) ) continue; 

    /** check whether we have a PDB ATOM record with a water molecule **/
    /** a non-water PDB record gets printed without further action    **/
    if( strncmp(s,"ATOM",4) && strncmp(s,"HETATM",6) ) {
	fputs(s,pdb_outfile);
	continue;	
    }

    if( strncmp(s+17,"HOH",3) && strncmp(s+17,"WAT",3) && strncmp(s+17,"TIP3",4) ) {
	fputs(s,pdb_outfile);
	continue;
    }

    /** a water is found **/
	
    /** if the water atom is an oxygen, then: **/
    if( s[12] == 'O' || s[13] == 'O' || s[14] == 'O' || s[15] == 'O' )
	{
	  /** the printout blocking flag is dropped; **/
	  wat_on_hold = 0;
	  /** the water index is computed; **/
	  i_wat++;
	  /** if this water is to be replaced by an ion, **/
	  /** the printout blocking flag gets raised     **/
	  for( i_Na=0; i_Na<N_Na; i_Na++ ) {
	    if( i_wat == Na_waters[i_Na] ) {
		wat_on_hold = 1;
		break;
	    } }
	  if( !wat_on_hold ) {
	    for( i_Cl=0; i_Cl<N_Cl; i_Cl++ ) {
		if( i_wat == Cl_waters[i_Cl] ) {
		  wat_on_hold = 1;
		  break;
		} } }
	}

    /** the water atom coordinates are printed out only if the **/
    /** printout blocking flag is down for this water molecule **/
    if( !wat_on_hold ) 
	fputs(s,pdb_outfile);
  }
    

/************************************/
/*** Printout of the placed ions. ***/
/************************************/

  i_atom = N_atom - 3*(N_Na+N_Cl);

  for( i_Na=0; i_Na<N_Na; i_Na++, i_atom++ ) {
    if( i_atom < 100000 )
	fprintf(pdb_outfile,
		  "ATOM  %5d  SOD SOD %5d    %8.3f%8.3f%8.3f  1.00   .00      SOD\n",
		  i_atom, i_Na+1, Na_coor[i_Na][0], Na_coor[i_Na][1], Na_coor[i_Na][2]);
    else 
	fprintf(pdb_outfile,
		  "ATOM  *****  SOD SOD %5d    %8.3f%8.3f%8.3f  1.00   .00      SOD\n",
		  i_Na+1, Na_coor[i_Na][0], Na_coor[i_Na][1], Na_coor[i_Na][2]);    
  }

  for( i_Cl=0; i_Cl<N_Cl; i_Cl++, i_atom++ ) {
    if( i_atom < 100000 )
	fprintf(pdb_outfile,
		  "ATOM  %5d  CL  CL  %5d    %8.3f%8.3f%8.3f  1.00   .00      CL \n",
		  i_atom, i_Cl+1, Cl_coor[i_Cl][0], Cl_coor[i_Cl][1], Cl_coor[i_Cl][2]);
    else
	fprintf(pdb_outfile,
		  "ATOM  *****  CL  CL  %5d    %8.3f%8.3f%8.3f  1.00   .00      CL \n",
		  i_Cl+1, Cl_coor[i_Cl][0], Cl_coor[i_Cl][1], Cl_coor[i_Cl][2]);
  }

  fprintf(pdb_outfile, "END\n");
  fclose(pdb_outfile);


/******************************/
/*** Finishing the program. ***/
/******************************/

  printf("\n");
  printf("=========================================\n");
  printf(" The ions have been successfully placed. \n");
  printf(" Thank you for using WAT2IONS!           \n");
  printf("=========================================\n");

  return ERR_NO_ERROR;

}


/************************/
/*** THE SUBROUTINES. ***/
/************************/

/*****************************************************************/
/*** The following subroutines read configuration parameters   ***/
/*** from the given configuration line, if the required format ***/
/***       KEYWORD  PARAMETER                                  ***/
/*** is matched.                                               ***/
/*****************************************************************/

int read_config_line_int(char *line, const char *keyword, long int *num)
{
  char *sub, read_format[100];

  if( (sub=strstr(line,keyword)) == NULL )  return 0;
  
  sprintf(read_format,"%s %%ld",keyword);

  if( sscanf(sub,read_format,num) != 1 ) {
    printf("Wrong configuration line:\n%s\n",line);
    return 0;
  }
  else  return 1;

}

int read_config_line_float(char *line, const char *keyword, float *val)
{
  char *sub, read_format[100];

  if( (sub=strstr(line,keyword)) == NULL )  return 0;
  
  sprintf(read_format,"%s %%f",keyword);

  if( sscanf(sub,read_format,val) != 1 ) {
    printf("Wrong configuration line:\n%s\n",line);
    return 0;
  }
  else  return 1;

}

int read_config_line_string(char *line, const char *keyword, char *word)
{
  char *sub, read_format[100];

  if( (sub=strstr(line,keyword)) == NULL )  return 0;
  
  sprintf(read_format,"%s %%s",keyword);

  if( sscanf(sub,read_format,word) != 1 ) {
    printf("Wrong configuration line:\n%s\n",line);
    return 0;
  }
  else  return 1;

}

/***************************************************************/
/*** These subroutines are used to sort the water molecules: ***/
/***************************************************************/

/*** a) according to their coordinates; ***/

int wat_g_order(water *w1, water *w2)
{
  int i;

  for( i=0; i<3; i++) {
    if( w1->g[i] < w2->g[i] ) return -1;
    if( w1->g[i] > w2->g[i] ) return  1;
  }

  return 0;
}

/*** b) according to their electrostatic potential. ***/

int wat_phi_order(water *w1, water *w2)
{
  if( w1->phi < w2->phi ) return -1;
  if( w1->phi > w2->phi ) return  1;
  return 0;
}

/***********************************************************/
/*** This subroutine compares the sets of three numbers. ***/
/***********************************************************/

int node_compare(int n1[3], int n2[3])
{
  int i;

/*  printf(" .... Node 1: %ld, %ld, %ld \n",
	   n1[0],n1[1],n1[2]);
  printf(" .... Node 2: %ld, %ld, %ld \n", 
	   n2[0],n2[1],n2[2]); */


  for( i=0; i<3; i++) {
    if( n1[i] < n2[i] ) return -1;
    if( n1[i] > n2[i] ) return  1;
  }

  return 0;
}

/**********************************************************************************/
