$VERSION = "1.00";
package TCB::System::Machine;
our $VERSION = "1.00";

# -*- Perl -*- 		Thu May 13 13:19:53 CDT 2004
###############################################################################
# Written by Tim Skirvin <tskirvin@ks.uiuc.edu>
# Copyright 2000-2004, Tim Skirvin and UIUC Board of Trustees.
# Redistribution terms are below.
###############################################################################

=head1 NAME 

TCB::System::Machine - TCB Group Machine Database

=head1 SYNOPSIS

  use TCB::System::Machine;

See B<TCB::System::Obsolete> or B<TCB::System> for more information on how
to use this module.

=head1 DESCRIPTION

The 'Machine' table lists all existing TCB systems, their software
versions, etc.  

This table contains the following fields:

 Internal Information
  ID		Unique numeric ID - auto-generated (INT)

 Basic Information (TINYTEXT fields, unless noted)
  Name		Name of the system (generally a city or planet)
  Hardware	The hardware model of the system (
		  (SEE BELOW)
  OS		The version of the operating system on this computer
		  (SEE BELOW)
  3D		Is this system 3D capable?  (INT)
  RoomNumber	Which room number is the system in? (INT)
  Building	Which building is the system in? 
  Description	System description (TEXT)
  IP		IP Address of the system (TINYTEXT)

  PAS		Property Accounting Number of the system
  PowerProtect	Is the system on a UPS?  Which one?
  InService	Is the system in service? (INT)
  PrimaryUser	The username of the regular user.
  Alias		Other names for this system.
  
Key fields: 	Name

List items: 	Name, Room (number + building), OS, IP

Required: 	Same as Key fields.

Default order: 	IP

Admin Fields:	None

The 'SysLoads', 'Port', 'Installed', and 'Maintenance' tables depend on
this table directly.

Depends on the 'OS', 'Hardware', and 'Building' tables.

=head2 Variables

The following variables are set in the module, and are used as defaults
for creation and searching.

=over 4

=item @TCB::System::Equipment::POWER

Type of power conditioning, used with 'PowerProtect'. Default
possibilities: none, UPS, Isobar.

=back

=cut

###############################################################################
### Initialization ############################################################
###############################################################################
use vars qw( @ISA $FIELDS $KEYS $NAME $LIST $BIBFILE $ABSTRACTS $REQUIRED 
	     $ADMIN $ORDER $WEBROOT @POWER );
use strict;
use warnings;
use CGI;
use TCB::System;

push @ISA, qw( TCB::System );

###############################################################################
### Database Variables ########################################################
###############################################################################
$NAME = "Machine";
$FIELDS = {
  'ID'            =>  'INT NOT NULL PRIMARY KEY AUTO_INCREMENT',
  'Name'          =>  'TINYTEXT NOT NULL',  'OS' =>  'TINYTEXT',
  'Hardware'      =>  'TINYTEXT',      	    '3D' => 'INT',
  'RoomNumber'    =>  'INT',          'Building' => 'TINYTEXT',
  'Description'   =>  'TEXT',               'IP' => 'TINYTEXT',
  'PAS'           =>  'TINYTEXT',    'InService' => 'INT',
  'PowerProtect'  =>  'TINYTEXT',  'PrimaryUser' => 'TINYTEXT',
  'Alias'         =>  'TINYTEXT'
         };
$KEYS  = [ 'Name' ];
$LIST  = [ 
  'Name', 
  { 'Room' => '$$RoomNumber$$ $$Building$$' },
  { 'Hardware' => '$$Hardware$$' },
  'OS', 
  'IP'
         ];
$REQUIRED = $KEYS;
$ADMIN = [ ];
$ORDER = [ 'IP' ];

@POWER = qw( none UPS isobar );

###############################################################################
##### Functions ###############################################################
###############################################################################

=head2 Internal Functions 

=over 4

=item html ( ENTRY, TYPE, OPTIONS )

Prints the HTML version of the table.  Heavily customized for the TCB
environment, but then again so is the whole program.                            

=cut

sub html {
  my ($self, $entry, $type, $options, @rest) = @_;
  my $cgi = new CGI;
  if (lc $type eq 'create') {
    $entry ||= {
          'Building'  => "Beckman",
          'PowerProtect' => "isobar",
          'Hardware'  => "Sun Ultra 10 w/SunPCi",
          'OS'        => "Solaris-2.8",
          'InService' => 1,
          'IP'        => "130.126.120.",
          '3D'        => 0
             };
  } elsif (lc $type eq 'search') {
    $entry ||= { 
          'InService' => "",
          '3D'        => "",
               };
  } else { $entry ||= { } }

  my $extra = lc $type eq 'search' ? { '(any)' => "" } : { };

  my @os       = sort $self->select_fieldlist('OS', 'Name');
  my @hardware = sort $self->select_fieldlist('Hardware', 'Name');
  my @building = sort $self->select_fieldlist('Building', 'Name');
  my @power    = sort @POWER; 
  
  my %inservice = ( 1 => "Yes", 0 => "No", 3 => "Not TB" );
  my %threedee  = ( 1 => "Yes", 0 => "No" );

  my $searchservice = $$entry{'InService'} 
        ? "<input type=radio name=InService value='%' checked> Search" : "";

  if (lc $type eq 'search') {
    unshift @os, '';
    unshift @hardware, ''; 
    unshift @building, ''; 
    unshift @power, ''; 
    $inservice{''} = "*";
    $threedee{''} = "*";
  }


  my @return = <<HTML;

<div class="basetable">
 <div class="row2">
  <span class="label">Name</span>
  <span class="formw"> 
     @{[ $cgi->textfield('Name', $$entry{Name} || "", 20, 255) ]}
  </span>
  <span class="label">Hardware<br />OS</span>
  <span class="formw"> 
     @{[ $cgi->popup_menu('Hardware', \@hardware, $$entry{Hardware}) ]}
      <br />
     @{[ $cgi->popup_menu('OS', \@os, $$entry{OS}) ]}
  </span>
 </div>

 <div class="row2">
  <span class="label">Location</span>
  <span class="formw"> 
     @{[ $cgi->textfield('RoomNumber', $$entry{RoomNumber} || "", 8, 255) ]}
     @{[ $cgi->popup_menu('Building', \@building, $$entry{Building} || "") ]}
  </span>
  <span class="label">IP</span>
  <span class="formw"> 
     @{[ $cgi->textfield('IP', $$entry{IP} || "", 16, 255) ]}
  </span>
 </div>

 <div class="row1">
  <span class="label">Description</span>
  <span class="formw">
     @{[ $cgi->textarea(-name=>'Description', -rows=>5, -cols=>60, 
			-maxlength=>65535, -wrap=>'physical', 
			-default=>$$entry{Description} || "") ]}
  </span>
 </div>

 <div class="row1">
  <span class="label">Aliases</span>
  <span class="formw">
     @{[ $cgi->textfield('Alias', $$entry{Alias} || "", 80, 255) ]} 
  </span>
 </div>

 <div class="row2">
  <span class="label">Primary User</span>
  <span class="formw">
     @{[ $cgi->textfield('PrimaryUser', $$entry{PrimaryUser} || "", 15, 255) ]}
  </span>
  <span class="label">PAS</span>
  <span class="formw">
     @{[ $cgi->textfield('PAS', $$entry{PAS} || "", 8, 255) ]}
  </span>
  <span class="label">Power Protection</span>
  <span class="formw">
     @{[ $cgi->popup_menu('PowerProtect', \@power, $$entry{PowerProtect} || "") ]}
  </span>
 </div>
 
 <div class="row2">
  <span class="label">In Service?</span>
  <span class="formw">
     @{[ $cgi->radio_group('InService', [ keys %inservice ],
                        $$entry{InService} || "", 0, \%inservice ) ]} 
  </span>
  <span class="label">3D?</span>
  <span class="formw">
     @{[ $cgi->radio_group('3D', [ keys %threedee ],
                        $$entry{'3D'} || "", 0, \%threedee ) ]}
  </span>
 </div>
 <div class="submitbar"> @{[ $cgi->submit(-name=>"Submit") ]} </div>
</div>
HTML
  wantarray ? @return : join("\n", @return);
}

=back

There is no text() field at present.

=cut

###############################################################################
##### main() ##################################################################
###############################################################################

TCB::System->table_add($NAME, $FIELDS, $KEYS, $LIST, $ORDER, $ADMIN, 
			$REQUIRED, \&html, \&text);

=head1 NOTES

This table was originally put together to try to make an automatic floor
plan.  We never actually did this, which means that we never actually used
the table very much either.  Most of its functionality has later been
replaced by LDAP; but it still may be handy to someone.

The tables themselves were designed by the author, Tim Skirvin.

=head1 TODO

Making a nice user interface, rather than just an administrative
interface, would be helpful.

=head1 REQUIREMENTS

Perl 5.6.1 or greater, DBIx::Frame, TCB::System

=head1 SEE ALSO

B<TCB::Equipment>, B<TCB::System>, B<DBIx::Frame>

=head1 AUTHOR

Tim Skirvin <tskirvin@ks.uiuc.edu>

=head1 HOMEPAGE

B<http://www.ks.uiuc.edu/Development/MDTools/tcb-system-obsolete>

=head1 LICENSE

This code is distributed under the University of Illinois Open Source
License.  See
C<http://www.ks.uiuc.edu/Development/MDTools/tcb-system-obsolete/license.html>
for details.

=head1 COPYRIGHT

Copyright 2000-2004 by the University of Illinois Board of Trustees and
Tim Skirvin <tskirvin@ks.uiuc.edu>.

=cut

###############################################################################
### Version History ###########################################################
###############################################################################
# v0.2  	Tue Apr  2 15:35:25 CST 2002
### First version fully using DBI::Frame and commented.
# v1.00         Thu May 13 13:19:53 CDT 2004
### Ready to be a part of DBIx::Frame.
