$VERSION = "1.00";
package TCB::System::Installed;
our $VERSION = "1.00";

# -*- Perl -*- 		Thu May 13 13:19:53 CDT 2004
###############################################################################
# Written by Tim Skirvin <tskirvin@ks.uiuc.edu>
# Copyright 2000-2004, Tim Skirvin and UIUC Board of Trustees.
# Redistribution terms are below.
###############################################################################

=head1 NAME 

TCB::System::Installed - TCB Group Installed Software 

=head1 SYNOPSIS

  use TCB::System::Installed;

See B<TCB::System::Obsolete> or B<TCB::System> for more information on how
to use this module.

=head1 DESCRIPTION

The 'Installed' table lists installed software by system.  This was never
kept track of systematically, so is essentially worthless.

This table contains the following fields:

 Internal Information
  ID		Unique numeric ID - auto-generated (INT)

 Basic Information (TINYTEXT fields, unless noted)
  Program	Name of the installed program (taken from 'Package')
  Installed	Is it installed or not?  (INT)
  Notes		Any notes about the installation 
  Machine	Machine that this is installed on (taken from 'UsrLocal'
		  and 'Machine')
  Shared	Is this a shared install?  (INT)
  Encapped	Is this an encapped install?  (INT)
  InstalledBy	The user that installed the package (taken from 'Admins')
  
Key fields: 	Program, Machine

List items: 	Program, Installed On (Machine)

Required: 	Same as Key fields.

Default order: 	Program, Machine

Admin Fields:	None

No other tables depend on this table directly.

Depends on the 'Admins', 'UsrLocal', and 'Machine' tables.

=cut

###############################################################################
### Initialization ############################################################
###############################################################################
use vars qw( @ISA $FIELDS $KEYS $NAME $LIST $BIBFILE $ABSTRACTS $REQUIRED 
	     $ADMIN $ORDER $WEBROOT );
use strict;
use warnings;
use CGI;
use TCB::System;

push @ISA, qw( TCB::System );

###############################################################################
### Database Variables ########################################################
###############################################################################
$NAME = "Installed";
$FIELDS = {
  'ID'            =>  'INT NOT NULL PRIMARY KEY AUTO_INCREMENT',
  'Program'       =>  'TINYTEXT',   'Installed'     =>  'INT',
  'Notes'         =>  'TEXT',       'Machine'       => 'TINYTEXT',    
  'Shared'        =>  'INT',        'InstalledBy'   => 'TINYTEXT',
  'Encapped'      =>  'INT',     
         };
$KEYS  =  [ 'Program', 'Machine' ];
$LIST  =  [ 'Program', { 'Installed On' => '$$Machine$$' } ];                   
$REQUIRED = $KEYS;
$ADMIN = [ ];
$ORDER = [ 'Program', 'Machine' ];

###############################################################################
##### Functions ###############################################################
###############################################################################

=head2 Internal Functions 

=over 4

=item html ( ENTRY, TYPE, OPTIONS )

Prints the HTML version of the table.  Heavily customized for the TCB
environment, but then again so is the whole program.                            

=cut

sub html {
  my ($self, $entry, $type, $options, @rest) = @_;
  my $cgi = new CGI;	$entry ||= {};
  my $date = sprintf("%04d-%02d-%02d", (localtime)[5] + 1900,
                           (localtime)[4] + 1, (localtime)[3] );
  if (lc $type eq 'create') {
    $$entry{Installed} ||= 1;  $$entry{Encapped} ||= 0;  
    $$entry{Shared}    ||= 0;  
    $$entry{InstalledBy} ||= $ENV{'REMOTE_USER'} || "OTHER";
  } elsif (lc $type eq 'search') {
    $$entry{Installed} ||= '';  $$entry{Encapped}  ||= '';
  }

  my $extra = lc $type eq 'search' ? { '*' => "" } : { };

  my %programs;
  foreach my $entry ($self->select('Package', {})) {
    my $ID      = $$entry{ID};       my $OS      = $$entry{OS};
    my $program = $$entry{Program};  my $version = $$entry{Version};
    $programs{"$program - $version ($OS)"} = "$program - $version ($OS)";
  }

  my @machines = sort $self->select_fieldlist('UsrLocal', 'Name'),
                      $self->select_fieldlist('Machine', 'Name');
  unshift @machines, '' if (lc $type eq 'search');
  my @installers = sort $self->select_fieldlist('Admins', 'ShortName'), 
                                        "SYSTEM", "OTHER";

  my %installed = ( 1 => "Yes", 0 => "No" );
  my %shared    = ( 1 => "Yes", 0 => "No" );
  my %encapped  = ( 1 => "Yes", 0 => "No" );

  if (lc $type eq 'search') {
    unshift (@installers, '');
    $programs{$$entry{Program}} = $$entry{Program};
    $installed{''} = "*";
    $shared{''}    = "*";
    $encapped{''}  = "*";
  } elsif (lc $type eq 'create') {
    $programs{''}  = "";
  }

  my @return = <<HTML;
<div class="basetable">
 <div class="row1">
  <span class="label">Program</span>
  <span class="formw"> 
     @{[ $cgi->popup_menu('Program', [ sort keys %programs ], 
			$$entry{Program} || "", \%programs ) ]}
  </span>
 </div>
 <div class="row2">
  <span class="label">Installed By</span>
  <span class="formw"> 
    @{[ $cgi->popup_menu('InstalledBy', \@installers, $$entry{InstalledBy} || "") ]}
  </span>
  <span class="label">Installed On</span>
  <span class="formw"> 
     @{[ $cgi->popup_menu('Machine', \@machines, $$entry{Machine} || "") ]}
  </span>
 </div>
 <div class="row3">
  <span class="label">Installed?</span>
  <span class="formw">
   @{[ $cgi->radio_group('Installed', [ keys %installed ],
                        $$entry{Installed} || "", 0, \%installed) ]}
  </span>
  <span class="label">Shared?</span>
  <span class="formw">
   @{[ $cgi->radio_group('Shared', [ keys %shared ],
                        $$entry{Shared} || "", 0, \%shared) ]} 
  </span>
  <span class="label">Encapped?</span>
  <span class="formw">
   @{[ $cgi->radio_group('Encapped', [ keys %encapped ],
                        $$entry{Encapped} || "", 0, \%encapped) ]} 
  </span>
 </div>
 <div class="row1">
  <span class="label">Notes</span>
  <span class="formw">
     @{[ $cgi->textarea(-name=>'Notes', -default=>$$entry{Notes} || "",
                        -rows=>7, -cols=>60, -maxlength=>65535,
                        -wrap=>'physical') ]}
  </span>
 </div>
 <div class="submitbar"> @{[ $cgi->submit(-name=>"Submit") ]} </div>
</div>
HTML
  wantarray ? @return : join("\n", @return);
}

=back

There is no text() field at present.

=cut

###############################################################################
##### main() ##################################################################
###############################################################################

TCB::System->table_add($NAME, $FIELDS, $KEYS, $LIST, $ORDER, $ADMIN, 
			$REQUIRED, \&html, \&text);

=head1 NOTES

This table was never used because it was just plain too complicated to do
anything with.  It hasn't been much of a loss; it's easy enough
information to keep track of in bulk.

The tables themselves were designed by the author, Tim Skirvin.

=head1 TODO

Making a nice user interface, rather than just an administrative
interface, would be helpful.

=head1 REQUIREMENTS

Perl 5.6.1 or greater, DBIx::Frame, TCB::System

=head1 SEE ALSO

B<TCB::System::Obsolete>, B<TCB::System>, B<DBIx::Frame>

=head1 AUTHOR

Tim Skirvin <tskirvin@ks.uiuc.edu>

=head1 HOMEPAGE

B<http://www.ks.uiuc.edu/Development/MDTools/tcb-system-obsolete>

=head1 LICENSE

This code is distributed under the University of Illinois Open Source
License.  See
C<http://www.ks.uiuc.edu/Development/MDTools/tcb-system-obsolete/license.html>
for details.

=head1 COPYRIGHT

Copyright 2000-2004 by the University of Illinois Board of Trustees and
Tim Skirvin <tskirvin@ks.uiuc.edu>.

=cut

###############################################################################
### Version History ###########################################################
###############################################################################
# v0.2  	Tue Apr  2 15:35:25 CST 2002
### First version fully using DBI::Frame and commented.
# v1.00         Thu May 13 13:19:53 CDT 2004
### Ready to be a part of DBIx::Frame.
