$VERSION = "1.00";
package TCB::System::Maintenance;
our $VERSION = "1.00";

# -*- Perl -*- 		Thu May 13 13:19:53 CDT 2004
###############################################################################
# Written by Tim Skirvin <tskirvin@ks.uiuc.edu>
# Copyright 2000-2004, Tim Skirvin and UIUC Board of Trustees.
# Redistribution terms are below.
###############################################################################

=head1 NAME 

TCB::System::Maintenance - TCB Group Machine Maintenance Database

=head1 SYNOPSIS

  use TCB::System::Maintenance;

See B<TCB::System::Obsolete> or B<TCB::System> for more information on how
to use this module.

=head1 DESCRIPTION

The 'Maintenance' table lists all TCB hardware maintenance contracts.
This is used remarkably rarely. 

This table contains the following fields:

 Internal Information
  ID		Unique numeric ID - auto-generated (INT)

 Basic Information (TINYTEXT fields, unless noted)
  Machine	Name of the system being maintained
  Cost		Cost of the annual maintenance (dollars)
  PO		Purchase order of the maintenance
  Vendor	Vendor where the maintenance is coming from
  MaintenanceStart  Start of the maintenance (DATE)
  MaintenanceStop   Stop of the maintenance (DATE)
  Notes		Any notes about the maintenance (TEXT)
  
Key fields: 	Machine, MaintenanceStop, MaintenanceStart

List items: 	Machine, Maintenance (MaintenanceStart - MaintenanceStop)

Required: 	Same as Key fields.

Default order: 	MaintenanceStop (reverse), Machine

Admin Fields:	None

No other tables depend on this table directly.

Depends on the 'Machine' table.

=cut

###############################################################################
### Initialization ############################################################
###############################################################################
use vars qw( @ISA $FIELDS $KEYS $NAME $LIST $BIBFILE $ABSTRACTS $REQUIRED 
	     $ADMIN $ORDER $WEBROOT );
use strict;
use warnings;
use CGI;
use TCB::System;

push @ISA, qw( TCB::System );

###############################################################################
### Database Variables ########################################################
###############################################################################
$NAME = "Maintenance";
$FIELDS = {
  'ID'            =>  'INT NOT NULL PRIMARY KEY AUTO_INCREMENT',
  'Machine'       =>  'TINYTEXT',   'Cost '           => 'TINYTEXT',    
  'PO'            =>  'TINYTEXT',   'Vendor'          => 'TINYTEXT',   
  'MaintenanceStart' =>  'DATE',    'MaintenanceStop' => 'DATE',     
  'Notes'         =>  'TEXT',
         };
$KEYS  =  [ 'Machine', 'MaintenanceStart', 'MaintenanceStop' ];
$LIST  =  [ 
  { 'Machine' => '$$Machine$$' },
  { 'Maintenance' => '$$MaintenanceStart$$ - $$MaintenanceStop$$' } 
          ];
$REQUIRED = $KEYS;
$ADMIN = [ ];
$ORDER = [ '-MaintenanceStop', 'Machine' ];

###############################################################################
##### Functions ###############################################################
###############################################################################

=head2 Internal Functions 

=over 4

=item html ( ENTRY, TYPE, OPTIONS )

Prints the HTML version of the table.  Heavily customized for the TCB
environment, but then again so is the whole program.                            

=cut

sub html {
  my ($self, $entry, $type, $options, @rest) = @_;
  my $cgi = new CGI;	$entry ||= {};
  my $date = sprintf("%04d-%02d-%02d", (localtime)[5] + 1900,
                           (localtime)[4] + 1, (localtime)[3] );
  if (lc $type eq 'create') {
    $$entry{Machine}          ||= "";
    $$entry{MaintenanceStart} ||= $date;
  }

  my $extra = lc $type eq 'search' ? { '*' => "" } : { };

  my @machines = sort $self->select_fieldlist('Machine', 'Name');
  unshift @machines, '' if (lc $type eq 'search');

  my @return = <<HTML;

<div class="basetable">
 <div class="row1">
  <span class="label">Machine</span>
  <span class="formw"> 
     @{[ $cgi->popup_menu('Machine', \@machines, $$entry{Machine} || "") ]}
  </span>
 </div>
 <div class="row2">
  <span class="label">Maintenance</span>
  <span class="formw"> 
   @{[ $cgi->textfield('MaintenanceStart', $$entry{MaintenanceStart} || "", 
								     11, 11) ]}
      -
   @{[ $cgi->textfield('MaintenanceStop', $$entry{MaintenanceStop} || "", 
								     11, 11) ]}
  </span>
  <span class="label">Cost</span>
  <span class="formw"> 
     @{[ $cgi->textfield('Cost', $$entry{Cost} || "", 15, 255) ]}
  </span>
 </div>
 <div class="row2">
  <span class="label">Vendor</span>
  <span class="formw"> 
     @{[ $cgi->textfield('Vendor', $$entry{Vendor} || "", 25, 255) ]}
  </span>
  <span class="label">PO</span>
  <span class="formw"> 
     @{[ $cgi->textfield('PO', $$entry{PO} || "", 15, 255) ]}
  </span>
 </div>
 <div class="row1">
  <span class="label">Notes</span>
  <span class="formw">
     @{[ $cgi->textarea(-name=>'Notes', -default=>$$entry{Notes} || "",
                        -rows=>7, -cols=>60, -maxlength=>65535,
                        -wrap=>'physical') ]}
  </span>
 </div>
 <div class="submitbar"> @{[ $cgi->submit(-name=>"Submit") ]} </div>
</div>
HTML
  wantarray ? @return : join("\n", @return);
}

=back

There is no text() field at present.

=cut

###############################################################################
##### main() ##################################################################
###############################################################################

TCB::System->table_add($NAME, $FIELDS, $KEYS, $LIST, $ORDER, $ADMIN, 
			$REQUIRED, \&html, \&text);

=head1 NOTES

This table was never used because it was just plain too complicated to do
anything with.  It hasn't been much of a loss; it's easy enough
information to keep track of in bulk.

The tables themselves were designed by the author, Tim Skirvin.

=head1 TODO

Making a nice user interface, rather than just an administrative
interface, would be helpful.

=head1 REQUIREMENTS

Perl 5.6.1 or greater, DBIx::Frame, TCB::System

=head1 SEE ALSO

B<TCB::System::Obsolete>, B<TCB::System>, B<DBIx::Frame>

=head1 AUTHOR

Tim Skirvin <tskirvin@ks.uiuc.edu>

=head1 HOMEPAGE

B<http://www.ks.uiuc.edu/Development/MDTools/tcb-system-obsolete>

=head1 LICENSE

This code is distributed under the University of Illinois Open Source
License.  See
C<http://www.ks.uiuc.edu/Development/MDTools/tcb-system-obsolete/license.html>
for details.

=head1 COPYRIGHT

Copyright 2000-2004 by the University of Illinois Board of Trustees and
Tim Skirvin <tskirvin@ks.uiuc.edu>.

=cut

###############################################################################
### Version History ###########################################################
###############################################################################
# v0.2  	Tue Apr  2 15:35:25 CST 2002
### First version fully using DBI::Frame and commented.
# v1.00         Thu May 13 13:19:53 CDT 2004
### Ready to be a part of DBIx::Frame.
