$VERSION = "1.00";
package TCB::System::SuperComp;
our $VERSION = "1.00";

# -*- Perl -*- 		Thu May 13 16:56:10 CDT 2004
###############################################################################
# Written by Tim Skirvin <tskirvin@ks.uiuc.edu>
# Copyright 2000-2004, Tim Skirvin and UIUC Board of Trustees.
# Redistribution terms are below.
###############################################################################

=head1 NAME 

TCB::System::SuperComp - Supercomputer Center database

=head1 SYNOPSIS

  use TCB::System::SuperComp;

See B<TCB::SysLoads> or B<TCB::System> for more information on how to use
this module.

=head1 DESCRIPTION

The 'SuperComp' table lists remaining supercomputer time by day and
computer.  It is populated nightly, and several scripts are used to parse
it into graphs and other useful information.

This table contains the following fields:

 Internal Information
  ID		Unique numeric ID - auto-generated (INT)

 Basic Information (TINYTEXT fields, unless noted)
  Machine	Name of the supercomputer (should be in Centers)
  TimeStamp	Date that the information was entered (DATE)
  SUs		Service Units (CPU-Hours) remaining on the account
  MaxSUs	Maximum SUs 
 
Key fields: 	Machine, TimeStamp

List items: 	TimeStamp, Machine, SUs (SUs / MaxSUs)

Required: 	Same as Key fields.

Default order: 	TimeStamp (reverse)

Admin Fields:	None

No other tables depend on this table directly.

Depends on the 'Centers' table.

=cut

###############################################################################
### Initialization ############################################################
###############################################################################
use vars qw( @ISA $FIELDS $KEYS $NAME $LIST $BIBFILE $ABSTRACTS $REQUIRED 
	     $ADMIN $ORDER $WEBROOT );
use strict;
use warnings;
use CGI;
use TCB::System;

push @ISA, qw( TCB::System );

###############################################################################
### Database Variables ########################################################
###############################################################################
$NAME = "SuperComp";
$FIELDS = {
  'ID'            =>  'INT NOT NULL PRIMARY KEY AUTO_INCREMENT',
  'Machine'       =>  'TINYTEXT',  'TimeStamp'  =>  'DATE',
  'SUs'           =>  'TINYTEXT',  'MaxSUs'     =>  'TINYTEXT'
         };
$KEYS  = [ 'Machine', 'TimeStamp' ];
$LIST  = [ 'TimeStamp', 'Machine', { 'SUs' => '$$SUs$$ / $$MaxSUs$$' } ];
$REQUIRED = $KEYS;
$ADMIN = [ ];
$ORDER = [ '-TimeStamp' ];

###############################################################################
##### Functions ###############################################################
###############################################################################

=head2 Internal Functions 

=over 4

=item html ( ENTRY, TYPE, OPTIONS )

Prints the HTML version of the table.  Heavily customized for the TCB
environment, but then again so is the whole program.                            

=cut

sub html {
  my ($self, $entry, $type, $options, @rest) = @_;
  my $cgi = new CGI;	
  my $date = sprintf("%04d-%02d-%02d", (localtime)[5] + 1900, 
                                       (localtime)[4] + 1, (localtime)[3]);

  if (lc $type eq 'create') {
    $entry ||= { 'TimeStamp' => $date };
  } else { $entry ||= { } }

  my @machines = sort $self->select_fieldlist('Centers', 'Name');
  unshift @machines, '' if (lc $type eq 'search');

  my @return = <<HTML;
<div class="basetable">
 <div class="row2">
  <span class="label">Machine</span>
  <span class="formw"> 
   @{[ $cgi->popup_menu('Machine', \@machines, $$entry{Machine} || "") ]}
  </span>
  <span class="label">Time Stamp</span>
  <span class="formw"> 
   @{[ $cgi->textfield('TimeStamp', $$entry{TimeStamp} || "", 11, 10) ]}
  </span>
 </div>
 <div class="row1">
  <span class="label">Time Remaining</span>
  <span class="formw"> 
     @{[ $cgi->textfield('SUs', $$entry{SUs} || "", 8, 255) ]} 
        /
     @{[ $cgi->textfield('MaxSUs', $$entry{MaxSUs} || "", 8, 255) ]} 
  </span>
 </div>

 <div class="submitbar"> @{[ $cgi->submit(-name=>"Submit") ]} </div>

</div>

HTML
  wantarray ? @return : join("\n", @return);
}

=back

There is no text() field at present.

=cut

###############################################################################
##### main() ##################################################################
###############################################################################

TCB::System->table_add($NAME, $FIELDS, $KEYS, $LIST, $ORDER, $ADMIN, 
			$REQUIRED, \&html, \&text);

=head1 NOTES

This is one of the tables more extensively used by scripts; it's also one
of the most basically designed.  Oh well.

The table itself was designed by the author, Tim Skirvin.

=head1 TODO

Making a nice user interface, rather than just an administrative
interface, would be helpful.

=head1 REQUIREMENTS

Perl 5.6.1 or greater, DBIx::Frame, TCB::System

=head1 SEE ALSO

B<TCB::SysLoads>, B<TCB::System>, B<DBIx::Frame>

=head1 AUTHOR

Tim Skirvin <tskirvin@ks.uiuc.edu>

=head1 HOMEPAGE

B<http://www.ks.uiuc.edu/Development/MDTools/tcb-sysloads>

=head1 LICENSE

This code is distributed under the University of Illinois Open Source
License.  See
C<http://www.ks.uiuc.edu/Development/MDTools/tcb-sysloads/license.html>
for
details.

=head1 COPYRIGHT

Copyright 2000-2004 by the University of Illinois Board of Trustees and
Tim Skirvin <tskirvin@ks.uiuc.edu>.

=cut

###############################################################################
### Version History ###########################################################
###############################################################################
# v0.2  	Tue Apr  2 15:35:25 CST 2002
### First version fully using DBI::Frame and commented.
# v1.00         Thu May 13 16:56:10 CDT 2004
### Ready to be a part of DBIx::Frame.
