$VERSION = "1.01";
package TCB::SysLoads;
our $VERSION = "1.01";

# -*- Perl -*-		Fri May 14 10:48:52 CDT 2004 
###############################################################################
# Written by Tim Skirvin <tskirvin@ks.uiuc.edu>
# Copyright 2000-2004, Tim Skirvin and UIUC Board of Trustees.
# Redistribution terms are below.
###############################################################################

=head1 NAME

TCB::SysLoads - a basic system load database, written for DBIx::Frame

=head1 SYNOPSIS

  use TCB::SysLoads;

See the included CGI scripts to see how it works.

=head1 DESCRIPTION

TCB::SysLoads is a DBIx::Frame object written to keep track of system
loads, both local and remote, at the Theoretical and Computational
Biophysics Group at UIUC.  It provides an administrative web interface for
adding and reading this information, plus a series of scripts for loading
this information and a series of CGIs for extracting the information into
graphs.  

Specific tables:

=over 4

=item TCB::System::SysLoad - local system loads

=item TCB::System::SuperComp - daily remaining time on supercomputers

=item TCB::System::Centers - list of supercomputer center machines

=back

See each table's documentation for more information.

=head2 Variables

=over 4

=item %TCB::SysLoads::NCSAFIX 

A hash for converting the short NCSA machine names to longer ones.  Used
internally only.

=back

=cut

use strict;
use vars qw( @MODS $SERVER $DATABASE $DBTYPE $USER $PASS $DEBUG @ISA @PROBS 
	     %NCSAFIX );

### USER VARIABLES ############################################################
@MODS     = qw( DBIx::Frame Exporter TCB::System TCB::System::Centers 
		TCB::System::SuperComp TCB::System::SysLoad );
@ISA      = qw( DBIx::Frame Exporter TCB::System );
$SERVER   = "db.ks.uiuc.edu";	  
$DATABASE = "System";
$DBTYPE	  = "mysql";
$USER     = "guest";
$PASS     = "";
$DEBUG    = 0;
%NCSAFIX = ( 
        'modi'     => "NCSA Origin 2000", 
        'platinum' => "NCSA Platinum",
        'titan'    => "NCSA Titan",
        'copper'   => "NCSA p690",
        'tungsten' => "NCSA Tungsten",
        'teragrid' => "NCSA Teragrid (Local)",
           );
###############################################################################

###############################################################################
### main () ###################################################################
###############################################################################
# Load the necessary modules from @MODS
foreach (@MODS) { local $@; eval "use $_"; push @PROBS, "$@" if $@; }
die @PROBS if @PROBS;

# Initialize DBIx::Frame
DBIx::Frame->init($SERVER, $DBTYPE, $DATABASE, $USER, $PASS, $SERVER);

###############################################################################
### Subroutines ###############################################################
###############################################################################

=head2 Subroutines

TCB::SysLoads has several subroutines used within the supercomputer
databaes to parse information from the supercomputer centers' usage
databases.

=over 4

=cut

=item usage ( LINES )

Parses the information out of the 'usage' command from NCSA.  Returns an
array of arrayrefs, each of which contain the following four variables:

  Remaining Time, Allocated Time, Percentage Remaining, Machine Name

Machine Name is generated with the NCSAFIX hash.  

Note that the 'usage' command returns several pieces of information at a
time, which is why this is necessary.  

=cut

sub usage {
  my $self = shift;  my @return;
  foreach (@_) {
    chomp; $_ =~ s/^\s+//;      # Clear leading whitespace
    next if /^(\s*$|Proj)/;
    my ($proj,$mach,$login,$userusage,$status,$alloc,$usage,$expire,$name)
                        = split(/\s+/, $_);

    # Update the machine names to something regular
    foreach my $pat (sort keys %NCSAFIX) { 
      next unless $mach;
      $mach =~ s/$pat/$NCSAFIX{$pat}/ if ($pat && $NCSAFIX{$pat}); 
    }

    next unless $alloc;  $usage ||= 0;
    my $remain = $alloc - $usage;

    my $remainperc = 100 * $remain / $alloc ;
    push @return, [ $remain, $alloc, $remainperc, $mach ] if ($mach && $alloc);
  }
  @return;
}

=item tgusage() 

Parses the information out of the TeraGrid 'tgusage' command.  

=cut

sub tgusage {
  my ($self, @lines) = @_;
  # foreach (@lines) { warn "$_\n" }
  foreach my $line (@lines) { 
    my ($name, $alloc, $remain, $used, $limit) = $line =~ 
        m/^(.{25})\s*(\d+)\sSU\s+(\d+)\sSU\s+(\d+)\sSU\s+([\d\.]+)\%\s*$/;
    $name ||= ""; $name =~ s/^\s+|\s+//g;
    next unless ($alloc);
    my $remainperc = 100 * $remain / $alloc ;
    return ( $remain, $alloc, $remainperc );
  }
  ( 0, 0, 0 );
}

=item xbanner ( LINES )

Parses the information out of the 'xbanner' command from PSC.  Returns an
array of arrayrefs, each of which contain these three variables:

Note that xbanner returns only one machine's worth of information at a
time.

=cut

sub xbanner {
  my $self = shift;  # return (0,0,0) unless ($self && ref $self);
  my $allocation = $_[5] || ""; $allocation =~ s/^[^\d\-]*([-.\d]+)\s*$/$1/;
  my $remaining  = $_[6] || ""; $remaining  =~ s/^[^\d\-]*([-\d\.]+)\s*$/$1/;
  my $update     = $_[8] || ""; $update     =~ s/^[^\d\-]*(\d+\S+)\s*$/$1/;
  my $percentage = $allocation > 0 ? 100 * $remaining / $allocation 
                                   : 0 ;
  ($remaining || 0, $allocation || 0, $percentage || 0);
}

=back

=cut

1;

=head1 NOTES

This is a part of the TCB::System database.  Databases were designed by
Tim Skirvin <tskirvin@ks.uiuc.edu>.  See each table's page for information
on specific packages.

=head1 REQUIREMENTS

Perl 5.6.0 or better, MySQL, C<DBIx::Frame>, B<TCB::System>, the sysload
package

=head1 SEE ALSO

B<DBIx::Frame>, B<TCB::System>, B<TCB::System::SysLoad>,
B<TCB::System::SuperComp>, B<TCB::System::Centers>

System Loads: B<sysload>, B<syslog.cgi>, B<add-sysload>, B<convert>

SuperComputers: B<admin.cgi>, B<scstatus.pl>, B<scstatus-db.pl>,
B<sc-times.cgi>

=head1 TODO

=head1 AUTHOR

Written by Tim Skirvin <tskirvin@ks.uiuc.edu>.

=head1 HOMEPAGE

B<http://www.ks.uiuc.edu/Development/MDTools/tcb-sysloads>

=head1 LICENSE

This code is distributed under the University of Illinois Open Source
License.  See
C<http://www.ks.uiuc.edu/Development/MDTools/tcb-sysloads/license.html> for
details.

=head1 COPYRIGHT

Copyright 2000-2004 by the University of Illinois Board of Trustees and
Tim Skirvin <tskirvin@ks.uiuc.edu>.

=cut

###############################################################################
### Version History ###########################################################
###############################################################################
# v0.2		Mon Oct 13 13:42:41 CDT 2003 
### Setting up for real DBI::Frame.
# v0.95		Mon May 10 16:48:34 CDT 2004 
### Getting ready to use DBIx::Frame instead of DBI::Frame; this is a good
### excuse to comment everything and just generally make it all work.
### Comments are still based on old TCB::Library code, however.
# v1.00         Thu May 13 16:53:34 CDT 2004 
### Packaging up and distributing.
# v1.01		Fri May 14 10:48:41 CDT 2004 
### Put in the usage() type scripts.
