$VERSION = "0.95";
package TCB::Seminar::CheckList;
our $VERSION = "0.95";

# -*- Perl -*- 		Fri May 21 11:37:43 CDT 2004 
###############################################################################
# Written by Tim Skirvin <tskirvin@ks.uiuc.edu>
# Copyright 2002-2004, Tim Skirvin and UIUC Board of Trustees.
###############################################################################

=head1 NAME

TCB::Seminar::CheckList - checklist table for TCB::Seminar

=head1 SYNOPSIS

  use TCB::Seminar::CheckList;

See TCB::Seminar for more information.

=head1 DESCRIPTION

The CheckList is a list of items that should be completed for a specific
seminar.  Ideally, all of the questions (in [QUESTIONS]) should be filled
out.  The list of questions is contained in this module, along with
instructions of how to use them.

This table contains the following fields:
  
 Internal Information (TINYTEXT fields, unless noted)
  ID            Unique numeric ID - auto-generated (INT)
  CreatedBy     Username of person who created entry     \   Auto-created using 
  CreateDate    Date that this entry was created (DATE)   \  $ENV{REMOTE_USER} 
  ModifiedBy    Username who last modified this entry     /  and current time
  ModifyDate    Date that this entry was modified (DATE) /   in html() 

 Basic Information (TINYTEXT fields, unless noted)
  SeminarID	ID of the Seminar that this entry refers to (INT)
  Notes		Any general notes that should be looked at (TEXT)
  [QUESTIONS]	Each question has a field which holds the date that the 
	        item was completed.  Contained in @QUESIONS.  (DATE)

Key fields:     SeminarID

List items:     Seminar Name, Seminar Summary, Updated 

Required:       Same as Key fields.

Default order:  SeminarID (reverse)

Admin Fields:   None

No other tables depend on this table directly.

Depends on the 'Lecture' table.

=head2 Variables

The following variables are set in the module, and are used as defaults
for creation and searching.

=over 4

=item @TCB::Seminar::CheckList::QUESTIONS - the list of questions to ask.
Default possibilities are listed in the source code.

=back

=cut

###############################################################################
### Initialization ############################################################
###############################################################################
use vars qw( @ISA $FIELDS $KEYS $NAME $LIST $ORDER $ADMIN $REQUIRED 
	     %QUESTIONS @QUESTIONS );
use strict;
use warnings;
use CGI;
use TCB::Seminar qw( seminar_name_by_id seminar_summary updated );

unshift @ISA, "TCB::Seminar";

###############################################################################
### Database Variables ########################################################
###############################################################################
$NAME = "CheckList";	
$FIELDS = {
  # Non User-Modified Information
  'ID'          => 'INT NOT NULL PRIMARY KEY AUTO_INCREMENT',
  'CreatedBy'   => 'TINYTEXT',  'ModifiedBy'    => 'TINYTEXT',
  'CreateDate'  => 'DATE',      'ModifyDate'    => 'DATE',

  'SeminarID'   => 'INT',	'Notes'		=> 'TEXT'
	 };
$KEYS  = [ 'SeminarID' ];
$LIST  = [ {'Seminar' => [ \&seminar_name_by_id, '$$SeminarID$$' ] }, 
	   { 'Summary' => [ \&seminar_summary, '$$SeminarID$$' ] }, 
	   { 'Updated' => [ \&updated, '$$ModifyDate$$', '$$CreateDate$$' ] } ];
$ORDER = [ '-SeminarID' ];
$REQUIRED = $KEYS;
$ADMIN = [];

## The following arrays are used to list what questions we're going to ask
## in the check list. 

use vars qw( @FIRST @ACCEPT @SIXWEEKS @FOURWEEKS @TWOWEEKS @ONEWEEK );
@FIRST     = qw( SeminarRoom InviteLetter );
@ACCEPT    = qw( OpenFile HotelReserve ConfirmLetter CalendarInfo WebInfo );
@SIXWEEKS  = qw( SpeakerCheck GetAbstract ApproveAbstract WebAbstract
		 PrintAbstract CopyAbstract DistribAbstract);
@FOURWEEKS = qw( HostSchedule MakeAgenda );
@TWOWEEKS  = qw( MainAnnounce AVAids ContactHosts AgendaHosts );
@ONEWEEK   = qw( PostAnnounce SendAnnounce HotelAgenda CV_Klaus Reimbursement );
@QUESTIONS = ( @FIRST, @ACCEPT, @SIXWEEKS, @FOURWEEKS, @TWOWEEKS, @ONEWEEK);
foreach (@QUESTIONS) { $$FIELDS{$_} = 'DATE'; }

###############################################################################
##### Functions ###############################################################
###############################################################################

### questions() 
# Returns the list of questions.  We might want to export this in later
# versions.

sub questions { %QUESTIONS }

=head2 Internal Functions 

=over 4

=item html ( ENTRY, TYPE, OPTIONS )

Prints the HTML version of the table.  Heavily customized for the TCB
environment, but then again so is the whole program.                            

=cut

sub html {
  my ($self, $entry, $type, $options, @rest) = @_;
  my $cgi = new CGI;  $entry ||= {};  $options ||= {};
  my $admin = $$options{'admin'} ? 1 : 0;

  $entry ||= {};
  if (lc $type eq 'create') {
    $$entry{'CreatedBy'}  ||= $ENV{'REMOTE_USER'} || "unknown";
    $$entry{'CreateDate'} ||= $self->_date_mysql(time);
    $$entry{'Date'}       ||= $self->_date_mysql(time);
  } elsif (lc $type eq 'search') {
  } elsif (lc $type eq 'edit' || lc $type eq 'update') {
    $$entry{'ModifiedBy'} = $ENV{'REMOTE_USER'} || "unknown";
    $$entry{'ModifyDate'} = $self->_date_mysql(time);
  } else { }

  my %seminars;
  foreach my $entry ($self->select('Lecture', {})) { 
    my $id = $$entry{ID} || next;
    $seminars{$id} = $self->seminar_name_short($entry) || next;
  }

  my %reverse = reverse %seminars; my @sorted;
  foreach (reverse sort keys %reverse) { push @sorted, $reverse{$_} }

  $seminars{''} = "*" if (lc $type eq 'search');
  my @return = <<HTML;
<div class="basetable">
 <div class="row1">
  <span class="label">Seminar</span>
  <span class="formw">
   @{[ $cgi->popup_menu('SeminarID',\@sorted,$$entry{SeminarID} || "",\%seminars) ]}
  </span>
 </div>

 <div class="tablerow">
  <table width=100%>
   <tr> <th> Status </th> <th> Description </th> </tr>
   <tr> @{[ join("\n", checklist( $entry, @FIRST ) ) ]} </tr>
   <tr> <th align=left colspan=4> When the speaker has accepted... </td> </tr>
   <tr> @{[ join("\n", checklist( $entry, @ACCEPT ) ) ]} </tr>
   <tr> <th align=left colspan=4> 4-6 Weeks prior... </td> </tr>
   <tr> @{[ join("\n", checklist( $entry, @SIXWEEKS ) ) ]} </tr>
   <tr> <th align=left colspan=4> 3-4 Weeks prior... </td> </tr>
   <tr> @{[ join("\n", checklist( $entry, @FOURWEEKS ) ) ]} </tr>
   <tr> <th align=left colspan=4> 2 Weeks prior... </td> </tr>
   <tr> @{[ join("\n", checklist( $entry, @TWOWEEKS ) ) ]} </tr>
   <tr> <th align=left colspan=4> 1 Week prior... </td> </tr>
   <tr> @{[ join("\n", checklist( $entry, @ONEWEEK ) ) ]} </tr>
  </table>
 </div>

 <div class="row1">
  <span class="label">Notes</span>
  <span class="formw">
     @{[ $cgi->textarea(-name=>'Notes', -default=>$$entry{Notes} || "",
                        -rows=>4, -cols=>70, -maxlength=>65535,
                        -wrap=>'physical') ]}
  </span>
 </div>

 <div class="row2">
  <div class="modbox">
   @{[ $self->created_html( $$entry{CreateDate} || "", 
                            $$entry{ModifyDate} || "",
                            $$entry{CreatedBy}  || "", 
                            $$entry{ModifiedBy} || "", !$admin ) ]}
  </div>
 </div>

 <div class="submitbar"> @{[ $cgi->submit(-name=>"Submit") ]} </div>
</div>
HTML
  wantarray ? @return : join("\n", @return);
}

=back

=cut

sub text { }

### checklist( QUESTIONS )
# Makes the HTML-formatted checklist for each of QUESTIONS.  
sub checklist {  
  my @return;
  my ($entry, @questions) = @_;
  my $cgi = new CGI;
  foreach (@questions) {
    my %list = ( '0000-00-00' => "To Do", _date_mysql() => "Done - Today" );
    $list{$$entry{$_}} = "Done - $$entry{$_}" if ( $$entry{$_} && 
		$$entry{$_} ne '0000-00-00' && $$entry{$_} ne _date_mysql() );
    my $menu = $cgi->popup_menu($_, [sort keys %list], 
		$$entry{$_} || '0000-00-00', \%list );
    push @return, "  <td align=center> $menu </td>";
    my $text = $QUESTIONS{$_} || "";
    push @return, "  <td align=left> $text </td>";
    push @return, " </tr><tr>";
  }
  wantarray ? @return : join("\n", @return);
}

### _date_mysql( [TIME] )
# Returns the date based on TIME for use with mysql.
sub _date_mysql {
  my ($self, $time) = @_;  $time ||= time;
  sprintf("%04d-%02d-%02d", (localtime($time))[5] + 1900,
                        (localtime($time))[4] + 1, (localtime($time))[3]);
}

###############################################################################
##### main() ##################################################################
###############################################################################

# Parse __DATA__ for the actual text to put in the HTML, and store it in
# %QUESTIONS.  
my $last;
while (my $line = <DATA>) {
  chomp $line;  $line =~ s/\s*\#.*$//g; # Strip comments
  if ($line =~ /^(\S+)?\s*(.*)$/) { 
    my $text = $2;                next unless $text;
    my $key = $1 || $last || "";  next unless $key;
    $QUESTIONS{$key} = $QUESTIONS{$key} 
		? join("\n", $QUESTIONS{$key}, $text)
		: $text ;
    $last = $key;
  }
}

TCB::Seminar->add_table($NAME, $FIELDS, $KEYS, $LIST, \&html, \&text);

=head1 COMMENTS

=head1 TODO

As long as this list of questions is done, we're good to go.

=head1 REQUIREMENTS

Perl 5.6.0 or better, DBIx::Frame, TCB::Seminar.

=head1 SEE ALSO

B<TCB::Seminar>, B<DBIx::Frame>, B<TCB::Seminar::Functions>

=head1 AUTHOR

Tim Skirvin <tskirvin@ks.uiuc.edu>

=head1 HOMEPAGE

B<http://www.ks.uiuc.edu/Development/MDTools/tcb-seminar/>

=head1 LICENSE

This code is distributed under the University of Illinois Open Source
License.  See
C<http://www.ks.uiuc.edu/Development/MDTools/tcb-seminar/license.html>
for details.

=head1 COPYRIGHT

Copyright 2002-2004 by the University of Illinois Board of Trustees and
Tim Skirvin <tskirvin@ks.uiuc.edu>.

=cut

###############################################################################
### Version History ###########################################################
###############################################################################
# v0.1 		Fri Mar 22 14:09:27 CST 2002
### First functional version, not approved by anybody yet, this will likely 
### still need a lot of work.
# v0.9  	Wed Aug 14 13:51:37 CDT 2002
### Getting ready for publishing.  Documentation complete.
# v0.91 	Mon Oct  7 16:43:23 CDT 2002
### Slight fix with 'MODIFIED_BY'.
# v0.95		Fri May 21 11:37:43 CDT 2004 
### Updated for DBIx::Frame.  Getting ready for stylesheets.

###############################################################################
### DATA ######################################################################
###############################################################################
__DATA__ 
# The actual, HTML-formatted text for the checklist.  Formatted into a
# hash table; the key is the first word on the line; all else is the
# value.  Note that this bit must be formatted; if there's leading
# whitespace, then we'll append to the last key, otherwise we'll use the
# first word as the current key.  So at least include some whitespace at
# the beginning of the line.  Comments act like you'd expect.
SeminarRoom	Reserve seminar room for the semester (Rm. 3269 BI),
		currently each Monday at 3:00pm; 
		http://www.beckman.uiuc.edu/reserve/reserveform.html
InviteLetter	Send seminar information metter; seminar dates must be
		coordinated with Schulten's calendar and room 3269 
		calendar.
OpenFile	Open a file (lavendar) with speaker's name and date
		when confirmed.
HotelReserve	Check in manila folder for open reservation at Illini
		Union.  If one is available, call IU (3-1241) and change
	 	name from Marilyn to speaker.  Make hotel reservations
		for Sun pm to Mon pm at Illini Union (1st choice) with 
		seminar account (1-1-12057 programs support).  Hampton
		Inn is second choice, and they directly bill Beckman Inst.
ConfirmLetter	Send confirmation letter with necessary info to speaker.
CalendarInfo	Add the seminar to Klaus' appointment book + group calendar.
WebInfo		Send TCB webmaster the seminar information (date, speaker's 
		full name, affiliation) as you have the finalized.
SpeakerCheck	Check with the speaker to see if the reservations are okay.
GetAbstract	Ask speaker via email for title, abstract (about 10 lines),
		and most recent CV (inform speaker this is just to introduce 
		them the day of the seminar), and discuss travel arrangements
		(very seldom do we make these arrangements, but we need to
		know for the agenda); on occasion we are asked to purchase
		the ticket and fed ex it to them.
ApproveAbstract Upon receipt of title and abstract, get Klaus' approval 
		then continue with final version.
WebAbstract	Send TCB webmaster the abstract and title as soon as you 
		have it.
PrintAbstract	As soon as title and abstract are finalized, print out 
		abstract page, then copy onto border form which is in a file
		on desk (original border).
CopyAbstract	Copy announcement in copy center, 250 copies, using
		Program Support account (1-1-12057).
DistribAbstract Place announcements in TCB group mailboxes and on all bulletin
		boards in this area (there are 7 all total).
HostSchedule	Inform (seminar) mailing list of confirmed speaker and urge 
		them to let ou kniw if interested to meet with him/her and to 
		mark their calendars; also specify our speaker URL in your
		correspondence (http://www.ks.uiuc.edu/Services/Seminar/); 
		ask confirmed speaker who he/she would like to meet while here.
MakeAgenda	Agenda for speaker (coordinate with speaker, Klaus, others if 
		involved, be INCLUSIVE): make meals' reservations, arrangements
		for walking the speaker between appointments, driving speaker
		to and from airport; include hotel information on agenda
MainAnnounce	Send announcement (if sent much earlier than this, people tend 
		to forget) to the basic mail list (labels are available in file
		on desk as well as the number of copies to be sent to each
		address)
AVAids		Check with the speaker and then use following form to reserve 
		equipment and audiovisual aids; keep copies of all confirmations
		and correspondence on file:
		http://www.beckman.uiuc.edu/reserve/reserveform.html
ContactHosts	Cc: designated TCB Hosts on seminar related messages.
AgendaHosts	Involve TCB hosts in developing the agenda.
PostAnnounce	Post TCB seminar announcements a week prior to the seminar in:
		<ul>
 		<li> Group calendar (yourself) by Monday
	 	<li> BI calendar (weeklycal@beckman.uiuc.edu) by Monday @ 12pm
		<li> Physics blue sheet (bluesheet@mail.physics.uiuc.edu)
		     by Thursday noon
		<li> School of Chem Sciences (rauchfub@scs.uiuc.edu) by
		     Thursday
		<li> Engineering calendar (engcal@uiuc.edu) by Tuesday @ 4pm 
		<li> Add. Eng. calendar (b-irwin@uiuc.edu) by Tuesday @ 3pm 
		</ul>
SendAnnounce	Mail/fax/email agenda to speaker and ALL people involved
		(including hosts and TCB members who give demos) no later than
		Wednesday before the seminar.
HotelAgenda	Leave hard copy of agenda for speaker in hotel's front desk
		(Bree on Friday or host on Sunday)
CV_Klaus	Get the speaker's most updated CV to hand to Klaus just before
		the seminar (host or Marilyn).
Reimbursement	Reimbursements - FOR NON-US CITIZENS - the following
		information is needed before the visitor arrives:
		<ul>
		<li> Country of citizenship
		<li> Type of visa
		<li> Length of visit
		<li> Type of support committed (if honorarium was promised,
		     make it clear to Dale
 		</ul>
		Take this information to Dale Johnston to confirm how to handle
		the reimbursement in detail. 
		<br>
		For non-US citizens, we would normally need a copy of
		passport photo page, I94, visa status page, and SS# if they
		have one, in order to reimburse.  We will get these copies 
		made while the visitor is still here and will take all the
		copies to verify with Dale that nothing else is needed.  If
		more information is needed, we would still have the opportunity
		to get it from the visitor while they are here.  If needed,
		schedule time for Marilyn and the speaker.  Of course, you
		always need ORIGINAL receipts.
