$VERSION = "1.00";
package TCB::System::Port;
our $VERSION = "1.00";

# -*- Perl -*- 		Thu May 13 15:52:19 CDT 2004 
###############################################################################
# Written by Tim Skirvin <tskirvin@ks.uiuc.edu>
# Copyright 2000-2004, Tim Skirvin and UIUC Board of Trustees.
# Redistribution terms are below.
###############################################################################

=head1 NAME 

TCB::System::Port - TCB Group Port Database

=head1 SYNOPSIS

  use TCB::System::Port;

See B<TCB::System> for more information on how to use this module.

=head1 DESCRIPTION

The 'Port' table is used to store port activation information throughout
the TCB Group.  

This table contains the following fields:

 Internal Information
  ID		Unique numeric ID - auto-generated (INT)

 Basic Information (TINYTEXT fields, unless noted)
  BoxNumber	Box number of the port
  PortNumber	Port number of the port
  PortSpeed	Port Speed (0, 10, 100, 1000) (INT)
  PortMachine	Machine connected to this port
  RoomNumber	Room number this port is in
  Building	Building this port is in
  Notes		Notes about this port (TEXT)
  
Key fields: 	PortNumber, BoxNumber

List items: 	Room (Room + Building), PortSpeed, BoxNumber, PortNumber

Required: 	Same as Key fields.

Default order: 	Building, Room, BoxNumber, PortNumber

Admin Fields:	None

No other tables depend on this table directly.

Doesn't depend on any other table.

=cut

###############################################################################
### Initialization ############################################################
###############################################################################
use vars qw( @ISA $FIELDS $KEYS $NAME $LIST $BIBFILE $ABSTRACTS $REQUIRED 
	     $ADMIN $ORDER $WEBROOT );
use strict;
use warnings;
use CGI;
use TCB::System;

push @ISA, qw( TCB::System );

###############################################################################
### Database Variables ########################################################
###############################################################################
$NAME = "Port";
$FIELDS = {
  'ID'            =>  'INT NOT NULL PRIMARY KEY AUTO_INCREMENT',
  'BoxNumber'     =>  'TINYTEXT', 'PortNumber'    =>  'TINYTEXT',
  'PortSpeed'     =>  'INT',      'PortMachine'   =>  'TINYTEXT',
  'RoomNumber'    =>  'TINYTEXT', 'Building'      =>  'TINYTEXT',
  'Notes'         =>  'TEXT',
         };
$KEYS  = [ 'PortNumber', 'BoxNumber' ];
$LIST  = [ 
  { 'Room' => '$$RoomNumber$$ $$Building$$' },
  { 'Port Speed' => '$$PortSpeed$$' || 0 } ,
  'BoxNumber', 'PortNumber', 'PortMachine' ];
$REQUIRED = $KEYS;
$ADMIN = [ ];
$ORDER = [ 'Building', 'RoomNumber', 'BoxNumber', 'PortNumber' ];

###############################################################################
##### Functions ###############################################################
###############################################################################

=head2 Internal Functions 

=over 4

=item html ( ENTRY, TYPE, OPTIONS )

Prints the HTML version of the table.  Heavily customized for the TCB
environment, but then again so is the whole program.                            

=cut

sub html {
  my ($self, $entry, $type, $options, @rest) = @_;
  my $cgi = new CGI;
  if ( lc $type eq 'create' ) { 
    $entry ||= { 'PortSpeed' => 100, 'Building' => 'Beckman' }; 
  } else { $entry ||= {} }

  my %speeds = ( 1000 => "gigabit" , 
                 100  => "100bT"   , 
                 10   => "10bT"    , 
                 0    => "off"     );
  $speeds{''} = "*" if (lc $type eq 'search' );
  
  my @return = <<HTML;
<div class="basetable">
 <div class="row3">
  <span class="label">Room</span>
  <span class="formw"> 
    @{[ $cgi->textfield('RoomNumber', $$entry{RoomNumber} || "", 8, 255) ]}
  </span>
  <span class="label">Building</span>
  <span class="formw"> 
    @{[ $cgi->textfield('Building', $$entry{Building} || "", 8, 255) ]}
  </span>
  <span class="label">Box Number</span>
  <span class="formw"> 
    @{[ $cgi->textfield('BoxNumber', $$entry{BoxNumber} || "", 8, 255) ]}
  </span>
 </div>

 <div class="row3">
  <span class="label">Port Number</span>
  <span class="formw"> 
    @{[ $cgi->textfield('PortNumber', $$entry{PortNumber} || "", 8, 255) ]}
  </span>
  <span class="label">Port Speed</span>
  <span class="formw"> 
      @{[ $cgi->popup_menu('PortSpeed', [keys %speeds], 
                        $$entry{PortSpeed} || "", \%speeds)   ]}
  </span>
  <span class="label">Machine</span>
  <span class="formw"> 
    @{[ $cgi->textfield('PortMachine', $$entry{PortMachine} || "", 20, 255) ]}
  </span>
 </div>

 <div class="row1">
  <span class="label">Notes</span>
  <span class="formw">
     @{[ $cgi->textarea(-name=>'Notes', -default=>$$entry{Notes} || "", 
		        -rows=>5, -cols=>70, -maxlength=>65535,
			-wrap=>'physical') ]}
  </span>
 </div>

 <div class="submitbar"> @{[ $cgi->submit(-name=>"Submit") ]} </div>

</div>

HTML
  wantarray ? @return : join("\n", @return);
}

=back

There is no text() field at present.

=cut

###############################################################################
##### main() ##################################################################
###############################################################################

TCB::System->table_add($NAME, $FIELDS, $KEYS, $LIST, $ORDER, $ADMIN, 
			$REQUIRED, \&html, \&text);

=head1 NOTES

This table hasn't been very heavily used for its original purpose (part of
an automatic floorplan), but it has been handy to just have this
information around.

=head1 TODO

Making a nice user interface, rather than just an administrative
interface, would be helpful.

=head1 REQUIREMENTS

Perl 5.6.1 or greater, DBIx::Frame, TCB::System, TCB::System::PortMap

=head1 SEE ALSO

B<TCB::System::PortMap>, B<TCB::System>, B<DBIx::Frame>

=head1 AUTHOR

Tim Skirvin <tskirvin@ks.uiuc.edu>

=head1 HOMEPAGE

B<http://www.ks.uiuc.edu/Development/MDTools/tcb-portmap>

=head1 LICENSE

This code is distributed under the University of Illinois Open Source
License.  See
C<http://www.ks.uiuc.edu/Development/MDTools/tcb-portmap/license.html> for
details.

=head1 COPYRIGHT

Copyright 2000-2004 by the University of Illinois Board of Trustees and
Tim Skirvin <tskirvin@ks.uiuc.edu>.

=cut

###############################################################################
### Version History ###########################################################
###############################################################################
# v0.2  	Tue Apr  2 15:35:25 CST 2002
### First version fully using DBI::Frame and commented.
# v1.00         Thu May 13 15:52:23 CDT 2004 
### Ready to be a part of DBIx::Frame.
