$VERSION = "1.00";
package TCB::System::License;
our $VERSION = "1.00";

# -*- Perl -*- 		Thu May 13 15:30:01 CDT 2004
###############################################################################
# Written by Tim Skirvin <tskirvin@ks.uiuc.edu>
# Copyright 2000-2004, Tim Skirvin and UIUC Board of Trustees.
# Redistribution terms are below.
###############################################################################

=head1 NAME 

TCB::System::License - Software License Database

=head1 SYNOPSIS

  use TCB::System::License;

See B<TCB::Equipment> or B<TCB::System> for more information on how to use
this module.

=head1 DESCRIPTION

The 'License' table describes the licenses used by the TCB Group, both in
quantities and duration.  It is far more convenient to use this than the
equipment database for these types of information

This table contains the following fields:

 Internal Information
  ID		Unique numeric ID - auto-generated (INT)

 Basic Information (TINYTEXT fields, unless noted)
  Program	Name of the licensed program (from 'Package')
  Cost		Cost of the license
  LicenseStart	Start of the licensing period (DATE)
  LicenseStop	End of the licenseing period (DATE)
  Vendor	Who the license was purchased from
  PO		PO or other order number the license was purchased with
  Notes		Notes about the license (TEXT)

Key fields: 	Program, LicenseStart, LicenseStop

List items: 	Program, License Length (LicenseStart - LicenseStop)

Required: 	Same as Key fields.

Default order:  LicenseStop (reverse), Program

Admin Fields:	None

No other tables depend on this table directly.

Depends on the 'Package' table.

=cut

###############################################################################
### Initialization ############################################################
###############################################################################
use vars qw( @ISA $FIELDS $KEYS $NAME $LIST $BIBFILE $ABSTRACTS $REQUIRED 
	     $ADMIN $ORDER $WEBROOT );
use strict;
use warnings;
use CGI;
use TCB::System;

push @ISA, qw( TCB::System );

###############################################################################
### Database Variables ########################################################
###############################################################################
$NAME = "License";
$FIELDS = {
  'ID'            =>  'INT NOT NULL PRIMARY KEY AUTO_INCREMENT',
  'Program'       =>  'TINYTEXT',   'LicenseStart'     =>  'DATE',        
  'LicenseStop'   =>  'DATE',       'Cost '       => 'TINYTEXT',    
  'Vendor'        =>  'TINYTEXT',   'Notes'       => 'TEXT',
  'PO'            =>  'TINYTEXT'
         };
$KEYS  =  [ 'Program', 'LicenseStart', 'LicenseStop' ];
$LIST  =  [ 'Program', { 'License' => '$$LicenseStart$$ - $$LicenseStop$$' } ];
$REQUIRED = $KEYS;
$ADMIN = [ ];
$ORDER = [ '-LicenseStop', 'Program' ];

###############################################################################
##### Functions ###############################################################
###############################################################################

=head2 Internal Functions 

=over 4

=item html ( ENTRY, TYPE, OPTIONS )

Prints the HTML version of the table.  Heavily customized for the TCB
environment, but then again so is the whole program.                            

=cut

sub html {
  my ($self, $entry, $type, $options, @rest) = @_;
  my $cgi = new CGI;	$entry ||= {};
  my $date = sprintf("%04d-%02d-%02d", (localtime)[5] + 1900, 
		  		       (localtime)[4] + 1, (localtime)[3]);
  
  my $extra = lc $type eq 'search' ? { '*' => "" } : { };
  my %programs;
  foreach my $entry ($self->select('Package', {})) {
    my $ID      = $$entry{ID};       my $OS      = $$entry{OS};
    my $program = $$entry{Program};  my $version = $$entry{Version};
    $programs{"$program - $version ($OS)"} = "$program - $version ($OS)";
  }
  $programs{''} = "";
  if (lc $type eq 'create') {
    $$entry{Program}      ||= "";
    $$entry{LicenseStart} ||= $date;
  }

  my @return = <<HTML;
<div class="basetable">
 <div class="row1">
  <span class="label">Program</span>
  <span class="formw"> 
     @{[ $cgi->popup_menu('Program', [ sort keys %programs ], 
                        $$entry{Program} || "", \%programs ) ]}
  </span>
 </div>
 <div class="row2">
  <span class="label">License</span>
  <span class="formw"> 
     @{[ $cgi->textfield('LicenseStart', $$entry{LicenseStart} || "", 11, 10) ]}
        -
     @{[ $cgi->textfield('LicenseStop', $$entry{LicenseStop} || "", 11, 10) ]}
  </span>
  <span class="label">Cost</span>
  <span class="formw"> 
     @{[ $cgi->textfield('Cost', $$entry{Cost} || "", 15, 255) ]}
  </span>
 </div>

 <div class="row2">
  <span class="label">Vendor</span>
  <span class="formw"> 
     @{[ $cgi->textfield('Vendor', $$entry{Vendor} || "", 25, 10) ]}
  </span>
  <span class="label">PO</span>
  <span class="formw"> 
     @{[ $cgi->textfield('PO', $$entry{PO} || "", 15, 255) ]}
  </span>
 </div>

 <div class="row1">
  <span class="label">Notes</span>
  <span class="formw">
     @{[ $cgi->textarea(-name=>'Notes', -default=>$$entry{Notes} || "", 
		        -rows=>10, -cols=>70, -maxlength=>65535,
			-wrap=>'physical') ]}
  </span>
 </div>

 <div class="submitbar"> @{[ $cgi->submit(-name=>"Submit") ]} </div>

</div>

HTML
  wantarray ? @return : join("\n", @return);
}

=back

There is no text() field at present.

=cut

###############################################################################
##### main() ##################################################################
###############################################################################

TCB::System->table_add($NAME, $FIELDS, $KEYS, $LIST, $ORDER, $ADMIN, 
			$REQUIRED, \&html, \&text);

=head1 NOTES

This table was designed by Tim Skirvin <tskirvin@ks.uiuc.edu>.

=head1 TODO

Making a nice user interface, rather than just an administrative
interface, would be helpful.

=head1 REQUIREMENTS

Perl 5.6.1 or greater, DBIx::Frame, TCB::System

=head1 SEE ALSO

B<TCB::Equipment>, B<TCB::System>, B<DBIx::Frame>

=head1 AUTHOR

Tim Skirvin <tskirvin@ks.uiuc.edu>

=head1 HOMEPAGE

B<http://www.ks.uiuc.edu/Development/MDTools/tcb-equipment/>

=head1 LICENSE

This code is distributed under the University of Illinois Open Source
License.  See
C<http://www.ks.uiuc.edu/Development/MDTools/tcb-equipment/license.html>
for details.

=head1 COPYRIGHT

Copyright 2000-2004 by the University of Illinois Board of Trustees and
Tim Skirvin <tskirvin@ks.uiuc.edu>.

=cut

###############################################################################
### Version History ###########################################################
###############################################################################
# v0.2  	Tue Apr  2 15:35:25 CST 2002
### First version fully using DBI::Frame and commented.
# v1.00         Thu May 13 15:30:01 CDT 2004
### Ready to be a part of DBIx::Frame.
# v1,01		Wed May 19 09:35:58 CDT 2004 
### Fixes in html()
