$VERSION = "1.00";
package TCB::System::Equipment;
our $VERSION = "1.00";

# -*- Perl -*- 		Thu May 13 15:30:01 CDT 2004
###############################################################################
# Written by Tim Skirvin <tskirvin@ks.uiuc.edu>
# Copyright 2000-2004, Tim Skirvin and UIUC Board of Trustees.
# Redistribution terms are below.
###############################################################################

=head1 NAME 

TCB::System::Equipment - TCB Group Equipment Database

=head1 SYNOPSIS

  use TCB::System::Equipment;

See B<TCB::Equipment> or B<TCB::System> for more information on how to use
this module.

=head1 DESCRIPTION

The 'Equipment' table describes a given piece of equipment purchased by
the TCB Group, including order information, warranties, and so forth.
This includes hardware items like computers and telephones, software
licenses, and other such items.  

This table contains the following fields:

 Internal Information
  ID		Unique numeric ID - auto-generated (INT)

 Basic Information (TINYTEXT fields, unless noted)
  Manufacturer	Equipment manufacturer 
  Model		Equipment model number 
  Description	One-line description of the equipment 
  FullDesc	Long description of the equipment (TEXT)
  Type		What type of equipment is this (SEE BELOW)
  Location	Where this equipment is located (SEE BELOW)
  History	The item's history (TEXT)
  
 Accounting Information (TINYTEXT fields)
  Serial	Serial Number 
  PAS		UIUC Property Accounting Number

 Order Information (TINYTEXT fields, unless noted)
  Vendor	Who the equipment was purchased from
  Price		The (original) price of the equipment
  Quote		The quote number for the purchase
  PO		The purchase order number for the order
  Account	The account number used to pay for the equipment
  OrderDate	The date the equipment was ordered
  Received	The date the equipment was received

 Support Information (TINYTEXT fields, unless noted)
  SupportBegins	The date that support begins
  SupportEnds	The date that support ends

Key fields: 	Description, Manufacturer, Model, Serial

List items: 	Item (Manufacturer + Model), Description, PAS, Location

Required: 	Same as Key fields.

Default order: 	Received (reverse)

Admin Fields:	None

No other tables depend on this table directly.

Doesn't depend on any other table.

=head2 Variables

The following variables are set in the module, and are used as defaults
for creation and searching.

=over 4

=item @TCB::System::Equipment::TYPE

Type of equipment, used with 'Type'.  Default possibilities: Hardware,
Software, Other.

=item @TCB::System::Equipment::BUILDING

Possible locations of the equipment, used with 'Location'.  Default
possibilities: Beckman, Attic, Surplus, Off-Site, Lost, Broken.

=back

=cut

###############################################################################
### Initialization ############################################################
###############################################################################
use vars qw( @ISA $FIELDS $KEYS $NAME $LIST $BIBFILE $ABSTRACTS $REQUIRED 
	     $ADMIN $ORDER $WEBROOT @TYPE @BUILDING );
use strict;
use warnings;
use CGI;
use TCB::System;

push @ISA, qw( TCB::System );

###############################################################################
### Database Variables ########################################################
###############################################################################
$NAME = "Equipment";
$FIELDS = {

  # Non User-Modified Information
  'ID'            =>  'INT NOT NULL PRIMARY KEY AUTO_INCREMENT',

  # Item Description
  'Manufacturer' => 'TINYTEXT NOT NULL',    'Model'    => 'TINYTEXT',  
  'Description'  => 'TINYTEXT NOT NULL',    'FullDesc' => 'TEXT', 
  'Type'         => 'TINYTEXT NOT NULL',    'Location' => 'TINYTEXT',
  'History'      => 'TEXT',

  # Tagging information
  'Serial' 	 => 'TINYTEXT',        'PAS'      => 'TINYTEXT', 

  # Order Information
  'Vendor'       => 'TINYTEXT',        'Quote'    => 'TINYTEXT', 
  'PO'           => 'TINYTEXT',        'Price'    => 'TINYTEXT', 
  'OrderDate'    => 'DATE', 	       'Received' => 'DATE',
  'Account'      => 'TINYTEXT', 
  
  # Support Information
  'SupportBegins' => 'DATE', 'SupportEnds' => 'DATE', 

  # Obsolete Information (not used)
  'Pending' => 'INT',  

	 };
$KEYS  = [ 'Description', 'Manufacturer', 'Model', 'Serial' ];
$LIST  = [ 
   { 'Item' => '$$Manufacturer$$ - $$Model$$' }, 'Description', 
	 	'PAS', 'Serial', 'Location'
	 ];
$REQUIRED = $KEYS;
$ADMIN = [ ];
$ORDER = [ '-Received' ];

@TYPE     = qw ( Hardware Software Other );
@BUILDING = qw ( Beckman Attic Surplus Off-Site Lost Broken ) ;

###############################################################################
##### Functions ###############################################################
###############################################################################

=head2 Internal Functions 

=over 4

=item html ( ENTRY, TYPE, OPTIONS )

Prints the HTML version of the table.  Heavily customized for the TCB
environment, but then again so is the whole program.                            

=cut

sub html {
  my ($self, $entry, $type, $options, @rest) = @_;
  my $cgi = new CGI;
  my $date = sprintf("%04d-%02d-%02d", (localtime)[5] + 1900, 
		  		       (localtime)[4] + 1, (localtime)[3]);
  if (lc $type eq 'create') {
    $entry ||= {
	'Type'	   => 'Hardware',
	'Location' => 'Beckman',
	'Account'  => '1-495873-392030-191100',
	'Received' => $date,
	'SupportBegins' => $date,
	       };
  } else { $entry ||= {}; }
  
  my @building = sort @BUILDING;  my @type = sort @TYPE;

  if (lc $type eq 'search') {
    unshift @building, '';
    unshift @type, '';
  }

  my @return = <<HTML;
<div class="basetable">
 <h3 class="tablehead"> Equipment Information 
		@{[ $$entry{ID} ? "(Entry $$entry{ID})" : "" ]} </h3>

 <div class="row1">
  <span class="label">Description</span>
  <span class="formw"> @{[ $cgi->textfield('Description',
				$$entry{Description} || "", 80, 255) ]} </span>
 </div>
 <div class="row2">
  <span class="label">Manufacturer</span>
  <span class="formw"> @{[ $cgi->textfield('Manufacturer',
				$$entry{Manufacturer} || "", 30, 255) ]} </span>
  <span class="label">Model</span>
  <span class="formw"> @{[ $cgi->textfield('Model',
				$$entry{Model} || "", 30, 255) ]} </span>
 </div>
 <div class="row1">
  <span class="label">Full Description</span>
  <span class="formw">
     @{[ $cgi->textarea(-name=>'FullDesc', -default=>$$entry{FullDesc} || "", 
		        -rows=>5, -cols=>70, -maxlength=>65535,
			-wrap=>'physical') ]}
  </span>
 </div>

 <div class="row4">
  <span class="label">Type</span> 
  <span class="formw">
     @{[ $cgi->popup_menu('Type', \@type, $$entry{Type} || "") ]}
  </span>
  <span class="label">Location</span>
  <span class="formw">
     @{[ $cgi->popup_menu('Location', \@building, $$entry{Location} || "") ]}
  </span>
  <span class="label">Serial</span>
  <span class="formw">
     @{[ $cgi->textfield('Serial', $$entry{Serial} || "", 20, 255) ]}
  </span>
  <span class="label">PAS</span>
  <span class="formw">
     @{[ $cgi->textfield('PAS', $$entry{PAS} || "", 7, 255) ]}
  </span>
 </div>

 <p> </p>

 <h3 class="tablehead"> Order Information </h3>
 <div class="row2">
  <span class="label">Vendor</span>
  <span class="formw">
     @{[ $cgi->textfield('Vendor', $$entry{Vendor} || "", 30, 255) ]}
  </span>
  <span class="label">Quote</span>
  <span class="formw">
     @{[ $cgi->textfield('Quote', $$entry{Quote} || "", 30, 255) ]}
  </span>
 </div>

 <div class="row3">
  <span class="label">Price</span>
  <span class="formw">
     \$ @{[ $cgi->textfield('Price', $$entry{Price} || "", 8, 255) ]}
  </span>
  <span class="label">PO</span>
  <span class="formw">
     @{[ $cgi->textfield('PO', $$entry{PO} || "", 10, 255) ]}
  </span>
  <span class="label">Account #</span>
  <span class="formw">
     @{[ $cgi->textfield('Account', $$entry{Account} || "", 30, 255) ]}
  </span>
 </div>
 
 <div class="row3">
  <span class="label">Ordered</span>
  <span class="formw">
    @{[ $cgi->textfield('OrderDate', $$entry{OrderDate} || "", 11, 255)]} 
  </span>
  <span class="label">Received</span>
  <span class="formw">
    @{[ $cgi->textfield('Received', $$entry{Received} || "", 11, 255)]}
  </span>
  <span class="label">Support</span>
  <span class="formw">
     @{[ $cgi->textfield('SupportBegins', $$entry{SupportBegins} || "", 11, 255) ]} 
    -
     @{[ $cgi->textfield('SupportEnds', $$entry{SupportEnds} || "", 11, 255) ]}
  </span>
 </div>

 <div class="row1">
  <span class="label">History</span>
  <span class="formw">
     @{[ $cgi->textarea(-name=>'History', -default=>$$entry{History} || "", 
		        -rows=>5, -cols=>70, -maxlength=>65535,
			-wrap=>'physical') ]}
  </span>
 </div>

 <div class="submitbar"> @{[ $cgi->submit(-name=>"Submit") ]} </div>

</div>

HTML
  wantarray ? @return : join("\n", @return);
}

=back

There is no text() field at present.

=cut

###############################################################################
##### main() ##################################################################
###############################################################################

TCB::System->table_add($NAME, $FIELDS, $KEYS, $LIST, $ORDER, $ADMIN, 
			$REQUIRED, \&html, \&text);

=head1 NOTES

The fundamentals of this table were created before Charlie Brown, before
my time.  It was originally an msql table containing the same fields as
above; it's worked out pretty well, I suppose.  

=head1 TODO

Making a nice user interface, rather than just an administrative
interface, would be helpful.

=head1 REQUIREMENTS

Perl 5.6.1 or greater, DBIx::Frame, TCB::System

=head1 SEE ALSO

B<TCB::Equipment>, B<TCB::System>, B<DBIx::Frame>

=head1 AUTHOR

Tim Skirvin <tskirvin@ks.uiuc.edu>

=head1 HOMEPAGE

B<http://www.ks.uiuc.edu/Development/MDTools/tcb-equipment/>

=head1 LICENSE

This code is distributed under the University of Illinois Open Source
License.  See
C<http://www.ks.uiuc.edu/Development/MDTools/tcb-equipment/license.html>
for details.

=head1 COPYRIGHT

Copyright 2000-2004 by the University of Illinois Board of Trustees and
Tim Skirvin <tskirvin@ks.uiuc.edu>.

=cut

###############################################################################
### Version History ###########################################################
###############################################################################
# v0.2  	Tue Apr  2 15:35:25 CST 2002
### First version fully using DBI::Frame and commented.
# v1.00         Thu May 13 15:30:01 CDT 2004
### Ready to be a part of DBIx::Frame.
