$VERSION = "0.51";
package TCB::Conference::Register;
our $VERSION = "0.51";

# -*- Perl -*- 		Fri May 21 16:46:50 CDT 2004 
###############################################################################
# Written by Tim Skirvin <tskirvin@ks.uiuc.edu> Copyright 2003-2004, Tim
# Skirvin and UIUC Board of Trustees.
###############################################################################

=head1 NAME 

TCB::Conference::Register - the Registration table 

=head1 SYNOPSIS

  use TCB::Conference::Register;

see TCB::Conference for more information.

=head1 DESCRIPTION

The 'Register' table is the main table of the Conference database - users
used it to register for the conference, and the local adminstrators used
it to actually keep track of everyone.

This table contains the following fields (not all of which were actually
used):

 Internal Information (TINYTEXT fields, unless noted)
  ID            Unique numeric ID - auto-generated (INT)
  CreatedBy     Person who created this entry       \  Auto-created using
  ModifiedBy    Person who last modified this entry  \ $ENV{REMOTE_USER}
  CreateDate    Time this entry was created          / and current time in
  ModifyDate    Time this entry was last modified   /  html()

 General Inforamtion (TINYTEXT fields, unless noted)
  LastName	Last name of registrant
  FirstName	First name of registrant
  Institution	Institution the registrant came from
  Department	Department within the institution
  Title		Title within the department
 
 Contact Information (TINYTEXT fields, unless noted)
  Email		Email address of the registrant
  StreetAdd	Street Address of the registrant
  City		City of the registrant
  State		State of the registrant
  Zip		Zip/postal code of the registrant
  Country	Country of the registrant

 Housing Information (TINYTEXT fields, unless noted)
  Housing 	Type of housing requested
  Roommate	Name of a requested roommate
  ArrivalDate	Scheduled date of arrival (DATE)
  DepartDate 	Scheduled date of departure (DATE)
  HouseNotes	Additional notes on housing (TEXT)

 Events (general yes/no questions to ask; INT, unless noted)
  Event1	Experience with Unix 
  Event2	Experience with NAMD
  Event3	Experience with VMD
  Event4	Paid?
  Event5	Not used.

 Computer Accounts (TINYTEXT, unless noted)
  NCSALogin	Existing login name on NCSA systems
  SCSLogin	Existing login name on SCS systems
  Laptop	Will this registrant bring a laptop?  (INT)

 Additional Notes (TINYTEXT, unless noted)
  FoodType	Meant to track vegetarian/kosher/etc; not used.  (INT)
  Notes		General notes.  (TEXT)

 Administrative Fields (TINYTEXT, unless noted)
  GotMoney	When did we get their registration money?  (DATE)
  Approved	Are they approved?  (INT)
  AdminNotes	Administrative Notes (TEXT)
  
 NSF Fields (TINYTEXT, unless noted)
  Gender	Registrant gender (INT)
  Citizenship	Registrant's country of citizenship
  Race		Registrant's race

Key fields:	FirstName, LastName, Emal

List items:	Name (FirstName, LastName), Institution, Approved (Accepted)

Required:	LastName, FirstName, Email, Title, Institution, Department, 
		Housing, Gender, Citizenship

Default order:	LastName, FirstName, Email

Admin Fields:	Approved, Event4

No other table directly depends on this table.

Doesn't depend on any other table.

=head1 USAGE

=cut

###############################################################################
### Initialization ############################################################
###############################################################################
use vars qw( @ISA $FIELDS $KEYS $NAME $LIST $VERSION $REQUIRED $ADMIN $ORDER ); 
use strict;
use warnings;
use TCB::Conference;
use CGI;
unshift @ISA, "TCB::Conference";

###############################################################################
### Database Variables ########################################################
###############################################################################
$NAME = "Register";
$FIELDS = {

  # Non User-Modified Information
  'ID'          => 'INT NOT NULL PRIMARY KEY AUTO_INCREMENT',
  'CreatedBy'   => 'TINYTEXT',  'ModifiedBy'    => 'TINYTEXT',
  'CreateDate'  => 'DATE',      'ModifyDate'    => 'DATE',

  # General Information - Name, Title, Employer
  'LastName' 	=> 'TINYTEXT', 	'FirstName'	=> 'TINYTEXT',
  'Department'	=> 'TINYTEXT', 	'Title'		=> 'TINYTEXT',
  'Institution' => 'TINYTEXT',  

  # Contact Information
  'Email'	=> 'TINYTEXT', 	'StreetAdd'	=> 'TINYTEXT',	
  'City'	=> 'TINYTEXT',	'State'		=> 'TINYTEXT',
  'Zip'		=> 'TINYTEXT', 	'Country'	=> 'TINYTEXT',
  
  # Housing Info
  'Housing'	=> 'TINYTEXT',	'Roommate'	=> 'TINYTEXT',
  'ArrivalDate'	=> 'DATE',	'DepartDate'	=> 'DATE',	
  'HouseNotes'  => 'TEXT',

  # Events (up to 5)
  'Event1' => 'INT', 'Event2' => 'INT', 'Event3' => 'INT', 
  'Event4' => 'INT', 'Event5' => 'INT', 

  # Computer accounts
  'NCSALogon' 	=> 'TINYTEXT',	'SCSLogon' 	=> 'TINYTEXT',	
  'Laptop'	=> 'INT',	

  # Additional Notes
  'FoodType'	=> 'INT',	'Notes'		=> 'TEXT',

  # Administrative Fields
  'GotMoney'	=> 'DATE',	'Approved'	=> 'INT',
  'AdminNotes'  => 'TEXT',
  
  # Fields that the NSF needs
  'Gender'	=> 'INT', 	'Citizenship'	=> 'TINYTEXT',
  'Race'	=> 'TINYTEXT',

          };
$KEYS = [ 'FirstName', 'LastName', 'Email' ];
$LIST = [ { 'Name' => '$$FirstName$$ $$LastName$$' }, 
		'Institution', { 'Approved' => \&accepted } ];
$REQUIRED = [ 'LastName', 'FirstName', 'Email', 'Title', 'Institution', 
	      'Department', 'Housing', 'Gender', 'Citizenship' ];
$ADMIN    = [ 'Approved', 'Event4' ];
$ORDER    = [ 'LastName', 'FirstName', 'Email' ];

###############################################################################
##### Functions ###############################################################
###############################################################################

=head1 Internal Functions

=over 4

=item html ( ENTRY, TYPE, OPTIONS )

Returns the HTML necessary for database manipulation, as detailed in
DBI::Frame::CGI.

=cut

sub html {
  my ($self, $entry, $type, $options, @rest) = @_;
  my $cgi = new CGI; $entry ||= {};  $options ||= {};

  if (lc $type eq 'create') {
    $$entry{'CreatedBy'}  ||= $ENV{'REMOTE_USER'} || "unknown";
    $$entry{'CreateDate'} ||= $self->date_mysql(time);
    $$entry{'Gender'}     ||= '0';
    $$entry{'Citizen'}    ||= 1;
    $$entry{'FoodType'}   ||= 0;
    $$entry{'Laptop'}     ||= 0;
    $$entry{'Approved'}   ||= 0;
    $$entry{'Title'}      ||= "Graduate Student";
    $$entry{'Housing'}    ||= 'Hendrick';
  } elsif (lc $type eq 'search') {
    $$entry{'Gender'}     ||= ''; $$entry{'Citizen'}    ||= '';
    $$entry{'FoodType'}   ||= ''; $$entry{'Laptop'}     ||= '';
  } elsif (lc $type eq 'edit' || lc $type eq 'update') {
    $$entry{'ModifiedBy'} = $ENV{'REMOTE_USER'} || "unknown";
    $$entry{'ModifyDate'} = $self->date_mysql(time);
  } else { }

  my $add = $$options{'simple'} ? 0 : 1;	# Should we leave out 
						# some info?  Probably.
  
  my %genders = ( 1 => "Male", 0 => "Female" ) ;
  my %yesno   = ( 1 => "Yes", 0 => "No" ) ;
  my @title   = ( 'Undergraduate', 'Graduate Student', 'PostDoc', 
		  'Research Scientist', 'Assistant Professor', 
		  'Associate Professor', 'Professor', 'Other' );

  my %housing = (
	'Hendrick' => 'Hendrick House',
	'Other'    => 'Other/None (ie you will handle it)',
		);

  my $admin = $$options{'admin'} ? 1 : 0;

  if (lc $type eq 'search') {
    $genders{''} = "*"; $yesno{''} = "*"; 
    $housing{''} = "*"; 
    unshift @title, '';
  } 

  my @return = <<HTML;
<div class="basetable">
 <div class="row1">
  <span class="label">First Name</span>
  <span class="formw">
   @{[ $cgi->textfield('FirstName', $$entry{FirstName} || "", 60, 255) ]}
  </span>
 </div>

 <div class="row1">
  <span class="label">Last Name</span>
  <span class="formw">
   @{[ $cgi->textfield('LastName', $$entry{LastName} || "", 60, 255) ]}
  </span>
 </div>

 <div class="row1">
  <span class="label">E-Mail</span>
  <span class="formw">
   @{[ $cgi->textfield('Email', $$entry{Email} || "", 60, 255) ]}
  </span>
 </div>

 <div class="row1">
  <span class="label">Company/ Institution</span>
  <span class="formw">
   @{[ $cgi->textfield('Institution', $$entry{Institution} || "", 60, 255) ]}
  </span>
 </div>

 <div class="row1">
  <span class="label">Department</span>
  <span class="formw">
   @{[ $cgi->textfield('Department', $$entry{Department} || "", 60, 255) ]}
  </span>
 </div>

 <div class="row1">
  <span class="label">Title</span>
  <span class="formw">
   @{[ $cgi->popup_menu('Title', \@title, $$entry{Title} || "") ]}
  </span>
 </div>

 <div class="row1">
  <span class="label">Street Address</span>
  <span class="formw">
   @{[ $cgi->textarea(-name=>'StreetAdd', -default=>$$entry{StreetAdd} || "",
                        -rows=>3, -cols=>50, -maxlength=>255,
                        -wrap=>'off') ]}
  </span>
 </div>

 <div class="row4">
  <span class="label">City</span>
  <span class="formw">
     @{[ $cgi->textfield('City', $$entry{'City'} || "", 20, 255 )]} 
  </span>
  <span class="label">State</span>
  <span class="formw">
     @{[ $cgi->textfield('State', $$entry{'State'} || "", 4, 255 )]} 
  </span>
  <span class="label">Zip</span>
  <span class="formw">
     @{[ $cgi->textfield('Zip', $$entry{'Zip'} || "", 10, 255 )]} 
  </span>
  <span class="label">Country</span>
  <span class="formw">
     @{[ $cgi->textfield('Country', $$entry{Country} || "", 10, 255) ]} 
  </span>
 </div>

 <div class="fullrow">
   <b>Applicants are responsible for obtaining a proper visa to the 
    USA; <br />we are not able to assist with this or any related INS matters</b>
 </div>

 <div class="row1">
  <span class="label">Citizenship</span>
  <span class="formw">
   @{[ $cgi->textfield('Citizenship', $$entry{Citizenship} || "", 60, 255) ]} 
  </span>
 </div>


 <div class="row2">
  <span class="label">Gender</span>
  <span class="formw">
    @{[ $cgi->radio_group(-name => 'Gender', -values => [ keys %genders ],
        -default => $$entry{Gender} || "", -labels => \%genders ) ]}
  </span>
  <span class="label">Race (Optional)</span>
  <span class="formw">
   @{[ $cgi->textfield('Race', $$entry{Race} || "", 20, 255) ]} 
  </span>
 <div>

 <div class="fullrow">
    Registration fee covers a shared room in Hendrick House. 
 </div>

 <div class="row1">
  <span class="label">I will stay in</span>
  <span class="formw">
    @{[ $cgi->popup_menu('Housing', [ sort keys %housing ], 
			 $$entry{'Housing'} || "", \%housing ) ]}
  </span>
 </div>

 <div class="row1">
  <span class="label">Roommate Preference</span>
  <span class="formw">
   @{[ $cgi->textfield('Roommate', $$entry{Roommate} || "", 30, 255) ]}
   (full name)
   </span>
 </div>

 <div class="fullrow">
  Outline your relevant experience and your expectations from this program
  <br />
     @{[ $cgi->textarea(-name=>'Notes', -default=>$$entry{Notes} || "",
                        -rows=>5, -cols=>60, -maxlength=>255,
                        -wrap=>'physical') ]}
 </div>
HTML

  push @return, addhtml  ($self, $entry, $type, $options, @rest) if $add;
  push @return, adminhtml($self, $entry, $type, $options, @rest) if $admin;

  push @return, <<HTML;

 <div class="submitbar"> 
  @{[ iftext(!$admin, "<i>This form cannot be modified after submission<br />Please review carefully before submitting</i>") ]}
  @{[ $cgi->submit(-name=>"Submit") ]} </div>
</div>
HTML
  wantarray ? @return : join("\n", @return);
}

sub iftext { if (shift) { return join(" ", @_) } else { "" } }

=item text ( )

Not currently populated.

=cut

sub text { } 

=item adminhtml ( ENTRY, TYPE, OPTIONS )

Called within html() only if the 'admin' flag is set in C<OPTIONS>, this
offers the fields that shouldn't show up in all searches.

=cut

sub adminhtml {
  my ($self, $entry, $type, $options, @rest) = @_;
  my $cgi = new CGI;
  $entry ||= {};  $options ||= {};

  my %yesno   = ( 1 => "Yes", 0 => "No" ) ;
  my %approved = ( -1 => 'Rejected', 0 => 'Waitlist', 1 => 'Accepted',
		   2 => 'Confirmed' );
  if (lc $type eq 'search') { $approved{''} = "*"; $yesno{''} = '*' }

  return <<HTML;
 <!-- Admin Information - only appears for administrative use -->

 <hr style="clear: both" />
 <h3 class="tablehead">Admin Information</h3>

 <div class="row1">
  <span class="label">Approved?</span>
  <span class="formw">
   @{[ $cgi->radio_group( -name => 'Approved', 
       -values => [ sort keys %approved ], 
       -default => $$entry{Approved} || "", -labels => \%approved,
       -linebreak => 0 ) ]}
  </span>
 </div>

 <div class="row1">
  <span class="label">Paid?</span>
  <span class="formw">
   @{[ $cgi->radio_group('Event4', [ keys %yesno ],
                                 $$entry{Event4} || "", 0, \%yesno) ]}
  </span>
 </div>

 <div class="row1">
  <span class="label">Admin Notes</span>
  <span class="formw">
   @{[ $cgi->textarea(-name=>'AdminNotes', -default=>$$entry{AdminNotes} || "",
                     -rows=>5, -cols=>60, -maxlength=>255, -wrap=>'physical') ]}
  </span>
 </div>
 <div style="clear: both">
  <div class="modbox">
    @{[ $self->created_html( $$entry{CreateDate} || "", 
                             $$entry{ModifyDate} || "",
                             $$entry{CreatedBy}  || "", 
                             $$entry{ModifiedBy} || "" ) ]}
  </div>
 </div>
HTML

}

=item addhtml ( ENTRY, TYPE, OPTIONS )

Called within html() only if the 'simple' flag is not set in C<OPTIONS>,
this offers the fields that we decided weren't always important.

=cut

sub addhtml {
  my ($self, $entry, $type, $options, @rest) = @_;
  my $cgi = new CGI;    $entry ||= {};    $options ||= {};

  my %yesno   = ( 1 => "Yes", 0 => "No" ) ;
  my %foods   = ( 1 => "Vegetarian", 0 => "Normal", -1 => "See Notes" );
  my %experience = ( 0 => "None", 1 => "Moderate", 2 => "High" );

  if (lc $type eq 'search') { $yesno{''} = '*'; $foods{''} = '*'; 
			      $experience{''} = '*' }

  return <<HTML;

 <!-- Additional Information - semi-public, but not entered by users 
      when they're registering -->

 <hr style="clear: both" />

 <h3 class="tablehead">Second-Round Information</h3>

 <div class="row1">
  <span class="label">Dietary Requirements? </span>
  <span class="formw">
    @{[ $cgi->radio_group(-name => 'FoodType', -values => [ keys %foods ],
              -default => $$entry{FoodType} || "", -labels => \%foods,
	      -linebreak=>0 ) ]}
  </span>
 </div>

 <div class="row1">
  <span class="label">Rate your experience with Unix</span>
  <span class="formw">
       @{[ $cgi->radio_group(-name => 'Event1', -values => [ keys %experience ],
                -default => $$entry{Event1} || "", -labels => \%experience ) ]}
	<br />
  </span>
 </div>

 <div class="row1">
  <span class="label">Rate your experience with NAMD</span>
  <span class="formw">
       @{[ $cgi->radio_group(-name => 'Event2', -values => [ keys %experience ],
                -default => $$entry{Event2} || "", -labels => \%experience ) ]}
  </span>
 </div>
  
 <div class="row1">
  <span class="label">Rate your experience with VMD</span>
  <span class="formw">
       @{[ $cgi->radio_group(-name => 'Event3', -values => [ keys %experience ],
                -default => $$entry{Event3} || "", -labels => \%experience ) ]}
  </span>
 </div>

HTML
}

=item accepted ( INFO ) 

Returns a nice string interpreting $$info{'Approved'}.  Used by the list()
functions.

=cut

sub accepted {
  my ($self, $info) = @_;
  $info ||= {};
  return "(not set)" unless defined $$info{'Approved'};
  my $approved = $$info{'Approved'} || 0;
  return "Confirmed" if $approved eq 2;
  return "Accepted"  if $approved eq 1;
  return "Rejected"  if $approved eq -1;
  return "Waitlist"  if $approved eq 0;
  return "(not set)"; 
}

=back

=cut

###############################################################################
##### main() ##################################################################
###############################################################################

TCB::Conference->table_add($NAME, $FIELDS, $KEYS, $LIST, $ORDER, 
			  $ADMIN, $REQUIRED, \&html, \&text);

=head1 NOTES

=head1 TODO

=head1 REQUIREMENTS

Perl 5.6.1 or better, DBIx::Frame, TCB::Conference.

=head1 SEE ALSO

B<TCB::Conference>, B<DBIx::Frame>, B<DBIx::Frame::CGI>

=head1 AUTHOR

Tim Skirvin <tskirvin@ks.uiuc.edu>

=head1 HOMEPAGE

B<http://www.ks.uiuc.edu/Development/MDTools/tcb-conference/>

=head1 LICENSE

This code is distributed under the University of Illinois Open Source
License.  See
C<http://www.ks.uiuc.edu/Development/MDTools/tcb-conference/license.html>
for details.

=head1 COPYRIGHT

Copyright 2003-2004 by the University of Illinois Board of Trustees and
Tim Skirvin <tskirvin@ks.uiuc.edu>.

=cut

###############################################################################
### Version History ###########################################################
###############################################################################
# v0.50		Fri May 21 16:44:44 CDT 2004 
### Starting work on actually commenting this and getting it ready for
### distribution.
# v0.51		Wed May 26 11:49:17 CDT 2004 
### Corrected for DBIx::Frame.
