#!/usr/local/bin/perl 
# -*- Perl -*- Wed Feb 04 13:22:56 CST 2004 
###############################################################################
# Written by Tim Skirvin <tskirvin@ks.uiuc.edu>
# Copyright 2000-2004, Tim Skirvin and UIUC Board of Trustees.
# Redistribution terms are below.
###############################################################################

use vars qw( $VERSION $DEBUG $TEST %PASSWORDS $NOCHECK );
$VERSION = "1.12";

###############################################################################
### Default Configuration #####################################################
###############################################################################

$DEBUG = 1; 	    ## Print debugging information?  Can also be set with '-V'
$TEST  = 0;	    ## Just test the script?  Can also be set with '-t'
$NOCHECK = 0;	    ## Don't check the password?  Can be set with '-n'; not 
		    ## generally advertised, however

###############################################################################
### main() ####################################################################
###############################################################################

use strict;				# With perl
use Getopt::Std;			# With perl
# use lib '/home/tskirvin/dev/tb-adduser';
use TCB::AddUser "prompt_password";	# Stand-alone module
use vars qw( $opt_h $opt_v $opt_a $opt_s $opt_y $opt_w $opt_V $opt_t $opt_n );

delete $ENV{PATH};              # For setgid
$0 =~ s%.*/%%;			# Shorten program name
getopts('hvasywVtn');

$DEBUG++  if $opt_V;		# Verbose mode - print debugging information
$TEST++   if $opt_t;		# Testing mode - don't actually change PW's
$NOCHECK++ if $opt_n;		# Don't check the passwords

$PASSWORDS{'htpasswd'}++  if ($opt_w || $opt_a);	# Set web password?
$PASSWORDS{'yppasswd'}++  if ($opt_y || $opt_a);	# Set YP password?
$PASSWORDS{'smbpasswd'}++ if ($opt_s || $opt_a);	# Set SMB password?

Version() if $opt_v;
Usage()   if $opt_h || !(keys %PASSWORDS >= 1);

$TCB::AddUser::DEBUG = 1 if $DEBUG;
$TCB::AddUser::TEST  = 1 if $TEST;

### Get username
my $username;
if ($< eq 0) { 
  my $base = shift;  
  ($username) = $base =~ /^(.*)$/;	# Untaint the input nae
} else { $username = (getpwuid($<))[0] }
warn "Updating for username: '$username'\n";
Exit('No username') unless $username;

### Set passwords - YP, Samba, Web
my ($oldyp, $oldsmb, $newpass);
if ($PASSWORDS{'yppasswd'}) {
  $oldyp = $< ne 0 ? prompt_password("Old YP (UNIX) password", -1) : ""; 
} 
if ($PASSWORDS{'smbpasswd'}) {
  $oldsmb = $< ne 0 ? prompt_password("Old SMB (Windows) password", -1) : "";	
}

my @list;  
push @list, 'YP'  if $PASSWORDS{'yppasswd'};  
push @list, 'SMB' if $PASSWORDS{'smbpasswd'};  

if ($PASSWORDS{'yppasswd'} || $PASSWORDS{'smbpasswd'}) { 
  my $string = join("/", @list);
  print "Make sure this password is secure!\n";
  $newpass = prompt_password("New $string password", $NOCHECK ? 0 : 1 );
}

if ($PASSWORDS{'yppasswd'}) {
  warn "Setting YP password for '$username'\n" if $DEBUG;
  TCB::AddUser->set_yppasswd( $username, $newpass, $oldyp )
	or warn "Failed to set YP password\n";
}
if ($PASSWORDS{'smbpasswd'}) {
  warn "Setting SMB password for '$username'\n" if $DEBUG;
  TCB::AddUser->set_smbpasswd( $username, $newpass, $oldsmb )
	or warn "Failed to set SMB password\n";
}

if ($PASSWORDS{'htpasswd'}) {
  warn "Setting web password for '$username'\n" if $DEBUG;
  warn "Make sure this is different than your main password\n";
  TCB::AddUser->set_htpasswd( $username, prompt_password("New web password", 0) )
	or warn "Failed to set web password\n";
}

exit 0;

###############################################################################
### Internal Functions ########################################################
###############################################################################

### Usage ()
# Prints out a short help file and exits.
sub Usage { 
  print <<EOM;
$0 v$VERSION
Password setting utility for the TCB group
Usage: $0 [-hv] [-Vt] [-ysw|a] [username]

$0 sets the various passwords available within the TCB group.  There are
three main passwords: YP (Unix logins), SMB (Windows file shares), and Web
(remote web server access).  

	-h	Prints this message and exits
	-v	Prints out the version number and exits
	-V	Verbose.  Prints out debugging information. 
	-t	Test mode.  Don't actually set the passwords, just 
	 	prompt for them.
	-y	Sets the YP password.  
	-s	Sets the SMB password.
	-w	Sets the web password.
	-a 	Sets all of the above passwords.
     username	Username to modify the password(s) of.  Only works if running
		as root, otherwise runs as yourself.

Examples:
  $0 -s		Sets SMB password
  $0 -y -w	Sets YP and web passwords
  $0 -a   	Sets all passwords (YP, SMB, web)
EOM

  Exit();
}

### Version ()
# Prints out the version number and exits
sub Version { Exit("$0 v$VERSION"); }

### Exit ( REASON )
# Exits the program cleanly with a warned error message
sub Exit { warn "@_\n" if (scalar @_);  exit; }

=head1 NAME

passwd.pl - a script for modifying TB system passwords

=head1 SYNOPSIS

  passwd.pl [-hv] [-Vt] [-ysw|a] [username]

=head1 DESCRIPTION

passwd.pl sets the various passwords available within the TB group.
There are three main passwords: YP (Unix logins), SMB (Windows file
shares), and Web (remote web server access).

        -h      Prints this message and exits
        -v      Prints out the version number and exits
        -V      Verbose.  Prints out debugging information.
        -t      Test mode.  Don't actually set the passwords, just 
		prompt for them.
        -y      Sets the YP password.
        -s      Sets the SMB password.
        -w      Sets the web password.
        -a      Sets all of the above passwords.
     username	Username to modify the password of.  Only works if running
		as root, otherwise runs as yourself.

=head1 REQUIREMENTS

Getopt::Std, TCB::AddUser 

=head1 SEE ALSO

B<TCB::AddUser>, B<webpasswd>

=head1 AUTHOR

Tim Skirvin <tskirvin@ks.uiuc.edu>

=head1 LICENSE

Currently for internal TB use only - and, honestly, that's probably where
it's going to stay.  Code snippets may be used wherever you'd like, however.

=head1 COPYRIGHT

Copyright 2001-2004 by the University of Illinois Board of Trustees and Tim
Skirvin <tskirvin@ks.uiuc.edu>.

=cut


##### Version History 
# v0.1a		Thu Oct 11 10:46:22 CDT 2001
### Initial version
# v1.0 		Thu Oct 11 13:46:37 CDT 2001
### Commented, included as part of TB::AddUser
# v1.01		Mon Oct 15 15:16:00 CDT 2001
### Fixed bug with setgid - had to delete $ENV{PATH}
# v1.02 	Mon Nov 12 17:10:04 CST 2001
### Untainted 'username' when running as root
# v1.10		Wed Feb 04 13:23:05 CST 2004 
### Updated for use with new TB::AddUser (v1.02).  Added $NOCHECK stuff,
### but didn't document it.  Only asks for the new password once if you're 
### changing both YP and SMB passwords.
# v1.11		Wed Feb 04 14:03:25 CST 2004 
### Got the 'NOCHECK' thing wrong.
# v1.12		Thu Apr 22 14:26:02 CDT 2004 
### Now using TCB::AddUser
