$VERSION = "1.05";
package TCB::AddUser;
our $VERSION = "1.05";

# -*- Perl -*- 		Mon Jun 14 13:41:18 CDT 2004 
###############################################################################
# Written by Tim Skirvin <tskirvin@ks.uiuc.edu>
# Copyright 2001-2004, Tim Skirvin and UIUC Board of Trustees.  Redistribution 
# terms are below.
###############################################################################

use vars qw( $DEBUG $TEST $SKELETON @GROUPS @MAIL $PPAT $ALIASDIR $YPFILES 
	     $HTPASSWD $YPPASSWD $SMBPASSWD @YPSERVER $YPSERVER $HTPASSWDFILE 
             $CHECKPASSWD $TEMPLATE $SHELL $SMBHOST $HOMEDIRS $NPASSWD );

###############################################################################
### Administrator Configuration ###############################################
###############################################################################

## Print debugging information?  This variable is shared.
$DEBUG    = 0;

## Are we just testing?  This variable is shared, and should probably be
## off by default.
$TEST     = 0;

## Where is the YP server?  We have to be logged onto this machine to perform
## a lot of the operations.  
@YPSERVER = qw( taipei edmonton );		# taipei.ks.uiuc.edu
$YPSERVER = join(' or ', @YPSERVER);
$YPFILES  = "/etc/ypfiles";

## What table are the home directories stored in?
$HOMEDIRS = "auto.home";

## Valid default user groups 
@GROUPS   = qw( tbres tbadm tbguest );

## Where are the mail aliases kept?
$ALIASDIR = "/Common/mail_aliases";

## Valid mail aliases - there are more available, which can be added by hand.
@MAIL     = qw( g_students g_faculty g_admin g_staff sysadmin 
	        ks_research g_devel g_guests );

## Where is the skeleton directory for new users?
$SKELETON    = "/Projects/system/newuser/skeleton";

## We choose the valid home directory partitions by looking through the
## 'auto.HomeDirs' YP table, and matching to this pattern, using
## prompt_partitions().  You should modify this pattern as you feel
## necessary.
$PPAT        = "riga|rolla";

## Programs used to change passwords.  HTPASSWD is for changing the web
## password, SMBPASSWD is used for changing the Windows password, and
## YPPASSWD is used for changing the Unix password.  CHECKPASSWD is used
## to confirm that the password is fairly secure; NPASSWD is a replacement
## password program that works better than yppasswd in this case.

$HTPASSWD    = "/usr/local/bin/htpasswd";
$SMBPASSWD   = "/usr/local/bin/smbpasswd";
$YPPASSWD    = "/usr/bin/yppasswd";
$CHECKPASSWD = "/usr/local/bin/checkpassword -o";
$NPASSWD     = "/usr/local/bin/npasswd";

## Where is the web password file?  This isn't assumed by its utility.
$HTPASSWDFILE = "/home/webserver/security/htpasswd";

## Which host is the SMB server?
$SMBHOST     = "TBFILES";

## Web template file, used by create_webpage().
$TEMPLATE    = "/Projects/system/newuser/files/sample.html";

## Default user shell.
$SHELL       = "/usr/local/bin/tcsh";

###############################################################################
### Initialization Code #######################################################
###############################################################################

use strict;
use File::NCopy qw(copy);	# On CPAN
use Net::NIS::Table;		# On CPAN
use Filesys::DiskSpace;		# On CPAN
use Expect;			# On CPAN, may have to install manually
use File::Find;			# In base perl package
use Term::ReadKey;		# In base perl package
use IPC::Open3;			# In base perl package
use FileHandle;			# In base perl package
use Sys::Hostname;		# In base perl package
use Exporter;			# In base perl package

use vars qw( @EXPORT @EXPORT_OK @ISA );
@ISA       = qw( Exporter );
@EXPORT    = qw( );
@EXPORT_OK = qw( chown_by_name chown_recursive prompt prompt_multiline 
	         prompt_password check_password );

###############################################################################
### Initial Documentation #####################################################
###############################################################################

=head1 NAME 

TCB::AddUser - a module for new user creation and user management

=head1 SYNOPSIS

  my ($addedby,$username,$uid,$group,$gid,$partition,$mailgroups,$passwords)
	= TCB::AddUser->prompt_admininfo;
  my ($realname, $address, $phone, $homedept, $position, $boss)
        = TCB::AddUser->prompt_userinfo;

  TCB::AddUser->add_yppasswd($username, $uid, $gid, $realname,
                "/HomeDirs/$partition/$username");
  TCB::AddUser->add_hometable($username, $partition);
  TCB::AddUser->push_yptables;

  TCB::AddUser->create_homedir($username, $partition) ||print 
                                warn "Couldn't create home directory\n";
  TCB::AddUser->populate_homedir($username, $partition) ||
                                warn "Couldn't populate home directory\n";
  TCB::AddUser->create_webpage($partition, $username, $realname) ||
                                warn "Couldn't create homepage\n";

  my ($yppasswd, $smbpasswd, $htpasswd) = TCB::AddUser->get_passwords(
        $$passwords{'System'} || $$passwords{'Samba'}, $$passwords{'Web'});
  ($htpasswd  && TCB::AddUser->set_htpasswd($username, $htpasswd))
        or warn "Web password not set\n";
  ($smbpasswd && TCB::AddUser->set_smbpasswd($username, $smbpasswd))
        or warn "SMB password not set\n";
  ($yppasswd  && TCB::AddUser->set_yppasswd($username, $yppasswd))
        or warn "YP password not set\n";

  TCB::AddUser->add_mailalias($username, @{$mailgroups});

Note that a fair number of administrative defaults should be configured at
install-time.

=head1 DESCRIPTION

TCB::AddUser is a module for creating and, to some extent, maintaning user
accounts on TCB systems.  It is essentially a converstion and rewrite of 
the Group's previous 'adduser' script, which was written by Charles Brown 
back in 1997.  A separate script actually takes care of the rest of the
functionality.
  
=head1 USAGE

There are three types of functions in this module: Information Gathering,
System Modification, and Shared Functions.  Each is described below.

=cut

###############################################################################
### Information Gathering #####################################################
###############################################################################

=head2 Information Gathering

These functions are internal to TCB::AddUser, and are not exported.  They
are used to gather the information necessary to add a user.  

=over 4

=item prompt_userinfo ( )

Prompts for the user's personal information, which is generally provided
by the user.  Returns an array containing the elements REALNAME, ADDRESS,
PHONE, HOMEDEPT, POSITION, and BOSS.  The ADDRESS element is an array
reference containing the full address, the rest are scalars.

=cut

sub prompt_userinfo { 
  my ($self) = @_;
  my ($realname, $address, $phone, $homedept, $position, $boss);
  while (1) {
    $realname = prompt('Real Name');
    $address  = prompt_multiline('Full Address') || [];
    $phone    = prompt('Home Phone');
    $homedept = prompt('Home Department');
    $position = prompt('Position (grad, postdoc, etc)');
    $boss     = prompt('Advisor/Sponsor');
  
    last if prompt('Is the above information correct?', 1, 'y', 'n') eq 'y';
  }
  ($realname, $address, $phone, $homedept, $position, $boss);
}

=item prompt_admininfo ( )

Prompts for the administrative information, which is generally provided by
Gila and Tim.  Returns an array containing the elements ADDEDBY, USERNAME, UID,
GROUP, GID, PARTITION, MAILGROUPS, and PASSWORDS.  MAILGROUPS is an array
reference, PASSWORDS is a hash reference, and the rest are scalars.

=cut

sub prompt_admininfo { 
  my ($self) = @_;
  my ($addedby,$username,$uid,$group,$gid,$partition,$mailgroups,%passwords);
  while (1) {
    $addedby  = prompt('User Added By (your login)');
    $username = prompt('Username');
    $uid = $self->newuid($username);
    unless ($uid) { print "Username '$username' already exists\n"; next }
    print "UID: $uid\n";

    $group    = prompt('Group', 1, @GROUPS);
    $gid = $self->getgid($group);
    unless ($gid) { print "Invalid Group: '$group'\n"; next }
    print "GID: $gid\n";
  
    $self->prompt_partitions();
    my @partitions = $self->partitions();
    $partition = prompt('Home directory partition', 1, @partitions);
  
    $mailgroups = [];
    print "Entering a list of mail groups, type 'exit' to stop\n";
    while (1) {
      my $answer = prompt('  Mail groups', 1, @MAIL, 'exit');
      $answer eq 'exit' ? last : push @{$mailgroups}, $answer;
    }
    print "Final mail groups: ", join(', '), @{$mailgroups}, "\n";

    $passwords{'System'}++;
    $passwords{'Samba'}++ if prompt("Set Samba password?", 1, 'y', 'n') eq 'y';
    $passwords{'Web'}++   if prompt("Set Web password?", 1, 'y', 'n') eq 'y';

    last if prompt('Is the above information correct?', 1, 'y', 'n') eq 'y';
  }
  ($addedby,$username,$uid,$group,$gid,$partition,$mailgroups,\%passwords);
}

=item prompt_partitions ( )

Prompts for the partition on which to install the user's home directory.  
This is invoked by B<prompt_admininfo()>, but may be useful for other
purposes.

=cut

sub prompt_partitions {
  my ($self) = @_;
  my $homedirs = Net::NIS::Table->new("auto.HomeDirs")->list;
  return "" unless ($homedirs && ref $homedirs);
  my @partitions = $self->partitions;
  print "Valid partitions and disk usage are:\n";
  print "-------------------------------------------------------------------\n";
  foreach my $amdir (sort @partitions) {
    next unless "$amdir $$homedirs{$amdir}" =~ /$PPAT/;
    my $fulldir = $$homedirs{$amdir};  $fulldir =~ s/^.*\s+//g;
    push @partitions, $amdir;
    my $dir = "/HomeDirs/$amdir";   chdir($dir);
    my ($fs_type, $fs_desc, $used, $avail, $fused, $favail) = df $dir;
    my $left = $avail / ($used + $avail) * 100;
    printf("%-20s %-31s %3.0f%% free (%6d/%6dMB)\n", $dir, $fulldir,
        $left, $used / 1024, ($used + $avail) / 1024);
  }
  print "-------------------------------------------------------------------\n";
  @partitions;
}

=item get_passwords ( [GETYP/SMB [, GETWEB]] )

Asks for the various passwords necessary for the Group, using
B<prompt_password>.  Returns an array containing the passwords for YP,
SMB, and Web, respectively.  Note that it only asks for two; YP and SMB
should be the same.

If C<GETYP/SMB> or C<GETWEB> is set, then it only asks for those that are
set.  Otherwise, it defaults to ask for both.

=cut

sub get_passwords {
  my ($self, @ask) = @_;
  unless (scalar @ask) { @ask = (1,1); }
  
  my $passwd = undef;
  my $htpasswd = undef;

  if ($ask[0]) {
    print "Make sure this password can't be cracked!\n";
    $passwd = prompt_password('Main password', 1);
  }

  if ($ask[1]) {
    print "Your web password should be different than your main password.\n";
    $htpasswd = prompt_password('Web password');
  }

  ($passwd, $passwd, $htpasswd);
}

=back

=cut

###############################################################################
### System Modification #######################################################
###############################################################################

=head2 System Modification

These functions actually modify the system, and are not exported.  They 
generally do not operate directory if $TEST is set; instead, they offer a
few diagnostics.

=over 4

=item add_mailalias ( USERNAME, ALIAS [, ALIAS [...]] )

Adds USERNAME to the mail alias ALIAS.  Must be root, and doesn't make any
changes if $TEST is set.

=cut

sub add_mailalias {
  my ($self, $username, @alias) = @_;
  return 0 unless $username;
  my @added;
  warn "Editing mail aliases: ", join(', ', @alias), "\n" if $DEBUG;
  foreach (@alias) {
    my $alias = "$ALIASDIR/$_";
    my %entries;

    warn "Reading $alias\n" if $DEBUG;
    # Read the current alias file
    open(ALIASFILE, $alias) 
	|| ( warn "Can't open $alias: $!\n" and next );
    while (<ALIASFILE>) { foreach (split(/,|\s+/)) { $entries{$_}++ if $_ } };
    close(ALIASFILE);

    # Adding username to the list of entries
    $entries{$username}++;

    if ($TEST) { warn "Skipping edits of mail files.\n"; next }

    unless (-w $alias) { print "Can't write to $alias.\n"; next }
    warn "Writing $alias\n" if $DEBUG;

    # Make a backup
    copy($alias, "$alias.bak");

    # Write the alias file
    open(ALIASFILE, ">$alias") 
	|| ( print "Can't modify mail alias $alias: $!\n" and next );
    print ALIASFILE join(',', sort keys %entries), "\n";
    close(ALIASFILE);
    
    push @added, $alias;
  }
  scalar @added;
}

=item add_yppasswd ( USERNAME, UID, GID, FULLNAME, HOMEDIR [, SHELL] )

Creates a password entry.  The above fields are all required, and should
be self-explanatory.  Returns 1 if successful in adding the field, 0 if
the entry already exists or the function is otherwise unsuccessful.
Doesn't write out if $TEST is set.

=cut

sub add_yppasswd {
  my ($self, $username, $uid, $gid, $fullname, $homedir, $shell) = @_;
  $shell ||= $SHELL;
  unless ($username && $uid && $gid && $fullname && $homedir && $shell) { 
    warn "Need all information to create password entry:\n";
    warn " USERNAME, UID, GID, NAME, HOMEDIR [, SHELL]\n";
    return 0;
  }
  my $entry = join(':', $username, '', $uid, $gid, $fullname, $homedir, $shell);
  warn "Entry: $entry\n" if $DEBUG;

  # Can only do this if we're on YPSERVER
  unless ( _check_hostname(hostname, @YPSERVER) ) {
    warn "Must be logged into $YPSERVER to update password file\n";
    return 0;
  }

  # Make sure the entry doesn't already exist
  open(PASSWD, "$YPFILES/passwd")
	or ( warn "Couldn't read from password file: $!\n" and return 0 );
  while (<PASSWD>) {
    if (m/^$username:/) { warn "Entry already exists!\n";  return 0 }
  }
  close(PASSWD);

  if ($TEST) { warn "Not writing to password file\n";  return 1; }
  
  open(PASSWD, ">>$YPFILES/passwd") 
	or ( warn "Couldn't write to password file: $!\n" and return 0 );
  print PASSWD "$entry\n";
  close(PASSWD);

  1;
}

=item add_hometable ( USERNAME, PARTITION )

Creates a home directory automount entry in the home directory table.
The above fields are required, and should be self-explanatory.  Returns 1
if successful in adding the entry, 0 if the entry already exists or the
function is otherwise unsuccessful.  Doesn't write out if $TEST is set.

=cut

sub add_hometable {
  my ($self, $username, $partition) = @_;
  unless ($username && $partition) { 
    warn "Need all information to create password entry: USERNAME, PARTITION\n";
    return 0;
  }

  my $partitions = partitions();
  my $fullhomedir = $$partitions{$partition};
  unless ($fullhomedir) { 
    warn "No '$partition' partition found\n";
    return 0;
  }

  my $entry = "$username	$fullhomedir/$username";
  warn "Entry: $entry\n" if $DEBUG;

  # Can only do this if we're on YPSERVER
  unless ( _check_hostname(hostname, @YPSERVER) ) {
    warn "Must be logged into $YPSERVER to update $HOMEDIRS file\n";
    return 0;
  }

  if ($TEST) { warn "Not writing to $HOMEDIRS\n";  return 1; }

  # Make sure the entry doesn't already exist
  open(HOME, "$YPFILES/$HOMEDIRS")
	or ( warn "Couldn't read from $HOMEDIRS: $!\n" and return 0 );
  while (<HOME>) {
    if (m/^$username:/) { warn "Entry already exists!\n";  return 0 }
  }
  close(PASSWD);
  
  open(HOME, ">>$YPFILES/$HOMEDIRS")
	or ( warn "Couldn't write to $HOMEDIRS: $!\n" and return 0 );
  print HOME "$entry\n"; 
  close(HOME);
  
  1;
}

=item push_yptables ()

Pushes the YP tables, unless $TEST is set.  You'll want to watch the
status yourself.

=cut

sub push_yptables { 
  # Can only do this if we're on YPSERVER
  unless ( _check_hostname(hostname, @YPSERVER) ) {
    warn "Must be logged into $YPSERVER to update password file\n";
    return 0;
  }
  $TEST ? warn "Testing, won't push tables\n"
	: system("cd /etc/ypfiles; make yp; make checkin") ;
	# : system("cd /var/yp; make") ;
  1;
}

=item create_homedir ( USERNAME, PARTITION )

Creates the home directory for USERNAME, which must already be in the YP
maps.  The home directory goes into PARTITION.  Note that this does not
actually populate the home directory, you'll want B<populate_homedir()> for
that.  Returns 1 if successful, 0 otherwise.  Returns 1 and doesn't create
the directory if $TEST is set.

=cut

sub create_homedir {
  my ($self, $username, $partition) = @_;
  return 0 unless ($username && $partition);
  
  my $fulldir = "/HomeDirs/$partition/$username";

  print "Creating directory $fulldir\n" if $DEBUG;
  if (-d $fulldir) { warn "Directory $fulldir already exists!\n" }
  else { 
    $TEST ? warn "Skipping mkdir\n" && return 0
          : mkdir($fulldir,0777) || ( warn "mkdir failed\n" and return 0 ); 
  }
  
  print "Fixing permissions of home directory\n";
  unless ($TEST) {
    chmod(0755, $fulldir) || (warn "chmod failed\n" and return 0);
    chown_by_name($username, $fulldir) || (warn "chown failed\n" and return 0);
  }

  1;
}

=item populate_homedir ( USERNAME, PARTITION [, SKELETON] )

Populates the home directory for USERNAME (in PARTITION) with the files in
the directory SKELETON.  If B<SKELETON> is not offered, it uses the
default value set in the module.

=cut

sub populate_homedir {
  my ($self, $username, $partition, $skeleton) = @_;
  $skeleton ||= $SKELETON;
  return 0 unless ($username && $partition && $skeleton);

  my $fulldir = "/HomeDirs/$partition/$username";
  foreach ($fulldir, $skeleton) {
    unless (-d $_) { warn "Directory $_ doesn't exist!\n"; return 0 }
  }
  
  print "Setting up login skeleton\n";
  my $file = File::NCopy->new('recursive' => 1, );
  opendir(SKELETON, $skeleton) 
		or (warn "Couldn't open $skeleton: $!\n" and return 0);
  foreach (readdir SKELETON) {
    next if $_ eq '.';  next if $_ eq '..';
    if ($TEST) { warn "Skipping copy and chown for '$_'\n"; next }
    $file->copy( "$SKELETON/$_", $fulldir ) || 
		(warn "Couldn't copy $SKELETON/$_: $!\n" and return 0);
  }
  closedir(SKELETON);
  unless($TEST) { chown_recursive($username, $fulldir) }
  
  1;
}

=item create_forward( PARTITION, USERNAME, ADDRESS )

Creates a .forward message in the user's home directory.  C<ADDRESS> is
the address to send to; the text of the .forward message will read:

  \USERNAME,ADDRESS

Returns 1 if successful, 0 otherwise.  Listens to $TEST.

=cut

sub create_forward {
  my ($self, $partition, $username, $address) = @_;
  return 0 unless ($partition && $username && $address);

  my $forward = "/HomeDirs/$partition/$username/.forward";
  (print ".forward file already exists for $username\n" and return 0) 
  							   if (-f $forward);
  if ($TEST) { warn "Not creating .forward file\n";  return 1 }
  print "Creating .forward file\n" if $DEBUG;
  open FORWARD, ">$forward" 
                  or (warn "Couldn't write to $forward: $!\n" and return 0);
  print FORWARD "\\$username,$address\n";
  close FORWARD;

  print "Fixing permissions of .forward\n";
  chmod(0644, $forward) || (warn "chmod failed: $!\n" and return 0);
  chown_by_name($username, $forward) 
                                || (warn "chown failed: $!\n" and return 0);
  
  1;
}

=item create_webpage( PARTITION, USERNAME, FULLNAME [, TEMPLATE] )

Creates a basic web page from C<TEMPLATE> for C<USERNAME>.  C<TEMPLATE> is
optional, and defaults to the value of $TCB::AddUser::TEMPLATE;

=cut

sub create_webpage {
  my ($self, $partition, $username, $fullname, $template) = @_;
  return 0 unless ($partition && $username && $fullname);
  $template ||= $TEMPLATE;
  unless (-r $template) { warn "'$template' isn't readable\n";  return 0 }

  if ($TEST) { warn "Not creating webpage\n";  return 1 }

  print "Creating public_html directory\n" if $DEBUG;
  my $directory = "/HomeDirs/$partition/$username/public_html";
  unless (-d $directory) { 
    mkdir($directory,0777) || ( warn "mkdir failed\n" and return 0 ); 
    chmod(0755, $directory) || (warn "chmod failed\n" and return 0);
    chown_by_name($username,$directory) || (warn "chown failed\n" and return 0);
  }
  print "Creating template index.html file\n" if $DEBUG;
  open (TEMPLATE, $template)
        or (warn "Couldn't open $template: $!\n" and return 0);
  open (WEB, ">$directory/index.html")
     or (warn "Couldn't write to $directory/index.html: $!\n" and return 0);
  while (<TEMPLATE>) { s/NAME/$fullname/g; print WEB; }
  close(TEMPLATE); close(WEB);

  print "Fixing permissions of web page\n";
  chmod(0644, "$directory/index.html") || (warn "chmod failed\n" and return 0);
  chown_by_name($username, "$directory/index.html") 
				       || (warn "chown failed\n" and return 0);
  1;
}

=item set_htpasswd ( USERNAME, PASSWORD [, HTPASSWD] )

Sets the web password for USERNAME to PASSWORD.  HTPASSWD is the name of
the password file to update, and defaults to B<$TCB::AddUser::HTPASSWD>.
Returns 1 if successful, undef otherwise.

=cut

sub set_htpasswd {
  my ($self, $username, $passwd, $htpasswdfile) = @_;
  return undef unless ($username && $passwd);
  $htpasswdfile ||= $HTPASSWDFILE;

  ## Begin HTPASSWD
  my $program = "$HTPASSWD $htpasswdfile $username";
  warn "Opening '$program'\n" if $DEBUG;
  if ($TEST) { warn "Not setting web password\n";  return 1 }
  my $PROGRAM = Expect->spawn("$program") ||
	(warn "Unable to spawn $program\n" and return undef);

  ## Enter new password
  (warn "Unable to set web password for $username\n" and return undef) 
  			unless ($PROGRAM->expect(5, "New password: "));
  print $PROGRAM "$passwd\r";

  ## Confirm new password
  (warn "Unable to set web password for $username\n" and return undef) 
  			unless ($PROGRAM->expect(5, "Re-type new password: "));
  print $PROGRAM "$passwd\r";

  ## Make sure if worked
  (warn "Unable to set web password for $username\n" and return undef) 
  	    unless ($PROGRAM->expect(5, "Adding password for user $username",
  	    			       "Updating password for user $username"));

  ## Close and exit successfully
  $PROGRAM->soft_close;
  1;
}

=item set_yppasswd ( USERNAME, PASSWORD [, OLDPASSWD] )

Sets the YP password for USERNAME to PASSWORD.  OLDPASSWD is the old
password for USERNAME, and is necessary unless we're running as root.
Runs with NPASSWD if we have it, otherwise YPPASSWD.  Returns 1 if
successful, undef otherwise.

It's worth noting that this doesn't work anywhere near as well as I'd like
it to, because they keep changing the way that yppasswd works.  Hopefully
I'll eventually replace this with set_ldappasswd and call it done.

=cut

sub set_yppasswd { 
  my ($self, $username, $passwd, $oldpasswd) = @_;
  ( warn "\$username not passed to set_yppasswd()" && return undef ) 
							unless $username;
  ( warn "\$passwd not passed to set_yppasswd()" && return undef )  
							unless $passwd;

  my $isroot = ($< eq 0);

  ## Must offer old password if we're not root
  unless ($isroot || $oldpasswd) { 
    warn "Must either offer old password, or run as root\n"; return undef;
  }

  my $PROGRAM;
  if ($NPASSWD) { 
    ## Begin NPASSWD to change YP password
    my $program = $isroot ? "$NPASSWD $username" : $NPASSWD;
    warn "Opening '$program'\n" if $DEBUG;
    if ($TEST) { warn "Not setting YP password\n";  return 1 }
    $PROGRAM = Expect->spawn("$program") ||
	(warn "Unable to spawn $program\n" and return undef);

    ## Enter old password
    if (!$isroot) {
      (warn "Bad old password for $username\n" and return undef)
              unless ($PROGRAM->expect(15, 
			"Current password: " ));
      print $PROGRAM "$oldpasswd\r";
    }

    ## Enter new password
    (warn "Unable to set YP password for $username\n" and return undef)
            unless ($PROGRAM->expect(15, 
			/\s+Changing password for $username.*/,
		"New password (\? for help): " ));
    print $PROGRAM "$passwd\r";

    ## Confirm new password
    (warn "Unable to set YP password for $username\n" and return undef)
            unless ($PROGRAM->expect(15, "New password (again): "));
    print $PROGRAM "$passwd\r";

  } else { 
    ## Begin YPPASSWD
    my $program = $isroot ? "$YPPASSWD $username" : $YPPASSWD;
    warn "Opening '$program'\n" if $DEBUG;
    if ($TEST) { warn "Not setting YP password\n";  return 1 }
    $PROGRAM = Expect->spawn("$program") ||
	(warn "Unable to spawn $program\n" and return undef);

    ## Enter old password
    if (!$isroot) {
      (warn "Bad old password for $username\n" and return undef)
              unless ($PROGRAM->expect(15, "Enter login(NIS) password: ", 
					"Enter existing login password: ",
					"Please enter old password:" ));
      print $PROGRAM "$oldpasswd\r";
    }

    ## Enter new password
    (warn "Unable to set YP password for $username\n" and return undef)
            unless ($PROGRAM->expect(15, "New password:", "New Password: ",
		"Please enter new password:" ));
    print $PROGRAM "$passwd\r";

    ## Confirm new password
    (warn "Unable to set YP password for $username\n" and return undef)
            unless ($PROGRAM->expect(15, "Re-enter new Password: "));
    print $PROGRAM "$passwd\r";
  }

  ## Close and exit successfully
  $PROGRAM->soft_close;
  1;
 
}

=item set_smbpasswd ( USERNAME, PASSWORD [, OLDPASSWD] )

In theory, this sets the SMB password for USERNAME to PASSWORD.  However,
given how SMB works, it's a bit more complicated than that.

When run as a user, or if OLDPASSWD is offered, this will connect to the
appropriate SMB domain (as set in the module defualts) and change the
password for USERNAME - that is, no priveleges are required.  OLDPASSWD is
required instead.

If run as root, then this will set the USERNAME without needing an old
password.  However, it must be run on one of the machines that has SMB
configured properly to edit the password file directly.  If the username
doesn't exist, it will be created (though it must be in the regular Unix
password file first).

Returns 1 if successful, undef otherwise.

=cut

sub set_smbpasswd {
  my ($self, $username, $passwd, $oldpasswd) = @_;
  ( warn "\$username not passed to set_smbpasswd()" && return undef ) 
							unless $username;
  ( warn "\$passwd not passed to set_smbpasswd()" && return undef )  
							unless $passwd;
  my $isroot = ($< eq 0);

  my $program;
  if ($oldpasswd) { $program = "$SMBPASSWD -r $SMBHOST -U $username" }
  elsif ($isroot) { $program = "$SMBPASSWD -a $username" }
  else { warn "Must either offer old password or run as root\n"; return undef }

  warn "Opening '$program'\n" if $DEBUG;
  if ($TEST) { warn "Not setting SMB password\n";  return 1 }
  my $PROGRAM = Expect->spawn("$program") ||
	(warn "Unable to spawn $program\n" and return undef);
 
  ## Enter old password
  if ($oldpasswd) {
    (warn "Bad old password for $username\n" and return undef)
              unless ($PROGRAM->expect(15, "Old SMB password:"));
    print $PROGRAM "$oldpasswd\r";
  }

  ## Enter new password
  (warn "Unable to set SMB password for $username\n" and return undef) 
  		unless ($PROGRAM->expect(15, "New SMB password:"));
  print $PROGRAM "$passwd\r";

  ## Confirm new password
  (warn "Unable to set SMB password for $username\n" and return undef) 
  		unless ($PROGRAM->expect(15, "Retype new SMB password:"));
  print $PROGRAM "$passwd\r";
  
  ## Close and exit successfully
  $PROGRAM->soft_close;
  1;
}

=back

=cut

###############################################################################
### Shared Functions ##########################################################
###############################################################################

=head2 Shared Functions 

These functions are exported with @EXPORT_OK, and can therefore be
imported by other functions that want to use them.  They're also split out
in the code, so they can be transferred to other pieces of code later.
Basically, they're handy little pieces of code that will probably come in
handy later.  Enjoy them.

=over 4

=item chown_by_name( USER, FILES )

Changes file ownership (UID, GID) by username.  From Camel Book, 3rd Ed,
p691.

=cut

sub chown_by_name {
  my($user, @files) = @_;
  chown((getpwnam($user))[2,3], @files) == @files
	  or (warn "Can't chown @files: $!\n" and return 0);
}

=item chown_recursive( USER, FILES )

Uses File::Find to do a recursive chown.  Uses a global USER variable,
which is unfortunate but necessary if we want to use File::Find.  The
actual work is done by _chown(), which invokes chown_by_name appropriately.

=cut

use File::Find;
use vars qw( $USER );
sub chown_recursive {
  my ($user, @files) = @_;
  return undef unless $user;
  $USER = $user;
  my @return = find(\&_chown, @files);
  $USER = "";
  @return;
}
sub _chown { chown_by_name($USER, $_); }

=item prompt( STRING, CONFIRM [, ANSWER [, ANSWER [...]]] )

Prompt the user for input.  STRING is the prompting information, and each
ANSWER is a possible value.  If no ANSWERs are offered, prompts with 'y,
n' instead.  CONFIRM asks if we should check this stuff.

=cut

sub prompt {
  my ($string, $confirm, @answers) = @_;
  while (1) {
    scalar @answers ? print "$string [" , join(', ', @answers), "] --> "
                    : print "$string --> ";
    my $answer = <STDIN>;  chomp $answer;  next unless $answer;
    return $answer unless $confirm;
    if (scalar @answers) {
      foreach (@answers) { return $_ if $_ eq $answer; }
    } else {
      return $answer if prompt("Are you sure?", 0, 'y', 'n') eq 'y';
    }
  }
  0;
}

=item prompt_multiline( STRING, CONFIRM )

Prompt the user for input that spans lines.  STRING is the prompting
information, and CONFIRM asks if we should check to confirm the answer
afterwards.  Allows for multi-line inputs; '.' on an empty line
terminates.

=cut

sub prompt_multiline {
  my ($string, $confirm) = @_;
  my $answer;
  while (1) {
    $answer = ();
    print "$string ('.' on an empty line to stop).\n";
   LOOP:
    while (<STDIN>) { 
      chomp;
      $_ eq '.' ? last LOOP : push(@{$answer}, $_);
    }
    return $answer if !$confirm || prompt("Are you sure?", 0, 'y', 'n') eq 'y';
  }
  [];
}

=item prompt_password( STRING [, CHECK] )

Prompts the user for a password.  If CHECK is positive, then it runs the
password through C<check_password> before accepting it.  If CHECK is
negative, then we don't even ask for a 'confirm password' step.

=cut

use Term::ReadKey;
sub prompt_password {
  my ($string, $check) = @_;
  my $password;
  while (1) {
    print "$string --> ";
    ReadMode 'noecho'; $password = ReadLine 0; ReadMode 0; chomp $password;
    print "\n";  next if $password eq '';
  
    unless ($check < 0) {
      print "Confirm --> ";
      ReadMode 'noecho'; my $confirm = ReadLine 0; ReadMode 0; chomp $confirm;
      print "\n";
      unless ($password eq $confirm) {
        print "Passwords don't match, starting over\n";
        next;
      }
    }

    if ($check >= 1) {	
      warn "Testing password\n" if $DEBUG;
      check_password($password, 1) 
	        ? ( print "Password acceptable\n" and last )
		: ( print "Password not acceptable, starting over\n" and next );
    } else { last }
  }
  $password;
}

=item check_password ( PASSWORD [, VERBOSE ] )

Checks to see if the password is "good enough" to be used.  Uses
'checkpassword' (as set in the module), part of the 'npasswd'.

=cut

use FileHandle;		# Comes with Perl
use IPC::Open3;		# Comes with Perl
sub check_password {
  my ($password, $verbose) = @_;
  return 0 unless ($password);  $verbose ||= 0;

  my $program = $CHECKPASSWD;
  my $out = new FileHandle;  my $in = new FileHandle;
  my $pid = eval { open3($in, $out, $out, $program) };
  print $in "$password\n";
  waitpid($pid, 0);
  while (<$out>) { chomp; print "$_\n" if $verbose }
  $in->close;  $out->close;
   
  return $? eq '0' ? 1 : 0;
}

=back

=cut

###############################################################################
### Helper Functions ##########################################################
###############################################################################

### We're not going to document these, they're just internal.

### newuid( USERNAME )
# Get the next avaiable UID for USERNAME.  If USERNAME already exists,
# then return 0.  Uses YP.
sub newuid {
  my ($self, $username) = @_;
  return 0 if $username && scalar getpwnam($username);

  my $maxuid = 0;
  my $passwd = Net::NIS::Table->new("passwd.byname")->list;
  return 0 unless ($passwd && ref $passwd);

  foreach (keys %{$passwd}) {
    my @values = split(':', $$passwd{$_});
    $maxuid = $values[2] if $values[2] > $maxuid;
  }
  $maxuid + 1;
}

### getgid( GROUP ) 
# Returns the GID for GROUP, or 0 if unsuccessful.  Uses YP.
sub getgid {
  my ($self, $group) = @_; return 0 unless $group;
  my $groups = Net::NIS::Table->new("group.byname")->list;
  return 0 unless ($groups && ref $groups);
  return 0 unless $$groups{$group};
  my @values = split(':', $$groups{$group});
  $values[2];
}

### partitions
# Returns a hash of valid partitions, based on YP and matching the
# pattern in $PPAT.
sub partitions {
  my ($self) = @_;
  my $homedirs = Net::NIS::Table->new("auto.HomeDirs")->list;
  return {} unless ($homedirs && ref $homedirs);
  my %partitions;
  foreach my $amdir (sort keys %{$homedirs}) {
    next unless "$amdir $$homedirs{$amdir}" =~ /$PPAT/;
    $partitions{$amdir} = $$homedirs{$amdir};
  }
  wantarray ? keys %partitions : \%partitions;
}

sub _check_hostname {
  my ($hostname, @servers) = @_;
  $hostname =~ s/\..*$//;
  my $yes = 0;
  foreach (@servers) { $yes++ if $_ eq $hostname; }
  $yes;
}

###############################################################################
### Final Documentation #######################################################
###############################################################################

=head1 SHARED VARIABLES

=over 4

=item $TCB::AddUser::DEBUG

Module functions print debugging information if this variable is set.

=item $TCB::AddUser::TEST

Module functions don't actually modify the system if this is set, so you
can just make sure your script generally works.

=back

=head1 NOTES

This module is mostly intended for internal TCB use, and is designed as
such.  However, many of the functions in here would be of use to anybody
else attempting to make user manipulation scripts and such.  So keep that
in mind - this code may be handy!

Some portions of this code may eventually be used for a web-based (or
Tcl-based, possibly) user creation script setup.  Eventually.

=head1 REQUIREMENTS

File::NCopy (from CPAN), Net::NIS::Table (CPAN), Expect (CPAN, may have
to install manually), File::Find, Filesys::DiskSpace, Term::ReadKey,
IPC::Open3, FileHandle, Sys::Hostname, and Exporter.

Also requires the 'npasswd' package, for the password checking scripts.

=head1 SEE ALSO

/Projects/system/newuser contains some scripts that use these functions.

=head1 TODO

Make some new scripts with this module, to modify passwords and such.

Make a DBI back-end to the whole user creation process, using this module
as a start.

Would be nice to let users adjust their shells and such too.

=head1 AUTHOR

Written by Tim Skirvin <tskirvin@ks.uiuc.edu>.

=head1 HOMEPAGE
                                                                                
B<http://www.ks.uiuc.edu/Development/MDTools/tcb-adduser/>
                                                                                
=head1 LICENSE
 
This code is distributed under the University of Illinois Open Source
License.  See
C<http://www.ks.uiuc.edu/Development/MDTools/tcb-adduser/license.html>
for details.
 
=head1 COPYRIGHT

Copyright 2001-2004 by the University of Illinois Board of Trustees and
Tim Skirvin <tskirvin@ks.uiuc.edu>.

=cut

1;

###############################################################################
### Version History ###########################################################
###############################################################################
### v0.1a	Wed Oct 10 09:52:12 CDT 2001
# Initial code complete, commenting and documentation in progress.

### v0.91b	Thu Oct 11 14:20:20 CDT 2001
# Fixed Exporter code, standardized name on "TCB::AddUser", changed
# behavior of prompt_password to do a better job with old passwords.
#
# Let set_smbpasswd work on remote machines, and added SMBHOST accordingly.
#
# Fixed set_htpasswd for case that you're updating the password instead of
# adding it.
#
# Getting ready for a real internal release.
#
# Fixed Makefile.pl to look for prereqs

### v0.92b 	Fri Oct 12 11:06:59 CDT 2001
# Fixed a small bug in add_ypppasswd

### v0.93b      Mon Nov 12 17:09:14 CST 2001
# Changed how root users change passwords with add_smbpasswd

### v0.94b	Tue Nov 27 14:19:00 CST 2001
# Fixed problem with prompt_admininfo()

### v0.95b 	Wed Jan  2 10:43:13 CST 2002
# Changed get_password() to allow to only get one of web or YP/SMB passwords.

### v0.96b 	Tue Jan 15 16:09:25 CST 2002
# prompt_multiline() now correctly returns an empty list if no input is offered.

### v0.97b	Fri Feb 15 14:05:35 CST 2002
# add_yppasswd() exits if not enough information is offered.  Created 
# add_hometable().

### v0.98b 	Wed Feb 20 10:22:38 CST 2002
# Use auto.HomeDirs instead of auto.Home to figure out the appropriate
# home directory for adding users.  Don't put into place until after the 
# site visit!

### v0.99b	Tue Feb 18 09:28:09 CST 2003 
# Trying to get SMB to work properly.  Using auto.home where necessary.
# Changed some debugging information stuff.
# set_smbpasswd() now lets you set the username remotely

### v1.00	Mon Oct 13 09:04:41 CDT 2003
# It works properly.  Let's just call it v1.0 and call it done.
# Changed default $PPAT, and formatting for prompt_partitions().

### v1.01	Fri Jan 16 09:48:00 CST 2004 
# Added create_forward().  Updated some functions to use $TEST properly
# and use better error messages.

### v1.02	Wed Feb 04 11:22:50 CST 2004 
# Fixed some errors in chown() stuff.  Allowed for multiple YP servers.
# Changes to set_yppasswd() and set_smbpasswd().

# v1.03		Thu Apr 22 14:27:15 CDT 2004 
### Changed to use TCB::AddUser instead of TB::AddUser.

# v1.04		Wed May 19 17:55:49 CDT 2004 
### Licensed as part of MDTools.

# v1.05		Mon Jun 14 13:40:36 CDT 2004 
### Expanded the use of npasswd.
