/*
 * sysmon.c -- system load monitoring
 *
 *     Gather machine load information and send a UDP packet to
 *     collection server at regular time intervals.
 *
 * author: David Hardy
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <errno.h>
#include <ctype.h>
#include <time.h>
#include <unistd.h>
#include <sys/wait.h>
#include <sys/time.h>
#include <sys/resource.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <netdb.h>
#include "getdate.h"
#include "defopts.h"


#define  OPT_HELP     'h'
#define  OPT_NAME     'm'
#define  OPT_PORT     'p'
#define  OPT_SLEEP    't'
#define  OPT_DIAGNOSE 'd'
#define  OPT_LOOPS    'n'


int get_sysload_str(char *strbuf, const char *cmd)
{
  int fdpipe[2], childpid, n;
  int strbufsz = BUFLEN;

  pipe(fdpipe);
  fflush(stdout);
  if ((childpid = fork()) > 0) {  /* parent */
    close(fdpipe[1]);
    do {
      if (strbufsz > 0) {
        n = read(fdpipe[0], strbuf, strbufsz);
        strbuf += n;
        strbufsz -= n;
      }
    } while (wait3(NULL, WNOHANG, NULL) == 0);
    close(fdpipe[0]);
    if (strbufsz > 0)  *strbuf = '\0';
    else  *(--strbuf) = '\0';
  }
  else {  /* child */
    close(1);  /* stdout */
    close(2);  /* stderr */
    close(fdpipe[0]);
    dup2(fdpipe[1], 1);
    dup2(fdpipe[1], 2);
    system(cmd);
    exit(0);
  }
  return 0;
}


int get_loaddata(char *loaddata, int sleeptime)
{
  char buf[BUFLEN], *sysload_str;
  char *p;
  int chksum;

  /* parse for date and time */
  if (get_date(buf)) {
    fprintf(stderr, "call to get_date() failed\n");
    exit(1);
  }
  strcpy(loaddata, buf);

  /* parse for most recent load average */
#ifdef SCYLD
/*
 * filtering 'beostat -l' through the given awk command makes string of form:
 * " 0.990645 31\n"
 *
 * which load avg value (1 min, 5 min, or 15 min) is determined by awk cmd
 * second value is number of nodes that are up
 */
  if (sleeptime > 450) {
    /* use 15 min load averages */
    get_sysload_str(buf, "beostat -l | awk '/^[0-9]+/ "
          "{ total += $3 ; num++ } END { print \" \" total \" \" num }'");
  }
  else if (sleeptime > 150) {
    /* use 5 min load averages */
    get_sysload_str(buf, "beostat -l | awk '/^[0-9]+/ "
          "{ total += $2 ; num++ } END { print \" \" total \" \" num }'");
  }
  else {
    /* use 1 min load averages */
    get_sysload_str(buf, "beostat -l | awk '/^[0-9]+/ "
          "{ total += $1 ; num++ } END { print \" \" total \" \" num }'");
  }
  /* can't put number of nodes in database yet, just get load avg */
  p = strrchr(buf, ' ');
  *p = '\0';
  strcat(loaddata, buf);
#else
/*
 * string of form:
 * "  2:18pm  up 45 day(s),  5:58,  4 users,  load average: 0.82, 0.91, 0.88\n"
 *
 * load averages over last 1 min, 5 min, 15 min
 */
  get_sysload_str(buf, "uptime");
  p = strstr(buf, "average:");
  sysload_str = strchr(p, ' ');
  if (sleeptime <= 150) {
    /* use 1 min load averages */
    p = strchr(sysload_str, ',');
    *p = '\0';
  }
  else if (sleeptime <= 450) {
    /* use 5 min load averages */
    p = sysload_str + 1;
    sysload_str = strchr(p, ' ');
    p = strchr(sysload_str, ',');
    *p = '\0';
  }
  else {
    /* use 15 min load averages */
    p = sysload_str + 1;
    sysload_str = strchr(p, ' ');
    p = sysload_str + 1;
    sysload_str = strchr(p, ' ');
    p = strchr(sysload_str, '\n');
    *p = '\0';
  }
  strcat(loaddata, sysload_str);
#endif

  /* simple checksum to be verified by server */
  /* (shouldn't be longer than 8 chars) */
  for (p = loaddata, chksum = 0;  *p != '\0';  p++) {
    chksum += (int) *p;
  }
  sprintf(buf, " %d", chksum);
  strcat(loaddata, buf);
  return 0;
}


void print_help(const char *progname)
{
  printf(
      "\nUsage for %s:\n\n"
      "%s  [ -%c | [-%c <machine_name>] [-%c <port_number>]\n"
      "\t[-%c <sleep_time>] [-%c] [-%c <number>] ]\n\n"
      "Options:\n"
      "\t-%c\thelp (print this information)\n\n"
      "\t-%c\tspecify machine name of system load collector\n"
      "\t\t(defaults to \"%s\")\n\n"
      "\t-%c\tspecify port number of system load collector\n"
      "\t\t(defaults to \"%s\")\n\n"
      "\t-%c\tspecify sleep time (in seconds) between system load reporting\n"
      "\t\t(defaults to \"%s\")\n\n"
      "(the other options are for debugging purposes)\n\n"
      "\t-%c\tprint diagnostic message on each reporting\n"
      "\t\t(turned off by default)\n\n"
      "\t-%c\tnumber of times to report load averages\n"
      "\t\t(otherwise, loops forever)\n\n"
      , progname
      , progname, OPT_HELP, OPT_NAME, OPT_PORT, OPT_SLEEP
      , OPT_DIAGNOSE, OPT_LOOPS
      , OPT_HELP
      , OPT_NAME, DEFAULT_SERVER
      , OPT_PORT, DEFAULT_PORTNUM
      , OPT_SLEEP, DEFAULT_SLEEPTIME
      , OPT_DIAGNOSE, OPT_LOOPS
      );
}


int main(int argc, char **argv)
{
  char loaddata_str[MAXBUFLEN];
  int n, cnt = 0;
  int tsec;
  char *port_str = NULL, *server_str = NULL, *time_str = NULL;
  int sockfd, numbytes;
  struct sockaddr_in server_addr;
  struct hostent *he;
  const char *progname = argv[0];
  int isdmsg = 0;
  int isinf = 1;
  int numloops = 0;

  /* get hostname and port of sysload server from command line */
  while ((++argv, --argc) > 0) {
    if (**argv == '-') {
      switch (*(++(*argv))) {
      case OPT_HELP:
        print_help(progname);
        exit(0);
      case OPT_PORT:
        if (!(*(++(*argv)))) ++argv, --argc;
        port_str = *argv;
        break;
      case OPT_NAME:
        if (!(*(++(*argv)))) ++argv, --argc;
        server_str = *argv;
        break;
      case OPT_SLEEP:
        if (!(*(++(*argv)))) ++argv, --argc;
        time_str = *argv;
        break;
      case OPT_DIAGNOSE:
        isdmsg = 1;
        break;
      case OPT_LOOPS:
        if (!(*(++(*argv)))) ++argv, --argc;
        isinf = 0;
        numloops = atoi(*argv);
        break;
      default:
        fprintf(stderr, "Error ... unknown option\n");
        print_help(progname);
        exit(1);
      }
    }
    else {
      fprintf(stderr, "Error ... illegal use\n");
      print_help(progname);
      exit(1);
    }
  }
  if (!port_str) port_str = DEFAULT_PORTNUM;
  if (!server_str) server_str = DEFAULT_SERVER;
  if (!time_str) time_str = DEFAULT_SLEEPTIME;

  printf(
      "%s ... sending system load information to collection server\n"
      "\tsystem collector machine \"%s\"\n"
      "\tport number \"%s\"\n"
      "\tsleep time between load monitoring \"%s\" (seconds)\n"
      , progname, server_str, port_str, time_str
      );

  /* sleep time in seconds */
  tsec = atoi(time_str);

  /* setup to send UDP packet to sysload server */
  if ((he = gethostbyname(server_str)) == NULL) {
    herror("gethostbyname");
    exit(1);
  }
  if ((sockfd = socket(AF_INET, SOCK_DGRAM, 0)) == -1) {
    perror("socket");
    exit(1);
  }
  server_addr.sin_family = AF_INET;
  server_addr.sin_port = htons(atoi(port_str)); /* short, network byte order */
  server_addr.sin_addr = *((struct in_addr *)he->h_addr);
  memset(server_addr.sin_zero, 0, 8);           /* zero rest of struct */

  /* get hostname of our machine */
  /* (need room for simple checksum at end of string) */
  gethostname(loaddata_str, 2 * BUFLEN - 10);
  loaddata_str[2 * BUFLEN - 10] = '\0';
  n = strlen(loaddata_str);
  loaddata_str[n] = ' ';
  loaddata_str[++n] = '\0';

  while (1) {  /* repeat forever */

    /* append info about date, time, and load after hostname */
    loaddata_str[n] = '\0';
    get_loaddata(&loaddata_str[n], tsec);

    /* send UDP packet to sysload server */
    if ((numbytes = sendto(sockfd, loaddata_str, strlen(loaddata_str), 0,
          (struct sockaddr *) &server_addr, sizeof(struct sockaddr))) == -1
          && errno != ECONNREFUSED) {
      perror("sendto");
      exit(1);
    }

    if (isdmsg)  printf("%s\n", loaddata_str);
    if (!isinf && ++cnt == numloops)  break;
    sleep(tsec);
  }

  close(sockfd);

  return 0;
}
