/*
 * ihash.h - A simple hash table
 * 
 * Uses null terminated strings as the keys for the table.
 * Stores an integer value with the string key.  It would be
 * easy to change to use void *'s instead of ints.  Maybe rewrite
 * as a C++ template??
 * 
 * Donated by John Stone
 *
 * The integer value should be an array index (non-negative).  
 * Otherwise, the value will conflict with the IHASH_FAIL flag.  
 * Also note that ihash_insert SUCCEEDS if it returns IHASH_FAIL, 
 * meaning that the key used was not found in the table.  
 * The const char * strings passed in as keys are ALIASED by the 
 * table, so they must PERSIST at least as long as the table does.
 *
 * Error handling is updated to return IHASH_MEMALLOC whenever memory 
 * allocation for the bucket array fails from any calls to ihash_init 
 * or ihash_insert.  If IHASH_MEMALLOC occurs on call to ihash_init, 
 * you should immediately call ihash_destroy to cleanup gracefully.
 * If IHASH_MEMALLOC occurs on call to ihash_insert, this means that 
 * this insert failed due to memory allocation error.  However, all 
 * access to the previously existing table should still be preserved.
 *
 * Modified by David Hardy
 */

#ifndef H_IHASH
#define H_IHASH

#ifdef __cplusplus
extern "C" {
#endif

typedef struct Ihash_Tag {
  struct hash_node_t **bucket;        /* array of hash nodes */
  int size;                           /* size of the array */
  int entries;                        /* number of entries in table */
  int downshift;                      /* shift cound, used in hash function */
  int mask;                           /* used to select bits for hashing */
  char buf[64];                       /* used to store buffer for stats */
} Ihash;

#define  IHASH_FAIL      -1
#define  IHASH_MEMALLOC  -2

int ihash_init(Ihash *, int);
int ihash_lookup (Ihash *, const char *);
int ihash_update (Ihash *, const char *, int);
int ihash_insert (Ihash *, const char *, int);
int ihash_delete (Ihash *, const char *);
void ihash_destroy(Ihash *);
const char *ihash_stats (Ihash *);

#ifdef __cplusplus
}
#endif

#endif /* H_IHASH */
