/*
 * syscol.c -- system load collector
 *
 *     Collect system load UDP messages from client machines.
 *     Average results at every date change and store in
 *     system loads database.
 *
 * author: David Hardy
 */

#include <stdio.h> 
#include <stdlib.h> 
#include <string.h> 
#include <signal.h>
#include <errno.h> 
#include <sys/types.h> 
#include <netinet/in.h> 
#include <sys/socket.h> 
#include <sys/wait.h> 
#include "getdate.h"
#include "ihash.h"
#include "defopts.h"


#define  MAX_MACH  2

#define  OPT_HELP       'h'
#define  OPT_PORT       'p'
#define  OPT_LOG        'l'
#define  OPT_OLDLOG     'o'
#define  OPT_AVERAGE    'a'
#define  OPT_DBUPDATE   'u'
#define  OPT_DIAGNOSE   'd'
#define  OPT_LOOPS      'n'


typedef struct Mach_Tag {
  char *name;
  double xmin, x;
  double ymin, ymax, y;
  double area_sum;
  int n;
} Mach;


int process_logfile(char *fname, char *yesterday, int isdbupdate)
{
#ifdef DBUPDATE
  extern char *dbupdate_str;
#endif
  char buf[MAXBUFLEN];
  char formatbuf[BUFLEN];
  int namelen, max_namelen = 0;
  char *p, *mach_str, *date_str, *time_str, *load_str;
  FILE *fp;
  int index;
  double xnew, ynew, daily_avg;
  Mach *mach = NULL;
  int num_mach = 0;
  int max_mach = MAX_MACH;
  Ihash table;

  /* allocate memory for mach array */
  mach = (Mach *) malloc(max_mach * sizeof(Mach));
  if (mach == NULL) {
    fprintf(stderr, "can't allocate memory\n");
    exit(1);
  }

  /* initialize hash table */
  if (ihash_init(&table, max_mach) != 0) {
    fprintf(stderr, "can't initialize hash table\n");
    exit(1);
  }

  /* open yesterday's log file */
  fp = fopen(fname, "r");
  if (fp == NULL) {
    fprintf(stderr, "can't open file %s\n", fname);
    exit(1);
  }

  /* process log file */
  while (fgets(buf, MAXBUFLEN, fp) != NULL) {
    buf[MAXBUFLEN - 1] = '\0';

    /*
     * buffer message should be in form:
     *
     *     machine_name YYYY-MM-DD HH:MM:SS loadavg
     */

    /* peel off machine name, date, time, and load */
    mach_str = buf;
    if ((p = strchr(buf, ' ')) == NULL)  continue;
    *p++ = '\0';
    date_str = p;
    if ((p = strchr(p, ' ')) == NULL)  continue;
    *p++ = '\0';
    time_str = p;
    if ((p = strchr(p, ' ')) == NULL)  continue;
    *p++ = '\0';
    load_str = p;

    /* sanity check on correct date */
    if (strcmp(yesterday, date_str) != 0)  continue;

    /* parse time string, express as seconds (from previous midnight) */
    if ((p = strchr(time_str, ':')) == NULL)  continue;
    xnew = 60.0 * atof(time_str);  /* hours to minutes */
    time_str = p + 1;
    if ((p = strchr(time_str, ':')) == NULL)  continue;
    xnew = 60.0 * (xnew + atof(time_str));  /* minutes to seconds */
    time_str = p + 1;
    xnew += atof(time_str);  /* add in seconds */
    ynew = atof(load_str);

    /* lookup name in hash table */
    index = ihash_lookup(&table, mach_str);
    if (index == IHASH_FAIL) {  /* was not already in hash table */
      /* extend Mach array if we need to */
      if (num_mach == max_mach) {
        void *tmp = realloc(mach, 2 * max_mach * sizeof(Mach));
        if (tmp == NULL) {
          fprintf(stderr, "out of memory for machine list\n");
          exit(1);
        }
        mach = (Mach *) tmp;
        max_mach *= 2;
      }
      namelen = strlen(mach_str);
      if (max_namelen < namelen)  max_namelen = namelen;
      if ((mach[num_mach].name = (char *) malloc(namelen + 1)) == NULL) {
        fprintf(stderr, "out of memory for machine name\n");
        exit(1);
      }
      strcpy(mach[num_mach].name, mach_str);
      /* insert name into hash table */
      index = ihash_insert(&table, mach[num_mach].name, num_mach);
      if (index != IHASH_FAIL) {
        fprintf(stderr, "out of memory for hash table\n");
        exit(1);
      }
      index = num_mach++;
      mach[index].n = 1;
      mach[index].area_sum = 0.0;
      mach[index].xmin = xnew;
      mach[index].ymax = mach[index].ymin = ynew;
    }
    else {  /* accumulate area using trapezoid rule */
      mach[index].n++;
      mach[index].area_sum += (xnew - mach[index].x) 
                                * 0.5 * (ynew + mach[index].y);
      if (mach[index].ymax < ynew)  mach[index].ymax = ynew;
      else if (mach[index].ymin > ynew)  mach[index].ymin = ynew;
    }
    mach[index].x = xnew;
    mach[index].y = ynew;
  }

  /* finished reading file */
  fclose(fp);

  /* finish computing daily averages */
  printf("\nSystem load averages for %s:\n", yesterday);
  sprintf(formatbuf, "%%-%ds  min=%%6.2f  max=%%6.2f  avg=%%6.2f\n",
          max_namelen);
  for (index = 0;  index < num_mach;  index++) {
    if (mach[index].n == 1) {
      daily_avg = mach[index].y;
    }
    else {
      daily_avg = mach[index].area_sum / (mach[index].x - mach[index].xmin);
    }
    if (isdbupdate) {
#ifdef DBUPDATE
      /* update the system loads database */
      sprintf(buf, "%s %s %4.2f %4.2f %s", dbupdate_str, mach[index].name,
          daily_avg, mach[index].ymax, yesterday);
      system(buf);
#endif
    }
    /* print daily average */
    printf(formatbuf,
           mach[index].name, mach[index].ymin, mach[index].ymax, daily_avg);
  }

  /* destroy hash table */
  ihash_destroy(&table);

  /* deallocate memory */
  for (index = 0;  index < num_mach;  index++) {
    free(mach[index].name);
  }
  free(mach);
  return 0;
}


void print_help(const char *progname)
{
  printf(
      "\nUsage for %s:\n\n"
      "%s  [ -%c | [-%c <port_number>] [-%c <log_file>]\n"
      "\t[-%c <old_log_file>] [-%c] "
#ifdef DBUPDATE
      "[-%c <update_script>] "
#endif
      "[-%c] [-%c <number>] ]\n\n"
      "Options:\n"
      "\t-%c\thelp (print this information)\n\n"
      "\t-%c\tspecify port number for receiving system load messages\n"
      "\t\t(defaults to \"%s\")\n\n"
      "\t-%c\tspecify log file for storing system load messages\n"
      "\t\t(defaults to \"%s\")\n\n"
      "\t-%c\tspecify name for yesterday's log file\n"
      "\t\t(defaults to \"%s\")\n\n"
      "\t-%c\tcompute current load averages from log file then exit\n"
      "\t\t(not the default)\n\n"
#ifdef DBUPDATE
      "\t-%c\tspecify script used to update system load database\n"
      "\t\t(defaults to \"%s\")\n\n"
#endif
      "(the other options are for debugging purposes)\n\n"
      "\t-%c\tprint diagnostic message on each system load message received\n"
      "\t\t(turned off by default)\n\n"
      "\t-%c\tnumber of system load messages to receive before quitting\n"
      "\t\t(otherwise, loops forever)\n\n"
      , progname
      , progname, OPT_HELP, OPT_PORT, OPT_LOG, OPT_OLDLOG
      , OPT_AVERAGE
#ifdef DBUPDATE
      , OPT_DBUPDATE
#endif
      , OPT_DIAGNOSE, OPT_LOOPS
      , OPT_HELP
      , OPT_PORT, DEFAULT_PORTNUM
      , OPT_LOG, DEFAULT_LOG
      , OPT_OLDLOG, DEFAULT_OLDLOG
      , OPT_AVERAGE
#ifdef DBUPDATE
      , OPT_DBUPDATE, DEFAULT_DBUPDATE
#endif
      , OPT_DIAGNOSE, OPT_LOOPS
      );
}


/*
 * file pointer to the log file
 */
FILE *fp_log;

/* flush output to logfile on SIGTERM */
void terminate(int n)
{
  extern FILE *fp_log;

  printf("system load collector received terminate signal %d, "
         "flushing log file\n", n);
  fflush(stdout);
  fclose(fp_log);
  exit(0);
}


#ifdef DBUPDATE
/* need this globally visible */
char *dbupdate_str = NULL;
#endif


int main(int argc, char **argv)
{
  char buf[MAXBUFLEN];
  char namebuf[MAXBUFLEN];
  char *port_str = NULL, *log_str = NULL, *oldlog_str = NULL;
#ifdef DBUPDATE
  extern char *dbupdate_str;
#endif
  int sockfd;
  struct sockaddr_in my_addr;    /* my address information */
  struct sockaddr_in their_addr; /* connector's address information */
  int addr_len, numbytes;
  char *p, *data;
  int chksum, sum;
  char date_str[BUFLEN];
  char today_str[BUFLEN], yesterday_str[BUFLEN];
  char movelogfile_cmd[MAXBUFLEN];
  int i, n;
  const char *progname = argv[0];
  int isdmsg = 0;
  int isinf = 1;
  int numloops = 0;
  int cnt = 0;
  extern FILE *fp_log;
  int child_pid = 0;
  int retval = 0;
  int stat = 0;
  char eraseoldlog_cmd[MAXBUFLEN];
  void (*old_handler)(int);
  int iscompavg = 0;

  /* get hostname and port of sysload server from command line */
  while ((++argv, --argc) > 0) {
    if (**argv == '-') {
      switch (*(++(*argv))) {
      case OPT_HELP:
        print_help(progname);
        exit(0);
      case OPT_PORT:
        if (!(*(++(*argv)))) ++argv, --argc;
        port_str = *argv;
        break;
      case OPT_LOG:
        if (!(*(++(*argv)))) ++argv, --argc;
        log_str = *argv;
        break;
      case OPT_OLDLOG:
        if (!(*(++(*argv)))) ++argv, --argc;
        oldlog_str = *argv;
        break;
      case OPT_AVERAGE:
        iscompavg = 1;
        break;
#ifdef DBUPDATE
      case OPT_DBUPDATE:
        if (!(*(++(*argv)))) ++argv, --argc;
        dbupdate_str = *argv;
        break;
#endif
      case OPT_DIAGNOSE:
        isdmsg = 1;
        break;
      case OPT_LOOPS:
        if (!(*(++(*argv)))) ++argv, --argc;
        isinf = 0;
        numloops = atoi(*argv);
        break;
      default:
        fprintf(stderr, "Error ... unknown option\n");
        print_help(progname);
        exit(1);
      }
    }
    else {
      fprintf(stderr, "Error ... illegal use\n");
      print_help(progname);
      exit(1);
    }
  }
  if (!port_str) port_str = DEFAULT_PORTNUM;
  if (!log_str) log_str = DEFAULT_LOG;
  if (!oldlog_str) oldlog_str = DEFAULT_OLDLOG;
#ifdef DBUPDATE
  if (!dbupdate_str) dbupdate_str = DEFAULT_DBUPDATE;
#endif
  if (strcmp(log_str, oldlog_str) == 0) {
    fprintf(stderr, "Error ... old log file must have different name "
        "than current log file\n");
    print_help(progname);
    exit(1);
  }

  /* get today's date */
  if (get_date(today_str)) {
    fprintf(stderr, "call to get_date() failed\n");
    exit(1);
  }
  p = strchr(today_str, ' ');    /* truncate time-of-day */
  *p = '\0';

  if (iscompavg) {
    process_logfile(log_str, today_str, 0);
    exit(0);
  }

  printf(
      "%s ... receiving system load information, writing log file\n"
      "\tport number \"%s\"\n"
      "\tlog file \"%s\"\n"
      "\told log file \"%s\"\n"
#ifdef DBUPDATE
      "\tupdate script \"%s\"\n"
#endif
      , progname, port_str, log_str, oldlog_str
#ifdef DBUPDATE
      , dbupdate_str
#endif
      );

  /* open log file */
  if ((fp_log = fopen(log_str, "a")) == NULL) {
    fprintf(stderr, "can't open log file\n");
    exit(1);
  }

  /* install new SIGTERM handler */
  if ((old_handler = signal(SIGTERM, terminate)) == SIG_ERR) {
    perror("signal");
    exit(1);
  }

  /* establish command to move log files */
  sprintf(movelogfile_cmd, "mv -f %s %s", log_str, oldlog_str);
  sprintf(eraseoldlog_cmd, "rm -f %s", oldlog_str);

  /* setup to receive UDP packets on specified port */
  if ((sockfd = socket(AF_INET, SOCK_DGRAM, 0)) == -1) {
    perror("socket");
    exit(1);
  }
  my_addr.sin_family = AF_INET;              /* host byte order */
  my_addr.sin_port = htons(atoi(port_str));  /* short, network byte order */
  my_addr.sin_addr.s_addr = INADDR_ANY;      /* auto-fill with my IP */
  memset(my_addr.sin_zero, 0, 8);            /* zero rest of struct */
  if (bind(sockfd, (struct sockaddr *) &my_addr, 
        sizeof(struct sockaddr)) == -1) {
    perror("bind");
    exit(1);
  }

  while (1) {  /* repeat forever */

    /* exit loop if we are counting loops and reached the max number */
    if (!isinf && cnt++ >= numloops)  break;

    /* wait for child process if one exists, erase old log file */
    if (child_pid != 0) {
      retval = waitpid(child_pid, &stat, WNOHANG);
      if (retval == -1) {
        perror("waitpid");
        exit(1);
      }
      else if (retval != 0) {
        if (stat != 0) {
          fprintf(stderr, "error processing log file\n");
        }
        child_pid = 0;
        if (system(eraseoldlog_cmd) == -1) {
          fprintf(stderr, "erase command failed\n");
          perror(eraseoldlog_cmd);
          exit(1);
        }
      }
    }

    /* wait for a UDP packet */
    addr_len = sizeof(struct sockaddr);
    if ((numbytes = recvfrom(sockfd, buf, MAXBUFLEN-1, 0,
          (struct sockaddr *) &their_addr, &addr_len)) == -1) {
      perror("recvfrom");
      exit(1);
    }
    buf[numbytes] = '\0';

    /* validate checksum of message */
    if ((p = strrchr(buf, ' ')) == NULL)  continue;
    *p++ = '\0';
    chksum = atoi(p);
    if ((p = strchr(buf, ' ')) == NULL)  continue;
    for (sum = 0, p++;  *p != '\0';  p++) {
      sum += (int) *p;
    }
    if (sum != chksum)  continue;

    /* peel off date substring from buf */
    sscanf(buf, "%*s %s", date_str);

    /* verify that date_str is same as today's date */
    if (strcmp(date_str, today_str) != 0) {
      strcpy(yesterday_str, today_str);
      if (get_date(today_str)) {
        fprintf(stderr, "call to get_date() failed\n");
        exit(1);
      }
      p = strchr(today_str, ' ');    /* truncate time-of-day */
      *p = '\0';
      if (strcmp(date_str, today_str) != 0)  continue;
          /* if not equal, message's date is still wrong, so throw away */

      /* otherwise, the date has changed, need to process all data so far */
      /* start by closing and moving log file */
      fclose(fp_log);

      /* still need to wait on child, erase old log file */
      if (child_pid != 0) {
        if (waitpid(child_pid, &stat, 0) == -1) {
          perror("waitpid");
          exit(1);
        }
        else if (stat != 0) {
          fprintf(stderr, "error processing log file\n");
        }
        child_pid = 0;
        if (system(eraseoldlog_cmd) == -1) {
          fprintf(stderr, "erase command failed\n");
          perror(eraseoldlog_cmd);
          exit(1);
        }
      }

      /* need to move current log file to old log file */
      if (system(movelogfile_cmd) == -1) {
        fprintf(stderr, "move command failed\n");
        perror(movelogfile_cmd);
        exit(1);
      }

      /* now fork off a process to take care of processing old log file */
      fflush(stdout);
      if ((child_pid = fork()) == 0) {  /* child */
        process_logfile(oldlog_str, yesterday_str, 1);
        exit(0);
      }

      /* open new log file */
      if ((fp_log = fopen(log_str, "w")) == NULL) {
        fprintf(stderr, "can't re-open log file\n");
        exit(1);
      }
    }

    /* write line into log file */
    fprintf(fp_log, "%s\n", buf);

    /* see if we need to print diagnostic message */
    if (isdmsg)  printf("%s\n", buf);
  }

  /*
   * if we make it this far, we were counting messages received
   * process what we have and quit
   */

  /* close socket */
  close(sockfd);

  /* close log file and process system load averages */
  fclose(fp_log);

  /* see if we still need to wait on child, erase old log file */
  if (child_pid != 0) {
    if (waitpid(child_pid, &stat, 0) == -1) {
      perror("waitpid");
      exit(1);
    }
    else if (stat != 0) {
      fprintf(stderr, "error processing log file\n");
    }
    child_pid = 0;
    if (system(eraseoldlog_cmd) == -1) {
      fprintf(stderr, "erase command failed\n");
      perror(eraseoldlog_cmd);
      exit(1);
    }
  }

  /* need to move current log file to old log file */
  if (system(movelogfile_cmd) == -1) {
    fprintf(stderr, "move command failed\n");
    perror(movelogfile_cmd);
    exit(1);
  }

  /* now process old log file */
  process_logfile(oldlog_str, today_str, 1);

  /* erase old log file now that we are done with it */
  if (system(eraseoldlog_cmd) == -1) {
    fprintf(stderr, "erase command failed\n");
    perror(eraseoldlog_cmd);
    exit(1);
  }

  return 0;
}
