/*
 * Copyright (C) 2004-2005 by David J. Hardy.  All rights reserved.
 *
 * timer.c
 */

#include <stdio.h>
#include <string.h>
#include "timer/timer.h"
#include "debug/debug.h"

#define FAIL  (-1)

/*
 * Note: gettimeofday() is low resolution for performance timing,
 * in the sense that it measures wall clock time, not process time.
 */


Timer *timer_create(void)
{
  Timer *t;

  t = (Timer *) malloc(sizeof(Timer));
  if (t == NULL) {
    return NULL;
  }
  else if (timer_initialize(t)) {
    timer_cleanup(t);
    free(t);
    return NULL;
  }
  return t;
}


int timer_initialize(Timer *t)
{
  struct timeval v;

  ASSERT(t != NULL);
  memset(t, 0, sizeof(Timer));    /* clear memory */
  if (gettimeofday(&v, NULL)) return FAIL;  /* init start time */
  t->tv = v;    /* store it */
  return 0;
}


void timer_cleanup(Timer *t)
{
  ASSERT(t != NULL);
  memset(t, 0, sizeof(Timer));    /* clear memory */
  /* no initialization allocations to cleanup! */
}


void timer_destroy(Timer *t)
{
  ASSERT(t != NULL);
  timer_cleanup(t);
  free(t);
}


double timer_click(Timer *t)
{
  struct timeval v;

  ASSERT(t != NULL);
  if (gettimeofday(&v, NULL)) return FAIL;  /* read the stopwatch */
  t->delta = (double)(v.tv_sec - t->tv.tv_sec)  /* part in seconds */
    + 1e-6 * (v.tv_usec - t->tv.tv_usec);  /* plus microseconds, converted */
  t->total += t->delta;  /* accumulate delta into total */
  t->tv = v;    /* store this stopwatch reading */
  return t->delta;
}


double timer_total(Timer *t)
{
  ASSERT(t != NULL);
  return t->total;
}


const char *timer_msg_delta(Timer *t)
{
  sprintf(t->msg, "%g seconds", t->delta);
  return (const char *)(t->msg);
}


const char *timer_msg_total(Timer *t)
{
  sprintf(t->msg, "%g seconds", t->total);
  return (const char *)(t->msg);
}
