/*
 * Copyright (C) 2007 by David J. Hardy.  All rights reserved.
 *
 * step_setup.c
 */

#include <stdlib.h>
#include <string.h>
#include "step/step_defn.h"
#define DEBUG_WATCH
#undef DEBUG_WATCH
#include "debug/debug.h"


Step *new_Step(void)
{
  return (Step *) malloc(sizeof(Step));
}


void delete_Step(Step *s)
{
  free(s);
}


int step_setup(Step *s, StepParam *param,
    MD_Dvec *pos, MD_Dvec *vel, MD_Dvec *force)
{
  int32 i, retval;

  memset(s, 0, sizeof(Step));

  /* set default error message */
  sprintf(s->errmsg, "no error");

  /* check arguments */
  if (NULL==param || NULL==pos) {
    return step_error(s, "invalid argument");
  }

  /* check basic parameter initialization */
  if (param->method <= 0 || param->method >= STEP_METHOD_MARKER
      || (param->options & ~STEP_OPTION_MARKER) != 0
      || NULL==param->output_message
      || NULL==param->compute_force
      || NULL==param->submit_results
      || NULL==param->atom
      || param->natoms <= 0) {
    return step_error(s, "StepParam parameters are not correctly initialized");
  }
  /* shallow copy StepParam object */
  s->param = param;

  /* allocate StepSystem */
  s->system = (StepSystem *) calloc(1, sizeof(StepSystem));
  if (NULL==s->system) {
    return step_error(s, "calloc() failed to allocate StepSystem");
  }
  s->system->pos = pos;
  s->system->vel = vel;
  s->system->force = force;
  if (NULL==vel) {
    s->system->vel = (MD_Dvec *) calloc(param->natoms, sizeof(MD_Dvec));
    if (NULL==s->system->vel) {
      return step_error(s, "calloc() failed to allocate %d-element "
	  "\"vel\" array", param->natoms);
    }
    s->cleanup_vel = STEP_TRUE;
  }
  if (NULL==force) {
    s->system->force = (MD_Dvec *) calloc(param->natoms, sizeof(MD_Dvec));
    if (NULL==s->system->pos) {
      return step_error(s, "calloc() failed to allocate %d-element "
	  "\"force\" array", param->natoms);
    }
    s->cleanup_force = STEP_TRUE;
  }

  /* setup random number generator */
  random_initseed(&(s->random), param->random_seed);

  /* allocate some common internal arrays */
  s->half_vel = (MD_Dvec *) calloc(param->natoms, sizeof(MD_Dvec));
  if (NULL==s->half_vel) {
    return step_error(s, "calloc() failed to allocate %d-element "
	"\"half_vel\" array", param->natoms);
  }
  s->scal_inv_mass = (double *) malloc(param->natoms * sizeof(double));
  if (NULL==s->scal_inv_mass) {
    return step_error(s, "malloc() failed to allocate %d-element "
	"\"scal_inv_mass\" array", param->natoms);
  }
  for (i = 0;  i < param->natoms;  i++) {
    s->scal_inv_mass[i] = MD_FORCE_CONST / param->atom[i].m;
  }

  /* determine number of fixed atoms */
  s->system->nfixedatoms = 0;
  if (param->options & STEP_FIXEDATOMS) {
    s->doFixedAtoms = STEP_TRUE;
    for (i = 0;  i < param->natoms;  i++) {
      if (param->atom[i].is_fixed) s->system->nfixedatoms++;
    }
    if (step_output(s, "%d fixed atoms\n", s->system->nfixedatoms)) {
      return STEP_FAILURE;
    }
  }

  /* determine number of rigid waters */
  s->system->nrigidwaters = 0;
  if (param->options & STEP_SETTLE) {
    s->doRigidWaters = STEP_TRUE;
    for (i = 0;  i < param->natoms;  i++) {
      if (param->atom[i].is_water) s->system->nrigidwaters++;
    }
    if (step_output(s, "%d rigid waters\n", s->system->nrigidwaters)) {
      return STEP_FAILURE;
    }
  }

  /* determine number of degrees of freedom */
  s->system->ndegfreedom = 3 * param->natoms;
  if (s->system->nfixedatoms > 0) {
    s->system->ndegfreedom -= 3 * s->system->nfixedatoms;
  }
  if (s->system->nrigidwaters > 0) {
    s->system->ndegfreedom -= 3 * s->system->nrigidwaters;
  }
  if (0==s->system->nfixedatoms //&& 0==s->system->nrigidwaters
      && 0==(param->options & STEP_CONSTRAINTS)
      && (param->options & (STEP_REMOVECOM | STEP_ZEROLINMO))) {
    s->system->ndegfreedom -= 3;
  }
  if (s->system->ndegfreedom <= 0) {
    return step_error(s, "no degrees of freedom in system");
  }
  else if (step_output(s, "%d degrees of freedom\n", s->system->ndegfreedom)) {
    return STEP_FAILURE;
  }

  /* determine temperature constant */
  s->tempkonst = 2.0 / (s->system->ndegfreedom * MD_BOLTZMAN);

  /* initialize velocity */
  if (param->options & STEP_INITVEL) {
    if (step_set_random_velocities(s)) return STEP_FAILURE;
  }
  if (((param->options & STEP_RESTARTVEL)==0
        && (param->options & STEP_REMOVECOM))
      || (param->options & STEP_ZEROLINMO)) {
    step_remove_com_motion(s);
  }

  /* set fixed atom velocities to zero */
  if (s->system->nfixedatoms > 0) {
    for (i = 0;  i < param->natoms;  i++) {
      if (param->atom[i].is_fixed) {
        s->system->vel[i].x = 0.;
        s->system->vel[i].y = 0.;
        s->system->vel[i].z = 0.;
      }
    }
  }

  /* setup method */
  switch (param->method) {
    case STEP_VERLET:
      s->compute = step_compute_verlet;
      s->done = step_done_verlet;
      retval = step_init_verlet(s);
      break;
    case STEP_SHADOW:
      s->compute = step_compute_shadow;
      s->done = step_done_shadow;
      retval = step_init_shadow(s);
      break;
    case STEP_TEMPBATH:
      s->compute = step_compute_tempbath;
      s->done = step_done_tempbath;
      retval = step_init_tempbath(s);
      break;
    case STEP_NHEXP:
      s->compute = step_compute_nosehoover_explicit;
      s->done = step_done_nosehoover_explicit;
      retval = step_init_nosehoover_explicit(s);
      break;
    case STEP_DRUDE_NH:
      s->compute = step_compute_drude_thermal;
      s->done = step_done_drude_thermal;
      retval = step_init_drude_thermal(s);
      break;
    case STEP_DRUDE_ROUX:
      s->compute = step_compute_drude_roux;
      s->done = step_done_drude_roux;
      retval = step_init_drude_roux(s);
      break;
    case STEP_DRUDE_CHEN:
      s->compute = step_compute_drude_chen;
      s->done = step_done_drude_chen;
      retval = step_init_drude_chen(s);
      break;
    case STEP_CGMIN:
      s->compute = step_compute_cgmin;
      s->done = step_done_cgmin;
      retval = step_init_cgmin(s);
      break;
    default:
      return step_error(s, "method is not supported");
  }

  /* setup SETTLE if requested */
  if (s->doRigidWaters) {
    if (step_setup_settle(s)) return STEP_FAILURE;
  }

  return retval;
}


void step_cleanup(Step *s)
{
  /* cleanup particular integration method */
  s->done(s);

  if (s->doRigidWaters) step_cleanup_settle(s);

  /* free array space */
  free(s->scal_inv_mass);
  free(s->half_vel);
  if (s->cleanup_force) {
    free(s->system->force);
  }
  if (s->cleanup_vel) {
    free(s->system->vel);
  }
  free(s->system);
}


int step_compute(Step *s, int32 numsteps)
{
  return s->compute(s, numsteps);
}


const char *step_errmsg(Step *s)
{
  return (const char *)(s->errmsg);
}


int step_set_random_velocities(Step *s)
{
  const double init_temp = s->param->initial_temperature;
  const double kbtemp = MD_BOLTZMAN * MD_ENERGY_CONST * init_temp;
  double sqrt_kbtemp_div_mass;
#ifndef STEP_ALT_INITTEMP
  double rnum;
#endif
  const MD_Atom *atom = s->param->atom;
  MD_Dvec *vel = s->system->vel;
  Random *r = &(s->random);
  const int32 natoms = s->param->natoms;
  int32 n, k;

  /* make sure initial temperature is valid */
  if (init_temp < 0.0) {
    return step_error(s, "initial temperature must be nonnegative");
  }

  for (n = 0;  n < natoms;  n++) {
    sqrt_kbtemp_div_mass = sqrt(kbtemp / atom[n].m);

#ifndef STEP_ALT_INITTEMP
    /*
     * The following method and comments taken from NAMD WorkDistrib.C:
     *
     * //  The following comment was stolen from X-PLOR where
     * //  the following section of code was adapted from.
     *
     * //  This section generates a Gaussian random
     * //  deviate of 0.0 mean and standard deviation RFD for
     * //  each of the three spatial dimensions.
     * //  The algorithm is a "sum of uniform deviates algorithm"
     * //  which may be found in Abramowitz and Stegun,
     * //  "Handbook of Mathematical Functions", pg 952.
     */
    rnum = -6.0;
    for (k = 0;  k < 12;  k++) {
      rnum += random_uniform(r);
    }
    vel[n].x = sqrt_kbtemp_div_mass * rnum;

    rnum = -6.0;
    for (k = 0;  k < 12;  k++) {
      rnum += random_uniform(r);
    }
    vel[n].y = sqrt_kbtemp_div_mass * rnum;

    rnum = -6.0;
    for (k = 0;  k < 12;  k++) {
      rnum += random_uniform(r);
    }
    vel[n].z = sqrt_kbtemp_div_mass * rnum;
#else
    /*
     * Alternate method from NAMD Sequencer.C:
     */
    vel[n].x = sqrt_kbtemp_div_mass * random_gaussian(r);
    vel[n].y = sqrt_kbtemp_div_mass * random_gaussian(r);
    vel[n].z = sqrt_kbtemp_div_mass * random_gaussian(r);
#endif
  }
  return 0;
}


void step_remove_com_motion(Step *s)
{
  const MD_Atom *atom = s->param->atom;
  MD_Dvec *vel = s->system->vel;
  const int32 natoms = s->param->natoms;
  int32 i;
  MD_Dvec mv = { 0.0, 0.0, 0.0 };  /* accumulate net momentum */
  double mass = 0.0;               /* accumulate total mass */

  /* compute net momentum and total mass */
  for (i = 0;  i < natoms;  i++) {
    mv.x += atom[i].m * vel[i].x;
    mv.y += atom[i].m * vel[i].y;
    mv.z += atom[i].m * vel[i].z;
    mass += atom[i].m;
  }

  /* scale net momentum by total mass */
  mv.x /= mass;
  mv.y /= mass;
  mv.z /= mass;

  /* remove from atom velocities */
  for (i = 0;  i < natoms;  i++) {
    vel[i].x -= mv.x;
    vel[i].y -= mv.y;
    vel[i].z -= mv.z;
  }
}
