/*
 * Copyright (C) 2007 by David J. Hardy.  All rights reserved.
 */

/**@file    step.h
 * @brief   Time stepping library.
 * @author  David J. Hardy
 * @date    2005
 *
 * The @c Step_t class performs time integration or energy minimization
 * of a system of atoms.  Choose between various methods:
 * @li velocity-Verlet (NVE)
 * @li shadow Hamiltonian (NVE)
 * @li weakly coupled temperature bath
 * @li Nose-Hoover thermostat (NVT)
 * @li Drude oscillators with dual thermostating (NVT,T*)
 * @li energy minimization using conjugate gradient search
 *
 * and various options:
 * @li assign initial velocities from initial temperature distribution
 * @li remove center of mass motion from initial velocity assignment
 * @li zero linear momentum
 * @li rigid waters
 *
 * The @c StepParam_t structure is setup by the caller to configure
 * the @c Step_t class.  Three callback functions are required:
 * one providing force evaluation to @c Step_t for given positions,
 * one collecting data from @c Step_t at the end of a step, and
 * one receiving text output.
 *
 * The data collection callback receives a @c StepSystem_t structure
 * that contains (among other things) current positions, velocities,
 * and energy reductions.
 *
 * The units are as outlined by the MDAPI documentation:
 * @li distance in Angstroms
 * @li time in femtoseconds
 * @li velocity in Angstroms/femtosecond
 * @li energy in kcal/mol
 * @li force in kcal/mol/Angstrom
 * @li temperature in degrees Kelvin
 */

#ifndef STEP_H
#define STEP_H

#include "mdapi/mdtypes.h"

#ifdef __cplusplus
extern "C" {
#endif


  /**@brief Constant values.
   *
   */
  enum {
    STEP_FAILURE = -1,
    /**< Return value indicating an error occurred. */

    STEP_SUCCESS = 0
    /**< Return value indicating all is well. */
  };


  /**@brief Available methods.
   *
   */
  typedef enum StepMethod_t {
    STEP_VERLET = 1,
    /**< velocity Verlet */
    
    STEP_SHADOW,
    /**< computes high-order approximations to the shadow Hamiltonian */
                        
    STEP_TEMPBATH,
    /**< weak coupling to a temperature bath (Berendsen, et al., 1984) */

    STEP_NHEXP,
    /**< Nose-Hoover thermostat with explicit steps */
    
    STEP_DRUDE_NH,
    /**< thermalized Drude oscillators using dual Nose-Hoover thermostats */

    STEP_DRUDE_ROUX,
    /**< thermalized Drude oscillators using Roux's Nose-Hoover integrator */

    STEP_DRUDE_CHEN,
    /**< thermalized Drude oscillators using Chen's method */

    STEP_CGMIN,
    /**< energy minimization using conjugate gradient search directions */

    STEP_METHOD_MARKER

  } StepMethod;


  /**@brief Available options.
   *
   */
  typedef enum StepOption_t {
    STEP_INITVEL   = 0x001,
    /**< assign random velocities from initial temperature distribution */

    STEP_RESTARTVEL= 0x002,
    /**< restarting from velocity file, i.e. prevent STEP_REMOVECOM from
     *   modifying the velocities */

    STEP_REMOVECOM = 0x004,
    /**< remove center of mass motion of velocities assigned
     *   from an initial temperature distribution,
     *   and decrease ndegfreedom by 3
     *   (do not modify velocities if STEP_RESTARTVEL is set) */

    STEP_ZEROLINMO = 0x008,
    /**< impose holonomic constraint to keep linear momentum zero;
     *   should be used to correct for PME or MSM approximations
     *   only when linear momentum would otherwise be conserved */

    STEP_SETTLE    = 0x010,
    /**< use SETTLE (analytical RATTLE) for rigid water */

    STEP_VIRIAL    = 0x020,
    /**< compute kinetic contribution to virial */

    STEP_MOMENTUM  = 0x040,
    /**< compute linear and angular momentum */

    STEP_FIXEDATOMS= 0x080,
    /**< pay attention to fixed atom status flag; fixed atoms are
     *   assigned zero velocity, their positions are left unchanged,
     *   and the number of fixed atoms is considered in determining
     *   number of degrees of freedom */

    STEP_CONSTRAINTS=0x100,
    /**< harmonic constraints/restraints are to be applied to some
     *   atoms in the system */

    STEP_OPTION_MARKER = 0x1FF

  } StepOption;


  struct StepSystem_t;
  typedef struct StepSystem_t StepSystem;


  /**@brief Parameters for configuring Step class.
   *
   * User must provide values where appropriate, depending on selected
   * method.  Must provide callbacks.  Retains ownership of topology
   * and force field arrays.
   */
  typedef struct StepParam_t {

    StepMethod method;
    /**< Choose a method for integration or minimization. */

    StepOption options;
    /**< Select one or more options by bitwise ORing. */

    unsigned long random_seed;
    /**< Seed for the random number generator. */

    double timestep;
    /**< Time step (in femtoseconds). */

    double initial_temperature;
    /**< Initial temperature (in Kelvin).  This is used to initialize
     *   velocities unless they are assigned in step_setup(). */

/*** callbacks ***/

    void *messageObjPtr;
    /**< Generic handle passed to output_message() callback. */

    int (*output_message)(void *messageObjPtr, const char *msg);
    /**< Routine to receive messages from Step library. */

    void *forceObjPtr;
    /**< Generic handle passed to compute_force() callback. */

    int (*compute_force)(void *forceObjPtr,
        double *pe, MD_Dvec *force, const MD_Dvec *pos);
    /**< Computes @c force array for given @c pos array.
     *   Returns 0 for success.  Must be provided. */

    void *resultsObjPtr;
    /**< Generic handle passed to submit_results() callback. */

    int (*submit_results)(void *resultsObjPtr, StepSystem *sys);
    /**< Collects results at end-of-step intervals.
     *   Returns 0 for success.  Must be provided. */

    int32 resultsFreq;
    /**< Gives number of steps between each submit_results() call. */

    int32 init_stepnum;
    /**< Gives the initial step number. */

/*** topology ***/

    MD_Atom *atom;
    /**< Array of atom data for system.  (Only the mass is needed.) */

    int32 natoms;
    /**< Number of atoms in array.  Used as length for position, velocity,
     *   and force arrays. */

#if 0
    MD_Bond *bond;
    /**< Array of spring bonds for system.  (Needed for Drude and SETTLE.) */

    int32 nbonds;
    /**< Number of bonds in array. */

    MD_Angle *angle;
    /**< Array of angle bonds for system.  (Needed for SETTLE.) */

    int32 nangles;
    /**< Number of angles in array. */

/*** force field ***/

    MD_BondPrm *bondprm;
    /**< Array of bond force field parameters.  (Needed for SETTLE.) */

    int32 nbondprms;
    /**< Number of bond force field parameters in array. */

    MD_AnglePrm *angleprm;
    /**< Array of angle force field parameters.  (Needed for SETTLE.) */

    int32 nangleprms;
    /**< Number of angle force field parameters in array. */
#endif

/*** parameters for STEP_TEMPBATH ***/

    double bath_temperature;
    /**< Temperature of the bath (in Kelvin).  Can be set to 0. */

    double relaxation_time;
    /**< Relaxation time to determine coupling to temperature bath.
     *   Units in femtoseconds.  Need relaxation_time >= timestep.
     *   Use of relaxation_time == timestep gives simple velocity
     *   rescaling. (Leach recommends 400 fs for timestep==1fs.) */

/*** parameters for STEP_NHEXP ****/

    double nh_temperature;
    /**< Nose-Hoover thermostat temperature for canonical ensemble (K).
     *   Must be greater than 0. */

    double nh_timescale;
    /**< Nose-Hoover thermostat timescale.  Controls coupling of the system
     *   to the thermostat.  A reasonable choice is  50 fs, the period of a
     *   harmonic oscillator. */
      
/*** parameters for STEP_DRUDE ***/

    double drude_com_temperature;
    /**< Thermostat temperature for atomic COMs (centers of mass),
     *   the hot degrees of freedom. */

    double drude_bond_temperature;
    /**< Thermostat temperature for Drude bond displacements,
     *   the cold degrees of freedom. */

    double drude_com_timescale;
    /**< Thermostat timescale.  Controls coupling of the COM system to
     *   its thermostat.  B. Roux recommends 100 fs. */

    double drude_bond_timescale;
    /**< Thermostat timescale.  Controls coupling of the bond system to
     *   its thermostat.  B. Roux recommends 7.07 fs for Drude particles
     *   with mass 0.8 AMU, and shorter for smaller mass values. */

    int32 drude_multisteps;
    /**< Number of multisteps for propagating extended system variables.
     *   Must be at least 1.  Recommended greater than 1, between 2 and 100. */

/*** parameters for STEP_CGMIN ***/

    double cgmin_dis;
    /**< Maximum displacement for line search; 1 A works OK. */

    double cgmin_tol;
    /**< Tolerance for line search; 0.01 A seems OK for most systems. */

    int32 cgmin_eval;
    /**< Maximum number of potential evaluations before giving up;
     *   50 seems to work well in practice. */

/*** parameters for STEP_SETTLE option ***/

    double settle_oh_dist;
    /**< Bond equilibrium length for water O-H. */

    double settle_hoh_angle;
    /**< Angle equilibrium length for water H-O-H. */

  } StepParam;


  /**@brief For indexing @c StepSystem_t::shadow_energy. */
  enum StepShadowIndex_t {
    STEP_SHADOW_4 = 0,    /**< 4th order approximation */
    STEP_SHADOW_8,        /**< 8th order approximation */
    STEP_SHADOW_12,       /**< 12th order approximation */
    STEP_SHADOW_16,       /**< 16th order approximation */
    STEP_SHADOW_20,       /**< 20th order approximation */
    STEP_SHADOW_24,       /**< 24th order approximation */
    STEP_SHADOW_MAXINDEX
  };


  /**@brief Provides system data through callback.
   *
   * These should be read from, not written to.
   */
  struct StepSystem_t {

    double kinetic_energy;
    /**< Kinetic energy of system (kcal/mol). */

    double temperature;
    /**< Temperature of system (K). */

    MD_Dvec linear_momentum;
    /**< Linear momentum of system. */

    MD_Dvec angular_momentum;
    /**< Angular momentum of system. */

    double kinetic_virial[9];
    /**< Kinetic contribution to the pressure virial. */

    double potential_energy;
    /**< Potential energy from most recent force evaluation (kcal/mol). */

    double shadow_energy[STEP_SHADOW_MAXINDEX];
    /**< Shadow energy of system (kcal/mol), very well-conserved. */

    double nh_extended_energy;
    /**< Nose-Hoover extended energy, a conserved quantity. */

    double nh_friction_coef;
    /**< Nose-Hoover friction coefficient. */

    double nh_log_s;
    /**< Nose-Hoover log(s). */

    double drude_com_energy;
    /**< Drude energy for COM degrees. */

    double drude_bond_energy;
    /**< Drude energy for bond displacement degrees. */

    double drude_extended_energy;
    /**< Drude extended energy, a conserved quantity. */

    double drude_com_temperature;
    /**< Drude temperature for COM degrees. */

    double drude_bond_temperature;
    /**< Drude temperature for bond displacement degrees. */

    MD_Dvec *force;
    /**< Contains most recent force evaluation. */

    MD_Dvec *vel;
    /**< System velocities. */

    MD_Dvec *pos;
    /**< System positions. */

    int32 stepnum;
    /**< The current step number. */

    int32 ndegfreedom;
    /**< Number of degrees of freedom determined by method and options.
     *   (Is 3N for a system with no constraints, less 3 for zero linear
     *   momentum or less 3 times number of fixed atoms or less 3 times
     *   number of rigid water molecules.) */

    int32 nfixedatoms;
    /**< Number of fixed atoms, indicated by status flag; this remains
     *   zero unless STEP_FIXEDATOMS option is enabled. */

    int32 nrigidwaters;
    /**< Number of rigid waters, indicated by status flag. */

  };


  /**@brief The Step class.
   */
  struct Step_t;
  typedef struct Step_t Step;


  /**@brief Constructor.
   */
  Step *new_Step(void);

  /**@brief Destructor.
   */
  void delete_Step(Step *);

  /**@brief Setup.
   *
   * @param[in] param      @c StepParam_t object for configuration.
   * @param[in,out] pos    @c Initial positions.
   * @param[in,out] vel    @c Initial velocities.
   * @param[in,out] force  @c Array buffer for forces.
   *
   * Initializes the @c Step_t object.  Must be called before
   * @c Step_t::step_compute().  Initial positions must be provided.
   * The other two arrays can be optionally assigned NULL.
   * (If @c vel is not provided, then the initial velocities are assigned
   * from the initial temperature distribution indicated in @c StepParam_t,
   * no matter the value of @c StepParam_t::option.)
   *
   * Caller is responsible for memory management for any provided arrays.
   *
   * @return @c STEP_SUCCESS, or @c STEP_FAILURE on error.
   */
  int step_setup(Step *, StepParam *param,
      MD_Dvec *pos, MD_Dvec *vel, MD_Dvec *force);


  /**@brief Performs the time stepping.
   *
   * @param[in] numsteps  Number of steps to integrate or minimize the system.
   *
   * Note that calls back to the @c StepParam_t::compute_force() routine
   * account for the majority of the computational cost.
   *
   * @return @c STEP_SUCCESS, or @c STEP_FAILURE on error.
   */
  int step_compute(Step *, int32 numsteps);


  /**@brief Error message.
   *
   * Provide text error message; call if an error is returned by
   * another @c Step_t class method.
   */
  const char *step_errmsg(Step *);


  /**@brief Cleanup.
   *
   * Call this to clean up internal state of @c Step_t object.
   */
  void step_cleanup(Step *);


#ifdef __cplusplus
}
#endif

#endif /* STEP_H */
