/*
 * Copyright (C) 2004-2006 by Wei Wang.  All rights reserved.
 * The dpsim implementation is based on NAMD2 implementation.
 *
 * Modified by David Hardy for use in MDX step library.
 */

/*
 * The original paper is
 * SETTLE: an analytical version of the SHAKE and RATTLE algorithm
 * for rigid water models 
 * Shuichi Miyamoto and Peter Kollman,  Journal of Computational Chemistry
 * Vol 13, 954--962, yr 1992
 *
 * Oxygen at a0, hydrogens at b0, c0, center of mass at the origin.
 *
 *                    |
 *                    |
 *                    |
 *                 a0 .-------
 *                    |   |
 *                    |   ra
 *                    |   | 
 * ---------------------------------------
 *            |       |        |
 *            rb      |---rc---|
 *            |       |        |
 *     b0 .-----------|        . c0 
 *   
 * SETTLE for step one of velocity verlet.
 * ref:  positions before unconstrained step
 * mass: masses
 * pos:  on input:  positions after unconstrained step;
 *       on output: the new positions
 * vel:  on input:  velocities after unconstrained step;
 *       on output: the new velocities
 * ra, rb, rc: canonical positions of water atoms; see above diagram
 */

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <math.h>
#include "step/step_defn.h"
#undef DEBUG_WATCH
#include "debug/debug.h"

#define EPSILON 1e-8


static int settle1(Step *s,
    const MD_Dvec ref[3], MD_Dvec rnew[3], MD_Dvec vel[3], const double dt);

static int settle2(Step *s, const MD_Dvec r[3], MD_Dvec v[3], const double dt);


#if 0
/* for debugging */
#undef VEC
#define VEC(v) printf("%s=%.10g %.10g %.10g\n", #v, (v).x, (v).y, (v).z);
#undef FLT
#define FLT(x) printf("%s=%g\n", #x, x);
#endif


int step_setup_settle(Step *s)
{
  const double oh_dist = s->param->settle_oh_dist;
  const double hoh_angle = s->param->settle_hoh_angle;
  double hh_dist;
  double t1;
  double o_mass = 15.9994;  /* default mass */
  double h_mass = 1.0079;
  const MD_Atom *atom = s->param->atom;
  const int32 natoms = s->param->natoms;
  int32 i;

  /* store reference positions */
  s->refpos = (MD_Dvec *) calloc(natoms, sizeof(MD_Dvec));
  if (NULL==s->refpos) {
    return step_error(s, "calloc() failed to allocate %d-element "
        "\"refpos\" array", natoms);
  }

  /* determine actual mass values */
  for (i = 0;  i < natoms-2;  i++) {
    if (atom[i].is_water) {
      if (s->is_drude) {
        if (i+3 < natoms && atom[i+3].is_hydrogen && atom[i+3].is_other) {
          o_mass = atom[i].m + atom[i+3].m;
          if (step_output(s, "using SETTLE for Drude polarized water,\n")
              || step_output(s,
                "setting SETTLE oxygen mass to total mass %g AMU\n", o_mass)) {
            return STEP_FAILURE;
          } 
        }
        else if (i+3 < natoms && atom[i+3].is_water) {
          o_mass = atom[i].m;
          if (step_output(s,
        "using SETTLE with Drude polarization, but can't find an attached\n")
              || step_output(s,
        "Drude particle, so assuming that waters are NOT polarized,\n")
              || step_output(s,
        "setting SETTLE oxygen mass to %g AMU\n", o_mass)) {
            return STEP_FAILURE;
          } 
        }
      }
      else {
        o_mass = atom[i].m;
      }
      h_mass = atom[i+1].m;
      break;
    }
  }

  /* set parameters */
  hh_dist = sqrt(2*oh_dist*oh_dist*(1.0 - cos(hoh_angle)));
  t1 = 0.5 * o_mass/h_mass;
  s->o_mass = o_mass;
  s->h_mass = h_mass;
  s->rc = 0.5 * hh_dist;
  s->ra = sqrt(oh_dist*oh_dist - s->rc*s->rc)/(1.0 + t1);
  s->rb = t1 * s->ra;

  if (step_output(s, "using SETTLE for rigid waters\n")) {
    return STEP_FAILURE;
  }
  return STEP_SUCCESS;
}


void step_cleanup_settle(Step *s)
{
  free(s->refpos);
}


/*
 * this is how NAMD starts rattle/settle
 *
 * df is used to compute the constraint virial
 */
int step_settle_startup(Step *s)
{
  const double timestep = s->param->timestep;
  const double half_timestep = 0.5 * timestep;
  MD_Dvec *f = s->system->force;
  MD_Dvec *vel = s->system->vel;
  MD_Dvec *pos = s->system->pos;
  const MD_Dvec *ref = s->refpos;
  const double *scal_inv_mass = s->scal_inv_mass;
  const int32 natoms = s->param->natoms;
  MD_Atom *atom = s->param->atom;
  double dt;
  MD_Dvec npos[3], nvel[3], df;
  int32 i, j;

  if (step_settle_prep(s)) return STEP_FAILURE;

  /* settle1() - keep new positions but throw away new velocities */
  dt = 0.;
  for (i = 0;  i < natoms-2;  i++) {
    if (atom[i].is_water) {
      for (j = 0;  j < 3;  j++) {
        nvel[j] = vel[i+j];
      }
      if (settle1(s, ref+i, pos+i, nvel, dt)) {
        return step_error(s, "error returned from settle1()");
      }
      i += 2;  /* skip 2, plus 1 from loop increment */
    }
  }

  /* compute initial force */
  if (step_force(s)) return STEP_FAILURE;

  /* half kick backwards */
  for (i = 0;  i < natoms;  i++) {
    double c = -half_timestep * scal_inv_mass[i];
    vel[i].x += c * f[i].x;
    vel[i].y += c * f[i].y;
    vel[i].z += c * f[i].z;
  }

  step_settle2(s, -half_timestep);

  /* settle1() - keep new velocities but throw away new positions */
  dt = -timestep;
  for (i = 0;  i < natoms-2;  i++) {
    if (atom[i].is_water) {
      for (j = 0;  j < 3;  j++) {
        nvel[j] = vel[i+j];
        npos[j].x = pos[i+j].x + nvel[j].x * dt;
        npos[j].y = pos[i+j].y + nvel[j].y * dt;
        npos[j].z = pos[i+j].z + nvel[j].z * dt;
      }
      if (settle1(s, pos+i, npos, nvel, dt)) {
        return step_error(s, "error returned from settle1()");
      }
      for (j = 0;  j < 3;  j++) {
        vel[i+j] = nvel[j];
      }
      i += 2;  /* skip 2, plus 1 from loop increment */
    }
  }

  /* whole kick forwards */
  for (i = 0;  i < natoms;  i++) {
    double c = timestep * scal_inv_mass[i];
    vel[i].x += c * f[i].x;
    vel[i].y += c * f[i].y;
    vel[i].z += c * f[i].z;
  }

  step_settle2(s, timestep);

  /* settle1() - keep new velocities but throw away new positions */
  dt = timestep;
  for (i = 0;  i < natoms-2;  i++) {
    if (atom[i].is_water) {
      for (j = 0;  j < 3;  j++) {
        nvel[j] = vel[i+j];
        npos[j].x = pos[i+j].x + nvel[j].x * dt;
        npos[j].y = pos[i+j].y + nvel[j].y * dt;
        npos[j].z = pos[i+j].z + nvel[j].z * dt;
      }
      if (settle1(s, pos+i, npos, nvel, dt)) {
        return step_error(s, "error returned from settle1()");
      }
      for (j = 0;  j < 3;  j++) {
        double c = atom[i+j].m / dt;
        df.x = (nvel[j].x - vel[i+j].x) * c;
        df.y = (nvel[j].y - vel[i+j].y) * c;
        df.z = (nvel[j].z - vel[i+j].z) * c;
        f[i+j].x += df.x;
        f[i+j].y += df.y;
        f[i+j].z += df.z;
        vel[i+j] = nvel[j];
      }
      i += 2;  /* skip 2, plus 1 from loop increment */
    }
  }

  /* half kick backwards */
  for (i = 0;  i < natoms;  i++) {
    double c = -half_timestep * scal_inv_mass[i];
    vel[i].x += c * f[i].x;
    vel[i].y += c * f[i].y;
    vel[i].z += c * f[i].z;
  }

  step_settle2(s, -half_timestep);

  return STEP_SUCCESS;
}


/* save reference position before unconstrained step */
int step_settle_prep(Step *s)
{
  memcpy(s->refpos, s->system->pos, s->param->natoms * sizeof(MD_Dvec));
  return STEP_SUCCESS;
}


/* settle pos and vel after unconstrained drift step */
int step_settle1(Step *s, const double dt)
{
  const MD_Dvec *ref = s->refpos;
  MD_Dvec *pos = s->system->pos;
  MD_Dvec *vel = s->system->vel;
  const MD_Atom *atom = s->param->atom;
  const int32 natoms = s->param->natoms;
  int32 i;

  if (s->is_drude) {
    /* adjust the attached Drude particle to move with the oxygen */
    const DrudeBond *drudeBond = s->drudeBond;
    int32 k = 0;
    TEXT("settle1 drude");
    for (i = 0;  i < natoms-2;  i++) {
      if (atom[i].is_water) {
        MD_Dvec o_pos = pos[i];
        MD_Dvec o_vel = vel[i];
        if (settle1(s, ref+i, pos+i, vel+i, dt)) {
          return step_error(s, "error returned from settle1()");
        }
        if (i == drudeBond[k].atomIndex) {
          const int32 j = drudeBond[k].drudeIndex;
          pos[j].x += (pos[i].x - o_pos.x);
          pos[j].y += (pos[i].y - o_pos.y);
          pos[j].z += (pos[i].z - o_pos.z);
          vel[j].x += (vel[i].x - o_vel.x);
          vel[j].y += (vel[i].y - o_vel.y);
          vel[j].z += (vel[i].z - o_vel.z);
          k++;
        }
        i += 2;  /* skip 2, plus 1 from loop increment */
      }
    }
  }
  else {
    for (i = 0;  i < natoms-2;  i++) {
      if (atom[i].is_water) {
        if (settle1(s, ref+i, pos+i, vel+i, dt)) {
          return step_error(s, "error returned from settle1()");
        }
        i += 2;  /* skip 2, plus 1 from loop increment */
      }
    }
  }
  return STEP_SUCCESS;
}


/* refine vel after final kick */
int step_settle2(Step *s, const double dt)
{
  MD_Dvec *pos = s->system->pos;
  MD_Dvec *vel = s->system->vel;
  const MD_Atom *atom = s->param->atom;
  const int32 natoms = s->param->natoms;
  int32 i;

  if (s->is_drude) {
    /* adjust the attached Drude particle to move with the oxygen */
    const DrudeBond *drudeBond = s->drudeBond;
    int32 k = 0;
    TEXT("settle2 drude");
    for (i = 0;  i < natoms-2;  i++) {
      if (atom[i].is_water) {
        MD_Dvec o_vel = vel[i];
        if (settle2(s, pos+i, vel+i, dt)) {
          return step_error(s, "error returned from settle2()");
        }
        if (i == drudeBond[k].atomIndex) {
          const int32 j = drudeBond[k].drudeIndex;
          vel[j].x += (vel[i].x - o_vel.x);
          vel[j].y += (vel[i].y - o_vel.y);
          vel[j].z += (vel[i].z - o_vel.z);
          k++;
        }
        i += 2;  /* skip 2, plus 1 from loop increment */
      }
    }
  }
  else {
    for (i = 0;  i < natoms-2;  i++) {
      if (atom[i].is_water) {
        if (settle2(s, pos+i, vel+i, dt)) {
          return step_error(s, "error returned from settle2()");
        }
        i += 2;  /* skip 2, plus 1 from loop increment */
      }
    }
  }
  return STEP_SUCCESS;
}



#define MD_vec_subtract(v1, v2, vdiff) { \
  (vdiff).x = (v1).x - (v2).x; \
  (vdiff).y = (v1).y - (v2).y; \
  (vdiff).z = (v1).z - (v2).z; \
}

#define MD_vec_mul(v1, c, vprod) { \
  (vprod).x = (c)*(v1).x; \
  (vprod).y = (c)*(v1).y; \
  (vprod).z = (c)*(v1).z; \
} 

#define MD_vec_dot(v1, v2) ((v1).x*(v2).x + (v1).y*(v2).y + (v1).z*(v2).z)

#define MD_vec_cross(v1, v2, v) { \
  (v).x = (v1).y * (v2).z - (v1).z * (v2).y; \
  (v).y = (v1).z * (v2).x - (v1).x * (v2).z; \
  (v).z = (v1).x * (v2).y - (v2).x * (v1).y; \
}

#define MD_vec_len(v) (sqrt((v).x*(v).x + (v).y*(v).y + (v).z*(v).z))


int settle1(Step *s,
    const MD_Dvec ref[3], MD_Dvec rnew[3], MD_Dvec vel[3],
    const double dt)
{
  const double mO = s->o_mass;
  const double mH = s->h_mass;
  const double ra = s->ra;
  const double rb = s->rb;
  const double rc = s->rc;
  MD_Dvec b0, c0, d0, a1, b1, c1, a2, b2, c2, a3, b3, c3;
  MD_Dvec n1, n2, n0, m1, m2, m0;
  MD_Dvec v;
  double sinphi, cosphi, sinpsi, cospsi;
  double rbsphi, rbcphi;
  double alpha, beta, gamma;
  double sintheta, costheta, a2b2;
  double tmp, tmp1, tmp2;

  /* vectors in the plane of the original positions */
  MD_vec_subtract(ref[1], ref[0], b0);
  MD_vec_subtract(ref[2], ref[0], c0);

  tmp = 1.0 / (mO+mH+mH);
  d0.x = (rnew[0].x*mO + rnew[1].x*mH + rnew[2].x*mH) * tmp;
  d0.y = (rnew[0].y*mO + rnew[1].y*mH + rnew[2].y*mH) * tmp;
  d0.z = (rnew[0].z*mO + rnew[1].z*mH + rnew[2].z*mH) * tmp;
  
  MD_vec_subtract(rnew[0], d0, a1);
  MD_vec_subtract(rnew[1], d0, b1);
  MD_vec_subtract(rnew[2], d0, c1);

  /* Vectors describing transformation from original coordinate system to
     the 'primed' coordinate system as in the diagram. */
  MD_vec_cross(b0, c0, n0);
  MD_vec_cross(a1, n0, n1);
  MD_vec_cross(n0, n1, n2);
  tmp = 1.0/MD_vec_len(n0); MD_vec_mul(n0, tmp, n0);
  tmp = 1.0/MD_vec_len(n1); MD_vec_mul(n1, tmp, n1);
  tmp = 1.0/MD_vec_len(n2); MD_vec_mul(n2, tmp, n2);

  v.x = MD_vec_dot(n1,b0); 
  v.y = MD_vec_dot(n2,b0);
  v.z = MD_vec_dot(n0,b0);
  b0.x=v.x; b0.y=v.y; b0.z=v.z;
  v.x = MD_vec_dot(n1,c0);
  v.y = MD_vec_dot(n2,c0);
  v.z = MD_vec_dot(n0,c0);
  c0.x=v.x; c0.y=v.y; c0.z=v.z;
  v.x = MD_vec_dot(n1,a1);
  v.y = MD_vec_dot(n2,a1);
  v.z = MD_vec_dot(n0,a1);
  a1.x=v.x; a1.y=v.y; a1.z=v.z;
  v.x = MD_vec_dot(n1,b1);
  v.y = MD_vec_dot(n2,b1);
  v.z = MD_vec_dot(n0,b1);
  b1.x=v.x; b1.y=v.y; b1.z=v.z;
  v.x = MD_vec_dot(n1,c1);
  v.y = MD_vec_dot(n2,c1);
  v.z = MD_vec_dot(n0,c1);
  c1.x=v.x; c1.y=v.y; c1.z=v.z;

  /* now we can compute positions of canonical water */
  sinphi = a1.z/ra;
  tmp = 1.0-sinphi*sinphi;
  ASSERT(tmp >= 0.0);
  cosphi = sqrt(tmp);
  sinpsi = (b1.z - c1.z)/(2.0*rc*cosphi);
  tmp = 1.0-sinpsi*sinpsi;
  ASSERT(tmp >= 0.0);
  cospsi = sqrt(tmp);

  rbcphi = -rb*cosphi;
  rbsphi = -rb*sinphi;
  tmp  = rc*sinpsi;
  tmp1 = tmp*sinphi;
  tmp2 = tmp*cosphi;
  a2.x=0.0;        a2.y=ra*cosphi;   a2.z=ra*sinphi;
  b2.x=-rc*cospsi; b2.y=rbcphi-tmp1; b2.z=rbsphi+tmp2; 
  c2.x= rc*cospsi; c2.y=rbcphi+tmp1; c2.z=rbsphi-tmp2; 

  /* there are no a0 terms because we've already subtracted the term off 
     when we first defined b0 and c0. */
  alpha = b2.x*(b0.x - c0.x) + b0.y*b2.y + c0.y*c2.y;
  beta  = b2.x*(c0.y - b0.y) + b0.x*b2.y + c0.x*c2.y;
  gamma = b0.x*b1.y - b1.x*b0.y + c0.x*c1.y - c1.x*c0.y;
 
  a2b2 = alpha*alpha + beta*beta;
  sintheta = (alpha*gamma - beta*sqrt(a2b2 - gamma*gamma))/a2b2;
  costheta = sqrt(1.0 - sintheta*sintheta);
  
  a3.x =-a2.y*sintheta;
  a3.y = a2.y*costheta;
  a3.z = a1.z;
  b3.x = b2.x*costheta - b2.y*sintheta;
  b3.y = b2.x*sintheta + b2.y*costheta;
  b3.z = b1.z;
  c3.x =-b2.x*costheta - c2.y*sintheta;
  c3.y =-b2.x*sintheta + c2.y*costheta;
  c3.z = c1.z;

  /* undo the transformation; generate new normal vectors from the transpose.*/
  m1.x=n1.x; m1.y=n2.x; m1.z=n0.x;
  m2.x=n1.y; m2.y=n2.y; m2.z=n0.y;
  m0.x=n1.z; m0.y=n2.z; m0.z=n0.z;

  rnew[0].x = MD_vec_dot(a3, m1) + d0.x;
  rnew[0].y = MD_vec_dot(a3, m2) + d0.y;
  rnew[0].z = MD_vec_dot(a3, m0) + d0.z;
  rnew[1].x = MD_vec_dot(b3, m1) + d0.x;
  rnew[1].y = MD_vec_dot(b3, m2) + d0.y;
  rnew[1].z = MD_vec_dot(b3, m0) + d0.z;
  rnew[2].x = MD_vec_dot(c3, m1) + d0.x;
  rnew[2].y = MD_vec_dot(c3, m2) + d0.y;
  rnew[2].z = MD_vec_dot(c3, m0) + d0.z;

#ifdef DEBUG_SETTLE
  tmp = sqrt((ra+rb)*(ra+rb) + rc*rc);
  MD_vec_subtract(rnew[0], rnew[1], v);
  ASSERT(fabs(MD_vec_dot(v,v) - tmp*tmp) < EPSILON);  
  MD_vec_subtract(rnew[0], rnew[2], v);
  ASSERT(fabs(MD_vec_dot(v,v) - tmp*tmp) < EPSILON);
  MD_vec_subtract(rnew[2], rnew[1], v);
  ASSERT(fabs(MD_vec_dot(v,v) - 4.0*rc*rc) < EPSILON);
#endif
  
  if (dt != 0) {
    tmp = 1.0/dt;
    vel[0].x = (rnew[0].x - ref[0].x) * tmp;  /* good ? */
    vel[0].y = (rnew[0].y - ref[0].y) * tmp;
    vel[0].z = (rnew[0].z - ref[0].z) * tmp;
    vel[1].x = (rnew[1].x - ref[1].x) * tmp;
    vel[1].y = (rnew[1].y - ref[1].y) * tmp;
    vel[1].z = (rnew[1].z - ref[1].z) * tmp;
    vel[2].x = (rnew[2].x - ref[2].x) * tmp;
    vel[2].y = (rnew[2].y - ref[2].y) * tmp;
    vel[2].z = (rnew[2].z - ref[2].z) * tmp;
  }

  return STEP_SUCCESS;
}


int settle2(Step *s, const MD_Dvec r[3], MD_Dvec v[3], const double dt)
{
  /* rigid water model parameters. d1 = d(O, H1) = d(O, H2),
   * d2 = d(H1, H2).  theta = angle(H1OH2). */
  const double errTol = 1e-6; /* bond length */
  const double inv_mo = 1.0 / s->o_mass;
  const double inv_mh = 1.0 / s->h_mass;
  MD_Dvec v10t, v20t, v21t;
  MD_Dvec r10, r20, r21;
  double c11, c12, c13, c21, c22, c23, c31, c32, c33;
  double inv_det;
  double b1, b2, b3;
  double inv_dt = 1.0 / dt;
  double u01, u02, u12;

  MD_vec_subtract(v[1], v[0], v10t);
  MD_vec_subtract(v[2], v[0], v20t);
  MD_vec_subtract(v[2], v[1], v21t);
  MD_vec_subtract(r[1], r[0], r10);
  MD_vec_subtract(r[2], r[0], r20);
  MD_vec_subtract(r[2], r[1], r21);

  /* must use the real values, not desired theoretical values */
  c11 = (inv_mo + inv_mh) * MD_vec_dot(r10, r10);
  c22 = (inv_mo + inv_mh) * MD_vec_dot(r20, r20);
  c33 = (inv_mh + inv_mh) * MD_vec_dot(r21, r21);
  c12 =  inv_mo * MD_vec_dot(r10, r20);
  c13 = -inv_mh * MD_vec_dot(r10, r21);
  c23 =  inv_mh * MD_vec_dot(r20, r21);
  c21 = c12;
  c22 = c11;
  c31 = c13;
  c32 = c23;
  
  inv_det = 1.0 /  (c11*c22*c33 + c21*c32*c13 + c31*c12*c23
                  - c13*c22*c31 - c23*c32*c11 - c33*c12*c21);

  /*
    fprintf(stderr, "<v1-v0, r1-r0> = %f\n", MD_vec_dot(v10t, r10));
    fprintf(stderr, "<v2-v0, r2-r0> = %f\n", MD_vec_dot(v20t, r20));
    fprintf(stderr, "<v2-v1, r2-r1> = %f\n", MD_vec_dot(v21t, r21));  
    
    fprintf(stderr, "|r1-r0| = %f\n", sqrt(MD_vec_dot(r10, r10)));
    fprintf(stderr, "|r2-r0| = %f\n", sqrt(MD_vec_dot(r20, r20)));
    fprintf(stderr, "|r2-r1| = %f\n", sqrt(MD_vec_dot(r21, r21)));
  */

  b1 = MD_vec_dot(v10t, r10) * inv_dt;
  b2 = MD_vec_dot(v20t, r20) * inv_dt;
  b3 = MD_vec_dot(v21t, r21) * inv_dt;

  u01 = (b1*c22*c33 + b2*c32*c13 + b3*c23*c12
      -  c13*c22*b3 - c23*c32*b1 - c33*b2*c12) * inv_det;
  u02 = (c11*b2*c33 + c21*b3*c13 + c31*c23*b1
      -  c13*b2*c31 - c23*b3*c11 - c33*c21*b1) * inv_det;
  u12 = (c11*c22*b3 + c21*c32*b1 + c31*b2*c12
      -  b1*c22*c31 - b2*c32*c11 - b3*c21*c12) * inv_det;

  v[0].x += dt*inv_mo * ( u01*r10.x + u02*r20.x);
  v[0].y += dt*inv_mo * ( u01*r10.y + u02*r20.y);
  v[0].z += dt*inv_mo * ( u01*r10.z + u02*r20.z);
  v[1].x += dt*inv_mh * (-u01*r10.x + u12*r21.x);
  v[1].y += dt*inv_mh * (-u01*r10.y + u12*r21.y);
  v[1].z += dt*inv_mh * (-u01*r10.z + u12*r21.z);
  v[2].x += dt*inv_mh * (-u02*r20.x - u12*r21.x);
  v[2].y += dt*inv_mh * (-u02*r20.y - u12*r21.y);
  v[2].z += dt*inv_mh * (-u02*r20.z - u12*r21.z);

  /* check orthogonality */
  MD_vec_subtract(v[1], v[0], v10t);
  MD_vec_subtract(v[2], v[0], v20t);
  MD_vec_subtract(v[2], v[1], v21t);
  if (fabs(MD_vec_dot(v10t, r10)) > errTol ||
      fabs(MD_vec_dot(v20t, r20)) > errTol ||
      fabs(MD_vec_dot(v21t, r21)) > errTol) {
    step_output(s, "ERROR:  SETTLE orthogonality condition violated\n");
    step_output(s, "ERROR:  <v1-v0, r1-r0> = %g\n", MD_vec_dot(v10t, r10));
    step_output(s, "ERROR:  <v2-v0, r2-r0> = %g\n", MD_vec_dot(v20t, r20));
    step_output(s, "ERROR:  <v2-v1, r2-r1> = %g\n", MD_vec_dot(v21t, r21));
    return STEP_FAILURE; 
  }

  return STEP_SUCCESS;
}
