/*
 * Copyright (C) 2004-2005 by David J. Hardy.  All rights reserved.
 *
 * run.c
 */

#include <stdlib.h>
#include <string.h>
#include <strings.h>
#include "mdapi/mdengine.h"
#include "force/force.h"
#include "simen/engine.h"
#include "debug/debug.h"


/* prototypes for internal functions */
static int32 reductions(Engine *);


int32 simen_run(MD_Front *front, int32 numsteps, int32 flags)
{
  Engine *eng = MD_engine_data(front);
  Force *force = eng->force;
  ForceResult *fr = &(eng->f_result);
  const MD_Atom *atom = (MD_Atom *) (eng->atom->buf);
  MD_Dvec *pos = (MD_Dvec *) (eng->pos->buf);
  MD_Dvec *vel = (MD_Dvec *) (eng->vel->buf);
  MD_Dvec *f = fr->f_total;
  double konst_div_mass;
  const double dt = eng->timestep;
  const double half_dt = 0.5 * dt;
  int32 natoms = eng->natoms;
  int32 step, k;

  /* update internal data and force library */
  if (flags & MD_UPDATE) {

    /* call update routine */
    if (simen_update(front)) return MD_FAIL;

    /* values might have changed by update(), re-init */
    f = fr->f_total;
    natoms = eng->natoms;

    /* previous force is invalid, must (re-)evaluate */
    if (force_compute(force, fr, pos)) {
      return MD_error(front, eng->err_force);
    }
  }

  /* see if callbacks need to be processed before stepping */
  if (MD_ready_callback(front)) {

    /* reductions might be needed by callback */
    if (reductions(eng)) return MD_FAIL;

    /* execute callbacks */
    if (MD_exec_callback(front) || MD_wait_callback(front)) return MD_FAIL;
  }

  /* leapfrog integration for numsteps */
  for (step = 0;  step < numsteps;  step++) {

    /* half kick */
    for (k = 0;  k < natoms;  k++) {
      konst_div_mass = MD_FORCE_CONST / atom[k].m;
      vel[k].x += half_dt * f[k].x * konst_div_mass;
      vel[k].y += half_dt * f[k].y * konst_div_mass;
      vel[k].z += half_dt * f[k].z * konst_div_mass;
    }

    /* drift */
    for (k = 0;  k < natoms;  k++) {
      pos[k].x += dt * vel[k].x;
      pos[k].y += dt * vel[k].y;
      pos[k].z += dt * vel[k].z;
    }

    /* evaluate forces */
    if (force_compute(force, fr, pos)) {
      return MD_error(front, eng->err_force);
    }

    /* half kick */
    for (k = 0;  k < natoms;  k++) {
      konst_div_mass = MD_FORCE_CONST / atom[k].m;
      vel[k].x += half_dt * f[k].x * konst_div_mass;
      vel[k].y += half_dt * f[k].y * konst_div_mass;
      vel[k].z += half_dt * f[k].z * konst_div_mass;
    }

    /* increment step counter with interface */
    MD_incrstep(front, 1);

    /* see if callbacks need to be processed for this step */
    if (MD_ready_callback(front)) {

      /* reductions might be needed by callback */
      if (reductions(eng)) return MD_FAIL;

      /* execute callbacks */
      if (MD_exec_callback(front) || MD_wait_callback(front)) return MD_FAIL;
    }
  }

  /* compute reductions before returning */
  if (reductions(eng)) return MD_FAIL;

  return 0;
}


int32 reductions(Engine *e)
{
  Result *result = &(e->result);
  ForceResult *fr = &(e->f_result);
  const MD_Atom *a = (MD_Atom *) (e->atom->buf);
  const MD_Dvec *v = (MD_Dvec *) (e->vel->buf);
  const int32 natoms = e->natoms;
  int32 k;
  double accum = 0.0;

  /* copy potential energies from force evaluation */
  result->pe = fr->u_total;
  result->bond = fr->u_bond;
  result->angle = fr->u_angle;
  result->dihed = fr->u_dihed;
  result->impr = fr->u_impr;
  result->elec = fr->u_elec;
  result->vdw = fr->u_vdw;
  result->bound = fr->u_bres;

  /* compute kinetic energy, total energy, and temperature */
  for (k = 0;  k < natoms;  k++) {
    accum += (v[k].x * v[k].x + v[k].y * v[k].y + v[k].z * v[k].z) * a[k].m;
  }
  result->ke = 0.5 * MD_KCAL_MOL * accum;
  result->energy = result->ke + result->pe;
  result->temp = e->tempkonst * result->ke;
  return 0;
}
