/*
 * Copyright (C) 2004-2005 by David J. Hardy.  All rights reserved.
 *
 * split.c
 */

#include "mgrid/split.h"
#include "debug/debug.h"


#ifndef MGRID_FAST_SPLIT


void mgrid_gamma(double *pg, double s, int32 split)
{
  double g;
  /* double s = r_a * r_a; */

  ASSERT(s <= 1);
  ASSERT(pg != NULL);
  switch (split) {
    case MGRID_TAYLOR1:
      g = 1 + (s-1)*(-1./2);
      break;
    case MGRID_TAYLOR2:
      g = 1 + (s-1)*(-1./2 + (s-1)*(3./8));
      break;
    case MGRID_TAYLOR3:
      g = 1 + (s-1)*(-1./2 + (s-1)*(3./8 + (s-1)*(-5./16)));
      break;
    case MGRID_TAYLOR4:
      g = 1 + (s-1)*(-1./2 + (s-1)*(3./8 + (s-1)*(-5./16
              + (s-1)*(35./128))));
      break;
    case MGRID_TAYLOR5:
      g = 1 + (s-1)*(-1./2 + (s-1)*(3./8 + (s-1)*(-5./16
              + (s-1)*(35./128 + (s-1)*(-63./256)))));
      break;
    case MGRID_TAYLOR6:
      g = 1 + (s-1)*(-1./2 + (s-1)*(3./8 + (s-1)*(-5./16
              + (s-1)*(35./128 + (s-1)*(-63./256
                  + (s-1)*(231./1024))))));
      break;
    case MGRID_TAYLOR7:
      g = 1 + (s-1)*(-1./2 + (s-1)*(3./8 + (s-1)*(-5./16
              + (s-1)*(35./128 + (s-1)*(-63./256
                  + (s-1)*(231./1024 + (s-1)*(-429./2048)))))));
      break;
    case MGRID_TAYLOR8:
      g = 1 + (s-1)*(-1./2 + (s-1)*(3./8 + (s-1)*(-5./16
              + (s-1)*(35./128 + (s-1)*(-63./256
                  + (s-1)*(231./1024 + (s-1)*(-429./2048
                      + (s-1)*(6435./32768))))))));
      break;
    case MGRID_ERRMIN3:
      g = MGRID_ERRMIN3_D + s*(MGRID_ERRMIN3_C + s*(MGRID_ERRMIN3_B
            + s*MGRID_ERRMIN3_A));
      break;
    default:
      BUG("invalid splitting function");
  }
  *pg = g;
}


void mgrid_dgamma(double *pg, double *pdg, double s, int32 split)
{
  double g, dg;
  /* double s = r_a * r_a; */

  ASSERT(s <= 1);
  ASSERT(pg != NULL);
  ASSERT(pdg != NULL);
  switch (split) {
    case MGRID_TAYLOR1:
      g = 1 + (s-1)*(-1./2);
      dg = -1./2;
      break;
    case MGRID_TAYLOR2:
      g = 1 + (s-1)*(-1./2 + (s-1)*(3./8));
      dg = -1./2 + (s-1)*(3./4);
      break;
    case MGRID_TAYLOR3:
      g = 1 + (s-1)*(-1./2 + (s-1)*(3./8 + (s-1)*(-5./16)));
      dg = -1./2 + (s-1)*(3./4 + (s-1)*(-15./16));
      break;
    case MGRID_TAYLOR4:
      g = 1 + (s-1)*(-1./2 + (s-1)*(3./8 + (s-1)*(-5./16
              + (s-1)*(35./128))));
      dg = -1./2 + (s-1)*(3./4 + (s-1)*(-15./16 + (s-1)*(35./32)));
      break;
    case MGRID_TAYLOR5:
      g = 1 + (s-1)*(-1./2 + (s-1)*(3./8 + (s-1)*(-5./16
              + (s-1)*(35./128 + (s-1)*(-63./256)))));
      dg = -1./2 + (s-1)*(3./4 + (s-1)*(-15./16 + (s-1)*(35./32
              + (s-1)*(-315./256))));
      break;
    case MGRID_TAYLOR6:
      g = 1 + (s-1)*(-1./2 + (s-1)*(3./8 + (s-1)*(-5./16
              + (s-1)*(35./128 + (s-1)*(-63./256
                  + (s-1)*(231./1024))))));
      dg = -1./2 + (s-1)*(3./4 + (s-1)*(-15./16 + (s-1)*(35./32
              + (s-1)*(-315./256 + (s-1)*(693./512)))));
      break;
    case MGRID_TAYLOR7:
      g = 1 + (s-1)*(-1./2 + (s-1)*(3./8 + (s-1)*(-5./16
              + (s-1)*(35./128 + (s-1)*(-63./256
                  + (s-1)*(231./1024 + (s-1)*(-429./2048)))))));
      dg = -1./2 + (s-1)*(3./4 + (s-1)*(-15./16 + (s-1)*(35./32
              + (s-1)*(-315./256 + (s-1)*(693./512
                  + (s-1)*(-3003./2048))))));
      break;
    case MGRID_TAYLOR8:
      g = 1 + (s-1)*(-1./2 + (s-1)*(3./8 + (s-1)*(-5./16
              + (s-1)*(35./128 + (s-1)*(-63./256
                  + (s-1)*(231./1024 + (s-1)*(-429./2048
                      + (s-1)*(6435./32768))))))));
      dg = -1./2 + (s-1)*(3./4 + (s-1)*(-15./16 + (s-1)*(35./32
              + (s-1)*(-315./256 + (s-1)*(693./512
                  + (s-1)*(-3003./2048 + (s-1)*(6435./4096)))))));
      break;
    case MGRID_ERRMIN3:
      g = MGRID_ERRMIN3_D + s*(MGRID_ERRMIN3_C + s*(MGRID_ERRMIN3_B
            + s*MGRID_ERRMIN3_A));
      dg = MGRID_ERRMIN3_C + s*(2*MGRID_ERRMIN3_B + s*3*MGRID_ERRMIN3_A);
      break;
    case MGRID_EXSELF1:
      if (s > MGRID_EXSELF1_swdis) {
        g = MGRID_EXSELF1_D + s*(MGRID_EXSELF1_C + s*(MGRID_EXSELF1_B
              + s*MGRID_EXSELF1_A));
        dg = MGRID_EXSELF1_C + s*(2*MGRID_EXSELF1_B + s*3*MGRID_EXSELF1_A);
      }
      else {
        g = MGRID_EXSELF1_F + s*MGRID_EXSELF1_E;
        dg = MGRID_EXSELF1_E;
      }
      break;
    default:
      BUG("invalid splitting function");
  }
  *pg = g;
  /* *pdg = dg * (2.0 * r_a); */
  *pdg = dg;
}


void mgrid_gamma_odd(double *pg, double _r, int32 split)
{
  double g;
  double _r2 = _r*_r;

  ASSERT(_r <= 1);
  ASSERT(pg != NULL);
  ASSERT(pdg != NULL);
  switch (split) {
    case MGRID_ODDPR1:
    case MGRID_LOWDEG1:
      _g = 3./2 + _r2*(-1./2);
      break;
    case MGRID_ODDPR2:
    case MGRID_LOWDEG2:
    case MGRID_EXACT0:
      _g = 2 + _r2*(-2 + _r);
      break;
    case MGRID_ODDPR3:
    case MGRID_LOWDEG3:
      _g = 9./4 + _r2*(-5./2 + _r2*(9./4 - _r));
      break;
    case MGRID_ODDPR4:
      _g = 21./8 + _r2*(-35./8 + _r2*(63./8 + _r*(-7 + _r*(15./8))));
      break;
    case MGRID_ODDPR5:
    case MGRID_LOWDEG5:
      _g = 45./16 + _r2*(-21./4 + _r2*(63./8 + _r2*(-45./4
              + _r*(9 + _r*(-35./16)))));
      break;
    case MGRID_ODDPR6:
      _g = 25./8 + _r2*(-15./2 + _r2*(63./4 + _r2*(-75./2
              + _r*(45 + _r*(-175./8 + _r*4)))));
      break;
    case MGRID_ODDPR7:
      _g = 105./32 + _r2*(-275./32 + _r2*(297./16 + _r2*(-495./16
              + _r2*(1925./32 + _r*(-66 + _r*(945./32 + _r*(-5)))))));
      break;
    case MGRID_ODDPR8:
      _g = 455./128 + _r2*(-715./64 + _r2*(3861./128 + _r2*(-2145./32
              + _r2*(25025./128 + _r*(-286 + _r*(12285./64 + _r*(-65
                      + _r*(1155./128))))))));
      break;
    case MGRID_LOWDEG4:
      _g = 5./2 + _r2*(-7./2 + _r2*(7./2 + _r2*(-5./2 + _r)));
      break;
    case MGRID_LOWDEG6:
      _g = 385./128 + _r2*(-825./128 + _r2*(693./64 + _r2*(-825./64
              + _r2*(1925./128 + _r*(-11 + _r*(315./128))))));
      break;
    case MGRID_LOWDEG7:
      _g = 819./256 + _r2*(-1001./128 + _r2*(3861./256
            + _r2*(-1287./64 + _r2*(5005./256 + _r2*(-2457./128
                  + _r*(13 + _r*(-693./256)))))));
      break;
    case MGRID_LOWDEG8:
      _g = 441./128 + _r2*(-637./64 + _r2*(3003./128
            + _r2*(-1287./32 + _r2*(7007./128 + _r2*(-5733./64
                  + _r*(91 + _r*(-4851./128 + _r*(6))))))));
      break;
    case MGRID_LOWALT5_1:
      _g = 175./64 + _r2*(-75./16 + _r2*(189./32 + _r2*(-75./16
              + _r2*(175./64 - _r))));
      break;
    case MGRID_LOWALT6_1:
      _g = 189./64 + _r2*(-385./64 + _r2*(297./32 + _r2*(-297./32
              + _r2*(385./64 + _r2*(-189./64 + _r)))));
      break;
    case MGRID_LOWALT7_1:
      _g = 1617./512 + _r2*(-1911./256 + _r2*(7007./512 + _r2*(-2145./128
              + _r2*(7007./512 + _r2*(-1911./256 + _r2*(1617./512 - _r))))));
      break;
    case MGRID_LOWALT8_1:
      _g = 3465./1024 + _r2*(-9555./1024 + _r2*(21021./1024
            + _r2*(-32175./1024 + _r2*(35035./1024 + _r2*(-28665./1024
                  + _r2*(24255./1024 + _r*(-15 + _r*(3003./1024))))))));
      break;
    case MGRID_LOWALT8_2:
      _g = 429./128 + _r2*(-1155./128 + _r2*(2457./128 + _r2*(-3575./128
              + _r2*(3575./128 + _r2*(-2457./128 + _r2*(1155./128
                    + _r2*(-429./128 + _r)))))));
      break;
    case MGRID_EXSFPC2:
      if (_r > MGRID_EXSFPC2_swdis) {
        _g = MGRID_EXSFPC2_A + _r*(MGRID_EXSFPC2_B + _r*(MGRID_EXSFPC2_C
              + _r*(MGRID_EXSFPC2_D + _r*MGRID_EXSFPC2_E)));
      }
      else {
        _g = MGRID_EXSFPC2_F;
      }
      break;
    case MGRID_EXSFPQ2:
      if (_r > MGRID_EXSFPQ2_swdis) {
        _g = MGRID_EXSFPQ2_A + _r*(MGRID_EXSFPQ2_B + _r*(MGRID_EXSFPQ2_C
              + _r*MGRID_EXSFPQ2_D));
      }
      else {
        _g = MGRID_EXSFPQ2_E + _r2*MGRID_EXSFPQ2_F;
      }
      break;
    case MGRID_EXSFRC2:
      if (_r > MGRID_EXSFRC2_swdis) {
        _g = MGRID_EXSFRC2_A/_r + MGRID_EXSFRC2_B + _r*(MGRID_EXSFRC2_C
              + _r*(MGRID_EXSFRC2_D + _r*MGRID_EXSFRC2_E));
      }
      else {
        _g = MGRID_EXSFRC2_F;
      }
      break;
    case MGRID_EXSFRQ2:
      if (_r > MGRID_EXSFRQ2_swdis) {
        _g = MGRID_EXSFRQ2_A/_r + MGRID_EXSFRQ2_B + _r*(MGRID_EXSFRQ2_C
              + _r*MGRID_EXSFRQ2_D);
      }
      else {
        _g = MGRID_EXSFRQ2_E + _r2*MGRID_EXSFRQ2_F;
      }
      break;
    case MGRID_EXACT3:
      if (_r > 7./8) {
        _g = -19./15 + _r*(49./5 + _r*(-59./5 + _r*(64./15)));
      }
      else {
        _g = 191./120 + _r2*(-3./5);
      }
      break;
    case MGRID_EXACT2:
      if (_r > 3./4) {
        _g = 5./7 + _r*(27./7 + _r*(-41./7 + _r*(16./7)));
      }
      else {
        _g = 47./28 + _r2*(-5./7);
      }
      break;
    case MGRID_EXACT1:
      if (_r > 1./2) {
        _g = 5./3 + _r + _r2*(-3 + _r*(4./3));
      }
      else {
        _g = 11./6 - _r2;
      }
      break;
    default:
      BUG("invalid splitting function");
  }
  *pg = g;
}


void mgrid_dgamma_odd(double *pg, double *pdg, double _r, int32 split)
{
  double _g, _dg;
  double _r2 = _r*_r;

  ASSERT(_r <= 1);
  ASSERT(pg != NULL);
  ASSERT(pdg != NULL);
  switch (split) {
    case MGRID_ODDPR1:
    case MGRID_LOWDEG1:
      _g = 3./2 + _r2*(-1./2);
      _dg = -_r;
      break;
    case MGRID_ODDPR2:
    case MGRID_LOWDEG2:
    case MGRID_EXACT0:
      _g = 2 + _r2*(-2 + _r);
      _dg = _r*(-4 + _r*3);
      break;
    case MGRID_ODDPR3:
    case MGRID_LOWDEG3:
      _g = 9./4 + _r2*(-5./2 + _r2*(9./4 - _r));
      _dg = r*(-5 + _r2*(9 + _r*(-5)));
      break;
    case MGRID_ODDPR4:
      _g = 21./8 + _r2*(-35./8 + _r2*(63./8 + _r*(-7 + _r*(15./8))));
      _dg = _r*(-35./4 + _r2*(63./2 + _r*(-35 + _r*(45./4))));
      break;
    case MGRID_ODDPR5:
    case MGRID_LOWDEG5:
      _g = 45./16 + _r2*(-21./4 + _r2*(63./8 + _r2*(-45./4
              + _r*(9 + _r*(-35./16)))));
      _dg = _r*(-21./2 + _r2*(63./2 + _r2*(-135./2
              + _r*(63 + _r*(-35./2)))));
      break;
    case MGRID_ODDPR6:
      _g = 25./8 + _r2*(-15./2 + _r2*(63./4 + _r2*(-75./2
              + _r*(45 + _r*(-175./8 + _r*4)))));
      _dg = _r*(-15 + _r2*(63 + _r2*(-225
              + _r*(315 + _r*(-175 + _r*36)))));
      break;
    case MGRID_ODDPR7:
      _g = 105./32 + _r2*(-275./32 + _r2*(297./16 + _r2*(-495./16
              + _r2*(1925./32 + _r*(-66 + _r*(945./32 + _r*(-5)))))));
      _dg = _r*(-275./16 + _r2*(297./4 + _r2*(-1485./8
              + _r2*(1925./4 + _r*(-594 + _r*(4725./16 + _r*(-55)))))));
      break;
    case MGRID_ODDPR8:
      _g = 455./128 + _r2*(-715./64 + _r2*(3861./128 + _r2*(-2145./32
              + _r2*(25025./128 + _r*(-286 + _r*(12285./64 + _r*(-65
                      + _r*(1155./128))))))));
      _dg = _r*(-715./32 + _r2*(3861./32 + _r2*(-6435./16
              + _r2*(25025./16 + _r*(-2574 + _r*(61425./32 + _r*(-715
                      + _r*(3465./32)))))))); 
      break;
    case MGRID_LOWDEG4:
      _g = 5./2 + _r2*(-7./2 + _r2*(7./2 + _r2*(-5./2 + _r)));
      _dg = _r*(-7 + _r2*(14 + _r2*(-15 + _r*(7))));
      break;
    case MGRID_LOWDEG6:
      _g = 385./128 + _r2*(-825./128 + _r2*(693./64 + _r2*(-825./64
              + _r2*(1925./128 + _r*(-11 + _r*(315./128))))));
      _dg = _r*(-825./64 + _r2*(693./16 + _r2*(-2475./32
              + _r2*(1925./16 + _r*(-99 + _r*(1575./64))))));
      break;
    case MGRID_LOWDEG7:
      _g = 819./256 + _r2*(-1001./128 + _r2*(3861./256
            + _r2*(-1287./64 + _r2*(5005./256 + _r2*(-2457./128
                  + _r*(13 + _r*(-693./256)))))));
      _dg = _r*(-1001./64 + _r2*(3861./64 + _r2*(-3861./32
              + _r2*(5005./32 + _r2*(-12285./64 + _r*(143
                    + _r*(-2079./64)))))));
      break;
    case MGRID_LOWDEG8:
      _g = 441./128 + _r2*(-637./64 + _r2*(3003./128
            + _r2*(-1287./32 + _r2*(7007./128 + _r2*(-5733./64
                  + _r*(91 + _r*(-4851./128 + _r*(6))))))));
      _dg = _r*(-637./32 + _r2*(3003./32 + _r2*(-3861./16
              + _r2*(7007./16 + _r2*(-28665./32 + _r*(1001
                    + _r*(-14553./32 + _r*(78))))))));
      break;
    case MGRID_LOWALT5_1:
      _g = 175./64 + _r2*(-75./16 + _r2*(189./32 + _r2*(-75./16
              + _r2*(175./64 - _r))));
      _dg = _r*(-75./8 + _r2*(189./8 + _r2*(-225./8 + _r2*(175./8
                + _r*(-9)))));
      break;
    case MGRID_LOWALT6_1:
      _g = 189./64 + _r2*(-385./64 + _r2*(297./32 + _r2*(-297./32
              + _r2*(385./64 + _r2*(-189./64 + _r)))));
      _dg = _r*(-385./32 + _r2*(297./8 + _r2*(-891./16 + _r2*(385./8
                + _r2*(-945./32 + _r*(11))))));
      break;
    case MGRID_LOWALT7_1:
      _g = 1617./512 + _r2*(-1911./256 + _r2*(7007./512 + _r2*(-2145./128
              + _r2*(7007./512 + _r2*(-1911./256 + _r2*(1617./512 - _r))))));
      _dg = _r*(-1911./128 + _r2*(7007./128 + _r2*(-6435./64 + _r2*(7007./64
                + _r2*(-9555./128 + _r2*(4851./128 + _r*(-13)))))));
      break;
    case MGRID_LOWALT8_1:
      _g = 3465./1024 + _r2*(-9555./1024 + _r2*(21021./1024
            + _r2*(-32175./1024 + _r2*(35035./1024 + _r2*(-28665./1024
                  + _r2*(24255./1024 + _r*(-15 + _r*(3003./1024))))))));
      _dg = _r*(-9555./512 + _r2*(21021./256 + _r2*(-96525./512
              + _r2*(35035./128 + _r2*(-143325./512 + _r2*(72765./256
                    + _r*(-195 + _r*(21021./512))))))));
      break;
    case MGRID_LOWALT8_2:
      _g = 429./128 + _r2*(-1155./128 + _r2*(2457./128 + _r2*(-3575./128
              + _r2*(3575./128 + _r2*(-2457./128 + _r2*(1155./128
                    + _r2*(-429./128 + _r)))))));
      _dg = _r*(-1155./64 + _r2*(2457./32 + _r2*(-10725./64
              + _r2*(3575./16 + _r2*(-12285./64 + _r2*(3465./32
                    + _r2*(-3003./64 + _r*(15))))))));
      break;
    case MGRID_EXSFPC2:
      if (_r > MGRID_EXSFPC2_swdis) {
        _g = MGRID_EXSFPC2_A + _r*(MGRID_EXSFPC2_B + _r*(MGRID_EXSFPC2_C
              + _r*(MGRID_EXSFPC2_D + _r*MGRID_EXSFPC2_E)));
        _dg = MGRID_EXSFPC2_B + _r*(2*MGRID_EXSFPC2_C
            + _r*(3*MGRID_EXSFPC2_D + _r*(4*MGRID_EXSFPC2_E)));
      }
      else {
        _g = MGRID_EXSFPC2_F;
        _dg = 0;
      }
      break;
    case MGRID_EXSFPQ2:
      if (_r > MGRID_EXSFPQ2_swdis) {
        _g = MGRID_EXSFPQ2_A + _r*(MGRID_EXSFPQ2_B + _r*(MGRID_EXSFPQ2_C
              + _r*MGRID_EXSFPQ2_D));
        _dg = MGRID_EXSFPQ2_B + _r*(2*MGRID_EXSFPQ2_C
            + _r*(3*MGRID_EXSFPQ2_D));
      }
      else {
        _g = MGRID_EXSFPQ2_E + _r2*MGRID_EXSFPQ2_F;
        _dg = _r*(2*MGRID_EXSFPQ2_F);
      }
      break;
    case MGRID_EXSFRC2:
      if (_r > MGRID_EXSFRC2_swdis) {
        _g = MGRID_EXSFRC2_A/_r + MGRID_EXSFRC2_B + _r*(MGRID_EXSFRC2_C
              + _r*(MGRID_EXSFRC2_D + _r*MGRID_EXSFRC2_E));
        _dg = -MGRID_EXSFRC2_A/_r2 + MGRID_EXSFRC2_C
            + _r*(2*MGRID_EXSFRC2_D + _r*(3*MGRID_EXSFRC2_E));
      }
      else {
        _g = MGRID_EXSFRC2_F;
        _dg = 0;
      }
      break;
    case MGRID_EXSFRQ2:
      if (_r > MGRID_EXSFRQ2_swdis) {
        _g = MGRID_EXSFRQ2_A/_r + MGRID_EXSFRQ2_B + _r*(MGRID_EXSFRQ2_C
              + _r*MGRID_EXSFRQ2_D);
        _dg = -MGRID_EXSFRQ2_A/_r2 + MGRID_EXSFRQ2_C
            + _r*2*MGRID_EXSFRQ2_D;
      }
      else {
        _g = MGRID_EXSFRQ2_E + _r2*MGRID_EXSFRQ2_F;
        _dg = _r*(2*MGRID_EXSFRQ2_F);
      }
      break;
    case MGRID_EXACT3:
      if (_r > 7./8) {
        _g = -19./15 + _r*(49./5 + _r*(-59./5 + _r*(64./15)));
        _dg = 49./5 + _r*(-118./5 + _r*(64./5));
      }
      else {
        _g = 191./120 + _r2*(-3./5);
        _dg = _r*(-6./5);
      }
      break;
    case MGRID_EXACT2:
      if (_r > 3./4) {
        _g = 5./7 + _r*(27./7 + _r*(-41./7 + _r*(16./7)));
        _dg = 27./7 + _r*(-82./7 + _r*(48./7));
      }
      else {
        _g = 47./28 + _r2*(-5./7);
        _dg = _r*(-10./7);
      }
      break;
    case MGRID_EXACT1:
      if (_r > 1./2) {
        _g = 5./3 + _r + _r2*(-3 + _r*(4./3));
        _dg = 1 + _r*(-6 + _r*(4));
      }
      else {
        _g = 11./6 - _r2;
        _dg = _r*(-2);
      }
      break;
    default:
      BUG("invalid splitting function");
  }
  *pg = _g;
  *pdg = _dg;
}


#endif /* not defined MGRID_FAST_SPLIT */
/* otherwise expand these functions as macros */


void mgrid_ndgamma(double *pg, int32 n, double s, int32 split)
{
  int32 k = 0;

  ASSERT(s >= 0);
  ASSERT(n >= 0);
  ASSERT(pg != NULL || n == 0);
  if (k == n) return;
  if (s <= 1) {
    /* compute derivatives of smoothed part */
    switch (split) {
      case MGRID_TAYLOR2:
        pg[k++] = 1 + (s-1)*(-1./2 + (s-1)*(3./8));
        if (k == n) break;
        pg[k++] = -1./2 + (s-1)*(3./4);
        if (k == n) break;
        pg[k++] = 3./4;
        break;
      case MGRID_TAYLOR3:
        pg[k++] = 1 + (s-1)*(-1./2 + (s-1)*(3./8 + (s-1)*(-5./16)));
        if (k == n) break;
        pg[k++] = -1./2 + (s-1)*(3./4 + (s-1)*(-15./16));
        if (k == n) break;
        pg[k++] = 3./4 + (s-1)*(-15./8);
        if (k == n) break;
        pg[k++] = -15./8;
        break;
      case MGRID_TAYLOR4:
        pg[k++] = 1 + (s-1)*(-1./2 + (s-1)*(3./8 + (s-1)*(-5./16
                + (s-1)*(35./128))));
        if (k == n) break;
        pg[k++] = -1./2 + (s-1)*(3./4 + (s-1)*(-15./16 + (s-1)*(35./32)));
        if (k == n) break;
        pg[k++] = 3./4 + (s-1)*(-15./8 + (s-1)*(105./32));
        if (k == n) break;
        pg[k++] = -15./8 + (s-1)*(105./16);
        if (k == n) break;
        pg[k++] = 105./16;
        break;
      case MGRID_TAYLOR5:
        pg[k++] = 1 + (s-1)*(-1./2 + (s-1)*(3./8 + (s-1)*(-5./16
                + (s-1)*(35./128 + (s-1)*(-63./256)))));
        if (k == n) break;
        pg[k++] = -1./2 + (s-1)*(3./4 + (s-1)*(-15./16 + (s-1)*(35./32
                + (s-1)*(-315./256))));
        if (k == n) break;
        pg[k++] = 3./4 + (s-1)*(-15./8 + (s-1)*(105./32 + (s-1)*(-315./64)));
        if (k == n) break;
        pg[k++] = -15./8 + (s-1)*(105./16 + (s-1)*(-945./64));
        if (k == n) break;
        pg[k++] = 105./16 + (s-1)*(-945./32);
        if (k == n) break;
        pg[k++] = -945./32;
        break;
      case MGRID_TAYLOR6:
        pg[k++] = 1 + (s-1)*(-1./2 + (s-1)*(3./8 + (s-1)*(-5./16
                + (s-1)*(35./128 + (s-1)*(-63./256 + (s-1)*(231./1024))))));
        if (k == n) break;
        pg[k++] = -1./2 + (s-1)*(3./4 + (s-1)*(-15./16 + (s-1)*(35./32
                + (s-1)*(-315./256 + (s-1)*(693./512)))));
        if (k == n) break;
        pg[k++] = 3./4 + (s-1)*(-15./8 + (s-1)*(105./32 + (s-1)*(-315./64
                + (s-1)*(3465./512))));
        if (k == n) break;
        pg[k++] = -15./8 + (s-1)*(105./16 + (s-1)*(-945./64
              + (s-1)*(3465./128)));
        if (k == n) break;
        pg[k++] = 105./16 + (s-1)*(-945./32 + (s-1)*(10395./128));
        if (k == n) break;
        pg[k++] = -945./32 + (s-1)*(10395./64);
        if (k == n) break;
        pg[k++] = 10395./64;
        break;
      case MGRID_TAYLOR7:
        pg[k++] = 1 + (s-1)*(-1./2 + (s-1)*(3./8 + (s-1)*(-5./16
                + (s-1)*(35./128 + (s-1)*(-63./256
                    + (s-1)*(231./1024 + (s-1)*(-429./2048)))))));
        if (k == n) break;
        pg[k++] = -1./2 + (s-1)*(3./4 + (s-1)*(-15./16 + (s-1)*(35./32
                + (s-1)*(-315./256 + (s-1)*(693./512
                    + (s-1)*(-3003./2048))))));
        if (k == n) break;
        pg[k++] = 3./4 + (s-1)*(-15./8 + (s-1)*(105./32 + (s-1)*(-315./64
                + (s-1)*(3465./512 + (s-1)*(-9009./1024)))));
        if (k == n) break;
        pg[k++] = -15./8 + (s-1)*(105./16 + (s-1)*(-945./64 + (s-1)*(3465./128
                + (s-1)*(-45045./1024))));
        if (k == n) break;
        pg[k++] = 105./16 + (s-1)*(-945./32 + (s-1)*(10395./128
              + (s-1)*(-45045./256)));
        if (k == n) break;
        pg[k++] = -945./32 + (s-1)*(10395./64 + (s-1)*(-135135./256));
        if (k == n) break;
        pg[k++] = 10395./64 + (s-1)*(-135135./128);
        if (k == n) break;
        pg[k++] = -135135./128;
        break;
      case MGRID_TAYLOR8:
        pg[k++] = 1 + (s-1)*(-1./2 + (s-1)*(3./8 + (s-1)*(-5./16
                + (s-1)*(35./128 + (s-1)*(-63./256
                    + (s-1)*(231./1024 + (s-1)*(-429./2048
                        + (s-1)*(6435./32768))))))));
        if (k == n) break;
        pg[k++] = -1./2 + (s-1)*(3./4 + (s-1)*(-15./16 + (s-1)*(35./32
                + (s-1)*(-315./256 + (s-1)*(693./512
                    + (s-1)*(-3003./2048 + (s-1)*(6435./4096)))))));
        if (k == n) break;
        pg[k++] = 3./4 + (s-1)*(-15./8 + (s-1)*(105./32 + (s-1)*(-315./64
                + (s-1)*(3465./512 + (s-1)*(-9009./1024
                    + (s-1)*(45045./4096))))));
        if (k == n) break;
        pg[k++] = -15./8 + (s-1)*(105./16 + (s-1)*(-945./64 + (s-1)*(3465./128
                + (s-1)*(-45045./1024 + (s-1)*(135135./2048)))));
        if (k == n) break;
        pg[k++] = 105./16 + (s-1)*(-945./32 + (s-1)*(10395./128
              + (s-1)*(-45045./256 + (s-1)*(675675./2048))));
        if (k == n) break;
        pg[k++] = -945./32 + (s-1)*(10395./64 + (s-1)*(-135135./256
              + (s-1)*(675675./512)));
        if (k == n) break;
        pg[k++] = 10395./64 + (s-1)*(-135135./128 + (s-1)*(2027025./512));
        if (k == n) break;
        pg[k++] = -135135./128 + (s-1)*(2027025./256);
        if (k == n) break;
        pg[k++] = 2027025./256;
        break;
      case MGRID_ERRMIN3:
        pg[k++] = MGRID_ERRMIN3_D + s*(MGRID_ERRMIN3_C + s*(MGRID_ERRMIN3_B
              + s*MGRID_ERRMIN3_A));
        if (k == n) break;
        pg[k++] = MGRID_ERRMIN3_C + s*(2*MGRID_ERRMIN3_B + s*3*MGRID_ERRMIN3_A);
        if (k == n) break;
        pg[k++] = 2*MGRID_ERRMIN3_B + s*2*3*MGRID_ERRMIN3_A;
        if (k == n) break;
        pg[k++] = 2*3*MGRID_ERRMIN3_A;
        break;
      case MGRID_EXSELF1:
        if (s > MGRID_EXSELF1_swdis) {
          pg[k++] = MGRID_EXSELF1_D + s*(MGRID_EXSELF1_C + s*(MGRID_EXSELF1_B
                + s*MGRID_EXSELF1_A));
          if (k == n) break;
          pg[k++] = MGRID_EXSELF1_C + s*(2*MGRID_EXSELF1_B
              + s*3*MGRID_EXSELF1_A);
          if (k == n) break;
          pg[k++] = 2*MGRID_EXSELF1_B + s*2*3*MGRID_EXSELF1_A;
          if (k == n) break;
          pg[k++] = 2*3*MGRID_EXSELF1_A;
        }
        else {
          pg[k++] = MGRID_EXSELF1_F + s*MGRID_EXSELF1_E;
          if (k == n) break;
          pg[k++] = MGRID_EXSELF1_E;
        }
        break;
      default:
        BUG("invalid splitting function");
    }
    /* higher derivatives are zero */
    while (k < n) pg[k++] = 0;
  }
  else { /*  (s > 1)  */
    /* compute derivatives of s^(-1/2) */
    const double s_1 = 1./s;
    double s_p = sqrt(s_1);
    double p = -0.5;
    double c = 1;
    pg[k++] = c * s_p;
    while (k < n) {
      s_p *= s_1;
      c *= p;
      p -= 1;
      pg[k++] = c * s_p;
    }
  }
}
