/*
 * Copyright (C) 2004-2005 by David J. Hardy.  All rights reserved.
 *
 * system.c
 */

#include <stdlib.h>
#include <string.h>
#include "mdio/mdio.h"
#include "deven/deven.h"
#include "mdsim/system.h"
#include "mdsim/simparam.h"
#include "mdsim/error.h"
#include "mdsim/const.h"
#undef DEBUG_WATCH
#include "debug/debug.h"


/*
 * System constructor
 * clear memory and initialize file readers
 */
int system_init(System *s)
{
  memset(s, 0, sizeof(System));

  if (mdio_initializeParam(&(s->prm))) {
    return error(MSG, "mdio_initializeParam() failed");
  }
  mdio_setErrorHandlerFile((mdio_File *) &(s->prm), mdio_errhandler);

  if (mdio_initializeTopo(&(s->topo))) {
    return error(MSG, "mdio_initializeTopo() failed");
  }
  mdio_setErrorHandlerFile((mdio_File *) &(s->topo), mdio_errhandler);

  if (mdio_initializePdbcoord(&(s->pdb))) {
    return error(MSG, "mdio_initializePdbcoord() failed");
  }
  mdio_setErrorHandlerFile((mdio_File *) &(s->pdb), mdio_errhandler);

  if (mdio_initializePdbcoord(&(s->pdbvel))) {
    return error(MSG, "mdio_initializePdbcoord() failed");
  }
  mdio_setErrorHandlerFile((mdio_File *) &(s->pdbvel), mdio_errhandler);

  if (mdio_initializeBincoord(&(s->bin))) {
    return error(MSG, "mdio_initializeBincoord() failed");
  }
  mdio_setErrorHandlerFile((mdio_File *) &(s->bin), mdio_errhandler);

  if (mdio_initializeBincoord(&(s->binvel))) {
    return error(MSG, "mdio_initializeBincoord() failed");
  }
  mdio_setErrorHandlerFile((mdio_File *) &(s->binvel), mdio_errhandler);

  if (adt_initializeList(&(s->potential_index), sizeof(PotentialIndex),
        0, NULL)) {
    return error(MSG_NOMEM);
  }

  if (adt_initializeList(&(s->fixedatoms_index), sizeof(int32), 0, NULL)) {
    return error(MSG_NOMEM);
  }

  /* assumes deven */
  if (adt_initializeList(&(s->constraints_data), sizeof(DevenConstraint), 0,
        NULL)) {
    return error(MSG_NOMEM);
  }
  return 0;
}


/*
 * validate access assumptions for engine
 * engine needs to have already been initialized
 */
int system_setup(System *s, MD_Engine *e)
{
  const int32 read = MD_READ;
  const int32 wlen = MD_WRITE | MD_SETLEN;
  const int32 rwlen = MD_READ | MD_WRITE | MD_SETLEN;

  /* store engine interface pointer */
  s->eng = e;

  /* check for essential engine data arrays */
  /*** treat "result" as essential ***/
  if ((s->atomprm_id = MD_idnum(e, "atomprm")) == MD_FAIL
      || (s->bondprm_id = MD_idnum(e, "bondprm")) == MD_FAIL
      || (s->angleprm_id = MD_idnum(e, "angleprm")) == MD_FAIL
      || (s->dihedprm_id = MD_idnum(e, "dihedprm")) == MD_FAIL
      || (s->imprprm_id = MD_idnum(e, "imprprm")) == MD_FAIL
      || (s->nbfixprm_id = MD_idnum(e, "nbfixprm")) == MD_FAIL
      || (s->atom_id = MD_idnum(e, "atom")) == MD_FAIL
      || (s->bond_id = MD_idnum(e, "bond")) == MD_FAIL
      || (s->angle_id = MD_idnum(e, "angle")) == MD_FAIL
      || (s->dihed_id = MD_idnum(e, "dihed")) == MD_FAIL
      || (s->impr_id = MD_idnum(e, "impr")) == MD_FAIL
      || (s->excl_id = MD_idnum(e, "excl")) == MD_FAIL
      || (s->pos_id = MD_idnum(e, "pos")) == MD_FAIL
      || (s->vel_id = MD_idnum(e, "vel")) == MD_FAIL
      || (s->result_id = MD_idnum(e, "result")) == MD_FAIL) {
    return error(MSG_ENGINE, "missing essential data array");
  }

  /* store attributes, might need to update later */
  s->atomprm_attr = MD_attrib(e, s->atomprm_id);
  s->bondprm_attr = MD_attrib(e, s->bondprm_id);
  s->angleprm_attr = MD_attrib(e, s->angleprm_id);
  s->dihedprm_attr = MD_attrib(e, s->dihedprm_id);
  s->imprprm_attr = MD_attrib(e, s->imprprm_id);
  s->nbfixprm_attr = MD_attrib(e, s->nbfixprm_id);
  s->atom_attr = MD_attrib(e, s->atom_id);
  s->bond_attr = MD_attrib(e, s->bond_id);
  s->angle_attr = MD_attrib(e, s->angle_id);
  s->dihed_attr = MD_attrib(e, s->dihed_id);
  s->impr_attr = MD_attrib(e, s->impr_id);
  s->excl_attr = MD_attrib(e, s->excl_id);
  s->pos_attr = MD_attrib(e, s->pos_id);
  s->vel_attr = MD_attrib(e, s->vel_id);
  s->result_attr = MD_attrib(e, s->result_id);

  /* check for type consistency */
  if (s->atomprm_attr.type != MD_ATOMPRM
      || s->bondprm_attr.type != MD_BONDPRM
      || s->angleprm_attr.type != MD_ANGLEPRM
      || s->dihedprm_attr.type != MD_TORSPRM
      || s->imprprm_attr.type != MD_TORSPRM
      || s->nbfixprm_attr.type != MD_NBFIXPRM
      || s->atom_attr.type != MD_ATOM
      || s->bond_attr.type != MD_BOND
      || s->angle_attr.type != MD_ANGLE
      || s->dihed_attr.type != MD_TORS
      || s->impr_attr.type != MD_TORS
      || s->excl_attr.type != MD_EXCL
      || (s->pos_attr.type != MD_DVEC /* && s->pos_attr.type != MD_FVEC */)
      || s->vel_attr.type != s->pos_attr.type
      || s->result_attr.type != MD_type(e, "Result")) {
    return error(MSG_ENGINE, "incorrect type on essential data array");
  }

  /* check minimal access permissions */
  if ((s->atomprm_attr.access & wlen) != wlen
      || (s->bondprm_attr.access & wlen) != wlen
      || (s->angleprm_attr.access & wlen) != wlen
      || (s->dihedprm_attr.access & wlen) != wlen
      || (s->imprprm_attr.access & wlen) != wlen
      || (s->nbfixprm_attr.access & wlen) != wlen
      || (s->atom_attr.access & wlen) != wlen
      || (s->bond_attr.access & wlen) != wlen
      || (s->angle_attr.access & wlen) != wlen
      || (s->dihed_attr.access & wlen) != wlen
      || (s->impr_attr.access & wlen) != wlen
      || (s->excl_attr.access & wlen) != wlen
      || (s->pos_attr.access & rwlen) != rwlen
      || (s->vel_attr.access & rwlen) != rwlen
      || (s->result_attr.access & read) != read) {
    return error(MSG_ENGINE,
        "incorrect access permissions on essential data array");
  }

  /* check attributes for result */
  if ((s->result_attr.access & (MD_RESIZE | MD_ERESIZE)) != 0
      || s->result_attr.len != 1
      || s->result_attr.max != 1) {
    return error(MSG_ENGINE, "incorrect attributes on \"result\"");
  }

  /* check for consistency of access permissions */
  if (s->atomprm_attr.access != s->bondprm_attr.access
      || s->atomprm_attr.access != s->angleprm_attr.access
      || s->atomprm_attr.access != s->dihedprm_attr.access
      || s->atomprm_attr.access != s->imprprm_attr.access
      || s->atomprm_attr.access != s->nbfixprm_attr.access
      || s->atom_attr.access != s->bond_attr.access
      || s->atom_attr.access != s->angle_attr.access
      || s->atom_attr.access != s->dihed_attr.access
      || s->atom_attr.access != s->impr_attr.access
      || s->atom_attr.access != s->excl_attr.access
      || s->pos_attr.access != s->vel_attr.access) {
    return error(MSG_ENGINE,
        "incorrect access permissions on essential data array");
  }

  /* check for availability of potential and force arrays */
  if ((s->u_bond_id = MD_idnum(e, "u_bond")) != MD_FAIL
      && (s->u_angle_id = MD_idnum(e, "u_angle")) != MD_FAIL
      && (s->u_dihed_id = MD_idnum(e, "u_dihed")) != MD_FAIL
      && (s->u_impr_id = MD_idnum(e, "u_impr")) != MD_FAIL
      && (s->u_elec_id = MD_idnum(e, "u_elec")) != MD_FAIL
      && (s->u_vdw_id = MD_idnum(e, "u_vdw")) != MD_FAIL
      && (s->u_bound_id = MD_idnum(e, "u_bound")) != MD_FAIL
      && (s->pot_elec_id = MD_idnum(e, "pot_elec")) != MD_FAIL) {

    /* store attributes, might need to update later */
    s->u_bond_attr = MD_attrib(e, s->u_bond_id);
    s->u_angle_attr = MD_attrib(e, s->u_angle_id);
    s->u_dihed_attr = MD_attrib(e, s->u_dihed_id);
    s->u_impr_attr = MD_attrib(e, s->u_impr_id);
    s->u_elec_attr = MD_attrib(e, s->u_elec_id);
    s->u_vdw_attr = MD_attrib(e, s->u_vdw_id);
    s->u_bound_attr = MD_attrib(e, s->u_bound_id);
    s->pot_elec_attr = MD_attrib(e, s->pot_elec_id);

    /* check for type consistency */
    if (s->u_bond_attr.type != MD_DOUBLE
        || s->u_angle_attr.type != MD_DOUBLE
        || s->u_dihed_attr.type != MD_DOUBLE
        || s->u_impr_attr.type != MD_DOUBLE
        || s->u_elec_attr.type != MD_DOUBLE
        || s->u_vdw_attr.type != MD_DOUBLE
        || s->u_bound_attr.type != MD_DOUBLE
        || s->pot_elec_attr.type != MD_DOUBLE) {
      return error(MSG_ENGINE, "incorrect type on potential or force array");
    }

    /* check for minimal access permissions */
    if ((s->u_bond_attr.access & read) != read
        || (s->u_angle_attr.access & read) != read
        || (s->u_dihed_attr.access & read) != read
        || (s->u_impr_attr.access & read) != read
        || (s->u_elec_attr.access & read) != read
        || (s->u_vdw_attr.access & read) != read
        || (s->u_bound_attr.access & read) != read
        || (s->pot_elec_attr.access & read) != read) {
      return error(MSG_ENGINE,
          "incorrect access permissions on potential or force array");
    }

    /* check for consistency of access permisions */
    if (s->u_bond_attr.access != s->u_angle_attr.access
        || s->u_bond_attr.access != s->u_dihed_attr.access
        || s->u_bond_attr.access != s->u_impr_attr.access
        || s->u_bond_attr.access != s->u_elec_attr.access
        || s->u_bond_attr.access != s->u_vdw_attr.access
        || s->u_bond_attr.access != s->u_bound_attr.access
        || s->u_bond_attr.access != s->pot_elec_attr.access) {
      return error(MSG_ENGINE,
          "incorrect access permissions on potential or force array");
    }

    /* indicate that force and potential arrays are available */
    s->is_potential_array = TRUE;
  }

  /* check for "fixedatomslist", support by engine is optional */
  if ((s->fixedatomslist_id = MD_idnum(e, "fixedatomslist")) != MD_FAIL) {
    s->fixedatomslist_attr = MD_attrib(e, s->fixedatomslist_id);
    /* make sure it is expected type */
    if (s->fixedatomslist_attr.type != MD_INT32) {
      return error(MSG_ENGINE, "unexpected type for fixed atoms list");
    }
    /* make sure we can write and resize */
    else if ((s->fixedatomslist_attr.access & wlen) != wlen) {
      return error(MSG_ENGINE,
          "incorrect access permissions on fixed atoms list");
    }
  }
  else if (MD_reset(e)) { /* otherwise we have to reset error indicator */
    return error(MSG_MDAPI, "from MD_reset()", MD_errmsg(e));
  }

  /* check for "constraintslist", support by engine is optional */
  if ((s->constraintlist_id = MD_idnum(e, "constraintlist")) != MD_FAIL) {
    s->constraintlist_attr = MD_attrib(e, s->constraintlist_id);
    /* make sure it is expected type */
    if (strcmp(MD_type_name(e, s->constraintlist_attr.type),
          "Constraint") != 0
        || MD_SIZEOF(s->constraintlist_attr.type) != sizeof(DevenConstraint)) {
      /* and we could check more thoroughly the members */
      return error(MSG_ENGINE, "unexpected type for constraint list");
    }
    /* make sure we can write and resize */
    else if ((s->constraintlist_attr.access & wlen) != wlen) {
      return error(MSG_ENGINE,
          "incorrect access permissions on constraint list");
    }
  }
  else if (MD_reset(e)) { /* otherwise we have to reset error indicator */
    return error(MSG_MDAPI, "from MD_reset()", MD_errmsg(e));
  }

  return 0;
}


/*
 * use MDIO to read files (with file names given by SimParam)
 * initialize engine data arrays
 */
int system_input(System *s, SimParam *p)
{
  /* MDIO file reading objects */
  mdio_Param *prm = &(s->prm);
  mdio_Topo *topo = &(s->topo);
  mdio_Pdbcoord *pdb = &(s->pdb);
  mdio_Pdbcoord *pdbvel = &(s->pdbvel);
  mdio_Bincoord *bin = &(s->bin);
  mdio_Bincoord *binvel = &(s->binvel);

  /* define pointers for brevity */
  MD_Engine *e = s->eng;
  adt_List *pa;
  int32 len;

  /* other declarations */
  int32 k, npos, nvel;


  /* read force field param files */
  pa = &(p->parameters);
  len = (int32) adt_getLengthList(pa);
  for (k = 0;  k < len;  k++) {
    const char *fname = *((const char **) adt_indexList(pa, k));
    /* any errors parsing parameter files are unrecoverable */
    if (mdio_readParam(prm, fname)) {
      return error(MSG_MDIO, mdio_getErrorMessageFile((mdio_File *) prm));
    }
    printf("# force field parameters file \"%s\"\n", fname);
  }

  /* read topology file */
  if (mdio_readTopo(topo, p->structure)) {
    return error(MSG_MDIO, mdio_getErrorMessageFile((mdio_File *) topo));
  }
  printf("# PSF file \"%s\"\n", p->structure);

  /* set topology indexing of params */
  if (mdio_indexParamTopo(topo, prm)) {
    return error(MSG_MDIO, mdio_getErrorMessageFile((mdio_File *) topo));
  }

  /* obtain param data */
  s->atomprm = mdio_getAtomParam(prm, &(s->atomprm_len));
  s->bondprm = mdio_getBondParam(prm, &(s->bondprm_len));
  s->angleprm = mdio_getAngleParam(prm, &(s->angleprm_len));
  s->dihedprm = mdio_getDihedParam(prm, &(s->dihedprm_len));
  s->imprprm = mdio_getImprParam(prm, &(s->imprprm_len));
  s->nbfixprm = mdio_getNbfixParam(prm, &(s->nbfixprm_len));

  /* obtain topology data */
  s->atom = mdio_getAtomTopo(topo, &(s->atom_len));
  s->bond = mdio_getBondTopo(topo, &(s->bond_len));
  s->angle = mdio_getAngleTopo(topo, &(s->angle_len));
  s->dihed = mdio_getDihedTopo(topo, &(s->dihed_len));
  s->impr = mdio_getImprTopo(topo, &(s->impr_len));
  s->excl = mdio_getExclTopo(topo, &(s->excl_len));

  /* check number of atoms and atomprms */
  if (s->atom_len <= 0) {
    return error("%d atoms in system", s->atom_len);
  }
  else if (s->atomprm_len <= 0) {
    return error("%d atom parameters", s->atomprm_len);
  }

  /* message to user */
  printf("# successfully combined force field parameters with topology\n");
  printf("# system has: %9d atoms\n", s->atom_len);
  printf("#             %9d bonds\n", s->bond_len);
  printf("#             %9d angles\n", s->angle_len);
  printf("#             %9d dihedrals\n", s->dihed_len);
  printf("#             %9d impropers\n", s->impr_len);
  printf("#             %9d explicit nonbonded exclusions\n", s->excl_len);

  /* read positions from PDB file for expected length */
  if (mdio_readPdbcoord(pdb, p->coordinates, s->atom_len)) {
    return error(MSG_MDIO, mdio_getErrorMessageFile((mdio_File *) pdb));
  }
  s->pdbatompos = mdio_getAtomPdbcoord(pdb, &npos);
  if (npos != s->atom_len) {
    return error("number of atoms %d in coordinate PDB differs from "
        "%d atoms in PSF", npos, s->atom_len);
  }

  /* message to user */
  printf("# PDB coordinates file \"%s\"\n", p->coordinates);

  /* set "pdbatomvel" to also point to extended atom position info (for now) */
  s->pdbatomvel = s->pdbatompos;

  if (p->bincoordinates) { 
    /* read positions from binary coordinate file for expected length */
    if (mdio_readBincoord(bin, p->bincoordinates, s->atom_len)) {
      return error(MSG_MDIO, mdio_getErrorMessageFile((mdio_File *) bin));
    }
    s->pos = mdio_getBincoord(bin, &npos);
    if (npos != s->atom_len) {
      return error("number of atoms %d in binary position file differs from "
          "%d atoms in PSF", npos, s->atom_len);
    }
    /* message to user */
    printf("# binary coordinates file \"%s\"\n", p->bincoordinates);
  }
  else {
    /* otherwise use atom positions from PDB file */
    s->pos = mdio_getPdbcoord(pdb, &npos);
    ASSERT(npos == s->atom_len);
  }

  /*
   * note that we check first for "velocities" file, then for "binvelocities"
   * (it is turned around from "coordinates" to imitate behavior of NAMD)
   */
  if (p->velocities) {
    /* read velocities from PDB file for expected length */
    if (mdio_readPdbcoord(pdbvel, p->velocities, s->atom_len)) {
      return error(MSG_MDIO, mdio_getErrorMessageFile((mdio_File *) pdbvel));
    }
    /* reset "pdbatomvel" to point to extended velocity info */
    s->pdbatomvel = mdio_getAtomPdbcoord(pdbvel, &nvel);
    s->vel = mdio_getPdbcoord(pdbvel, &nvel);
    if (nvel != s->atom_len) {
      return error("number of atoms %d in velocity PDB differs from "
          "%d atoms in PSF", nvel, s->atom_len);
    }
    /* must convert pdb velocities from A/ps to internal units A/fs */
    for (k = 0;  k < nvel;  k++) {
      s->vel[k].x *= MD_ANGSTROM_FS;
      s->vel[k].y *= MD_ANGSTROM_FS;
      s->vel[k].z *= MD_ANGSTROM_FS;
    }
    /* message to user */
    printf("# PDB velocities file \"%s\"\n", p->velocities);
  }
  else if (p->namdbinvelocities) {
    const double factor = NAMD_PDBVELFACTOR * 1e-3; /* unit convert to A/fs */
    /* read velocities from namd binary coordinate file for expected length */
    if (mdio_readBincoord(binvel, p->namdbinvelocities, s->atom_len)) {
      return error(MSG_MDIO, mdio_getErrorMessageFile((mdio_File *) binvel));
    }
    s->vel = mdio_getBincoord(binvel, &nvel);
    if (nvel != s->atom_len) {
      return error("number of atoms %d in NAMD binary velocity file differs "
          "from %d atoms in PSF", nvel, s->atom_len);
    }
    /* must convert units */
    for (k = 0;  k < nvel;  k++) {
      s->vel[k].x *= factor;
      s->vel[k].y *= factor;
      s->vel[k].z *= factor;
    }
    /* message to user */
    printf("# NAMD binary velocities file \"%s\"\n", p->namdbinvelocities);
  }
  else if (p->binvelocities) {
    /* read velocities from binary coordinate file for expected length */
    if (mdio_readBincoord(binvel, p->binvelocities, s->atom_len)) {
      return error(MSG_MDIO, mdio_getErrorMessageFile((mdio_File *) binvel));
    }
    s->vel = mdio_getBincoord(binvel, &nvel);
    if (nvel != s->atom_len) {
      return error("number of atoms %d in binary velocity file differs from "
          "%d atoms in PSF", nvel, s->atom_len);
    }
    /* message to user */
    printf("# binary velocities file \"%s\"\n", p->binvelocities);
  }
  else {
    /*
     * allocate zeroed velocity array and assume engine will set
     * using some initial temperature
     */
    nvel = s->atom_len;
    if ((s->vel = (MD_Dvec *) calloc(nvel, sizeof(MD_Dvec))) == NULL) {
      return error(MSG_NOMEM);
    }
    s->is_vel_alloc = TRUE;
  }

  ASSERT(npos == s->atom_len);
  ASSERT(nvel == s->atom_len);

  /* allocate extra buffers that might be needed */
  s->posbuf = (MD_Dvec *) calloc(s->atom_len, sizeof(MD_Dvec));
  s->velbuf = (MD_Dvec *) calloc(s->atom_len, sizeof(MD_Dvec));
  s->resultbuf = calloc(1, MD_SIZEOF(s->result_attr.type)); 
  if (s->posbuf == NULL || s->velbuf == NULL || s->resultbuf == NULL) {
    return error(MSG_NOMEM);
  }

  /* set force field parameter engine arrays */
  if (s->atomprm_attr.access & MD_SHARE) {
    /* all param arrays have SHARE access */
    if (MD_share(e, s->atomprm_id, s->atomprm, s->atomprm_len,
          s->atomprm_len)
        || MD_share(e, s->bondprm_id, s->bondprm, s->bondprm_len,
          s->bondprm_len)
        || MD_share(e, s->angleprm_id, s->angleprm, s->angleprm_len,
          s->angleprm_len)
        || MD_share(e, s->dihedprm_id, s->dihedprm, s->dihedprm_len,
          s->dihedprm_len)
        || MD_share(e, s->imprprm_id, s->imprprm, s->imprprm_len,
          s->imprprm_len)
        || MD_share(e, s->nbfixprm_id, s->nbfixprm, s->nbfixprm_len,
          s->nbfixprm_len)) {
      return error(MSG_MDAPI, "from MD_share()", MD_errmsg(e));
    }
  }
  else {
    /* fall back on WRITE access - but it increases memory use */
    /* resize arrays to desired length */
    if (MD_setlen(e, s->atomprm_id, s->atomprm_len)
        || MD_setlen(e, s->bondprm_id, s->bondprm_len)
        || MD_setlen(e, s->angleprm_id, s->angleprm_len)
        || MD_setlen(e, s->dihedprm_id, s->dihedprm_len)
        || MD_setlen(e, s->imprprm_id, s->imprprm_len)
        || MD_setlen(e, s->nbfixprm_id, s->nbfixprm_len)) {
      return error(MSG_MDAPI, "from MD_setlen()", MD_errmsg(e));
    }
    /* write contents from our buffers to engine buffers */
    if (MD_write(e, s->atomprm_id, s->atomprm, s->atomprm_len)
        || MD_write(e, s->bondprm_id, s->bondprm, s->bondprm_len)
        || MD_write(e, s->angleprm_id, s->angleprm, s->angleprm_len)
        || MD_write(e, s->dihedprm_id, s->dihedprm, s->dihedprm_len)
        || MD_write(e, s->imprprm_id, s->imprprm, s->imprprm_len)
        || MD_write(e, s->nbfixprm_id, s->nbfixprm, s->nbfixprm_len)) {
      return error(MSG_MDAPI, "from MD_write()", MD_errmsg(e));
    }
  }

  /* set topology engine arrays */
  if (s->atom_attr.access & MD_SHARE) {
    /* all topology arrays have SHARE access */
    if (MD_share(e, s->atom_id, s->atom, s->atom_len, s->atom_len)
        || MD_share(e, s->bond_id, s->bond, s->bond_len, s->bond_len)
        || MD_share(e, s->angle_id, s->angle, s->angle_len, s->angle_len)
        || MD_share(e, s->dihed_id, s->dihed, s->dihed_len, s->dihed_len)
        || MD_share(e, s->impr_id, s->impr, s->impr_len, s->impr_len)
        || MD_share(e, s->excl_id, s->excl, s->excl_len, s->excl_len)) {
      return error(MSG_MDAPI, "from MD_share()", MD_errmsg(e));
    }
  }
  else {
    /* fall back on WRITE access - but it increases memory use */
    /* resize arrays to desired length */
    if (MD_setlen(e, s->atom_id, s->atom_len)
        || MD_setlen(e, s->bond_id, s->bond_len)
        || MD_setlen(e, s->angle_id, s->angle_len)
        || MD_setlen(e, s->dihed_id, s->dihed_len)
        || MD_setlen(e, s->impr_id, s->impr_len)
        || MD_setlen(e, s->excl_id, s->excl_len)) {
      return error(MSG_MDAPI, "from MD_setlen()", MD_errmsg(e));
    }
    /* write contents from our buffers to engine buffers */
    if (MD_write(e, s->atom_id, s->atom, s->atom_len)
        || MD_write(e, s->bond_id, s->bond, s->bond_len)
        || MD_write(e, s->angle_id, s->angle, s->angle_len)
        || MD_write(e, s->dihed_id, s->dihed, s->dihed_len)
        || MD_write(e, s->impr_id, s->impr, s->impr_len)
        || MD_write(e, s->excl_id, s->excl, s->excl_len)) {
      return error(MSG_MDAPI, "from MD_write()", MD_errmsg(e));
    }
  }

  /* set trajectory engine arrays */
  if (s->pos_attr.access & MD_SHARE) {
    /* both pos and vel have SHARE access */
    if (MD_share(e, s->pos_id, s->pos, s->atom_len, s->atom_len)
        || MD_share(e, s->vel_id, s->vel, s->atom_len, s->atom_len)) {
      return error(MSG_MDAPI, "from MD_share()", MD_errmsg(e));
    }
  }
  else {
    /* fall back on WRITE access - but it increases memory use */
    /* resize arrays to desired length */
    if (MD_setlen(e, s->pos_id, s->atom_len)
        || MD_setlen(e, s->vel_id, s->atom_len)) {
      return error(MSG_MDAPI, "from MD_setlen()", MD_errmsg(e));
    }
    /* write contents from our buffers to engine buffers */
    if (MD_write(e, s->pos_id, s->pos, s->atom_len)
        || MD_write(e, s->vel_id, s->vel, s->atom_len)) {
      return error(MSG_MDAPI, "from MD_write()", MD_errmsg(e));
    }
  }

  /* need to refresh engine array attributes */
  s->atomprm_attr = MD_attrib(e, s->atomprm_id);
  s->bondprm_attr = MD_attrib(e, s->bondprm_id);
  s->angleprm_attr = MD_attrib(e, s->angleprm_id);
  s->dihedprm_attr = MD_attrib(e, s->dihedprm_id);
  s->imprprm_attr = MD_attrib(e, s->imprprm_id);
  s->nbfixprm_attr = MD_attrib(e, s->nbfixprm_id);
  s->atom_attr = MD_attrib(e, s->atom_id);
  s->bond_attr = MD_attrib(e, s->bond_id);
  s->angle_attr = MD_attrib(e, s->angle_id);
  s->dihed_attr = MD_attrib(e, s->dihed_id);
  s->impr_attr = MD_attrib(e, s->impr_id);
  s->excl_attr = MD_attrib(e, s->excl_id);
  s->pos_attr = MD_attrib(e, s->pos_id);
  s->vel_attr = MD_attrib(e, s->vel_id);

  /*** setup fixed atoms ***/
  if (p->is_fixedatoms) {
    mdio_Pdbcoord pdbfixedatoms;
    MD_Dvec *dvec;
    mdio_Pdbatom *pdbatom;
    int32 whichcol = p->fixedatomscol[0];  /* letter identifying column */
    int32 natoms;
    adt_List *fixedatomslist = &(s->fixedatoms_index);
    int32 *array;
    int32 arraylen;
    int32 is_atomfixed;

    /* engine must support */
    if (s->fixedatomslist_id == MD_FAIL) {
      return error(MSG_ENGINE, "engine does not support fixed atoms");
    }

    /* construct PDB file object */
    if (mdio_initializePdbcoord(&pdbfixedatoms)) {
      return error(MSG, "mdio_initializePdbcoord() failed");
    }
    mdio_setErrorHandlerFile((mdio_File *) &pdbfixedatoms, mdio_errhandler);

    /* read data from PDB file */
    if (mdio_readPdbcoord(&pdbfixedatoms, p->fixedatomsfile, s->atom_len)) {
      return error(MSG_MDIO,
          mdio_getErrorMessageFile((mdio_File *) &pdbfixedatoms));
    }
    /* get pointers to PDB file object data arrays */
    dvec = mdio_getPdbcoord(&pdbfixedatoms, &natoms);
    if (natoms != s->atom_len) {
      return error("number of atoms %d in fixed atoms PDB differs from "
          "%d atoms in PSF", natoms, s->atom_len);
    }
    pdbatom = mdio_getAtomPdbcoord(&pdbfixedatoms, &natoms);
    ASSERT(natoms == s->atom_len);

    /* message to user */
    printf("# PDB fixed atoms file \"%s\"\n", p->fixedatomsfile);

    /* construct list of fixed atoms */
    for (k = 0;  k < natoms;  k++) {
      is_atomfixed = FALSE;
      switch (whichcol) {
        case 'X':
          is_atomfixed = (dvec[k].x != 0);
          break;
        case 'Y':
          is_atomfixed = (dvec[k].y != 0);
          break;
        case 'Z':
          is_atomfixed = (dvec[k].z != 0);
          break;
        case 'O':
          is_atomfixed = (pdbatom[k].occupancy != 0);
          break;
        case 'B':
          is_atomfixed = (pdbatom[k].tempFactor != 0);
          break;
        default:
          return error(MSG, "incorrect column indicator for fixed atoms");
      }
      if (is_atomfixed && adt_appendList(fixedatomslist, &k)) {
        return error(MSG_NOMEM);
      }
    }
    printf("# using column \'%c\' to specify fixed atoms\n", (char)whichcol);

    /* no longer need PDB file data, free up some memory */
    mdio_cleanupPdbcoord(&pdbfixedatoms);

    /* set fixed atoms list engine array */
    array = (int32 *) adt_getDataList(fixedatomslist);
    arraylen = adt_getLengthList(fixedatomslist);
    if (s->fixedatomslist_attr.access & MD_SHARE) {
      /* use SHARE access if available */
      if (MD_share(e, s->fixedatomslist_id, array, arraylen, arraylen)) {
        return error(MSG_MDAPI, "from MD_share()", MD_errmsg(e));
      }
    }
    else {
      /* fall back on WRITE access - but it increases memory use */
      /* resize array to desired length */
      if (MD_setlen(e, s->fixedatomslist_id, arraylen)) {
        return error(MSG_MDAPI, "from MD_setlen()", MD_errmsg(e));
      }
      /* write contents from our buffer to engine buffer */
      if (MD_write(e, s->fixedatomslist_id, array, arraylen)) {
        return error(MSG_MDAPI, "from MD_write()", MD_errmsg(e));
      }
    }

    /* need to refresh engine array attributes */
    s->fixedatomslist_attr = MD_attrib(e, s->fixedatomslist_id);

    /* message to user */
    printf("# fixing %d atoms\n", arraylen);
  }

  /*** setup harmonic constraints ***/
  if (p->is_constraints) {
    mdio_Pdbcoord pdb_consref;
    mdio_Pdbcoord pdb_conskfile;
    DevenConstraint c;
    int32 whichcol;
    int32 natoms;
    adt_List *constraintlist = &(s->constraints_data);
    void *array;
    int32 arraylen;
    int32 is_same_file = (strcmp(p->consref, p->conskfile) == 0);
    MD_Dvec *refpos_dvec, *k_dvec;
    mdio_Pdbatom *k_pdbatom;

    TEXT("setting constraints");

    /* engine must support */
    if (s->constraintlist_id == MD_FAIL) {
      return error(MSG_ENGINE, "engine does not support constraints");
    }

    /* construct PDB file object */
    if (mdio_initializePdbcoord(&pdb_consref)) {
      return error(MSG, "mdio_initializePdbcoord() failed");
    }
    mdio_setErrorHandlerFile((mdio_File *) &pdb_consref, mdio_errhandler);

    /* read data from PDB file */
    if (mdio_readPdbcoord(&pdb_consref, p->consref, s->atom_len)) {
      return error(MSG_MDIO,
          mdio_getErrorMessageFile((mdio_File *) &pdb_consref));
    }
    /* get pointers to PDB file object coordinate array */
    refpos_dvec = mdio_getPdbcoord(&pdb_consref, &natoms);
    if (natoms != s->atom_len) {
      return error("number of atoms %d in constraint reference position PDB "
          "differs from %d atoms in PSF", natoms, s->atom_len);
    }

    /* message to user */
    printf("# PDB constraints reference file \"%s\"\n", p->consref);

    /* see if we need to open and read a second file for constants */
    if (is_same_file) {
      k_dvec = refpos_dvec;
      k_pdbatom = mdio_getAtomPdbcoord(&pdb_consref, &natoms);
      ASSERT(natoms == s->atom_len);
    }
    else {
      /* construct PDB file object */
      if (mdio_initializePdbcoord(&pdb_conskfile)) {
        return error(MSG, "mdio_initializePdbcoord() failed");
      }
      /* get pointers to PDB file object coordinate arrays */
      k_dvec = mdio_getPdbcoord(&pdb_conskfile, &natoms);
      if (natoms != s->atom_len) {
        return error("number of atoms %d in constraint reference position PDB "
            "differs from %d atoms in PSF", natoms, s->atom_len);
      }
      k_pdbatom = mdio_getAtomPdbcoord(&pdb_conskfile, &natoms);
      ASSERT(natoms == s->atom_len);
      /* message to user */
      printf("# PDB constraints force constant file \"%s\"\n", p->consref);
    }

    /* construct list of constraints */
    c.expo = p->consexp;
    STR(p->conskcol);
    whichcol = p->conskcol[0];  /* letter identifying column */
    for (k = 0;  k < natoms;  k++) {
      c.index = k;
      c.refpos = refpos_dvec[k];
      switch (whichcol) {
        case 'X':
          c.k = k_dvec[k].x;
          break;
        case 'Y':
          c.k = k_dvec[k].y;
          break;
        case 'Z':
          c.k = k_dvec[k].z;
          break;
        case 'O':
          c.k = k_pdbatom[k].occupancy;
          break;
        case 'B':
          c.k = k_pdbatom[k].tempFactor;
          break;
        default:
          return error(MSG, "incorrect column indicator for constraints");
      }
      if (c.k != 0.0 && adt_appendList(constraintlist, &c)) {
        return error(MSG_NOMEM);
      }
    }
    printf("# using column \'%c\' to specify constraints\n", (char)whichcol);

    /* no longer need PDB file(s), free up memory */
    mdio_cleanupPdbcoord(&pdb_consref);
    if ( ! is_same_file) {
      mdio_cleanupPdbcoord(&pdb_conskfile);
    }

    /* set constraints list engine array */
    array = adt_getDataList(constraintlist);
    arraylen = adt_getLengthList(constraintlist);
    if (s->constraintlist_attr.access & MD_SHARE) {
      /* use SHARE access if available */
      if (MD_share(e, s->constraintlist_id, array, arraylen, arraylen)) {
        return error(MSG_MDAPI, "from MD_share()", MD_errmsg(e));
      }
    }
    else {
      /* fall back on WRITE access - but it increases memory use */
      /* resize array to desired length */
      if (MD_setlen(e, s->constraintlist_id, arraylen)) {
        return error(MSG_MDAPI, "from MD_setlen()", MD_errmsg(e));
      }
      /* write constents from our buffer to engine buffer */
      if (MD_write(e, s->constraintlist_id, array, arraylen)) {
        return error(MSG_MDAPI, "from MD_write()", MD_errmsg(e));
      }
    }

    /* message to user */
    printf("# setting %d harmonic restraining forces\n", arraylen);

    /* need to refresh engine array attributes */
    s->constraintlist_attr = MD_attrib(e, s->constraintlist_id);

    TEXT("done setting constraints");
  }

  return 0;
}


int system_setup_potential(System *s, SimParam *p)
{
  MD_Engine *e = s->eng;

  PotentialIndex n;
  adt_List *pdex = &(s->potential_index);
  adt_List *psel = &(p->potential_select);
  int psel_len = adt_getLengthList(psel);

  adt_Table *t_bond = NULL;
  adt_Table *t_angle = NULL;
  adt_Table *t_dihed = NULL;
  adt_Table *t_impr = NULL;

  adt_List *l_bond = NULL;
  adt_List *l_angle = NULL;
  adt_List *l_dihed = NULL;
  adt_List *l_impr = NULL;

  const char *cs;
  char name[8];
  int is_bond = FALSE;
  int is_angle = FALSE;
  int is_dihed = FALSE;
  int is_impr = FALSE;

  int32 type, index;
  int32 a, b, c, d, i, j;
  char ch;

  if ( ! s->is_potential_array ) {
    warn("potential arrays are unavailable from engine");
    return 0;
  }

  for (i = 0;  i < psel_len;  i++) {
#define BUFLEN  36
    char buf[BUFLEN];
    cs = *((const char **) adt_indexList(psel, i));

    /* validate selection */
    if (sscanf(cs, "%7s", name) != 1) {
      warn("potential selection \"%s\" is not understood", cs);
      type = INVALID;
    }
    else if (strcasecmp(name, "bond") == 0
        && sscanf(cs, "%*s %d %d%c", &a, &b, &ch) == 2) {
      type = BOND;
      is_bond = TRUE;
    }
    else if (strcasecmp(name, "angle") == 0
        && sscanf(cs, "%*s %d %d %d%c", &a, &b, &c, &ch) == 3) {
      type = ANGLE;
      is_angle = TRUE;
    }
    else if (strcasecmp(name, "dihed") == 0
        && sscanf(cs, "%*s %d %d %d %d%c", &a, &b, &c, &d, &ch) == 4) {
      type = DIHED;
      is_dihed = TRUE;
    }
    else if (strcasecmp(name, "impr") == 0
        && sscanf(cs, "%*s %d %d %d %d%c", &a, &b, &c, &d, &ch) == 4) {
      type = IMPR;
      is_impr = TRUE;
    }
    else if (strcasecmp(name, "elec") == 0
        && sscanf(cs, "%*s %d%c", &a, &ch) == 1) {
      type = ELEC;
    }
    else if (strcasecmp(name, "vdw") == 0
        && sscanf(cs, "%*s %d%c", &a, &ch) == 1) {
      type = VDW;
    }
    else if (strcasecmp(name, "bound") == 0
        && sscanf(cs, "%*s %d%c", &a, &ch) == 1) {
      type = BOUND;
    }
    else if (strcasecmp(name, "epot") == 0
        && sscanf(cs, "%*s %d%c", &a, &ch) == 1) {
      type = EPOT;
    }
    else {
      warn("potential selection \"%s\" is not understood", cs);
      type = INVALID;
    }

    /* see if we need to create a hash table */
    if (is_bond && t_bond == NULL) {
      if ((t_bond = adt_createTable(0)) == NULL) {
        return error(MSG_NOMEM);
      }
      if ((l_bond = adt_createList(sizeof(char *), 0, NULL)) == NULL) {
        return error(MSG_NOMEM);
      }
      for (j = 0;  j < s->bond_len;  j++) {
#define BONDSTRLEN  20
        MD_Bond x = s->bond[j];
        char *str;
        if ((str = (char *) malloc(BONDSTRLEN)) == NULL) {
          return error(MSG_NOMEM);
        }
        snprintf(str, BONDSTRLEN, "%x %x", x.atom[0], x.atom[1]);
        if (adt_appendList(l_bond, &str)) {
          return error(MSG_NOMEM);
        }
        if (adt_insertTable(t_bond, str, j) != j) {
          return error(MSG_NOMEM);
        }
      }
    }
    if (is_angle && t_angle == NULL) {
      if ((t_angle = adt_createTable(0)) == NULL) {
        return error(MSG_NOMEM);
      }
      if ((l_angle = adt_createList(sizeof(char *), 0, NULL)) == NULL) {
        return error(MSG_NOMEM);
      }
      for (j = 0;  j < s->angle_len;  j++) {
#define ANGLESTRLEN  28
        MD_Angle x = s->angle[j];
        char *str;
        if ((str = (char *) malloc(ANGLESTRLEN)) == NULL) {
          return error(MSG_NOMEM);
        }
        snprintf(str, ANGLESTRLEN, "%x %x %x", x.atom[0], x.atom[1],
            x.atom[2]);
        if (adt_appendList(l_angle, &str)) {
          return error(MSG_NOMEM);
        }
        if (adt_insertTable(t_angle, str, j) != j) {
          return error(MSG_NOMEM);
        }
      }
    }
    if (is_dihed && t_dihed == NULL) {
      if ((t_dihed = adt_createTable(0)) == NULL) {
        return error(MSG_NOMEM);
      }
      if ((l_dihed = adt_createList(sizeof(char *), 0, NULL)) == NULL) {
        return error(MSG_NOMEM);
      }
      for (j = 0;  j < s->dihed_len;  j++) {
#define DIHEDSTRLEN  36
        MD_Tors x = s->dihed[j];
        char *str;
        if ((str = (char *) malloc(DIHEDSTRLEN)) == NULL) {
          return error(MSG_NOMEM);
        }
        snprintf(str, DIHEDSTRLEN, "%x %x %x %x", x.atom[0], x.atom[1],
            x.atom[2], x.atom[3]);
        if (adt_appendList(l_dihed, &str)) {
          return error(MSG_NOMEM);
        }
        if (adt_insertTable(t_dihed, str, j) != j) {
          return error(MSG_NOMEM);
        }
      }
    }
    if (is_impr && t_impr == NULL) {
      if ((t_impr = adt_createTable(0)) == NULL) {
        return error(MSG_NOMEM);
      }
      if ((l_impr = adt_createList(sizeof(char *), 0, NULL)) == NULL) {
        return error(MSG_NOMEM);
      }
      for (j = 0;  j < s->impr_len;  j++) {
#define IMPRSTRLEN  36
        MD_Tors x = s->impr[j];
        char *str;
        if ((str = (char *) malloc(IMPRSTRLEN)) == NULL) {
          return error(MSG_NOMEM);
        }
        snprintf(str, IMPRSTRLEN, "%x %x %x %x", x.atom[0], x.atom[1],
            x.atom[2], x.atom[3]);
        if (adt_appendList(l_impr, &str)) {
          return error(MSG_NOMEM);
        }
        if (adt_insertTable(t_impr, str, j) != j) {
          return error(MSG_NOMEM);
        }
      }
    }

    /* determine index of potential */
    /* have to decrease atom numbers by one */
    switch (type) {
      case BOND:
        snprintf(buf, BUFLEN, "%x %x", a-1, b-1);
        if ((index = adt_lookupTable(t_bond, buf)) == ADT_ERROR) {
          warn("unknown bond: \"%d %d\"", a, b);
          type = INVALID;
          index = -1;
        }
        else if (s->u_bond_attr.len != s->bond_len) {
          /* set expected length for u_bond */
          if (MD_setlen(e, s->u_bond_id, s->bond_len)) {
            return error(MSG_MDAPI, "from MD_setlen()", MD_errmsg(e));
          }
          s->u_bond_attr = MD_attrib(e, s->u_bond_id);
        }
        break;
      case ANGLE:
        snprintf(buf, BUFLEN, "%x %x %x", a-1, b-1, c-1);
        if ((index = adt_lookupTable(t_angle, buf)) == ADT_ERROR) {
          warn("unknown angle: \"%d %d %d\"", a, b, c);
          type = INVALID;
          index = -1;
        }
        else if (s->u_angle_attr.len != s->angle_len) {
          /* set expected length for u_angle */
          if (MD_setlen(e, s->u_angle_id, s->angle_len)) {
            return error(MSG_MDAPI, "from MD_setlen()", MD_errmsg(e));
          }
          s->u_angle_attr = MD_attrib(e, s->u_angle_id);
        }
        break;
      case DIHED:
        snprintf(buf, BUFLEN, "%x %x %x %x", a-1, b-1, c-1, d-1);
        if ((index = adt_lookupTable(t_dihed, buf)) == ADT_ERROR) {
          warn("unknown dihedral: \"%d %d %d %d\"", a, b, c, d);
          type = INVALID;
          index = -1;
        }
        else if (s->u_dihed_attr.len != s->dihed_len) {
          /* set expected length for u_dihed */
          if (MD_setlen(e, s->u_dihed_id, s->dihed_len)) {
            return error(MSG_MDAPI, "from MD_setlen()", MD_errmsg(e));
          }
          s->u_dihed_attr = MD_attrib(e, s->u_dihed_id);
        }
        break;
      case IMPR:
        snprintf(buf, BUFLEN, "%x %x %x %x", a-1, b-1, c-1, d-1);
        if ((index = adt_lookupTable(t_impr, buf)) == ADT_ERROR) {
          warn("unknown improper: \"%d %d %d %d\"", a, b, c, d);
          type = INVALID;
          index = -1;
        }
        else if (s->u_impr_attr.len != s->impr_len) {
          /* set expected length for u_impr */
          if (MD_setlen(e, s->u_impr_id, s->impr_len)) {
            return error(MSG_MDAPI, "from MD_setlen()", MD_errmsg(e));
          }
          s->u_impr_attr = MD_attrib(e, s->u_impr_id);
        }
        break;
      case ELEC:
        index = a-1;
        if (index < 0 || index >= s->atom_len) {
          warn("unknown atom: \"%d\"", a);
          type = INVALID;
          index = -1;
        }
        else if (s->u_elec_attr.len != s->atom_len) {
          /* set expected length for u_elec */
          if (MD_setlen(e, s->u_elec_id, s->atom_len)) {
            return error(MSG_MDAPI, "from MD_setlen()", MD_errmsg(e));
          }
          s->u_elec_attr = MD_attrib(e, s->u_elec_id);
        }
        break;
      case VDW:
        index = a-1;
        if (index < 0 || index >= s->atom_len) {
          warn("unknown atom: \"%d\"", a);
          type = INVALID;
          index = -1;
        }
        else if (s->u_vdw_attr.len != s->atom_len) {
          /* set expected length for u_vdw */
          if (MD_setlen(e, s->u_vdw_id, s->atom_len)) {
            return error(MSG_MDAPI, "from MD_setlen()", MD_errmsg(e));
          }
          s->u_vdw_attr = MD_attrib(e, s->u_vdw_id);
        }
        break;
      case BOUND:
        index = a-1;
        if (index < 0 || index >= s->atom_len) {
          warn("unknown atom: \"%d\"", a);
          type = INVALID;
          index = -1;
        }
        else if (s->u_bound_attr.len != s->atom_len) {
          /* set expected length for u_bound */
          if (MD_setlen(e, s->u_bound_id, s->atom_len)) {
            return error(MSG_MDAPI, "from MD_setlen()", MD_errmsg(e));
          }
          s->u_bound_attr = MD_attrib(e, s->u_bound_id);
        }
        break;
      case EPOT:
        index = a-1;
        if (index < 0 || index >= s->atom_len) {
          warn("unknown atom: \"%d\"", a);
          type = INVALID;
          index = -1;
        }
        else if (s->pot_elec_attr.len != s->atom_len) {
          /* set expected length for pot_elec */
          if (MD_setlen(e, s->pot_elec_id, s->atom_len)) {
            return error(MSG_MDAPI, "from MD_setlen()", MD_errmsg(e));
          }
          s->pot_elec_attr = MD_attrib(e, s->pot_elec_id);
        }
        break;
      default:
        type = INVALID;
        index = -1;
    }

    /* store index info into list */
    n.type = type;
    n.index = index;
    if (adt_appendList(pdex, &n)) {
      return error(MSG_NOMEM);
    }
  }

  /* free table and list memory */
  if (t_bond) {
    adt_destroyTable(t_bond);
    for (i = 0;  i < s->bond_len;  i++) {
      char *str = *((char **) adt_indexList(l_bond, i));
      free(str);
    }
    adt_destroyList(l_bond);
  }
  if (t_angle) {
    adt_destroyTable(t_angle);
    for (i = 0;  i < s->angle_len;  i++) {
      char *str = *((char **) adt_indexList(l_angle, i));
      free(str);
    }
    adt_destroyList(l_angle);
  }
  if (t_dihed) {
    adt_destroyTable(t_dihed);
    for (i = 0;  i < s->dihed_len;  i++) {
      char *str = *((char **) adt_indexList(l_dihed, i));
      free(str);
    }
    adt_destroyList(l_dihed);
  }
  if (t_impr) {
    adt_destroyTable(t_impr);
    for (i = 0;  i < s->impr_len;  i++) {
      char *str = *((char **) adt_indexList(l_impr, i));
      free(str);
    }
    adt_destroyList(l_impr);
  }

  return 0;
}


/*
 * System destructor
 * free array allocations (from MDIO and directly)
 * destroy engine before calling
 */
void system_done(System *s)
{
  /* cleanup index lists */
  adt_cleanupList(&(s->potential_index));
  adt_cleanupList(&(s->fixedatoms_index));
  adt_cleanupList(&(s->constraints_data));

  /* cleanup MDIO file reader objects */
  mdio_cleanupParam(&(s->prm));
  mdio_cleanupTopo(&(s->topo));
  mdio_cleanupPdbcoord(&(s->pdb));
  mdio_cleanupPdbcoord(&(s->pdbvel));
  mdio_cleanupBincoord(&(s->bin));
  mdio_cleanupBincoord(&(s->binvel));

  /* cleanup array allocations */
  free(s->posbuf);
  free(s->velbuf);
  free(s->resultbuf);
  if (s->is_vel_alloc) {
    free(s->vel);
  }

  /* clear memory */
  memset(s, 0, sizeof(System));
}
