/*
 * Copyright (C) 2004-2005 by David J. Hardy.  All rights reserved.
 *
 * topo.c - read X-Plor PSF containing topology of molecular system
 */

#define DAVE_DEBUG

#include <stdlib.h>
#include <string.h>
#include <strings.h>
#include <ctype.h>
#include "mdio/topo.h"
#define DEBUG_WATCH
#include "debug/debug.h"


#define INIT_BUFLEN  (100)


/*
 * keep track of dihedral and improper multiplicities present in
 * X-Plor compatible PSF files, maintain list of repeated interactions
 */
typedef struct TorsMult_t {
  int index;  /* index of torsion interaction */
  int mult;   /* number of times interaction is repeated in PSF */
} TorsMult;


/* prototypes of internal functions */
static int psf_init(mdio_Topo *);
static void psf_done(mdio_Topo *);
static const char *psf_getline(mdio_Topo *);
static int psf_read(mdio_Topo *);


mdio_Topo *mdio_createTopo(void)
{
  mdio_Topo *p;
  p = (mdio_Topo *) malloc(sizeof(mdio_Topo));
  if (p == NULL) {
    ERRMSG("out of memory");
    return NULL;
  }
  if (mdio_initializeTopo(p)) {
    free(p);
    return NULL;
  }
  return p;
}


int mdio_initializeTopo(mdio_Topo *p)
{
  ASSERT(p != NULL);
  memset(p, 0, sizeof(mdio_Topo));     /* zero memory */
  if (mdio_initializeFile(&(p->file))) {
    return MDIO_ERROR;
  }
  if (adt_initializeArray(&(p->atom), sizeof(MD_Atom), 0, NULL)) {
    mdio_setErrorMessageFile(&(p->file), MDIO_ERROR_NOMEM,
        "cannot initialize \"atom\" array");
    return MDIO_ERROR;
  }
  if (adt_initializeArray(&(p->bond), sizeof(MD_Bond), 0, NULL)) {
    mdio_setErrorMessageFile(&(p->file), MDIO_ERROR_NOMEM,
        "cannot initialize \"bond\" array");
    return MDIO_ERROR;
  }
  if (adt_initializeArray(&(p->angle), sizeof(MD_Angle), 0, NULL)) {
    mdio_setErrorMessageFile(&(p->file), MDIO_ERROR_NOMEM,
        "cannot initialize \"angle\" array");
    return MDIO_ERROR;
  }
  if (adt_initializeArray(&(p->dihed), sizeof(MD_Tors), 0, NULL)) {
    mdio_setErrorMessageFile(&(p->file), MDIO_ERROR_NOMEM,
        "cannot initialize \"dihed\" array");
    return MDIO_ERROR;
  }
  if (adt_initializeArray(&(p->impr), sizeof(MD_Tors), 0, NULL)) {
    mdio_setErrorMessageFile(&(p->file), MDIO_ERROR_NOMEM,
        "cannot initialize \"tors\" array");
    return MDIO_ERROR;
  }
  if (adt_initializeArray(&(p->excl), sizeof(MD_Excl), 0, NULL)) {
    mdio_setErrorMessageFile(&(p->file), MDIO_ERROR_NOMEM,
        "cannot initialize \"excl\" array");
    return MDIO_ERROR;
  }
  if (adt_initializeList(&(p->dihed_mult), sizeof(TorsMult), 0, NULL)) {
    mdio_setErrorMessageFile(&(p->file), MDIO_ERROR_NOMEM,
        "cannot initialize \"dihed_mult\" list");
    return MDIO_ERROR;
  }
  if (adt_initializeList(&(p->impr_mult), sizeof(TorsMult), 0, NULL)) {
    mdio_setErrorMessageFile(&(p->file), MDIO_ERROR_NOMEM,
        "cannot initialize \"impr_mult\" list");
    return MDIO_ERROR;
  }
  if (psf_init(p)) {
    ERRMSG("cannot initialize psf parser");
    psf_done(p);
    return MDIO_ERROR;
  }
  return 0;
}


void mdio_destroyTopo(mdio_Topo *p)
{
  ASSERT(p != NULL);
  mdio_cleanupTopo(p);
  free(p);
}


void mdio_cleanupTopo(mdio_Topo *p)
{
  ASSERT(p != NULL);
  psf_done(p);
  adt_cleanupArray(&(p->atom));
  adt_cleanupArray(&(p->bond));
  adt_cleanupArray(&(p->angle));
  adt_cleanupArray(&(p->dihed));
  adt_cleanupArray(&(p->impr));
  adt_cleanupArray(&(p->excl));
  adt_cleanupList(&(p->dihed_mult));
  adt_cleanupList(&(p->impr_mult));
  mdio_cleanupFile(&(p->file));
}


int mdio_readTopo(mdio_Topo *p, const char *name)
{
  ASSERT(p != NULL);
  ASSERT(name != NULL);

  /* open file */
  if (mdio_openFile(&(p->file), name, MDIO_FILE_TEXT | MDIO_FILE_READ)) {
    return MDIO_ERROR;
  }

  /* read file - no forgiveness for syntax errors! */
  if (psf_read(p)) {
    mdio_closeFile(&(p->file));
    return MDIO_ERROR;
  }

  /* close file */
  if (mdio_closeFile(&(p->file))) {
    return MDIO_ERROR;
  }
  return 0;
}


MD_Atom *mdio_getAtomTopo(mdio_Topo *p, int *nelems)
{
  ASSERT(p != NULL);
  ASSERT(nelems != NULL);
  *nelems = adt_getLengthArray(&(p->atom));
  return (MD_Atom *) adt_getDataArray(&(p->atom));
}


MD_Bond *mdio_getBondTopo(mdio_Topo *p, int *nelems)
{
  ASSERT(p != NULL);
  ASSERT(nelems != NULL);
  *nelems = adt_getLengthArray(&(p->bond));
  return (MD_Bond *) adt_getDataArray(&(p->bond));
}


MD_Angle *mdio_getAngleTopo(mdio_Topo *p, int *nelems)
{
  ASSERT(p != NULL);
  ASSERT(nelems != NULL);
  *nelems = adt_getLengthArray(&(p->angle));
  return (MD_Angle *) adt_getDataArray(&(p->angle));
}


MD_Tors *mdio_getDihedTopo(mdio_Topo *p, int *nelems)
{
  ASSERT(p != NULL);
  ASSERT(nelems != NULL);
  *nelems = adt_getLengthArray(&(p->dihed));
  return (MD_Tors *) adt_getDataArray(&(p->dihed));
}


MD_Tors *mdio_getImprTopo(mdio_Topo *p, int *nelems)
{
  ASSERT(p != NULL);
  ASSERT(nelems != NULL);
  *nelems = adt_getLengthArray(&(p->impr));
  return (MD_Tors *) adt_getDataArray(&(p->impr));
}


MD_Excl *mdio_getExclTopo(mdio_Topo *p, int *nelems)
{
  ASSERT(p != NULL);
  ASSERT(nelems != NULL);
  *nelems = adt_getLengthArray(&(p->excl));
  return (MD_Excl *) adt_getDataArray(&(p->excl));
}


int mdio_indexParamTopo(mdio_Topo *topo, mdio_Param *prm)
{
  mdio_File *f;
  adt_Table *table;
  adt_Array *keylist;
  MD_AtomPrm *atomprm;
  MD_BondPrm *bondprm;
  MD_AnglePrm *angleprm;
  MD_TorsPrm *dihedprm;
  MD_TorsPrm *imprprm;
  MD_NbfixPrm *nbfixprm;
  int n_atomprm;
  int n_bondprm;
  int n_angleprm;
  int n_dihedprm;
  int n_imprprm;
  int n_nbfixprm;
  MD_Atom *atom;
  MD_Bond *bond;
  MD_Angle *angle;
  MD_Tors *dihed;
  MD_Tors *impr;
  int n_atom;
  int n_bond;
  int n_angle;
  int n_dihed;
  int n_impr;
  int nkeys, cnt;
  typedef char KeyName[36];
  KeyName *key;
  KeyName keyname;
  int i, j, k;
  int retval = 0;
  TorsMult *tmlist;
  int tmlist_len;
  char msg[240];  /* for error message */

  ASSERT(prm != NULL);
  ASSERT(topo != NULL);

  /* report errors through topology */
  f = &(topo->file);

  /* check on consistency between parameters file and PSF */
  if (prm->filetype == MDIO_PARAM_CHARMM
      && (adt_getLengthList(&(topo->dihed_mult)) > 0
        || adt_getLengthList(&(topo->impr_mult)) > 0)) {
    mdio_setErrorMessageFile(f, MDIO_ERROR_WARN,
        "inconsistent file types:  using CHARMM parameters file "
        "with X-Plor PSF, all dihedral multiplicities will be reset to 1");
  }

  /* get topology data */
  atom = mdio_getAtomTopo(topo, &n_atom);
  bond = mdio_getBondTopo(topo, &n_bond);
  angle = mdio_getAngleTopo(topo, &n_angle);
  dihed = mdio_getDihedTopo(topo, &n_dihed);
  impr = mdio_getImprTopo(topo, &n_impr);

  /* get param data */
  atomprm = mdio_getAtomParam(prm, &n_atomprm);
  bondprm = mdio_getBondParam(prm, &n_bondprm);
  angleprm = mdio_getAngleParam(prm, &n_angleprm);
  dihedprm = mdio_getDihedParam(prm, &n_dihedprm);
  imprprm = mdio_getImprParam(prm, &n_imprprm);
  nbfixprm = mdio_getNbfixParam(prm, &n_nbfixprm);

  /* setup table and keylist array */
  nkeys = n_atomprm + 2 * n_bondprm + 2 * n_angleprm
    + 2 * n_dihedprm + 2 * n_imprprm;
  if ((keylist = adt_createArray(sizeof(KeyName), nkeys, NULL)) == NULL) {
    mdio_setErrorFile(f, MDIO_ERROR_NOMEM);
    return MDIO_ERROR;
  }
  if ((table = adt_createTable(2 * nkeys)) == NULL) {
    adt_destroyArray(keylist);
    mdio_setErrorFile(f, MDIO_ERROR_NOMEM);
    return MDIO_ERROR;
  }
  key = (KeyName *) adt_getDataArray(keylist);

  /* store atom param names */
  cnt = 0;
  for (k = 0;  k < n_atomprm;  k++) {
    ASSERT(cnt < nkeys);
    snprintf(key[cnt], sizeof(KeyName), "%s", atomprm[k].type);
    j = adt_insertTable(table, key[cnt], k);
    if (j == ADT_ERROR) {
      adt_destroyTable(table);
      adt_destroyArray(keylist);
      mdio_setErrorFile(f, MDIO_ERROR_NOMEM);
      return MDIO_ERROR;
    }
    else if (j != k) {
      if (adt_updateTable(table, key[cnt], k) == ADT_ERROR) {
        BUG("string should already be in table");
      }
      snprintf(msg, sizeof(msg),
          "redefined atom parameters for %s", key[cnt]);
      mdio_setErrorMessageFile(f, MDIO_ERROR_WARN, msg);
    }
    cnt++;
  }

  /* store bond param names, listed in both directions */
  for (k = 0;  k < n_bondprm;  k++) {
    ASSERT(cnt < nkeys);
    snprintf(key[cnt], sizeof(KeyName), "%s %s",
        bondprm[k].type[0], bondprm[k].type[1]);
    j = adt_insertTable(table, key[cnt], k);
    if (j == ADT_ERROR) {
      adt_destroyTable(table);
      adt_destroyArray(keylist);
      mdio_setErrorFile(f, MDIO_ERROR_NOMEM);
      return MDIO_ERROR;
    }
    else if (j != k) {
      if (adt_updateTable(table, key[cnt], k) == ADT_ERROR) {
        BUG("string should already be in table");
      }
      snprintf(msg, sizeof(msg),
          "redefined bond parameters for %s", key[cnt]);
      mdio_setErrorMessageFile(f, MDIO_ERROR_WARN, msg);
    }
    cnt++;
    snprintf(key[cnt], sizeof(KeyName), "%s %s",
        bondprm[k].type[1], bondprm[k].type[0]);
    j = adt_insertTable(table, key[cnt], k);
    if (j == ADT_ERROR) {
      adt_destroyTable(table);
      adt_destroyArray(keylist);
      mdio_setErrorFile(f, MDIO_ERROR_NOMEM);
      return MDIO_ERROR;
    }
    else if (j != k) {
      if (adt_updateTable(table, key[cnt], k) == ADT_ERROR) {
        BUG("string should already be in table");
      }
    }
    cnt++;
  }

  /* store angle param names, listed in both directions */
  for (k = 0;  k < n_angleprm;  k++) {
    ASSERT(cnt < nkeys);
    snprintf(key[cnt], sizeof(KeyName), "%s %s %s",
        angleprm[k].type[0], angleprm[k].type[1], angleprm[k].type[2]);
    j = adt_insertTable(table, key[cnt], k);
    if (j == ADT_ERROR) {
      adt_destroyTable(table);
      adt_destroyArray(keylist);
      mdio_setErrorFile(f, MDIO_ERROR_NOMEM);
      return MDIO_ERROR;
    }
    else if (j != k) {
      if (adt_updateTable(table, key[cnt], k) == ADT_ERROR) {
        BUG("string should already be in table");
      }
      snprintf(msg, sizeof(msg),
          "redefined angle parameters for %s", key[cnt]);
      mdio_setErrorMessageFile(f, MDIO_ERROR_WARN, msg);
    }
    cnt++;
    snprintf(key[cnt], sizeof(KeyName), "%s %s %s",
        angleprm[k].type[2], angleprm[k].type[1], angleprm[k].type[0]);
    j = adt_insertTable(table, key[cnt], k);
    if (j == ADT_ERROR) {
      adt_destroyTable(table);
      adt_destroyArray(keylist);
      mdio_setErrorFile(f, MDIO_ERROR_NOMEM);
      return MDIO_ERROR;
    }
    else if (j != k) {
      if (adt_updateTable(table, key[cnt], k) == ADT_ERROR) {
        BUG("string should already be in table");
      }
    }
    cnt++;
  }

  /* store dihedral param names, listed in both directions */
  for (k = 0;  k < n_dihedprm;  k++) {
    ASSERT(cnt < nkeys);
    snprintf(key[cnt], sizeof(KeyName), "d %s %s %s %s", dihedprm[k].type[0],
        dihedprm[k].type[1], dihedprm[k].type[2], dihedprm[k].type[3]);
    j = adt_insertTable(table, key[cnt], k);
    if (j == ADT_ERROR) {
      adt_destroyTable(table);
      adt_destroyArray(keylist);
      mdio_setErrorFile(f, MDIO_ERROR_NOMEM);
      return MDIO_ERROR;
    }
    else if (j != k) {
      /* always update table */
      /* (for multiplicities, we repeatedly store the same strings, but it
       * doesn't matter since we have already allocated the space) */
      if (adt_updateTable(table, key[cnt], k) == ADT_ERROR) {
        BUG("string should already be in table");
      }
      if (j < k && dihedprm[j].mult + (k-j) > dihedprm[k].mult) {
        /* not an additional term, give warning for redefined params */
        snprintf(msg, sizeof(msg),
            "redefined dihedral parameters for %s", key[cnt]);
        mdio_setErrorMessageFile(f, MDIO_ERROR_WARN, msg);
      }
    }
    cnt++;
    snprintf(key[cnt], sizeof(KeyName), "d %s %s %s %s", dihedprm[k].type[3],
        dihedprm[k].type[2], dihedprm[k].type[1], dihedprm[k].type[0]);
    j = adt_insertTable(table, key[cnt], k);
    if (j == ADT_ERROR) {
      adt_destroyTable(table);
      adt_destroyArray(keylist);
      mdio_setErrorFile(f, MDIO_ERROR_NOMEM);
      return MDIO_ERROR;
    }
    else if (j != k) {
      if (adt_updateTable(table, key[cnt], k) == ADT_ERROR) {
        BUG("string should already be in table");
      }
    }
    cnt++;
  }

  /* store improper param names, listed in both directions */
  for (k = 0;  k < n_imprprm;  k++) {
    ASSERT(cnt < nkeys);
    snprintf(key[cnt], sizeof(KeyName), "i %s %s %s %s", imprprm[k].type[0],
        imprprm[k].type[1], imprprm[k].type[2], imprprm[k].type[3]);
    j = adt_insertTable(table, key[cnt], k);
    if (j == ADT_ERROR) {
      adt_destroyTable(table);
      adt_destroyArray(keylist);
      mdio_setErrorFile(f, MDIO_ERROR_NOMEM);
      return MDIO_ERROR;
    }
    else if (j != k) {
      /* always update table */
      /* (for multiplicities, we repeatedly store the same strings, but it
       * doesn't matter since we have already allocated the space) */
      if (adt_updateTable(table, key[cnt], k) == ADT_ERROR) {
        BUG("string should already be in table");
      }
      if (j < k && imprprm[j].mult + (k-j) > imprprm[k].mult) {
        /* not an additional term, give warning for redefined params */
        snprintf(msg, sizeof(msg),
            "redefined improper parameters for %s", key[cnt]);
        mdio_setErrorMessageFile(f, MDIO_ERROR_WARN, msg);
      }
    }
    cnt++;
    snprintf(key[cnt], sizeof(KeyName), "i %s %s %s %s", imprprm[k].type[3],
        imprprm[k].type[2], imprprm[k].type[1], imprprm[k].type[0]);
    j = adt_insertTable(table, key[cnt], k);
    if (j == ADT_ERROR) {
      adt_destroyTable(table);
      adt_destroyArray(keylist);
      mdio_setErrorFile(f, MDIO_ERROR_NOMEM);
      return MDIO_ERROR;
    }
    else if (j != k) {
      if (adt_updateTable(table, key[cnt], k) == ADT_ERROR) {
        BUG("string should already be in table");
      }
    }
    cnt++;
  }

  /* check array length assertion */
  ASSERT(cnt == nkeys);

  /* connect nbfix params to atom params */
  for (k = 0;  k < n_nbfixprm;  k++) {
    for (j = 0;  j < 2;  j++) {
      i = adt_lookupTable(table, nbfixprm[k].type[j]);
      if (i == ADT_ERROR) {
        snprintf(msg, sizeof(msg),
            "no atom parameters exist for %s", nbfixprm[k].type[j]);
        mdio_setErrorMessageFile(f, MDIO_ERROR_BADVAL, msg);
        retval = MDIO_ERROR;
      }
      else {
        nbfixprm[k].prm[j] = i;
      }
    }
  }

  /* connect atom topology to atom params */
  for (k = 0;  k < n_atom;  k++) {
    snprintf(keyname, sizeof(KeyName), "%s", atom[k].type);
    if ((atom[k].prm = adt_lookupTable(table, keyname)) == ADT_ERROR) {
      snprintf(msg, sizeof(msg), "no atom parameters exist for %s", keyname);
      mdio_setErrorMessageFile(f, MDIO_ERROR_BADVAL, msg);
      retval = MDIO_ERROR;
    }
  }

  /* connect bond topology to bond params */
  for (k = 0;  k < n_bond;  k++) {
    if (bond[k].atom[0] < 0 || bond[k].atom[0] >= n_atom
        || bond[k].atom[1] < 0 || bond[k].atom[1] >= n_atom) {
      snprintf(msg, sizeof(msg),
          "bond %d has out-of-range atom number (%d,%d)",
          k+1, bond[k].atom[0]+1, bond[k].atom[1]+1);
      mdio_setErrorMessageFile(f, MDIO_ERROR_BADVAL, msg);
      retval = MDIO_ERROR;
      continue;
    }
    snprintf(keyname, sizeof(KeyName), "%s %s",
        atom[bond[k].atom[0]].type, atom[bond[k].atom[1]].type);
    if ((bond[k].prm = adt_lookupTable(table, keyname)) == ADT_ERROR) {
      snprintf(msg, sizeof(msg), "no bond parameters exist for %s", keyname);
      mdio_setErrorMessageFile(f, MDIO_ERROR_BADVAL, msg);
      retval = MDIO_ERROR;
    }
  }

  /* connect angle topology to angle params */
  for (k = 0;  k < n_angle;  k++) {
    if (angle[k].atom[0] < 0 || angle[k].atom[0] >= n_atom
        || angle[k].atom[1] < 0 || angle[k].atom[1] >= n_atom
        || angle[k].atom[2] < 0 || angle[k].atom[2] >= n_atom) {
      snprintf(msg, sizeof(msg),
          "angle %d has out-of-range atom number (%d,%d,%d)",
          k+1, angle[k].atom[0]+1, angle[k].atom[1]+1, angle[k].atom[2]+1);
      mdio_setErrorMessageFile(f, MDIO_ERROR_BADVAL, msg);
      retval = MDIO_ERROR;
      continue;
    }
    snprintf(keyname, sizeof(KeyName), "%s %s %s",
        atom[angle[k].atom[0]].type, atom[angle[k].atom[1]].type,
        atom[angle[k].atom[2]].type);
    if ((angle[k].prm = adt_lookupTable(table, keyname)) == ADT_ERROR) {
      snprintf(msg, sizeof(msg), "no angle parameters exist for %s", keyname);
      mdio_setErrorMessageFile(f, MDIO_ERROR_BADVAL, msg);
      retval = MDIO_ERROR;
    }
  }

  /* connect dihedral topology to dihedral params */
  tmlist = (TorsMult *) adt_getDataList(&(topo->dihed_mult));
  tmlist_len = adt_getLengthList(&(topo->dihed_mult));
  j = 0;
  for (k = 0;  k < n_dihed;  k++) {
    if (dihed[k].atom[0] < 0 || dihed[k].atom[0] >= n_atom
        || dihed[k].atom[1] < 0 || dihed[k].atom[1] >= n_atom
        || dihed[k].atom[2] < 0 || dihed[k].atom[2] >= n_atom
        || dihed[k].atom[3] < 0 || dihed[k].atom[3] >= n_atom) {
      snprintf(msg, sizeof(msg),
          "dihedral %d has out-of-range atom number (%d,%d,%d,%d)",
          k+1, dihed[k].atom[0]+1, dihed[k].atom[1]+1,
          dihed[k].atom[2]+1, dihed[k].atom[3]+1);
      mdio_setErrorMessageFile(f, MDIO_ERROR_BADVAL, msg);
      retval = MDIO_ERROR;
      continue;
    }
    /*
     * large "if" test expression:  short circuit termination provides
     *   first possible match from lookup table, then "else" section
     *   makes further adjustments based on PSF repeats
     */
    if ((snprintf(keyname, sizeof(KeyName), "d %s %s %s %s",
            atom[dihed[k].atom[0]].type, atom[dihed[k].atom[1]].type,
            atom[dihed[k].atom[2]].type, atom[dihed[k].atom[3]].type),
          (dihed[k].prm = adt_lookupTable(table, keyname)) < 0)
        &&
        /* try wildcard match */
        (snprintf(keyname, sizeof(KeyName), "d X %s %s X",
            atom[dihed[k].atom[1]].type, atom[dihed[k].atom[2]].type),
         (dihed[k].prm = adt_lookupTable(table, keyname)) < 0)) {
      snprintf(msg, sizeof(msg),
          "no parameters exist for dihedral %s %s %s %s",
          atom[dihed[k].atom[0]].type, atom[dihed[k].atom[1]].type,
          atom[dihed[k].atom[2]].type, atom[dihed[k].atom[3]].type);
      mdio_setErrorMessageFile(f, MDIO_ERROR_BADVAL, msg);
      retval = MDIO_ERROR;
    }
    else if (j < tmlist_len && tmlist[j].index == k) {
      /*
       * if PSF repeats more than param file gives, then use max
       * available multiplicity, print out warning, and continue
       */
      if (tmlist[j].mult > dihedprm[dihed[k].prm].mult) {
        snprintf(msg, sizeof(msg),
            "using %d terms for dihedral %d with atoms (%d,%d,%d,%d), "
            "less than multiplicity %d indicated in PSF",
            dihedprm[dihed[k].prm].mult, k,
            dihed[k].atom[0]+1, dihed[k].atom[1]+1,
            dihed[k].atom[2]+1, dihed[k].atom[3]+1,
            tmlist[j].mult);
        mdio_setErrorMessageFile(f, MDIO_ERROR_WARN, msg);
      }
      /*
       * if PSF repeats less than max available multiplicity from PSF, then
       * back step dihedprm index until we match indicated multiplicity
       */
      while (tmlist[j].mult < dihedprm[dihed[k].prm].mult) {
        TEXT("adjust");
        dihed[k].prm--;
      }
      /*
       * we've processed this entry in dihed_mult array,
       * increment the index
       */
      j++;
    }
    else if (prm->filetype == MDIO_PARAM_XPLOR) {
      /*
       * is not in PSF list of repeats, so make it multiplicity 1
       */
      while (dihedprm[dihed[k].prm].mult > 1) {
        dihed[k].prm--;
      }
    }
  }

  /* connect improper topology to improper params */
  tmlist = (TorsMult *) adt_getDataList(&(topo->impr_mult));
  tmlist_len = adt_getLengthList(&(topo->impr_mult));
  j = 0;
  for (k = 0;  k < n_impr;  k++) {
    if (impr[k].atom[0] < 0 || impr[k].atom[0] >= n_atom
        || impr[k].atom[1] < 0 || impr[k].atom[1] >= n_atom
        || impr[k].atom[2] < 0 || impr[k].atom[2] >= n_atom
        || impr[k].atom[3] < 0 || impr[k].atom[3] >= n_atom) {
      snprintf(msg, sizeof(msg),
          "improper %d has out-of-range atom number (%d,%d,%d,%d)",
          k+1, impr[k].atom[0]+1, impr[k].atom[1]+1,
          impr[k].atom[2]+1, impr[k].atom[3]+1);
      mdio_setErrorMessageFile(f, MDIO_ERROR_BADVAL, msg);
      retval = MDIO_ERROR;
      continue;
    }
    /*
     * large "if" test expression:  short circuit termination provides
     *   first possible match from lookup table, then "else" section
     *   makes further adjustments based on PSF repeats
     */
    if ((snprintf(keyname, sizeof(KeyName), "i %s %s %s %s",
            atom[impr[k].atom[0]].type, atom[impr[k].atom[1]].type,
            atom[impr[k].atom[2]].type, atom[impr[k].atom[3]].type),
          (impr[k].prm = adt_lookupTable(table, keyname)) < 0)
        &&
        /* try 1st wildcard match */
        (snprintf(keyname, sizeof(KeyName), "i %s X X %s",
            atom[impr[k].atom[0]].type, atom[impr[k].atom[3]].type),
          (impr[k].prm = adt_lookupTable(table, keyname)) < 0)
        &&
        /* try 2nd wildcard match */
        (snprintf(keyname, sizeof(KeyName), "i X %s %s %s",
            atom[impr[k].atom[1]].type, atom[impr[k].atom[2]].type,
            atom[impr[k].atom[3]].type),
          (impr[k].prm = adt_lookupTable(table, keyname)) < 0)
        &&
        /* try 3rd wildcard match */
        (snprintf(keyname, sizeof(KeyName), "i %s %s %s X",
            atom[impr[k].atom[0]].type, atom[impr[k].atom[1]].type,
            atom[impr[k].atom[2]].type),
          (impr[k].prm = adt_lookupTable(table, keyname)) < 0)
        &&
        /* try 4th wildcard match */
        (snprintf(keyname, sizeof(KeyName), "i X %s %s X",
            atom[impr[k].atom[1]].type, atom[impr[k].atom[2]].type),
          (impr[k].prm = adt_lookupTable(table, keyname)) < 0)
        &&
        /* try 5th wildcard match */
        (snprintf(keyname, sizeof(KeyName), "i X X %s %s",
            atom[impr[k].atom[2]].type, atom[impr[k].atom[3]].type),
          (impr[k].prm = adt_lookupTable(table, keyname)) < 0)
        &&
        /* try 6th wildcard match */
        (snprintf(keyname, sizeof(KeyName), "i %s %s X X",
            atom[impr[k].atom[0]].type, atom[impr[k].atom[1]].type),
          (impr[k].prm = adt_lookupTable(table, keyname)) < 0)) {
      snprintf(msg, sizeof(msg),
          "no parameters exist for improper %s %s %s %s",
          atom[impr[k].atom[0]].type, atom[impr[k].atom[1]].type,
          atom[impr[k].atom[2]].type, atom[impr[k].atom[3]].type);
      mdio_setErrorMessageFile(f, MDIO_ERROR_BADVAL, msg);
      retval = MDIO_ERROR;
    }
    else if (j < tmlist_len && tmlist[j].index == k) {
      /*
       * if PSF repeats more than param file gives, then use max
       * available multiplicity, print out warning, and continue
       */
      if (tmlist[j].mult > imprprm[impr[k].prm].mult) {
        snprintf(msg, sizeof(msg),
            "using %d terms for improper %d with atoms (%d,%d,%d,%d), "
            "less than multiplicity %d indicated in PSF",
            imprprm[impr[k].prm].mult, k,
            impr[k].atom[0]+1, impr[k].atom[1]+1,
            impr[k].atom[2]+1, impr[k].atom[3]+1,
            tmlist[j].mult);
        mdio_setErrorMessageFile(f, MDIO_ERROR_WARN, msg);
      }
      /*
       * if PSF repeats less than max available multiplicity from PSF, then
       * back step imprprm index until we match indicated multiplicity
       */
      while (tmlist[j].mult < imprprm[impr[k].prm].mult) {
        impr[k].prm--;
      }
      /*
       * we've processed this entry in impr_mult array,
       * increment the index
       */
      j++;
    }
    else if (prm->filetype == MDIO_PARAM_XPLOR) {
      /*
       * is not in PSF list of repeats, so make it multiplicity 1
       */
      while (imprprm[impr[k].prm].mult > 1) {
        impr[k].prm--;
      }
    }
  }

#if 0
#ifdef DAVE_DEBUG
  printf("\ndihedprm:\n");
  for (k = 0;  k < n_dihedprm;  k++) {
    printf("%d: mult=%d\n", k, dihedprm[k].mult);
  }
  printf("\ndihed:\n");
  for (k = 0;  k < n_dihed;  k++) {
    printf("%d: prm=%d mult=%d\n", k, dihed[k].prm,
        dihedprm[dihed[k].prm].mult);
  }
#endif
#ifdef DAVE_DEBUG
  tmlist = (TorsMult *) adt_getDataList(&(topo->dihed_mult));
  tmlist_len = adt_getLengthList(&(topo->dihed_mult));
  for (k = 0;  k < tmlist_len;  k++) {
    printf("dihed %d (%d,%d,%d,%d) repeated %d times in PSF\n",
        tmlist[k].index,
        dihed[tmlist[k].index].atom[0]+1, dihed[tmlist[k].index].atom[1]+1,
        dihed[tmlist[k].index].atom[2]+1, dihed[tmlist[k].index].atom[3]+1,
        tmlist[k].mult);
  }
#endif
#endif

  /* cleanup */
  adt_destroyTable(table);
  adt_destroyArray(keylist);
  return retval;
}


/*****************************************************************************
 *
 * Tokenize and parse PSF files
 *
 *****************************************************************************/


/* constructor */
int psf_init(mdio_Topo *topo)
{
  /* buffer array starts with INIT_BUFLEN elements */
  topo->buffer = (char *) malloc(INIT_BUFLEN);
  if (topo->buffer == NULL) {
    ERRMSG("out of memory");
    mdio_setErrorFile(&(topo->file), MDIO_ERROR_NOMEM);
    return MDIO_ERROR;
  }
  topo->bufferlen = INIT_BUFLEN;
  return 0;
}


/* destructor */
void psf_done(mdio_Topo *topo)
{
  /* cleanup buffer array */
  free(topo->buffer);
}


/*
 * called by psf_read() to get the next line
 * returned line is guaranteed to have non-white chars
 * except "" is returned for end-of-file
 * and NULL means an error has occurred
 */
const char *psf_getline(mdio_Topo *topo)
{
  int isblank = 1;
  int len;
  const char *c;

  /* skip over blank lines */
  while (isblank) {
    len = mdio_readLineFile(&(topo->file), &(topo->buffer), &(topo->bufferlen));
    if (len > 0) {
      for (c = topo->buffer;  *c != '\0';  c++) {
        if (!isspace(*c)) {
          isblank = 0;
          break;
        }
      }
    }
    else {
      break;
    }
  }
  return (len >= 0 ? topo->buffer : NULL);
}


/*
 * read PSF file
 * there is no means to keep going in case of an error
 * return 0 on success, -1 on failure
 */
int psf_read(mdio_Topo *topo)
{
  /* sscanf format for token */
#define  FMT  "%15[a-zA-Z]"

  /* number of elements for static arrays */
#define  NELEMS(a)  (sizeof(a) / sizeof((a)[0]))

  /* variable declarations */
  const char *line;
  char token[16] = "";
  const char *fmtsingle[] = {
    "%d",
    "%*d %d",
    "%*d %*d %d",
    "%*d %*d %*d %d",
    "%*d %*d %*d %*d %d",
    "%*d %*d %*d %*d %*d %d",
    "%*d %*d %*d %*d %*d %*d %d",
    "%*d %*d %*d %*d %*d %*d %*d %d"
  };
  const char *fmtpair[] = {
    "%d %d",
    "%*d %*d %d %d",
    "%*d %*d %*d %*d %d %d",
    "%*d %*d %*d %*d %*d %*d %d %d"
  };
  const char *fmttriple[] = {
    "%d %d %d",
    "%*d %*d %*d %d %d %d",
    "%*d %*d %*d %*d %*d %*d %d %d %d"
  };
  const char *fmtquad[] = {
    "%d %d %d %d",
    "%*d %*d %*d %*d %d %d %d %d"
  };
  int natoms, n, i, j, k, a, b, prev, next;
  MD_Atom atom;
  MD_Bond bond;
  MD_Angle angle;
  MD_Tors tors;
  MD_Excl excl;
  MD_Atom *atomlist;
  MD_Bond *bondlist;
  MD_Angle *anglelist;
  MD_Tors *torslist;
  MD_Excl *exclist;

  /* parse PSF keyword */
  if ((line = psf_getline(topo)) == NULL) return MDIO_ERROR;
  sscanf(line, " " FMT, token);
  if (strcasecmp(token, "PSF") != 0) {
    mdio_setErrorMessageFile(&(topo->file), MDIO_ERROR_SYNTAX,
        "failed to find PSF keyword");
    return MDIO_ERROR;
  }

  /* parse NTITLE section */
  if ((line = psf_getline(topo)) == NULL) return MDIO_ERROR;
  sscanf(line, "%d ! " FMT, &n, token);
  if (strcasecmp(token, "NTITLE") != 0) {
    mdio_setErrorMessageFile(&(topo->file), MDIO_ERROR_SYNTAX,
        "failed to find NTITLE keyword");
    return MDIO_ERROR;
  }
  /* find all REMARKS lines */
  for (i = 0;  i < n;  i++) {
    if ((line = psf_getline(topo)) == NULL) return MDIO_ERROR;
    if (sscanf(line, " " FMT, token) < 1 
        || (strcasecmp(token, "REMARKS") != 0
          && strcasecmp(token, "REMARK") != 0)) {
      mdio_setErrorMessageFile(&(topo->file), MDIO_ERROR_SYNTAX,
          "expected REMARKS line");
      return MDIO_ERROR;
    }
  }

  /* parse NATOM section */
  if ((line = psf_getline(topo)) == NULL) return MDIO_ERROR;
  sscanf(line, "%d ! " FMT, &natoms, token);
  if (strcasecmp(token, "NATOM") != 0) {
    mdio_setErrorMessageFile(&(topo->file), MDIO_ERROR_SYNTAX,
        "failed to find NATOM keyword");
    return MDIO_ERROR;
  }
  /* set atom array size */
  if (adt_resizeArray(&(topo->atom), natoms)) {
    ERRMSG("cannot resize atom array");
    mdio_setErrorFile(&(topo->file), MDIO_ERROR_NOMEM);
    return MDIO_ERROR;
  }
  /* init atom list */
  atomlist = (MD_Atom *) adt_getDataArray(&(topo->atom));
  /* get all atom records */
  for (i = 0;  i < natoms;  i++) {
    if ((line = psf_getline(topo)) == NULL) return MDIO_ERROR;
    /*
     * format of atom record:
     * # molecule_name segment residue atom_name atom_type charge mass
     */
    if (sscanf(line, "%d %*4s %*4s %*4s %4s %4s %lf %lf",
        &n, atom.name, atom.type, &atom.q, &atom.m) < 5) {
      char s[60];
      snprintf(s, sizeof(s), "unable to parse atom record %d", i+1);
      mdio_setErrorMessageFile(&(topo->file), MDIO_ERROR_SYNTAX, s);
      return MDIO_ERROR;
    }
    /* check validity */
    if (n != i+1) {
      char s[60];
      snprintf(s, sizeof(s), "atom record %d is mis-numbered as %d", i+1, n);
      mdio_setErrorMessageFile(&(topo->file), MDIO_ERROR_SYNTAX, s);
      return MDIO_ERROR;
    }
    atom.prm = -1;
#if 0
    atom.notused = 0;
#endif
    atomlist[i] = atom;
  }

  /* parse NBOND section */
  if ((line = psf_getline(topo)) == NULL) return MDIO_ERROR;
  sscanf(line, "%d ! " FMT, &n, token);
  if (strcasecmp(token, "NBOND") != 0) {
    mdio_setErrorMessageFile(&(topo->file), MDIO_ERROR_SYNTAX,
       	"failed to find NBOND keyword");
    return MDIO_ERROR;
  }
  /* set bond array size */
  if (adt_resizeArray(&(topo->bond), n)) {
    ERRMSG("cannot resize bond array");
    mdio_setErrorFile(&(topo->file), MDIO_ERROR_NOMEM);
    return MDIO_ERROR;
  }
  /* init bond list */
  bondlist = (MD_Bond *) adt_getDataArray(&(topo->bond));
  /* get all bond pairs */
  for (i = 0;  i < n; ) {
    if ((line = psf_getline(topo)) == NULL) return MDIO_ERROR;
    /*
     * format of bonds:  8 columns of ints, parse as pairs
     */
    for (j = 0;  j < NELEMS(fmtpair) && i < n;  j++, i++) {
      if (sscanf(line, fmtpair[j], &bond.atom[0], &bond.atom[1]) < 2) {
        char s[60];
       	snprintf(s, sizeof(s), "unable to parse bond %d", i+1);
       	mdio_setErrorMessageFile(&(topo->file), MDIO_ERROR_SYNTAX, s);
       	return MDIO_ERROR;
      }
      /* check validity */
      if (bond.atom[0] < 1 || bond.atom[0] > natoms
          || bond.atom[1] < 1 || bond.atom[1] > natoms
          || bond.atom[0] == bond.atom[1]) {
        char s[60];
       	snprintf(s, sizeof(s), "bond %d has illegal atom number (%d,%d)",
            i+1, bond.atom[0], bond.atom[1]);
       	mdio_setErrorMessageFile(&(topo->file), MDIO_ERROR_SYNTAX, s);
       	return MDIO_ERROR;
      }
      bond.atom[0]--;   /* adjust for C-based indexing */
      bond.atom[1]--;
      bond.prm = -1;  /* uninitialized */
      bondlist[i] = bond;
    }
  }

  /* parse NTHETA section */
  if ((line = psf_getline(topo)) == NULL) return MDIO_ERROR;
  sscanf(line, "%d ! " FMT, &n, token);
  if (strcasecmp(token, "NTHETA") != 0) {
    mdio_setErrorMessageFile(&(topo->file), MDIO_ERROR_SYNTAX,
        "failed to find NTHETA keyword");
    return MDIO_ERROR;
  }
  /* set angle array size */
  if (adt_resizeArray(&(topo->angle), n)) {
    ERRMSG("cannot resize angle array");
    mdio_setErrorFile(&(topo->file), MDIO_ERROR_NOMEM);
    return MDIO_ERROR;
  }
  /* init angle list */
  anglelist = (MD_Angle *) adt_getDataArray(&(topo->angle));
  /* get all angle triples */
  for (i = 0;  i < n; ) {
    if ((line = psf_getline(topo)) == NULL) return MDIO_ERROR;
    /*
     * format of angles:  9 columns of ints, parse as triples
     */
    for (j = 0;  j < NELEMS(fmttriple) && i < n;  j++, i++) {
      if (sscanf(line, fmttriple[j], &angle.atom[0], &angle.atom[1],
            &angle.atom[2]) < 3) {
        char s[60];
       	snprintf(s, sizeof(s), "unable to parse angle %d", i+1);
       	mdio_setErrorMessageFile(&(topo->file), MDIO_ERROR_SYNTAX, s);
       	return MDIO_ERROR;
      }
      /* check validity */
      if (angle.atom[0] < 1 || angle.atom[0] > natoms
          || angle.atom[1] < 1 || angle.atom[1] > natoms
          || angle.atom[2] < 1 || angle.atom[2] > natoms
          || angle.atom[0] == angle.atom[1]
          || angle.atom[0] == angle.atom[2]
          || angle.atom[1] == angle.atom[2]) {
        char s[60];
       	snprintf(s, sizeof(s), "angle %d has illegal atom number (%d,%d,%d)",
            i+1, angle.atom[0], angle.atom[1], angle.atom[2]);
       	mdio_setErrorMessageFile(&(topo->file), MDIO_ERROR_SYNTAX, s);
       	return MDIO_ERROR;
      }
      angle.atom[0]--;   /* adjust for C-based indexing */
      angle.atom[1]--;
      angle.atom[2]--;
      angle.prm = -1;  /* uninitialized */
      anglelist[i] = angle;
    }
  }

  /* parse NPHI section */
  if ((line = psf_getline(topo)) == NULL) return MDIO_ERROR;
  sscanf(line, "%d ! " FMT, &n, token);
  if (strcasecmp(token, "NPHI") != 0) {
    mdio_setErrorMessageFile(&(topo->file), MDIO_ERROR_SYNTAX,
        "failed to find NPHI keyword");
    return MDIO_ERROR;
  }
  /* set dihedral array size */
  if (adt_resizeArray(&(topo->dihed), n)) {
    ERRMSG("cannot resize dihedral array");
    mdio_setErrorFile(&(topo->file), MDIO_ERROR_NOMEM);
    return MDIO_ERROR;
  }
  /* init dihedral list */
  torslist = (MD_Tors *) adt_getDataArray(&(topo->dihed));
  /* get all dihedral quadruples */
  for (i = 0, k = 0;  i < n; ) {
    if ((line = psf_getline(topo)) == NULL) return MDIO_ERROR;
    /*
     * format of dihedrals:  8 columns of ints, parse as quadruples
     */
    for (j = 0;  j < NELEMS(fmtquad) && i < n;  j++, i++) {
      if (sscanf(line, fmtquad[j], &tors.atom[0], &tors.atom[1],
            &tors.atom[2], &tors.atom[3]) < 4) {
        char s[60];
       	snprintf(s, sizeof(s), "\nunable to parse dihedral %d", i+1);
       	mdio_setErrorMessageFile(&(topo->file), MDIO_ERROR_SYNTAX, s);
       	return MDIO_ERROR;
      }
      /* check validity */
      if (tors.atom[0] < 1 || tors.atom[0] > natoms
          || tors.atom[1] < 1 || tors.atom[1] > natoms
          || tors.atom[2] < 1 || tors.atom[2] > natoms
          || tors.atom[3] < 1 || tors.atom[3] > natoms
          || tors.atom[0] == tors.atom[1]
          || tors.atom[0] == tors.atom[2]
          || tors.atom[0] == tors.atom[3]
          || tors.atom[1] == tors.atom[2]
          || tors.atom[1] == tors.atom[3]
          || tors.atom[2] == tors.atom[3]) {
        char s[80];
       	snprintf(s, sizeof(s),
            "dihedral %d has illegal atom number (%d,%d,%d,%d)",
            i+1, tors.atom[0], tors.atom[1], tors.atom[2], tors.atom[3]);
       	mdio_setErrorMessageFile(&(topo->file), MDIO_ERROR_SYNTAX, s);
       	return MDIO_ERROR;
      }
      tors.atom[0]--;   /* adjust for C-based indexing */
      tors.atom[1]--;
      tors.atom[2]--;
      tors.atom[3]--;
      tors.prm = -1;  /* uninitialized */

      /*
       * if this is same as previous dihedral, then don't re-list it;
       * X-Plor PSF will repeat listed atoms if they are involved in
       * multiple terms
       */
      if (k == 0
          || tors.atom[0] != torslist[k-1].atom[0]
          || tors.atom[1] != torslist[k-1].atom[1]
          || tors.atom[2] != torslist[k-1].atom[2]
          || tors.atom[3] != torslist[k-1].atom[3]) {
        torslist[k] = tors;
        k++;
      }
      else {
        /*
         * otherwise keep track of its multiplicity indicated by PSF
         * repeats; any repeats are expected to be listed consecutively
         */
        TorsMult *tmlist = adt_getDataList(&(topo->dihed_mult));
        int len = adt_getLengthList(&(topo->dihed_mult));
        TorsMult tm;
        tm.index = k-1;
        tm.mult = 2;
        /*
         * if already listed simply add to the multiplicity count,
         * otherwise we need to append it to the list
         */
        if (len > 0 && tmlist[len-1].index == tm.index) {
          tmlist[len-1].mult++;
        }
        else if (adt_appendList(&(topo->dihed_mult), &tm)) {
          ERRMSG("cannot append to dihedral multiplicity list");
          mdio_setErrorFile(&(topo->file), MDIO_ERROR_NOMEM);
          return MDIO_ERROR;
        }
      }
    }
  }
  /* reset dihedral array size if there are fewer than n elements */
  if (k < n && adt_resizeArray(&(topo->dihed), k)) {
    ERRMSG("cannot resize dihedral array");
    mdio_setErrorFile(&(topo->file), MDIO_ERROR_NOMEM);
    return MDIO_ERROR;
  }

  /* parse NIMPHI section */
  if ((line = psf_getline(topo)) == NULL) return MDIO_ERROR;
  sscanf(line, "%d ! " FMT, &n, token);
  if (strcasecmp(token, "NIMPHI") != 0) {
    mdio_setErrorMessageFile(&(topo->file), MDIO_ERROR_SYNTAX,
        "failed to find NIMPHI keyword");
    return MDIO_ERROR;
  }
  /* set improper array size */
  if (adt_resizeArray(&(topo->impr), n)) {
    ERRMSG("cannot resize improper array");
    mdio_setErrorFile(&(topo->file), MDIO_ERROR_NOMEM);
    return MDIO_ERROR;
  }
  /* init improper list */
  torslist = (MD_Tors *) adt_getDataArray(&(topo->impr));
  /* get all improper quadruples */
  for (i = 0, k = 0;  i < n; ) {
    if ((line = psf_getline(topo)) == NULL) return MDIO_ERROR;
    /*
     * format of impropers:  8 columns of ints, parse as quadruples
     */
    for (j = 0;  j < NELEMS(fmtquad) && i < n;  j++, i++) {
      if (sscanf(line, fmtquad[j], &tors.atom[0], &tors.atom[1],
            &tors.atom[2], &tors.atom[3]) < 4) {
        char s[60];
       	snprintf(s, sizeof(s), "unable to parse improper %d", i+1);
       	mdio_setErrorMessageFile(&(topo->file), MDIO_ERROR_SYNTAX, s);
       	return MDIO_ERROR;
      }
      /* check validity */
      if (tors.atom[0] < 1 || tors.atom[0] > natoms
          || tors.atom[1] < 1 || tors.atom[1] > natoms
          || tors.atom[2] < 1 || tors.atom[2] > natoms
          || tors.atom[3] < 1 || tors.atom[3] > natoms
          || tors.atom[0] == tors.atom[1]
          || tors.atom[0] == tors.atom[2]
          || tors.atom[0] == tors.atom[3]
          || tors.atom[1] == tors.atom[2]
          || tors.atom[1] == tors.atom[3]
          || tors.atom[2] == tors.atom[3]) {
        char s[80];
       	snprintf(s, sizeof(s),
            "improper %d has illegal atom number (%d,%d,%d,%d)",
            i+1, tors.atom[0], tors.atom[1], tors.atom[2], tors.atom[3]);
       	mdio_setErrorMessageFile(&(topo->file), MDIO_ERROR_SYNTAX, s);
       	return MDIO_ERROR;
      }
      tors.atom[0]--;   /* adjust for C-based indexing */
      tors.atom[1]--;
      tors.atom[2]--;
      tors.atom[3]--;
      tors.prm = -1;  /* uninitialized */

      /*
       * if this is same as previous improper, then don't re-list it;
       * X-Plor PSF will repeat listed atoms if they are involved in
       * multiple terms
       */
      if (k == 0
          || tors.atom[0] != torslist[k-1].atom[0]
          || tors.atom[1] != torslist[k-1].atom[1]
          || tors.atom[2] != torslist[k-1].atom[2]
          || tors.atom[3] != torslist[k-1].atom[3]) {
        torslist[k] = tors;
        k++;
      }
      else {
        /*
         * otherwise keep track of its multiplicity indicated by PSF
         * repeats; any repeats are expected to be listed consecutively
         */
        TorsMult *tmlist = adt_getDataList(&(topo->impr_mult));
        int len = adt_getLengthList(&(topo->impr_mult));
        TorsMult tm;
        tm.index = k - 1;
        tm.mult = 2;
        /*
         * if already listed simply add to the multiplicity count,
         * otherwise we need to append it to the list
         */
        if (len > 0 && tmlist[len-1].index == tm.index) {
          tmlist[len-1].mult++;
        }
        else if (adt_appendList(&(topo->impr_mult), &tm)) {
          ERRMSG("cannot append to improper multiplicity list");
          mdio_setErrorFile(&(topo->file), MDIO_ERROR_NOMEM);
          return MDIO_ERROR;
        }
      }
    }
  }
  /* reset improper array size if there are fewer than n elements */
  if (k < n && adt_resizeArray(&(topo->impr), k)) {
    ERRMSG("cannot resize improper array");
    mdio_setErrorFile(&(topo->file), MDIO_ERROR_NOMEM);
    return MDIO_ERROR;
  }

  /* parse NDON section - read in but not used */
  if ((line = psf_getline(topo)) == NULL) return MDIO_ERROR;
  sscanf(line, "%d ! " FMT, &n, token);
  if (strcasecmp(token, "NDON") != 0) {
    mdio_setErrorMessageFile(&(topo->file), MDIO_ERROR_SYNTAX,
        "failed to find NDON keyword");
    return MDIO_ERROR;
  }
  /* get all donors */
  for (i = 0;  i < n; ) {
    if ((line = psf_getline(topo)) == NULL) return MDIO_ERROR;
    /*
     * format of donors:  8 columns of ints, parse as pairs
     */
    for (j = 0;  j < NELEMS(fmtpair) && i < n;  j++, i++) {
      if (sscanf(line, fmtpair[j], &a, &b) < 2) {
        char s[60];
       	snprintf(s, sizeof(s), "unable to parse donor pair %d", i+1);
       	mdio_setErrorMessageFile(&(topo->file), MDIO_ERROR_SYNTAX, s);
       	return MDIO_ERROR;
      }
      /* check validity */
      if (a < 0 || a > natoms || b < 0 || b > natoms || a == b) {
        char s[80];
       	snprintf(s, sizeof(s),
            "donor pair %d has illegal atom number (%d,%d)", i+1, a, b);
       	mdio_setErrorMessageFile(&(topo->file), MDIO_ERROR_SYNTAX, s);
       	return MDIO_ERROR;
      }
    }
  }

  /* parse NACC section - read in but not used */
  if ((line = psf_getline(topo)) == NULL) return MDIO_ERROR;
  sscanf(line, "%d ! " FMT, &n, token);
  if (strcasecmp(token, "NACC") != 0) {
    mdio_setErrorMessageFile(&(topo->file), MDIO_ERROR_SYNTAX,
        "failed to find NACC keyword");
    return MDIO_ERROR;
  }
  /* get all acceptors */
  for (i = 0;  i < n; ) {
    if ((line = psf_getline(topo)) == NULL) return MDIO_ERROR;
    /*
     * format of acceptors:  8 columns of ints, parse as pairs
     */
    for (j = 0;  j < NELEMS(fmtpair) && i < n;  j++, i++) {
      if (sscanf(line, fmtpair[j], &a, &b) < 2) {
        char s[60];
       	snprintf(s, sizeof(s), "unable to parse acceptor pair %d", i+1);
       	mdio_setErrorMessageFile(&(topo->file), MDIO_ERROR_SYNTAX, s);
       	return MDIO_ERROR;
      }
      /* check validity */
      if (a < 0 || a > natoms || b < 0 || b > natoms || a == b) {
        char s[80];
       	snprintf(s, sizeof(s),
            "acceptor pair %d has illegal atom number (%d,%d)", i+1, a, b);
       	mdio_setErrorMessageFile(&(topo->file), MDIO_ERROR_SYNTAX, s);
       	return MDIO_ERROR;
      }
    }
  }

  /* parse NNB section */
  if ((line = psf_getline(topo)) == NULL) return MDIO_ERROR;
  sscanf(line, "%d ! " FMT, &n, token);
  if (strcasecmp(token, "NNB") != 0) {
    mdio_setErrorMessageFile(&(topo->file), MDIO_ERROR_SYNTAX,
        "failed to find NNB keyword");
    return MDIO_ERROR;
  }
  /* set exclusion array size */
  if (adt_resizeArray(&(topo->excl), n)) {
    ERRMSG("cannot resize excl array");
    mdio_setErrorFile(&(topo->file), MDIO_ERROR_NOMEM);
    return MDIO_ERROR;
  }
  /* init exclusion list */
  exclist = (MD_Excl *) adt_getDataArray(&(topo->excl));
  /*
   * Rather than storing exclusions as pairs, they are stored in
   * a sparse-matrix-index format.  The first nexcl ints are the
   * second atom index of the exclusion pairs.  The next natoms
   * ints are a list of non-decreasing indices into the exclusion
   * list: the difference of a previous entry from the next entry
   * is the number of exclusions that that atom is to be included,
   * indexed from the previous entry to the next entry minus 1
   * (using zero-based indexing).
   *
   * Here is an example (from NAMD 1.5 source code comments):
   *
   *   3 !NNB
   *   3 4 5
   *   0 1 3 3 3
   *
   * This is a 5-atom system with 3 exclusions.  The exclusion list
   * generated is (2,3) (3,4) (3,5).  (Recall that since our data
   * arrays are C-based rather than Fortran-based, our exclusion
   * list pairs will actually be (1,2) (2,3) (2,4).)
   *
   * Note that although this seems like a strange way for storing
   * exclusion pairs, this will save space whenever the number of
   * explicit exclusions is greater than the number of atoms.
   */
  /* get exclusions (2nd atom of pair) */
  for (i = 0;  i < n; ) {
    if ((line = psf_getline(topo)) == NULL) return MDIO_ERROR;
    /*
     * format of exclusion list:  8 columns of ints, parse as singletons
     */
    for (j = 0;  j < NELEMS(fmtsingle) && i < n;  j++, i++) {
      if (sscanf(line, fmtsingle[j], &excl.atom[1]) < 1) {
        char s[60];
       	snprintf(s, sizeof(s), "unable to parse exclusion %d", i+1);
       	mdio_setErrorMessageFile(&(topo->file), MDIO_ERROR_SYNTAX, s);
       	return MDIO_ERROR;
      }
      /* check validity */
      if (excl.atom[1] < 1 || excl.atom[1] > natoms) {
        char s[80];
       	snprintf(s, sizeof(s),
            "exclusion %d has illegal atom value %d", i+1, excl.atom[1]);
       	mdio_setErrorMessageFile(&(topo->file), MDIO_ERROR_SYNTAX, s);
       	return MDIO_ERROR;
      }
      excl.atom[1]--;     /* adjust for C-based indexing */
      excl.atom[0] = -1;  /* leave unintialized for now */
      exclist[i] = excl;
    }
  }
  /* next read in list of indices into exclusion array */
  for (prev = 0, i = 0;  i < natoms; ) {
    if ((line = psf_getline(topo)) == NULL) return MDIO_ERROR;
    /*
     * format of index list:  8 columns of ints, parse as singletons
     */
    for (j = 0;  j < NELEMS(fmtsingle) && i < natoms;  j++, i++) {
      if (sscanf(line, fmtsingle[j], &next) < 1) {
        char s[60];
       	snprintf(s, sizeof(s), "unable to parse exclusion list index %d", i+1);
       	mdio_setErrorMessageFile(&(topo->file), MDIO_ERROR_SYNTAX, s);
       	return MDIO_ERROR;
      }
      /* check validity */
      if (next < prev || next > n) {
        char s[80];
       	snprintf(s, sizeof(s),
            "exclusion list index %d has illegal value %d", i+1, next);
       	mdio_setErrorMessageFile(&(topo->file), MDIO_ERROR_SYNTAX, s);
       	return MDIO_ERROR;
      }
      for (k = prev;  k < next;  k++) {
        exclist[k].atom[0] = i;
        /* check validity */
        if (i == exclist[k].atom[1]) {
	  char s[80];
	  snprintf(s, sizeof(s),
              "exclusion %d has illegal value (%d,%d)",
              k+1, exclist[k].atom[0]+1, exclist[k].atom[1]+1);
	  mdio_setErrorMessageFile(&(topo->file), MDIO_ERROR_SYNTAX, s);
	  return MDIO_ERROR;
        }
      }
      prev = next;
    }
  }
  /* check validity */
  if (next != n) {
    char s[80];
    snprintf(s, sizeof(s),
       	"only %d exclusions out of %d have been initialized", next, n);
    mdio_setErrorMessageFile(&(topo->file), MDIO_ERROR_SYNTAX, s);
    return MDIO_ERROR;
  }

  /* skip NGRP section */
  return 0;
}
