/*
 * Copyright (C) 2004-2005 by David J. Hardy.  All rights reserved.
 *
 * file.c
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "mdio/file.h"
#include "debug/debug.h"


#define MINLEN  100


#define NELEMS(x)  (sizeof(x)/sizeof(x[0]))


/*
 * error messages
 * must be kept in same order as error enum const array in file.h
 */
static const char *ErrorMessage[] = {
  "",
  "warning",
  "illegal value",
  "cannot allocate memory",
  "cannot open file",
  "cannot close file",
  "read error for file",
  "write error for file",
  "seek error for file",
  "syntax error in file",
  "unexpected end-of-file for file",
};


/*
 * default error handler
 * prints error message to stderr and returns error number
 */
static int default_errhandler(mdio_File *p)
{
  ASSERT(p != NULL);
  fprintf(stderr, "MDIO ERROR: %s\n", mdio_getErrorMessageFile(p));
  return p->errnum;
}


mdio_File *mdio_createFile(void)
{
  mdio_File *p;
  p = (mdio_File *) malloc(sizeof(mdio_File));
  if (p == NULL) {
    ERRMSG("out of memory");
    return NULL;
  }
  if (mdio_initializeFile(p)) {
    free(p);
    return NULL;
  }
  return p;
}


int mdio_initializeFile(mdio_File *p)
{
  ASSERT(NELEMS(ErrorMessage) == MDIO_LENGTH_ERRORLIST);
  ASSERT(p != NULL);
  memset(p, 0, sizeof(mdio_File));
  p->errhandler = default_errhandler;
  return 0;
}


void mdio_destroyFile(mdio_File *p)
{
  ASSERT(p != NULL);
  mdio_cleanupFile(p);
  free(p);
}


void mdio_cleanupFile(mdio_File *p)
{
  ASSERT(p != NULL);
}


int mdio_openFile(mdio_File *p, const char *name, int filetype)
{
  const char *filetype_str;
  ASSERT(p != NULL);
  ASSERT(name != NULL);
  switch (filetype) {
    case (MDIO_FILE_TEXT | MDIO_FILE_READ):
      filetype_str = "r";
      break;
    case (MDIO_FILE_TEXT | MDIO_FILE_WRITE):
      filetype_str = "w";
      break;
    case (MDIO_FILE_BINARY | MDIO_FILE_READ):
      filetype_str = "rb";
      break;
    case (MDIO_FILE_BINARY | MDIO_FILE_WRITE):
      filetype_str = "wb";
      break;
    default:
      BUG("incorrect \"filetype\" specified");
  }
  /* reset counters and status indicator */
  p->bytenum = 0;
  p->linenum = 0;
  p->status = 0;
  p->name = name;
  p->file = fopen(name, filetype_str);
  if (p->file == NULL) {
    mdio_setErrorFile(p, MDIO_ERROR_OPEN);
    return MDIO_ERROR;
  }
  p->filetype = filetype;
  /* reset error state */
  mdio_resetErrorFile(p);
  return 0;
}


int mdio_closeFile(mdio_File *p)
{
  ASSERT(p != NULL);
  if (fclose(p->file)) {
    mdio_setErrorFile(p, MDIO_ERROR_CLOSE);
    return MDIO_ERROR;
  }
  return 0;
}


int mdio_readLineFile(mdio_File *p, char **pbuf, int *pbuflen)
{
  char *s, *tmpbuf;
  int slen, total, isdone = 0;

  ASSERT(p != NULL);
  ASSERT(pbuf != NULL);
  ASSERT(pbuflen != NULL);
  ASSERT((*pbuflen) >= 0);
  ASSERT(p->filetype == (MDIO_FILE_TEXT | MDIO_FILE_READ));

  /* make sure char buffer has minimum length */
  if ((*pbuflen) < MINLEN) {
    tmpbuf = (char *) realloc((*pbuf), MINLEN);
    if (tmpbuf == NULL) {
      ERRMSG("out of memory");
      mdio_setErrorFile(p, MDIO_ERROR_NOMEM);
      return MDIO_ERROR;
    }
    *pbuf = tmpbuf;
    *pbuflen = MINLEN;
  }

  /* try to read line into buffer */
  s = fgets(*pbuf, *pbuflen, p->file);

  /* check for end-of-file or error */
  if (s == NULL) {
    if (feof(p->file)) {
      (*pbuf)[0] = '\0';
      p->status = MDIO_EOF;
      return 0;
    }
    mdio_setErrorFile(p, MDIO_ERROR_READ);
    return MDIO_ERROR;
  }

  /* see if entire line fits within buffer */
  slen = strlen(s);
  if (slen < (*pbuflen)-1 || s[(*pbuflen)-2] == '\n') {
    isdone = 1;
  }

  total = slen;
  while ( ! isdone ) {
    /* otherwise double length of buffer */
    tmpbuf = (char *) realloc(*pbuf, 2*(*pbuflen));
    if (tmpbuf == NULL) {
      ERRMSG("out of memory");
      mdio_setErrorFile(p, MDIO_ERROR_NOMEM);
      return MDIO_ERROR;
    }
    *pbuf = tmpbuf;

    /* try to read the remainder of line into buffer */
    s = fgets((*pbuf)+(*pbuflen)-1, (*pbuflen)+1, p->file);

    /* check error */
    if (s == NULL) {
      /* update buffer length before exiting */
      (*pbuflen) *= 2;
      mdio_setErrorFile(p, MDIO_ERROR_READ);
      return MDIO_ERROR;
    }

    /* see if entire line fits within (resized) buffer */
    slen = strlen(s);
    total += slen;
    isdone = (slen < (*pbuflen) || s[(*pbuflen)-1] == '\n');

    /* update buffer length */
    (*pbuflen) *= 2;
  }

  ASSERT(strlen(*pbuf) == total);
  ASSERT((*pbuf)[total-1] == '\n');
  p->linenum++;
  p->bytenum += total;
  return total;
}


int mdio_readTextFile(mdio_File *p, char *buf, int buflen)
{
  char *s;
  int slen;

  ASSERT(p != NULL);
  ASSERT(buf != NULL);
  ASSERT(buflen > 1);
  ASSERT(p->filetype == (MDIO_FILE_TEXT | MDIO_FILE_READ));
  s = fgets(buf, buflen, p->file);
  if (s == NULL) {
    if (feof(p->file)) {
      buf[0] = '\0';
      p->status = MDIO_EOF;
      return 0;
    }
    mdio_setErrorFile(p, MDIO_ERROR_READ);
    return MDIO_ERROR;
  }
  slen = strlen(s);
  ASSERT(slen > 0);
  ASSERT(slen < buflen);
  if (s[slen - 1] == '\n') {
    p->linenum++;
  }
  p->bytenum += slen;
  return slen;
}


int mdio_writeTextFile(mdio_File *p, const char *buf)
{
  int n;
  const char *s;

  ASSERT(p != NULL);
  ASSERT(buf != NULL);
  ASSERT(p->filetype == (MDIO_FILE_TEXT | MDIO_FILE_WRITE));
  if (fputs(buf, p->file) == EOF) {
    mdio_setErrorFile(p, MDIO_ERROR_WRITE);
    return MDIO_ERROR;
  }
  n = strlen(buf);
  p->bytenum += n;
  /* count number of lines written to file */
  for (s = buf;  (s = strchr(s, '\n')) != NULL;  p->linenum++, s++) ;
  return n;
}


int mdio_readBinaryFile(mdio_File *p, void *buffer, int elemsz, int n)
{
  int rn;

  ASSERT(p != NULL);
  ASSERT(p->filetype == (MDIO_FILE_BINARY | MDIO_FILE_READ));
  rn = fread(buffer, elemsz, n, p->file);
  if (rn < n) {
    if (feof(p->file)) {
      p->status = MDIO_EOF;
    }
    else {
      mdio_setErrorFile(p, MDIO_ERROR_READ);
    }
  }
  p->bytenum += rn * elemsz;
  return rn;
}


int mdio_writeBinaryFile(mdio_File *p, const void *buffer, int elemsz, int n)
{
  int rn;
  ASSERT(p != NULL);
  ASSERT(p->filetype == (MDIO_FILE_BINARY | MDIO_FILE_WRITE));
  rn = fwrite(buffer, elemsz, n, p->file);
  if (rn < n) {
    mdio_setErrorFile(p, MDIO_ERROR_WRITE);
  }
  p->bytenum += rn * elemsz;
  return rn;
}


int mdio_seekFile(mdio_File *p, long offset, int whence)
{
  ASSERT(p != NULL);
  ASSERT(p->filetype & MDIO_FILE_BINARY);
  if (fseek(p->file, offset, whence)
      || (p->bytenum = (int) ftell(p->file)) == -1) {
    mdio_setErrorFile(p, MDIO_ERROR_SEEK);
    return MDIO_ERROR;
  }
  return 0;
}


int mdio_isEndOfFile(mdio_File *p)
{
  ASSERT(p != NULL);
  return (p->status == MDIO_EOF);
}


int mdio_setErrorFile(mdio_File *p, int errnum)
{
  return mdio_setErrorMessageFile(p, errnum, "");
}


int mdio_setErrorMessageFile(mdio_File *p, int errnum, const char *msg)
{
  char *s;
  int slen, n;
  int ismsg = 0;

  ASSERT(p != NULL);
  ASSERT(msg != NULL);
  if (errnum < MDIO_ERROR_NONE || errnum >= MDIO_LENGTH_ERRORLIST) {
    BUG("invalid error number");
  }
  p->errnum = errnum;
  s = p->errmsg;
  slen = sizeof(p->errmsg);

  /* check for nonempty message string */
  if (strcmp(msg, "") != 0) {
    ismsg = 1;
  }

  /* obtain error message buffer space, write string */
  s = (char *) (p->errmsg);
  switch (errnum) {
    case MDIO_ERROR_NONE:
      if (ismsg) {
        /* note that the ERROR_NONE string is empty */
        n = snprintf(s, slen, "%s", msg);
      }
      else {
        n = snprintf(s, slen, "%s", ErrorMessage[errnum]);
      }
      break;
    case MDIO_ERROR_WARN:
    case MDIO_ERROR_BADVAL:
    case MDIO_ERROR_NOMEM: 
      if (ismsg) {
        n = snprintf(s, slen, "%s: %s", ErrorMessage[errnum], msg);
      }
      else {
        n = snprintf(s, slen, "%s", ErrorMessage[errnum]);
      }
      break;
    case MDIO_ERROR_OPEN:
    case MDIO_ERROR_CLOSE:
    case MDIO_ERROR_UNXEOF:
      if (ismsg) {
        n = snprintf(s, slen, "%s \"%s\": %s",
            ErrorMessage[errnum], p->name, msg);
      }
      else {
        n = snprintf(s, slen, "%s \"%s\"", ErrorMessage[errnum], p->name);
      }
      break;
    case MDIO_ERROR_READ:
    case MDIO_ERROR_WRITE:
    case MDIO_ERROR_SEEK:
    case MDIO_ERROR_SYNTAX:
      if (ismsg) {
        if (p->filetype & MDIO_FILE_TEXT) {
          n = snprintf(s, slen, "%s \"%s\", line number %d: %s",
              ErrorMessage[errnum], p->name, p->linenum, msg);
        }
        else {
          n = snprintf(s, slen, "%s \"%s\", byte number %d: %s",
              ErrorMessage[errnum], p->name, p->bytenum, msg);
        }
      }
      else {
        if (p->filetype & MDIO_FILE_TEXT) {
          n = snprintf(s, slen, "%s \"%s\", line number %d",
              ErrorMessage[errnum], p->name, p->linenum);
        }
        else {
          n = snprintf(s, slen, "%s \"%s\", byte number %d",
              ErrorMessage[errnum], p->name, p->bytenum);
        }
      }
      break;
    default:
      BUG("illegal error number value");
  }

  /* make sure that string is nil-terminated */
  s[slen-1] = '\0';
  if ( n < 0) {
    s[0] = '\0';
  }

  /* invoke error handler and return its value */
  if (p->errhandler != NULL) return p->errhandler(p);
  else return errnum;
}


void mdio_setErrorHandlerFile(mdio_File *p, int (*errhandler)(mdio_File *))
{
  ASSERT(p != NULL);
  p->errhandler = errhandler;
}


const char *mdio_getNameFile(mdio_File *p)
{
  ASSERT(p != NULL);
  return p->name;
}


int mdio_getTypeFile(mdio_File *p)
{
  ASSERT(p != NULL);
  return p->filetype;
}


int mdio_getErrorFile(mdio_File *p)
{
  ASSERT(p != NULL);
  return p->errnum;
}


int mdio_getBytenumFile(mdio_File *p)
{
  ASSERT(p != NULL);
  return p->bytenum;
}


int mdio_getLinenumFile(mdio_File *p)
{
  ASSERT(p != NULL);
  return p->linenum;
}


const char *mdio_getErrorMessageFile(mdio_File *p)
{
  ASSERT(p != NULL);
  return (const char *) (p->errmsg);
}


void mdio_resetErrorFile(mdio_File *p)
{
  ASSERT(p != NULL);
  clearerr(p->file);
  p->errnum = MDIO_ERROR_NONE;
  strcpy(p->errmsg, ErrorMessage[MDIO_ERROR_NONE]);
}
