/*
 * Copyright (C) 2004-2005 by David J. Hardy.  All rights reserved.
 */

/**@file    config.h
 * @brief   Configuration file reader.
 * @author  David J. Hardy
 * @date    2003-2005
 *
 * The @c mdio_Config class reads one or more NAMD-style configuration
 * files (@c keyword = @c value), compiling the results in an array of
 * @c mdio_ConfigData structures.  This array presents the results as
 * separate @c keyword and @c value strings, where it is up to the user
 * to further parse the @c value strings for any expected numeric values.
 *
 * The configuration file format is a text file that contains:
 * @verbatim
     # this is a comment
   keyword1=value1  # end-of-line comment
  
   keyword2  value2
   @endverbatim
 * where the comments are stripped during parsing and the = is optional.
 *
 * The @c mdio_Config class is derived from @c mdio_File so that error
 * handling is performed by simply typecasting (@c mdio_Config @c *)
 * to a (@c mdio_File @c *).
 */

#ifndef MDIO_CONFIG_H
#define MDIO_CONFIG_H

#include "adt/list.h"
#include "mdio/file.h"

#ifdef __cplusplus
extern "C" {
#endif

  /**@brief Configuration file data structure.
   *
   * This provides the data parsed from a line of a configuration file.
   * An array of @c mdio_ConfigData is returned after reading one
   * or more configuration files.  The members are accessed directly
   * to obtain the parsed file contents.
   */
  typedef struct mdio_ConfigData_t {
    const char *keyword;   /**< Keyword string from configuration file. */
    const char *value;     /**< Value string assigned to this keyword. */
    const char *filename;  /**< Name of this configuration file. */
    int linenum;           /**< Line number of configuration file. */
  } mdio_ConfigData;


  /**@brief Configuration file reader class.
   *
   * Members should be treated as private.
   */
  typedef struct mdio_Config_t {
    mdio_File file;
    adt_List configdata;
  } mdio_Config;


/* constructor and destructor */

  /**@brief Constructor.
   *
   * Creates dynamically allocated configuration file reader object.
   *
   * @return Pointer to new object or @c NULL on failure.
   */
  mdio_Config *mdio_createConfig(void);


  /**@brief Destructor.
   *
   * Frees dynamically allocated configuration file reader object
   * after freeing the @c mdio_ConfigData array.
   *
   * Note that the @c mdio_ConfigData array obtained does not persist
   * beyond destruction of the @c mdio_Config object.
   */
  void mdio_destroyConfig(mdio_Config *);


/* reading file and obtaining parsed data */

  /**@brief Read configuration file.
   *
   * @param[in] name  A nil-terminated string naming the configuration file.
   *
   * Opens the file, reads and parses each line of the file, and closes
   * the file.  Each non-comment line of the file is stored as an
   * @c mdio_ConfigData structure saved into a list.
   *
   * Each @c name string provided to this function must persist until
   * the configuration file reader object is destroyed.
   *
   * @return 0 on success, @c MDIO_ERROR on failure.
   */
  int mdio_readConfig(mdio_Config *, const char *name);


  /**@brief Obtain array of data.
   *
   * @param[out] len  Length of array returned into indicated variable.
   *
   * @return The array of @c mdio_ConfigData configuration file data
   * as a pointer.
   */
  mdio_ConfigData *mdio_getDataConfig(mdio_Config *, int *len);


/* constructor and destructor for preallocated mdio_Config object */

  /**@brief Alternative constructor.
   *
   * Use to construct a preallocated configuration file reader object.
   * See @c mdio_createConfig() for a description of expected arguments.
   */
  int mdio_initializeConfig(mdio_Config *);


  /**@brief Alternative destructor.
   *
   * Use to destroy a preallocated configuration file reader object
   * (i.e. one constructed using @c mdio_initializeConfig() ).
   */
  void mdio_cleanupConfig(mdio_Config *);

#ifdef __cplusplus
}
#endif

#endif /* MDIO_CONFIG_H */
