/*
 * Copyright (C) 2003-2004 by David J. Hardy.  All rights reserved.
 *
 * mdfront.h - API for front end
 */

#ifndef MDFRONT_H
#define MDFRONT_H

#include "mdapi/mdcommon.h"

#ifdef __cplusplus
extern "C" {
#endif


  /* front end manipulates an "MD_Engine" object */
  typedef MD_Interface MD_Engine;

  /*
   * Description:
   *
   * The MDAPI enables modular software design for molecular dynamics
   * simulations of biomolecular systems by separating the software
   * into a front end and an engine.  The front end is responsible for
   * reading and writing files, configuring and invoking the engine, and
   * post-processing the results.  The engine is the computational kernel
   * responsible for performing the work needed for each time step of the
   * simulation.
   *
   * The API for the front end enables the front end to run one engine
   * per MD_Engine object declaration.  Since engines are supposed to be
   * reentrant by design, it is possible for multiple simulations to be
   * managed by a single front end all running the same engine code but
   * with different MD_Engine instantiations.
   *
   * The main responsibility of the front end is to setup the data used
   * by the engine, then to run the simulation by calling MD_run()
   * one or more times, and finally to perform any post-processing or
   * output of results from the simulation.  Most of the API routines
   * deal with various ways of sharing data between the front end and
   * the engine.  There are alternative ways of sharing data, some of
   * which improve performance and decrease memory usage.
   *
   * The most basic front end API calls, along with their general order
   * of use, are:
   *
   *   MD_init() - constructor (essential)
   *   MD_idnum() - provides ID number for named array (essential)
   *   MD_write() - write data to engine arrays (some alternatives)
   *   MD_run() - run the simulation (essential)
   *   MD_read() - read data from engine arrays (some alternatives)
   *   MD_done() - destructor (essential)
   *
   * There are some initial concepts that need clarification.  In order
   * to perform a simulation, an engine needs data (a lot of it) to be
   * supplied by the front end.  Since the data varies quite a bit
   * depending on the type of simulation and computational methods used,
   * hardcoded variable names for the data arrays would be a poor choice.
   * These would require every engine to provide all functionality
   * desired (or at least define the entire set of variable names).
   * Furthermore, the introduction of any new features or methods would
   * then require a new version of the interface, as well as updates
   * to all front ends and engines.  Instead of using hardcoded variable
   * names, data arrays are identified by string names.  There is a core
   * set of names that a front end can expect an engine to define
   * (see mdcommon.h) for the data that is common to most molecular
   * dynamics simulations.  This provides a basic level of
   * interoperatility between front ends and engines, while also
   * enabling the development of more advanced front ends that can take
   * advantage of features offered by specialized engines.
   *
   * Advanced front ends cannot afford to give up control indefinitely
   * (e.g. imagine a GUI application that is unresponsive to the user).
   * At the same time, advanced implementations of the MDAPI layer are
   * expected to connect a front end to a remote engine.  For instance,
   * the front end might be VMD running on a graphics workstation
   * connected to the NAMD engine running remotely on a beowulf cluster.
   * This requires allowing for the front end asynchronous control of
   * the engine by defining calls with nonblocking calling semantics.
   * Such a routine returns immediately but is not necessarily finished
   * with its task.  API calls are provided to test to see whether or
   * not the call is finished or to wait for the call to finish.  In
   * order to simplify the MDAPI semantics, only one outstanding call
   * to a nonblocking routine is allowed and the call must complete
   * before making another API call.  The following routines are defined
   * with nonblocking semantics:
   *
   *   MD_init()
   *   MD_read() / MD_readsub()
   *   MD_direct()
   *   MD_update()
   *   MD_run()
   *
   * These are all routines that require data to be sent from the engine.
   * Note that MD_write() / MD_writesub() do not need to block because
   * they modify buffers local to the front end.  The engine is not
   * guaranteed to see the buffer modifications until the front end calls
   * MD_update() or MD_run().
   *
   * The call to MD_run() hides the majority of the work done by the
   * engine.  In order to improve performance for long simulations, it is
   * best to allow the engine to continue to run for a large number of
   * steps.  However, the front end will still need feedback during the
   * simulation, including intermediate trajectories to log for later
   * imaging and analysis and quantities to be monitored to ensure the
   * stability of the simulation.  An advanced front end might want to
   * enable realtime imaging of the simulation or additional interactions
   * between the user and the simulation, such as interactive molecular
   * dynamics to allow the user to supply an external force to the system
   * via a controller.  The MDAPI enables the communication of data
   * between the front end and engine during a running simulation through
   * the use of callbacks.  A callback is simply an entry point back into
   * the front end (i.e. a function) that is called during MD_run(),
   * supplying the front end with data from the simulation or providing
   * the engine with new data.
   *
   * There are predefined data types (in mdtypes.h) that facilitate the
   * computation of forces and the integration of a system of atoms.
   * An engine is also permitted to define new types (i.e. C structs) to
   * be offered through the MDAPI with routines that allow the front end
   * to "see" the definition of a previously unknown type and access its
   * members.
   *
   * Most of the calls below return zero on success or MD_FAIL on failure,
   * in which case an error status value is set (similar to errno from the
   * C library).  Additional API calls permit monitoring the error state
   * and attempting error recovery.
   *
   * The first argument to the routines below is a pointer to MD_Engine
   * that must point to a valid MD_Engine variable.  MD_init() is called
   * (as the constructor) to properly initialize the MD_Engine structure.
   * MD_done() is called (as the destructor) to free memory used by both
   * the engine and the interface when finished.
   */


/*** constructor ***/

#define MD_init(e, engname, flags, engine_init, engine_done) \
  MD_init_version(e, MDAPI_VERSION, engname, flags, engine_init, engine_done)
   
  int32 MD_init_version(MD_Engine *e, const char *mdapi_version,
      const char *engname, int32 flags,
      int32 (*engine_init)(MD_Engine *, int32 flags),
      void (*engine_done)(MD_Engine *));
  /*
   * The front end calls MD_init(), which is actually a macro expansion
   * that calls MD_init_version(), in order to initialize the MDAPI layer
   * and the specified engine.
   *
   *   e - points to a valid, uninitialized MD_Engine object
   *   engname - name of engine, nil-terminated string
   *   flags - implementation dependent, also passed to engine_init()
   *   engine_init - constructor for engine
   *   engine_done - destructor for engine
   *
   * For the case in which one or more engines are linked to the
   * front end, the engine constructor and destructor routines need
   * to be passed.  Here, the "engname" string is remembered, but useful
   * only for labeling a particular engine.
   *
   * For the case in which an engine is dynamically loaded or remotely
   * invoked, the "engine_init" and "engine_done" pointers should be NULL,
   * and behavior is determined by the "engname" string (and perhaps also
   * by the flags).  Suggested string interpretation:
   *
   *   [[user@]hostname:]pathname
   *
   * covers both dyanmically loadable engines on the local machine as
   * well as remote engines.
   *
   * No "flags" are (currently) defined by the API, but this value is
   * passed as an argument to "engine_init" to permit implementation
   * dependent flags.
   *
   * Call has nonblocking semantics (see below).
   *
   * Returns 0 on success or MD_FAIL on error.
   *
   * Error states:
   *   MD_ERR_VERSION - inconsistent version number
   *   MD_ERR_MEMALLOC - memory cannot be allocated
   *   MD_ERR_NEWDATA - (from engine_init()) cannot create new engine data
   *   MD_ERR_NEWTYPE - (from engine_init()) cannot create new type
   *   MD_ERR_INIT - default fatal error indicating that the engine
   *     initialization failed
   *
   * Note that any error state returned here is fatal.
   */


/*** destructor ***/

  void MD_done(MD_Engine *);
  /*
   * Calls engine_done() routine (see above), frees engine data array
   * buffers (where necessary), and frees memory allocations made within
   * MDAPI layer.
   *
   * Error states:
   *   none
   */


/*** data array names and ID numbers ***/

  int32 MD_idnum(MD_Engine *, const char *name);
  /*
   * Given the (nil-terminated string) name of a data array, return
   * its identification number.
   *
   *   name - data array name identifier, nil-terminated string
   *
   * Case sensitive string-matching is performed.
   *
   * Returns (nonzero) identification number corresponding to the
   * recognized data array name, or MD_FAIL if no match is found.
   *
   * Error states:
   *   MD_ERR_NAME - name does not match any engine data arrays
   */

  const char *MD_name(MD_Engine *, int32 idnum);
  /*
   * Given the identification number of a data array, return its
   * (primary) string name.
   *
   *   idnum - data array identification number
   *
   * Returns data array name identifier corresponding to the
   * recognized identification number, or NULL if no match.
   *
   * Error states:
   *   MD_ERR_IDNUM - invalid data ID number
   */

  const char **MD_namelist(MD_Engine *, int32 *listlen);
  /*
   * Obtain the list of data array names.
   *
   *   listlen - points to an integer variable
   *
   * Returns an array containing all data array names (as nil-terminated
   * strings) and sets the variable pointed to by "listlen" to the length
   * of the array.
   *
   * Error states:
   *   none
   */


/*** data array attributes ***/

  MD_Attrib MD_attrib(MD_Engine *, int32 idnum);
  /*
   * Obtain the attributes of an engine data array.
   *
   *   idnum - data array identification number
   *
   * Returns the MD_Attrib attribute structure for the indicated data
   * array.  If an error occurs, the return value is the MD_Attrib structre
   * with all fields set to MD_FAIL (e.g. test against "attrib.type" field).
   *
   * Error states:
   *   MD_ERR_IDNUM - invalid data ID number
   */


/*** resize data array ***/

  int32 MD_setlen(MD_Engine *, int32 idnum, int32 newlen);
  int32 MD_setmax(MD_Engine *, int32 idnum, int32 newmax);
  int32 MD_resize(MD_Engine *, int32 idnum, int32 newlen, int32 newmax);
  /*
   * These calls set attributes for an engine data array buffer.
   * MD_setlen() sets the length of the array in use, MD_setmax() sets
   * the maximum number of allocated array elements (greater than or
   * equal to length), and MD_resize() sets both.
   *
   *   idnum - data array identification number
   *   newlen - new length value
   *   newmax - new maximum array allocation value
   *   (should have 0 <= newlen <= newmax)
   *
   * These calls are permitted if attrib.access MD_SETLEN and/or MD_SETMAX
   * is set.  If newlen > attrib.max and MD_SETMAX is set, then buffer
   * allocation is extended to attrib.max = newlen.  If newmax < attrib.len
   * and MD_SETLEN is set, then buffer allocation is truncated with
   * attrib.len = newmax.
   *
   * Returns 0 on success or MD_FAIL on error.
   *
   * Error states:
   *   MD_ERR_IDNUM - invalid data ID number
   *   MD_ERR_ACCESS - access MD_SETLEN and/or MD_SETMAX not permitted
   *   MD_ERR_MEMALLOC - memory cannot be allocated
   *   MD_ERR_RANGE - newlen < 0 or newmax < 0 or newmax < newlen or access
   *     permissions do not allow newlen > attrib.max or newmax < attrib.len
   */


/*** asynchronous control ***/

  int32 MD_test(MD_Engine *);
  int32 MD_wait(MD_Engine *);
  /*
   * Some of the calls below (as well as MD_init() above) have
   * nonblocking calling semantics.  An implementation of MDAPI that
   * supports nonblocking calls enables use of MDAPI for advanced
   * front ends (e.g. GUI) which cannot afford to lose control for an
   * indefinite length of time.  Any call that might require a "response"
   * from the engine (i.e. receive remote communication) has nonblocking
   * semantics defined.
   *
   * A nonblocking call has not necessarily completed its function when
   * it returns.  The completion of the call must be either tested true
   * with MD_test() or waited for with MD_wait().
   *
   * MD_test() - Immediately returns without blocking, either true (> 0)
   *   if nonblocking call has completed or 0 if call has not yet completed.
   *   Returns MD_FAIL if an error occurred (i.e. completed but with error).
   *
   * MD_wait() - Blocks waiting for call to complete, returns 0 for success
   *   or MD_FAIL on error.
   *
   * Error states:
   *   none for single-threaded MDAPI implementation
   */


/*** read from engine data array ***/

#define MD_read(e, idnum, buf, len) \
  MD_readsub(e, idnum, buf, len, 0)
  /*
   * Read from the beginning of an engine data buffer "len" elements,
   * then copy into "buf".  Buffer must permit MD_READ access.
   *
   *   idnum - data array identification number
   *   buf - buffer to be filled from data array
   *   len - length of array in elements
   *
   * Call has nonblocking semantics.  Front end needs to have allocated
   * sufficient space for "buf" and retains control of this memory.
   * "buf" should not be accessed until after call completes.
   *
   * Returns 0 on success or MD_FAIL on error.
   *
   * Error states:
   *   MD_ERR_IDNUM - invalid data ID number
   *   MD_ERR_ACCESS - access MD_READ is not permitted
   *   MD_ERR_RANGE - len < 0 or len > attrib.len
   */

  int32 MD_readsub(MD_Engine *, int32 idnum, void *buf,
      int32 nelems, int32 first);
  /*
   * Read a subarray from an engine data buffer, then copy into "buf".
   * Buffer must permit MD_READ access.
   *
   *   idnum - data array identification number
   *   buf - buffer to be filled from data array
   *   nelems - number of elements to read
   *   first - first element to index from
   *
   * The subarray is defined by "first" as the first index and "nelems"
   * as the number of elements (or length).
   *
   * Call has nonblocking semantics.  Front end needs to have allocated
   * sufficient space for "buf" and retains control of this memory.
   * "buf" should not be accessed until after call completes.
   *
   * Returns 0 on success or MD_FAIL on error.
   *
   * Error states:
   *   MD_ERR_IDNUM - invalid data ID number
   *   MD_ERR_ACCESS - access MD_READ is not permitted
   *   MD_ERR_RANGE - first < 0 or nelems < 0 or first + nelems > attrib.len
   *
   * MD_read(e,id,buf,n) is equivalent to MD_readsub(e,id,buf,n,0).
   */


/*** write to engine data array ***/

#define MD_write(e, idnum, buf, len) \
  MD_writesub(e, idnum, buf, len, 0)
  /*
   * Write to the beginning of an engine data buffer "len" elements,
   * copied from "buf".  Buffer must permit MD_WRITE access.
   *
   *   idnum - data array identification number
   *   buf - buffer used for writing to data array
   *   len - length of array in elements
   *
   * Front end retains control of the "buf" memory.  This call does *not*
   * have nonblocking semantics.
   *
   * Returns 0 on success or MD_FAIL on error.
   *
   * Error states:
   *   MD_ERR_IDNUM - invalid data ID number
   *   MD_ERR_ACCESS - access MD_WRITE is not permitted
   *   MD_ERR_RANGE - len < 0 or len > attrib.len
   */

  int32 MD_writesub(MD_Engine *, int32 idnum, const void *buf,
      int32 nelems, int32 first);
  /*
   * Write to an engine data buffer subarray, copied from "buf".
   * Buffer must permit MD_WRITE access.
   *
   *   idnum - data array identification number
   *   buf - buffer used for writing to data array
   *   nelems - number of elements to write
   *   first - first element to index from
   *
   * The subarray is defined by "first" as the first index and "nelems"
   * as the number of elements (or length).
   *
   * Front end retains control of the "buf" memory.  This call does *not*
   * have nonblocking semantics.
   *
   * Returns 0 on success or MD_FAIL on error.
   *
   * Error states:
   *   MD_ERR_IDNUM - invalid data ID number
   *   MD_ERR_ACCESS - access MD_WRITE is not permitted
   *   MD_ERR_RANGE - first < 0 or nelems < 0 or first + nelems > attrib.len
   */


/*** front end allocated data buffer ***/

  int32 MD_share(MD_Engine *, int32 idnum, void *buf, int32 len, int32 max);
  /*
   * Front end shares its data buffer with the engine.  The engine
   * data array must permit MD_SHARE access.  Note that MD_SHARE
   * access is provided by the MDAPI layer, not by the engine.
   *
   *   idnum - data array identification number
   *   buf - allocated buffer
   *   len - number of elements initialized in buffer
   *   max - maximum number of elements available in buffer
   *
   * The front end provides the data buffer to be used for this data
   * array.  Control of the data buffer is yielded by the front end
   * until either MD_unshare() is called on this idnum or until MD_done().
   * However, the front end is still responsible for managing this memory
   * (i.e. freeing it) after control is regained.
   *
   * When the data buffer is established through MD_share(), MD_SETMAX is
   * no longer permitted (i.e. the memory allocation cannot be resized).
   * Until control of memory is regained, the front end should access it
   * through API calls rather than directly.
   *
   * Note that an engine data array may have MD_SHARE access only if
   * there is no buffer space yet allocated to it (i.e. attrib.max == 0).
   * Such an array will have MD_SHARE access disabled if the buffer is
   * resized by, say, MD_setmax().
   *
   * Call does not block, so it *does not* have nonblocking semantics.
   *
   * Returns 0 on success or MD_FAIL on error.
   *
   * Error states:
   *   MD_ERR_IDNUM - invalid data ID number
   *   MD_ERR_ACCESS - access MD_SHARE is not permitted
   *   MD_ERR_RANGE - len < 0 or max < len
   */

  void *MD_unshare(MD_Engine *, int32 idnum);
  /*
   * Front end regains control of shared data buffer previously offered
   * through MD_share().  MD_UNSHARE access is enabled on a data buffer
   * (by the MDAPI layer) immediately following a successful call to
   * MD_share().
   *
   *   idnum - data array identification number
   *
   * After successful completion, the engine data buffer will no longer
   * have any memory allocation (i.e. attrib.max == 0) and MD_SHARE
   * access will again be enabled, returning the engine data buffer to
   * its prior state before MD_share() was invoked on it.
   *
   * Call does not block, so it *does not* have nonblocking semantics.
   *
   * Returns 0 on success or MD_FAIL on error.
   *
   * Error states:
   *   MD_ERR_IDNUM - invalid data ID number
   *   MD_ERR_ACCESS - access MD_UNSHARE is not permitted
   */


/*** direct access to data buffer ***/

  void *MD_direct(MD_Engine *, int32 idnum);
  /*
   * Obtain direct access to an engine data buffer.  Buffer must permit
   * MD_DIRECT access.
   *
   *   idnum - data array identification number
   *
   * Returns either a pointer to the data array memory buffer or
   * NULL if data array has attrib.max == 0 or if an error occurs.
   *
   * Error states:
   *   MD_ERR_IDNUM - invalid data ID number
   *   MD_ERR_ACCESS - access MD_DIRECT is not permitted
   *   MD_ERR_MEMALLOC - local buffer space cannot be allocated
   *
   * This makes the data array buffer directly accessible to the front
   * end.  However, this must be used with caution.  For instance, if
   * MD_setmax() or MD_resize() resizes the memory buffer of a data
   * array, this will most likely invalidate any pointer returned for
   * that array prior to the resizing.
   *
   * This call has nonblocking semantics.  For a remote engine MDAPI
   * implementation, this routine might have to allocate local buffer
   * space for the array and populate it.
   */

  int32 MD_setmod(MD_Engine *, int32 idnum);
  /*
   * Indicate that a directly accessed engine data buffer has been
   * modified.  Buffer must permit MD_DIRECT access.
   *
   *   idnum - data array identification number
   *
   * This marks the "dirty bit" on the access flag, indicating that the
   * contents of the data array has been modified.  So this should be
   * called if the front end directly modifies the memory buffer obtained
   * from MD_direct().
   *
   * Returns 0 on success or MD_FAIL on error.
   *
   * Error states:
   *   MD_ERR_IDNUM - invalid data ID number
   *   MD_ERR_ACCESS - access MD_DIRECT is not permitted
   */


/*** update all data modifications to the engine ***/

  int32 MD_update(MD_Engine *);
  /*
   * Provide engine with data buffer modifications.
   *
   * For a front end connected to a remote engine, writing to data
   * buffers will (most likely) be cached local to the front end.
   * This call sends all updated data array buffers (i.e. those marked
   * with the "dirty bit") to the engine.
   *
   * Note that MD_run() must perform the functionality of MD_update()
   * before running the simulation.  Judicious use of MD_update() might
   * reduce the overhead required upon calling MD_run().
   *
   * This call has nonblocking semantics.
   *
   * Returns 0 on success or MD_FAIL on error.
   *
   * Error states:
   *   none for single-threaded MDAPI implementation (no functionality)
   */


/*** propagate system for given number of steps ***/

  int32 MD_firststep(MD_Engine *, int32 firststep);
  /*
   * Initialize the step number counter before running a simulation.
   *
   *   firststep - first step number
   *
   * This routine has the side effect of resetting callback processing:
   * all callbacks start their step increments together with respect to
   * this intial step number value.
   *
   * Step numbering defaults to start at value 0.
   *
   * Returns 0 (always succeeds).
   */

  int32 MD_stepnum(MD_Engine *);
  /*
   * Returns the step number counter.
   */

  int32 MD_run(MD_Engine *, int32 numsteps, int32 runflags);
  /*
   * Run a simulation, integrating the system for the specified
   * numbered steps.
   *
   *   numsteps - nonnegative integer
   *   runflags - runtime flags
   *
   * Calling with numsteps == 0 simply ensures that the engine is
   * fully initialized, which might include evaluation of the forces
   * for the current position configuration.  In this case, callbacks
   * are also processed if they have not already been for this step
   * (see callback notes below).
   *
   * The "runflags" may include MD_CBFIRST bitwise ORed with any
   * relevant engine-dependent flags.
   *
   * This call has nonblocking semantics.  Communication with a
   * running simulation is performed via pre-established callbacks.
   *
   * Returns 0 on success or MD_FAIL on error.
   *
   * Error states:
   *   MD_ERR_CALLBACK - callback routine returned nonzero
   *   MD_ERR_CBSHARE - callback shared buffer failed (access does not
   *     permit resizing of target engine data array buffer)
   *   MD_ERR_RANGE - numsteps < 0 or bug in engine resizing data arrays
   *   MD_ERR_MEMALLOC - memory cannot be allocated (resizing data arrays
   *     by engine or from callback shared buffer)
   *   MD_ERR_RUN - default error state if engine run routine returns
   *     nonzero without setting the error state
   *   MD_ERR_ACCESS - indicates bug in engine (incorrect attempt to
   *     resize data arrays or acknowledge data array modification)
   *   MD_ERR_CHECK - consistency check failed, indicates bug in engine
   *     (either step counter is not at projected value or callback not
   *     processed at expected step number)
   *
   * The engine might also set its own error state values.
   */


/*** establish callbacks ***/
  /*
   * The callback routines are called during the engine execution of
   * an MD_run() call.  They provide the means for the front end to
   * receive and send data to the engine without interrupting the
   * simulation.
   */

  int32 MD_callback(MD_Engine *,
      int32 (*cb)(void *info, MD_Cbdata *data, int32 len, int32 stepnum),
      void *info, MD_Cbdata *data, int32 datalen, int32 stepincr);
  /*
   * Register a standard callback.
   *
   *   cb - points to a standard callback function
   *     The callback function will receive the following arguments:
   *       info - pointer to front end specific information
   *       data - array of MD_Cbdata to receive engine data array contents
   *           (same data array as below with current data array elements)
   *       len - length of the MD_Cbdata array
   *       stepnum - the current step number
   *     (All of these, except stepnum, are from the arguments below.)
   *   info - (to be passed to callback function)
   *   data - array of MD_Cbdata to setup call to "cb"
   *   datalen - length of the MD_Cbdata array
   *   stepincr - how often to call back, counting from "firsttimestep"
   *
   * This establishes a callback to function "cb" providing engine data
   * array information through MD_Cbdata array "data" of length "datalen"
   * along with front end specific information pointed to through "info".
   * See the description of MD_Cbdata from mdcommon.h for an explanation
   * on referencing engine data subarrays and front end access rights.
   * A standard callback will be called every "stepincr" steps counting
   * from "firsttimestep".  The standard callbacks are called right after
   * the completion of a time integration step, so their purpose is to
   * log data such as atom trajectories, monitor energies or temperature,
   * etc.  A function "cb" can be registered to receive multiple callbacks
   * and even to receive different data sets through a different MD_Cbdata
   * array on each invocation.
   *
   * A callback function relinquishes any access rights to the MD_Cbdata
   * array data after returning.  This means, for instance, performing
   * asyncrhonous I/O to write trajectory files requires copying the
   * trajectory data received through the MD_Cbdata array into a separate
   * buffer before returning control back to the MDAPI layer.
   *
   * The MD_Cbdata array must persist until either this callback is
   * un-registered (see MD_callback_undo() below) or until cleanup
   * through MD_done().
   *
   * The return value of a callback function has significance.  Return
   * zero to indicate success and nonzero (preferably MD_FAIL) for an
   * error.  Upon failure of a callback, the engine is expected to stop
   * the simulation and return control of its run routine back to the
   * MDAPI layer, which in turn returns control back to the front end
   * with an error message from the MD_run() call.  So this provides
   * the front end with a mechanism to terminate the engine.
   *
   * Returns 0 on success or MD_FAIL on error.
   *
   * Error states:
   *   MD_ERR_MEMALLOC - memory cannot be allocated
   *   MD_ERR_IDNUM - invalid data ID number for some MD_Cbdata element
   *   MD_ERR_ACCESS - access MD_CBREAD and/or MD_CBWRITE or MD_CBSHARE not
   *     permitted, or attempt to combine MD_CBSHARE with other access
   *   MD_ERR_RANGE - stepincr <= 0 or, for some MD_Cbdata element, first < 0
   *     or nelems < -1 or first + nelems > attrib.len
   */

  int32 MD_callback_undo(MD_Engine *,
      int32 (*cb)(void *info, MD_Cbdata *data, int32 len, int32 stepnum));
  /*
   * Un-register a standard callback.
   *
   *   cb - points to a standard callback function
   *
   * Any previously registered callbacks to routine "cb" are removed from
   * the standard callback queue.  Will succeed (but without doing anything)
   * if there are no pointers to "cb" in the queue.  Calling with cb == NULL
   * will remove *all* callbacks from queue.
   *
   * Returns 0 on success or MD_FAIL on error.
   *
   * Error states:
   *   MD_ERR_MEMALLOC - memory cannot be allocated
   */


  int32 MD_fcallback(MD_Engine *,
      int32 (*fcb)(void*, MD_Cbdata*, int32 len, int32 stepnum, double frac),
      void *info, MD_Cbdata *data, int32 datalen);
  /*
   * Register a force callback.  Most of what is described for standard
   * callbacks applies to force callbacks.  Differences are listed below.
   *
   *   fcb - points to a force callback
   *     Additional argument received:
   *       frac - timestep fraction, meaning that the force is being
   *         evaluated for the atom position approximation at time
   *         t = dt*(stepnum+frac).  For instance, with leapfrog
   *         (velocity Verlet) integration, frac == 1.0, since the force
   *         is evaluated for the new positions, but since the step has
   *         not yet completed, stepnum retains the value for the current
   *         step.  Depending on the integrator used, frac could actually
   *         be greater than 1 or less than 0.
   *   (Note that the stepincr argument is no longer needed since force
   *   callbacks are processed on every force evaluation.)
   *
   * Force callbacks are the intended mechanism for the front end to
   * supply external forces to the simulation without having to know
   * details about the particular integrator being used to propagate the
   * system.  The main difference between force callbacks and standard
   * callbacks is when the callbacks are processed.  Standard callbacks
   * are processed at the beginning of a new step being reached.  Force
   * callbacks are processed during each force evaluation, which,
   * depending on the integration method, often occurs in the middle of
   * a step.
   *
   * Returns 0 on success or MD_FAIL on error.
   *
   * Error states:
   *   MD_ERR_MEMALLOC - memory cannot be allocated
   *   MD_ERR_IDNUM - invalid data ID number for some MD_Cbdata element
   *   MD_ERR_ACCESS - access MD_FCBREAD and/or MD_FCBWRITE or MD_FCBSHARE
   *     not permitted, or attempt to combine MD_FCBSHARE with other access
   *   MD_ERR_RANGE - stepincr <= 0 or, for some MD_Cbdata element, first < 0
   *     or nelems < -1 or first + nelems > attrib.len
   */

  int32 MD_fcallback_undo(MD_Engine *,
      int32 (*fcb)(void*, MD_Cbdata*, int32 len, int32 stepnum, double frac));
  /*
   * Un-register a force callback.
   *
   *   fcb - points to a force callback function
   *
   * Any previously registered callbacks to routine "fcb" are removed from
   * the force callback queue.  Will succeed (but without doing anything)
   * if there are no pointers to "fcb" in the queue.  Calling with
   * fcb == NULL will remove *all* callbacks from queue.
   *
   * Returns 0 on success or MD_FAIL on error.
   *
   * Error states:
   *   MD_ERR_MEMALLOC - memory cannot be allocated
   */


  int32 MD_msgcallback(MD_Engine *,
      int32 (*msgcb)(void *info, const char *msg, int32 stepnum),
      void *info);
  /*
   * Register a message callback.
   *
   *   msgcb - points to the message callback function
   *     The message callback receives the following arguments:
   *       info - pointer to front end specific information
   *       msg - the text message, a nil-terminated string
   *       stepnum - the current step number
   *   info - (to be passed to the message callback function)
   *
   * This establishes a function to receive message callbacks.  This is
   * simply a mechanism for the engine to provide the front end with a
   * status message (as a nil-terminated string) whenever it wants to
   * document an event.  The return value from a message callback routine
   * has the same significance as that from a force or standard callback.
   * However, unlike the other callback types, message callback routines
   * are not guaranteed to be called by the engine.
   *
   * Although for consistency with the other callback types, it is
   * possible to register multiple message callbacks, this feature does
   * not seem to serve much purpose in practice since they will all
   * receive exactly the same message.
   *
   * Returns 0 on success or MD_FAIL on error.
   *
   * Error states:
   *   MD_ERR_MEMALLOC - memory cannot be allocated
   */

  int32 MD_msgcallback_undo(MD_Engine *,
      int32 (*msgcb)(void *info, const char *msg, int32 stepnum));
  /*
   * Un-register a message callback.
   *
   *   msgcb - points to a message callback function
   *
   * Any previously registered callbacks to routine "msgcb" are removed
   * from the message callback queue.  Will succeed (but without doing
   * anything) if there are no pointers to "msgcb" in the queue.  Calling
   * with msgcb == NULL will remove *all* callbacks from queue.
   *
   * Returns 0 on success or MD_FAIL on error.
   *
   * Error states:
   *   MD_ERR_MEMALLOC - memory cannot be allocated
   */



/*** use engine defined types ***/

  int32 MD_type(MD_Engine *, const char *name);
  /*
   * Given the (nil-terminated string) name of a data type, return
   * its type number.
   *
   *   name - type name, nil-terminated string
   *
   * Case sensitive string-matching is performed.  Note that different
   * engines might define types of the same name, but this does not mean
   * that they will be assigned the same type ID number.
   *
   * Returns (nonzero) type identification number corresponding to
   * the recognized type name or MD_FAIL if no match is found.
   *
   * Error states:
   *   MD_ERR_NAME - name does not match any defined data types
   */

  const char *MD_type_name(MD_Engine *, int32 type);
  /*
   * Given the type identification number, return its string name.
   *
   *   type - type identification number
   *
   * Returns the type name corresponding to the recognized type
   * identification number, or NULL if no match.
   *
   * Error states:
   *   MD_ERR_TYPENUM - invalid type number
   */

  const char **MD_type_namelist(MD_Engine *, int32 *listlen);
  /*
   * Obtain the list of type names.
   *
   *   listlen - points to an integer variable
   *
   * Returns an array containing all type names (as nil-terminated
   * strings) and sets the variable pointed to by "listlen" to the
   * length of the array.
   *
   * Error states:
   *   none
   */

  const MD_Member *MD_type_memberlist(MD_Engine *, int32 type, int32 *listlen);
  /*
   * Obtain the MD_Member array for the defined type.
   *
   *   type - type identification number
   *   listlen - points to an integer variable
   *
   * Returns the array of MD_Member corresponding to the given type.
   * The MD_Member array provides information on the members of the
   * defined type (i.e. think C struct members).  See the description
   * of MD_Member in mdcommon.h.  The number of elements in the array
   * is returned in the variable pointed to by "listlen".  NULL is
   * returned for an invalid "type" number.
   *
   * Error states:
   *   MD_ERR_TYPENUM - invalid type number
   */

  void *MD_type_member(MD_Engine *, int32 type, const void *obj,
      const char *member_name, MD_Member *pmember);
  /*
   * Given the name of a member, obtain a pointer to that member within
   * an object of the defined type.
   *
   *   type - type identification number
   *   obj - points to an object instance of the indicated type
   *   member_name - nil-terminated string name of member from this type
   *   pmember - receive the complete MD_Member struct for the member
   *
   * Return a pointer to the member named "member_name" within the object
   * instance "obj" of the indicated type.  Case sensitive string-matching
   * is performed on the member name.
   *
   * The first time this routine is called for particular type, the MDAPI
   * layer builds a search table of the member names and their address
   * offsets from the start of the type.  Subsequent calls require just
   * O(1) work to search the existing table for a given name.  The
   * "pmember" field, if non-NULL, receives the entire MD_Member struct
   * for the named member.
   *
   * NULL is returned if an error occurs.
   *
   * Error states:
   *   MD_ERR_MEMALLOC - memory cannot be allocated
   *   MD_ERR_TYPENUM - invalid type number
   *   MD_ERR_NAME - name does not match any members of specified type
   *   MD_ERR_NEWTYPE - indicates bug in engine (occurs if more than one
   *     member has the same name, really indicating an error from engine
   *     initialization that was not discovered until this routine).
   */


/*** error handling and diagnostics ***/

  int32 MD_errnum(MD_Engine *);
  /*
   * Returns the (nonzero) error number if an error condition has been
   * set or 0 if there is no error.
   */

  const char *MD_errmsg(MD_Engine *);
  /*
   * Return the error message description (nil-terminated string) for
   * the current error condition.
   */

  int32 MD_reset(MD_Engine *);
  /*
   * Attempt to reset the current error state to MD_ERR_NONE.
   *
   * Returns 0 on success, in which case subsequent calls to MD_errnum()
   * will return 0 (until another error occurs).  Fails if the current
   * error state is considered unrecoverable (a.k.a. fatal); in this case,
   * MD_FAIL is returned and the front end should terminate use of this
   * engine object by calling MD_done().
   */

  const char *MD_engine_name(MD_Engine *);
  /*
   * Return the engine name, as specified in the MD_init() call.
   * Intended for diagnostic purposes.
   */


#ifdef __cplusplus
}
#endif

#endif  /* MDFRONT_H */
