/*
 * Copyright (C) 2003-2004 by David J. Hardy.  All rights reserved.
 *
 * mderror.c - function definitions for MDAPI error reporting and handling
 */

#include <stddef.h>
#include "adt/list.h"
#include "mdapi/mddef.h"
#include "debug/debug.h"


/*
 * error reporting - array needs to have same ordering as error
 *   numbers defined in mdcommon.h
 */

static const MD_Error ErrorList[MD_ERROR_LISTLEN] = {
  { "Success",                           0 },  /* MD_ERR_NONE     */
  { "Inconsistent version number",      -1 },  /* MD_ERR_VERSION  */
  { "Memory cannot be allocated",       -1 },  /* MD_ERR_MEMALLOC */
  { "Engine initialization failed",     -1 },  /* MD_ERR_INIT     */
  { "Engine run simulation failed",     -1 },  /* MD_ERR_RUN      */
  { "Consistency check failed",         -1 },  /* MD_ERR_CHECK    */
  { "Cannot create new engine data",    -1 },  /* MD_ERR_NEWDATA  */
  { "Cannot create new type",           -1 },  /* MD_ERR_NEWTYPE  */
  { "Unrecognized name string",          0 },  /* MD_ERR_NAME     */
  { "Invalid data ID number",            0 },  /* MD_ERR_IDNUM    */
  { "Invalid type number",               0 },  /* MD_ERR_TYPENUM  */
  { "Value is out of range",             0 },  /* MD_ERR_RANGE    */
  { "Access is denied",                  0 },  /* MD_ERR_ACCESS   */
  { "Callback routine failed",           0 },  /* MD_ERR_CALLBACK */
  { "Callback shared buffer failed",     0 },  /* MD_ERR_CBSHARE  */
};


/*
 * front end API methods
 */

int32 MD_errnum(MD_Engine *s)
{ 
  ASSERT(s != NULL);
  return s->errnum;
}

const char *MD_errmsg(MD_Engine *s)
{
  ASSERT(s != NULL);
  return s->error.msg;
}

int32 MD_reset(MD_Engine *s)
{
  ASSERT(s != NULL);
  if (s->error.isfatal) return MD_FAIL;
  MD_error(s, MD_ERR_NONE);
  return 0;
}


/*
 * engine API methods
 */

int32 MD_error(MD_Front *s, int32 errnum)
{
  adt_List *engerror;
  MD_Error *data;
  int32 len;

  ASSERT(s != NULL);
  engerror = &(s->engerror);
  len = (int32) adt_getLengthList(engerror);
  data = (MD_Error *) adt_getDataList(engerror);
  if (errnum >= 0 && errnum < MD_ERROR_LISTLEN) {
    s->errnum = errnum;
    s->error = ErrorList[errnum];
  }
  else if (errnum - MD_ERROR_LISTLEN < len) {
    s->errnum = errnum;
    s->error = data[errnum - MD_ERROR_LISTLEN];
  }
  else {
    BUG("Invalid error number");
  }
  return MD_FAIL;
}

int32 MD_new_error(MD_Front *s, const char *errmsg, int32 isfatal)
{
  adt_List *list;
  MD_Error err;
  int32 errnum, len;

  ASSERT(s != NULL);
  ASSERT(errmsg != NULL);
  list = &(s->engerror);
  len = (int32) adt_getLengthList(list);
  errnum = len + MD_ERROR_LISTLEN;
  err.msg = errmsg;
  err.isfatal = isfatal;
  if (adt_appendList(list, &err)) {
    return MD_error(s, MD_ERR_MEMALLOC);
  }
  return errnum;
}


/*
 * internal methods
 */

/* version number check failed */
int32 MD_error_version(MD_Interface *s)
{
  /* must set error here because we can't trust length of MD_Interface */
  s->errnum = MD_ERR_VERSION;
  s->error = ErrorList[MD_ERR_VERSION];
  return MD_FAIL;
}

int32 MD_init_error(MD_Interface *s)
{
  ASSERT(s != NULL);
  /* init error state */
  s->errnum = 0;
  s->error = ErrorList[0];
  /* init array for engine defined errors */
  if (adt_initializeList(&(s->engerror), sizeof(MD_Error), 0, NULL)) {
    return MD_error(s, MD_ERR_MEMALLOC);
  }
  return 0;
}

void MD_done_error(MD_Interface *s)
{
  ASSERT(s != NULL);
  /* free array for engine defined errors */
  adt_cleanupList(&(s->engerror));
}
