/*
 * Copyright (C) 2003-2004 by David J. Hardy.  All rights reserved.
 *
 * mdengine.h -- API for engine
 */

#ifndef MDENGINE_H
#define MDENGINE_H

#include "mdapi/mdcommon.h"

#ifdef __cplusplus
extern "C" {
#endif

  /* engine manipulates an "MD_Front"-end object */
  typedef MD_Interface MD_Front;


  /*
   * MD_Engdata provides information regarding an engine data array for
   * which the front end is permitted some kind of access.
   */
  typedef struct MD_Engdata_tag {
  /*** accessed by engine to set and get data ***/
  /*** generally engine should not redirect this pointer ***/
    void *buf;         /* points to the data */

  /*** the rest are not intended for engine ***/
    void *(*reallocmem)(void *ptr, size_t size);
        /* memory management routine to call for this data array */
        /* (has same semantics as C library realloc() but is not */
        /* necessarily the same function) */
    const char *name;  /* name of data array */
    MD_Attrib attrib;  /* attributes for the data array */
    int32 idnum;       /* associated identification number for this array */
    adt_List cback;    /* callbacks in which this data buffer is involved */
  } MD_Engdata;


  /*
   * MD_Callback stores data needed for invoking a callback.  One MD_Callback
   * object is allocated for each (standard and force) callback registered
   * by front end.  MDAPI layer keeps one array of MD_Callback for standard
   * callbacks and another array of MD_Callback for force callbacks.  Engine
   * can obtain either array in order to discover exactly what data is
   * needed by the front end and when.
   */

  /* status flag values for MD_Callback */
  enum {
    MD_CALLBACK_INIT  = 0x001,
    MD_CALLBACK_SETUP = 0x002,
    MD_CALLBACK_SHARE = 0x004
  };

  typedef struct MD_Callback_tag {
  /*** needed for both standard and force callbacks ***/
    union {
      int32 (*cb)(void *, MD_Cbdata *c, int32 clen, int32 stepnum);
      int32 (*fcb)(void *, MD_Cbdata *c, int32 clen, int32 step, double ds);
    } funcptr;          /* points to callback routine */
    void *info;         /* points to front end specific info */
    MD_Cbdata *cbarg;   /* specifies engine data arrays to pass to callback */
    int32 cbarglen;     /* length of array */

  /*** needed only for standard callbacks ***/
    int32 stepincr;     /* how often routine is to be called */
    int32 nextstepnum;  /* tells next step number routine will be called */
    int32 cbinfonum;    /* order that this callback was added to queue */

  /*** status flags ***/
    int32 flags;
  } MD_Callback;


  /*
   * Summary
   *
   * The engine has three entry points from MDAPI:
   * 1. initialization routine (called through MD_init())
   * 2. run routine (called through MD_run())
   * 3. cleanup routine (called through MD_done())
   *
   * Most of the API calls for the engine are made during intialization,
   * where the main goal is to setup all data array communication that
   * will take place between front end and engine.
   *
   * The run routine performs the integration and necessary force
   * evaluations for the system.  The main API calls here concern the
   * processing of callbacks.  Although the specific callbacks to the
   * front end are managed by the MDAPI layer, the engine is required to
   * initiate them.  There are three varieties of callbacks: standard
   * callbacks, force evaluation callbacks, and message callbacks.  The
   * standard callbacks need to be processed (when so indicated) at the
   * end of an integration step.  The force evaluation callbacks (for the
   * introduction of external forces) need to be processed during any
   * evaluation of the force to obtain the total forces.  The message
   * callback is simply for sending arbitrary text strings that report
   * the engine status to the front end.  These three callback types can
   * all be processed in parallel, with non-blocking calling semantics.
   * The only restrictions are that you cannot initiate another callback
   * of the same variety before the first callback of that variety has
   * completed.
   *
   * If the forces are stale (or have not yet been computed) when the
   * run routine is first called, then immediately after finishing any
   * remaining simulation initialization, the engine is expected to
   * perform an initial evaluation of the forces, then check for a
   * standard callback on the zeroth step.
   *
   * Also, upon completion of an integration step, the engine is
   * expected to increment its step counter.  (This is the mechanism
   * by which the MDAPI layer synchronizes the processing of callbacks.)
   *
   * The cleanup routine is required to free any memory allocations
   * made by the engine.  The engine should not free memory that it
   * does not explicitly allocate.
   */


/*****************************************************************************
 * calls during engine initialization
 *
 * Engine initialization routine prototype (called by MD_init()):
 *   int32 engine_init(MD_Front *, int32 flags);
 *
 * Any MDAPI calls here that fail are fatal, so engine should return value
 * MD_FAIL immediately.  Cleanup routine will still be called, so you can
 * delay freeing memory allocations until then.
 *****************************************************************************/

/*** first call made during engine initialization ***/

#define MD_setup_engine(f, engine) \
  MD_setup_engine_version(f, MDAPI_VERSION, engine)

  int32 MD_setup_engine_version(MD_Front *,
      const char *mdapi_version, void *engine);
  /*
   *   engine - points to main engine data structure
   *
   * This routine must be called before any other engine API routines,
   * because it checks the consistency of the MDAPI version number.
   * Engine calls MD_setup_engine(), which is actually a macro expansion
   * that calls MD_setup_engine_version(), in order to initialize the
   * engine API layer.
   *
   * The pointer value passed as "engine" will later be returned by
   * MD_engine_data() as the handle to the main engine data structure.
   *
   * Returns 0 on success and MD_FAIL on error.
   *
   * Error states:
   *   MD_ERR_VERSION - inconsistent version number
   */


/*** establish access to named engine data arrays ***/

  MD_Engdata *MD_engdata(MD_Front *, const char *name,
      int32 type, int32 access);
  /*
   * Create an MD_Engdata (engine data array) with an unbounded buffer
   * managed by MDAPI layer.
   *
   *   name - nil-terminated string identifier for this data array
   *   type - type of data (use type number constants from mdtypes.h or
   *     value returned by successful call to MD_new_type())
   *   access - bitwise flag of front end access rights to data array
   *     (use "set by engine" values defined in mdcommon.h)
   *
   * This is the typical way for the engine to setup a data array.  The
   * MDAPI performs memory management for this array, initially with 
   * attrib.len == attrib.max == 0.  Depending on other flags, MDAPI may
   * also enable MD_SHARE flag.  Note that this is the only constructor
   * for MD_Engdata that will permit MD_SHARE access.  The "name" must
   * persist until engine cleanup.
   *
   * Return value is pointer to the MD_Engdata object held by MDAPI layer
   * or NULL on error.
   *
   * Error states:
   *   MD_ERR_NEWDATA - cannot create new engine data (due to incorrect
   *     attributes or name conflict with another engine data array)
   *   MD_ERR_MEMALLOC - memory cannot be allocated
   *
   * Example:  Applies to most data arrays.  Length is unknown a priori,
   * instead determined by front end input files.  In this case, the front
   * end might be responsible to set length of data arrays like atom, bond,
   * angle, etc. based on size of the system, with memory allocation all
   * managed by MDAPI layer.
   */

  MD_Engdata *MD_engdata_buffer(MD_Front *, const char *name,
      MD_Attrib, void *buf);
  /*
   * Create an MD_Engdata (engine data array) with a preallocated buffer
   * supplied by the engine.
   *
   *   name - nil-terminated string identifier for this data array
   *   attrib - attribute for data array (includes setting len and max as
   *     well as type and access)
   *   buf - points to memory buffer (should be non-NULL)
   *
   * This lets engine setup data array with a static or preallocated buffer.
   * attrib.max is fixed (i.e. MD_SETMAX and MD_ESETMAX flags are zeroed).
   * The "name" must persist until engine cleanup.
   *
   * Return value is pointer to the MD_Engdata object held by MDAPI layer
   * or NULL on error.
   *
   * Error states:
   *   MD_ERR_NEWDATA - cannot create new engine data (due to incorrect
   *     attributes or name conflict with another engine data array)
   *   MD_ERR_MEMALLOC - memory cannot be allocated
   *
   * Example:  Engine permits access to a scalar quantity that has
   * fixed attrib.len == attrib.max == 1.
   *
   * Make sure that the static buffers accessed through MD_Engdata do not
   * overlap or reference each other.  Although it would work whenever
   * the front end and engine share the same memory address space,
   * it can produce unpredictable results whenever the front end and
   * engine are communicating remotely.
   */

  MD_Engdata *MD_engdata_manage(MD_Front *, const char *name,
      MD_Attrib, void *buf, void *(*realloc)(void *ptr, size_t size));
  /*
   * Create an MD_Engdata (engine data array) with buffer spaced managed
   * by engine.  Engine must provide its own "realloc" routine that is
   * required to have same semantics as C library realloc().
   *
   *   name - nil-terminated string identifier for this data array
   *   attrib - attribute for data array (includes setting len and max as
   *     well as type and access)
   *   buf - points to memory buffer (could be NULL)
   *   realloc - routine that will be called by MDAPI to manage resizing
   *     for this buffer (must have semantics identical with realloc()
   *     from C standard library)
   *
   * This is the (rare) case in which the engine needs to control the method
   * of memory management or needs to perform additional work related to the
   * management of memory.  The "name" should persist until engine cleanup.
   * The engine_done() cleanup routine might want to call MD_free_data() to
   * force data array buffer memory to be freed earlier, in the event that
   * the supplied "realloc" routine depends on the engine state.
   *
   * Return value is pointer to the MD_Engdata object held by MDAPI layer
   * or NULL on error.
   *
   * Error states:
   *   MD_ERR_NEWDATA - cannot create new engine data (due to incorrect
   *     attributes or name conflict with another engine data array)
   *   MD_ERR_MEMALLOC - memory cannot be allocated
   *
   * Example:  When computing shadow Hamiltonian, engine keeps an array of
   * previously computed position and velocity buffers needed for centered
   * differencing.  Doing this allows engine to switch the MD_Engdata buf
   * pointer to point to any of the buffers.  Then when memory is freed,
   * MDAPI calls the same memory management routine to free the "current"
   * buffer as does the engine to free all of the other buffers.
   */


/*** define alternate name for engine data buffer ***/

  int32 MD_engdata_alias(MD_Front *, const char *name, MD_Engdata *e);
  /*
   * Setup an alternate name for an engine data array.
   *
   *   name - nil-terminated string identifier to alias this MD_Engdata
   *   e - pointer to valid MD_Engdata object returned from one of
   *     MD_engdata(), MD_engdata_buffer(), MD_engdata_manage()
   *
   * Can establish a different name (alias) for an MD_Engdata item.
   * This new name can be used through front end API routines MD_idnum(),
   * MD_name(), and MD_namelist(), the same as the original name.
   *
   * Returns 0 on success or MD_FAIL on error.
   *
   * Error states:
   *   MD_ERR_NEWDATA - cannot create alias (due to name conflict, string
   *     name has already been used for some other engine data array)
   *   MD_ERR_MEMALLOC - memory cannot be allocated
   *
   * Example:  NAMD has its own names for engine parameters.  If these
   * conflict with MDAPI defined naming conventions, use both names to
   * avoid breaking compatibility.
   */


/*** define a new type ***/

  int32 MD_new_type(MD_Front *, const char *name,
      const MD_Member *member, int32 memberlen, int32 nbytes);
  /*
   * Engine-defined data type.
   *
   *   name - nil-terminated string identifier for this type
   *   member - array of MD_Member
   *   memberlen - length of array
   *   nbytes - size of the new type in bytes (use sizeof on the struct)
   *
   * The MDAPI layer internally calls this routine on all of its predefined
   * types from mdtypes.h before invoking engine_init().  The "member" array
   * describes each member field of the new type (i.e. C struct), where
   * MD_Member is described in mdcommon.h.  The "name" string and the
   * "member" array and elements must persist until engine cleanup.
   *
   * A new type can be constructed using member elements of any previously
   * defined type.  The only caveat is that members must be positioned so
   * that 4-byte numeric quantities (i.e. int32, float, MD_Fvec) start on
   * 4-byte boundaries and 8-byte numeric quantities (i.e. double, MD_Dvec)
   * start on 8-byte boundaries, counting from 0.
   *
   * Return value is the new (nonzero) type number on success or
   * MD_FAIL on error.
   *
   * Error states:
   *   MD_ERR_NEWTYPE - cannot create new type (due to invalid type numbers
   *     in member list, misaligned numeric types, or disagreement between 
   *     nbytes and internally computed size)
   *   MD_ERR_MEMALLOC - memory cannot be allocated
   *
   * Example:
   *
   * typedef struct MD_BondPrm_tag {
   *   double k;
   *   double r0;
   *   MD_Name type[2];
   * } MD_BondPrm;
   *
   * const MD_Member bondprm_member[] =
   *   { {MD_DOUBLE, 1, "k"}, {MD_DOUBLE, 1, "r0" }, {MD_NAME, 2, "type"} };
   *
   * bondprm_type = MD_new_type(frnt, bondprm_member,
   *   sizeof(bondprm_member)/sizeof(bondprm_member[0]), sizeof(MD_BondPrm));
   * if (bondprm_type != MD_BONDPRM) return FAIL;
   *
   * After doing this, MD_BondPrm can now be used for member of new type
   * definitions.
   *
   * So why create a new type?  Two recommended types are "Param" and
   * "Result".  The members of Param would include engine parameter values
   * that would be read from a NAMD-style "config" file and provided to
   * the front end with MD_READ and MD_WRITE access (the front end might
   * want to know parameter defaults).  The members of Result would be
   * quantities computed by the engine, provided to the front end with
   * MD_READ access.
   *
   * Example:
   *
   * struct Param {
   *   double timestep;     // time step
   *   double dielectric;   // dielectric constant
   *   double cutoff;       // cutoff for van der Waals (possibly for elec)
   *   double switchdist;   // switching distance for van der Waals
   *   double temperature;  // initial temperature
   *   ...  // etc.
   * };
   *
   * struct Result {
   *   double energy;  // total energy
   *   double ke;      // kinetic energy
   *   double pe;      // potential energy
   *   double temp;    // temperature
   *   double bond;    // bond potential
   *   double angle;   // angle potential
   *   ...  // etc.
   * };
   *
   * The advantage of having a new type is that communication between
   * the front end and engine would handle a larger quantity as a block
   * to improve efficiency.  For instance, imagine a callback that
   * monitors several "result" quantities.  If a single Result type is
   * defined, then only one MD_Cbdata structure is needed to refer to the
   * entire collection of quantities, all contiguous in memory, rather
   * than requesting each quantity individually.
   */


/*** define a new error ***/

  int32 MD_new_error(MD_Front *, const char *errmsg, int32 isfatal);
  /*
   * Engine-defined error condition.
   *
   *   errmsg - nil-terminated string of error description
   *   isfatal - zero indicates not fatal, nonzero indicates fatal
   *
   * This allows the engine to define its own errors.  Fatal errors are
   * usually (but not necessarily) used for system-related errors that
   * appear to be unrecoverable.  Error values are used in calls to
   * MD_error().  The errmsg string must persist until engine cleanup.
   *
   * Return new (nonzero) error number on success or MD_FAIL on error.
   *
   * Error states:
   *   MD_ERR_MEMALLOC - memory cannot be allocated
   *
   * Examples:
   *
   * too_hot = MD_new_error(frnt, "system temperature too hot", 0);
   * unstable = MD_new_error(frnt, "simulation is unstable", 1);
   *
   * ...
   *
   * return MD_error(frnt, unstable);
   */


/*** setup the run routine ***/

  int32 MD_setup_run(MD_Front *,
      int32 (*run)(MD_Front *front, int32 numsteps, int32 flags));
  /*
   * Setup the engine run routine.
   *
   *   run - pointer to run routine to be called by MD_run()
   *     The run routine will receive the following arguments:
   *       front - pointer to the MD_Front (MDAPI layer) object
   *       numsteps - number of time steps to integrate system
   *       flags - status flags
   *
   * The engine-supplied run routine is the entry point into the
   * compute-intensive portion of the MD code that is responsible for
   * evaluating the force field and propagating the molecular system
   * for the indicated number of steps.
   *
   * The "flags" argument is a bit field status indicator.  The MD_UPDATE
   * flag will be set if any array with MD_NOTIFY access has been modified
   * by the front end.  The engine should take appropriate action on any
   * modified arrays, then call MD_engdata_ackmod() for each such array.
   * The low order bits (i.e. 0x0000FFFF) of "flags" are reserved to the
   * engine to define its own status flags.
   *
   * Returns 0 (always succeeds).
   *
   * The run routine is responsible for initiating the processing of
   * callbacks.  Message callbacks are solely for the benefit of the
   * engine as a way to communicate text messages back to the front end.
   * Force callbacks (if any) must be processed whenever the force is
   * evaluated.  Standard callbacks (if any) are processed after the
   * completion of each step and also once before taking any steps.
   * Any data array buffers that were granted either standard or force
   * callback access might be involved in callback processing, so the
   * engine should not modify these buffers while callback processing
   * is active.
   *
   * If the force is stale when the run routine is invoked, the run
   * routine should first evaluate the force (which might involve a
   * force callback).
   *
   * After the completion of each step, before processing standard
   * callbacks, the run routine should call MD_incrstep() to update the
   * internal step counter that the MDAPI layer keeps.  This value is
   * available through MD_stepnum().
   */


/*****************************************************************************
 * calls during run routine
 *
 * Engine run routine prototype (routine to be called by MD_run()):
 * int32 engine_run(MD_Front *front, int32 numsteps, int32 flags);
 *
 * (Arguments passed to run routine were defined above.)
 *****************************************************************************/

/*** get data pointer to private engine data structure ***/

  void *MD_engine_data(MD_Front *);
  /*
   * Returns the pointer to the private engine data structure.
   * This was initially set by the call to MD_setup_engine().
   */


/*** step number counter ***/

  int32 MD_incrstep(MD_Front *, int32 delta);
  /*
   * Increments the internal step counter kept by MDAPI by amount delta.
   * Returns 0 (always succeeds).
   */

  int32 MD_stepnum(MD_Front *);
  /*
   * Returns the current step number counter value.
   */


/*** engine controls for resizing data array ***/

  int32 MD_engdata_setlen(MD_Front *, MD_Engdata *e, int32 newlen);
  int32 MD_engdata_setmax(MD_Front *, MD_Engdata *e, int32 newmax);
  int32 MD_engdata_resize(MD_Front *, MD_Engdata *e, int32 newlen,
      int32 newmax);
  /*
   * These calls set attributes for an engine data array buffer.
   * They are almost identical to the front end routines to resize
   * data arrays, except for the arguments.
   *
   * MD_engdata_setlen() sets the length of the array in use,
   * MD_engdata_setmax() sets the maximum number of allocated array
   * elements (greater than or equal to length), and MD_engdata_resize()
   * sets both.  These routines call the MD_Engdata "realloc" routine
   * for data buffer memory management.
   *
   *   e - pointer to valid MD_Engdata object returned from one of
   *     MD_engdata(), MD_engdata_buffer(), MD_engdata_manage()
   *   newlen - new length value
   *   newmax - new maximum array allocation value
   *   (should have 0 <= newlen <= newmax)
   *
   * These calls are permitted if attrib.access MD_ESETLEN and/or MD_ESETMAX
   * is set.  If newlen > attrib.max and MD_ESETMAX is set, then buffer
   * allocation is extended to attrib.max = newlen.  If newmax < attrib.len
   * and MD_ESETLEN is set, then buffer allocation is truncated with
   * attrib.len = newmax.
   *
   * Returns 0 on success or MD_FAIL on error.
   *
   * Error states:
   *   MD_ERR_ACCESS - access MD_ESETLEN and/or MD_ESETMAX not permitted
   *   MD_ERR_MEMALLOC - memory cannot be allocated
   *   MD_ERR_RANGE - newlen < 0 or newmax < 0 or newmax < newlen or access
   *     permissions do not allow newlen > attrib.max or newmax < attrib.len
   */


/*** working with access flags ***/

  int32 MD_engdata_ackmod(MD_Front *, MD_Engdata *e);
  /*
   * Acknowledge the front end modification of an engine data array.
   *
   *   e - pointer to valid MD_Engdata object returned from one of
   *     MD_engdata(), MD_engdata_buffer(), MD_engdata_manage()
   *
   * Clears the MD_MODIFY access flag, acknowledging that the front end
   * modified the data array.  The engine should acknowledge all such
   * MD_Engdata objects for which it had set MD_NOTIFY access.
   *
   * Returns 0 on success or MD_FAIL on error.
   *
   * Error states:
   *   MD_ERR_ACCESS - flag MD_MODIFY not set on this MD_Engdata object
   */


/*** reset all callbacks relying on an engine data buffer ***/

  int32 MD_engdata_cbsetup(MD_Front *, MD_Engdata *e);
  /*
   * This routine is normally called internally to signify that either
   * the MD_Engdata "buf" pointer has been changed or that the "attrib"
   * values (either attrib.len or attrib.max) have been changed.  The
   * purpose is to set some internal status flags for callback processing
   * (both standard and force callbacks) that indicate whether or not
   * the MDAPI can shortcut the setup process before invoking a registered
   * callback routine.
   *
   *   e - pointer to valid MD_Engdata object returned from one of
   *     MD_engdata(), MD_engdata_buffer(), MD_engdata_manage()
   *
   * The engine should call this on an MD_Engdata object after any
   * explicit manipulation of its "buf" or "attrib" values.  This
   * routine will succeed even if the MD_Engdata object is not involved
   * with any callbacks.
   *
   * Returns 0 (always succeeds).
   */


/*** perform callbacks ***/

  /*
   * Callback processing has non-blocking calling semantics and allows
   * processing of all three types of callbacks simultaneously.  The
   * only requirement is that a new set of callbacks of a given type
   * cannot be processed until the first set of callbacks of that type
   * has completed.
   *
   * Also keep in mind that data arrays being communicated through a
   * callback should not be altered by the engine until after the
   * processing of callbacks is complete.
   */

  int32 MD_ready_callback(MD_Front *);
  int32 MD_exec_callback(MD_Front *);
  int32 MD_test_callback(MD_Front *);
  int32 MD_wait_callback(MD_Front *);
  /*
   * These initiate the MDAPI layer processing of standard callbacks.
   * Standard callback processing should be performed after each
   * increment of the step counter (call to MD_incrstep()), where the
   * front end has specified the step number increment for each callback
   * that has been registered (e.g. invoke every 10 steps).
   *
   *   _ready_ - Returns nonzero if a callback is to be processed for
   *     current step number count.  This enables engine to perform any
   *     necessary internal configuration before callbacks are processed.
   *   _exec_ - Process the callbacks.
   *   _test_, _wait_ - For non-blocking calling semantics.  _test_
   *     returns immediately:  nonzero indicates callback processing is
   *     finished.  _wait_ is a synchronizing call that does not return
   *     until the callback processing is complete.
   *
   * _exec_ and _wait_ return 0 on success.  A return value MD_FAIL from
   * any of _exec_, _test_, or _wait_ indicates an error due to callback
   * processing.  Engine should immediately from within its run routine
   * return MD_FAIL back to MDAPI layer.
   *
   * Error states (from _exec_):
   *   MD_ERR_CALLBACK - callback routine returned nonzero
   *   MD_ERR_CBSHARE - callback shared buffer failed (access does not
   *     permit resizing of target engine data array buffer)
   *   MD_ERR_CHECK - consistency check failed, indicates bug in engine
   *     (callback not processed at expected step number)
   *   MD_ERR_MEMALLOC - memory cannot be allocated (resizing data arrays
   *     from callback shared buffer)
   */

  int32 MD_ready_fcallback(MD_Front *);
  int32 MD_exec_fcallback(MD_Front *, double timestepfrac);
  int32 MD_test_fcallback(MD_Front *);
  int32 MD_wait_fcallback(MD_Front *);
  /*
   * These initiate the MDAPI layer processing of force callbacks.
   * Force callback processing should be performed for every force
   * evaluation (even if each integration step requires multiple
   * force evaluations).
   *
   *   _ready_ - Returns nonzero if there are any force callbacks to be
   *     processed.  Note that it is only necessary to call this once
   *     during the run routine.
   *   _exec_, _test_, _wait_ - As defined above.
   *     Argument to _exec_:
   *       timestepfrac - Define the time t value corresponding to the
   *         position approximation, where timestepfrac is the (fractional)
   *         adjustment to the step number counter.  Note that the value(s)
   *         of timestepfrac depend(s) on the integrator being used by the
   *         engine (e.g. for leapfrog/Verlet integrator, timestepfrac==1.0
   *         since the force is evaluated using the updated positions, but
   *         the step number counter has not yet been incremented).
   * 
   * _exec_ and _wait_ return 0 on success.  A return value MD_FAIL from
   * any of _exec_, _test_, or _wait_ indicates an error due to callback
   * processing.  Engine should immediately from within its run routine
   * return MD_FAIL back to MDAPI layer.
   *
   * Error states (from _exec_):
   *   MD_ERR_CALLBACK - callback routine returned nonzero
   *   MD_ERR_CBSHARE - callback shared buffer failed (access does not
   *     permit resizing of target engine data array buffer)
   *   MD_ERR_MEMALLOC - memory cannot be allocated (resizing data arrays
   *     from callback shared buffer)
   */

  int32 MD_ready_msgcallback(MD_Front *);
  int32 MD_exec_msgcallback(MD_Front *, const char *msg);
  int32 MD_test_msgcallback(MD_Front *);
  int32 MD_wait_msgcallback(MD_Front *);
  /*
   * These initiate a message callback.  View this as a notification
   * system in which the engine can send status messages to the front end,
   * perhaps to be printed on the screen or saved in a log file.
   *
   *   _ready_ - Returns nonzero if there are any message callbacks.
   *   _exec_, _test_, _wait_ - As defined above.
   *
   * _exec_ and _wait_ return 0 on success.  A return value MD_FAIL from
   * any of _exec_, _test_, or _wait_ indicates an error due to callback
   * processing.  Engine should immediately from within its run routine
   * return MD_FAIL back to MDAPI layer.
   *
   * Error states (from _exec_):
   *   MD_ERR_CALLBACK - callback routine returned nonzero
   */


/*** obtain callback data requirements for front end ***/

  const MD_Callback **MD_callback_list(MD_Front *, int32 *listlen);
  const MD_Callback **MD_fcallback_list(MD_Front *, int32 *listlen);
  /*
   * Obtain the lists of MD_Callback for the standard and force callbacks
   * registered by the front end.
   *
   *   listlen - Points to an integer variable to receive list length.
   *
   *   _calback_ - Provides list of standard callbacks.
   *   _fcallback_ - Provides list of force callbacks.
   *
   * The MD_Callback structure is defined above.  The "cbarg" field is an
   * array of MD_Cbdata that specifies exactly what portions of engine
   * data arrays are needed by the callbacks.  Also, "stepincr" indicates,
   * for standard callbacks, how often the particular engine data arrays
   * will be accessed by the front end.
   *
   * These routines always succeed, with NULL returned to indicate an
   * empty list.
   */


/*** report that an error (exception) has occurred ***/

  int32 MD_error(MD_Front *, int32 errnum);
  /*
   * Report an error, setting the error state number to errnum.
   *
   *   errnum - Either predefined error numbers from mdcommon.h or
   *     error numbers defined from call to MD_new_error().
   *
   * Return value is always MD_FAIL, so it can be used as the return from
   * a failed engine function.
   *
   * Example:  return MD_error(frnt, MD_ERR_MEMALLOC);
   *
   * Note that in the case of cascading failures, MD_error() should be
   * called only once before control is returned back to MDAPI layer.
   * The MDAPI layer also uses this routine for error reporting.
   */

  int32 MD_errnum(MD_Front *);
  /*
   * Returns the (nonzero) error number if an error condition has been
   * set or 0 if there is no error.
   */

  const char *MD_errmsg(MD_Front *);
  /*
   * Return the error message description (nil-terminated string) for
   * the current error condition.
   */


/*****************************************************************************
 * calls during cleanup routine
 *
 * Engine cleanup routine prototype (called by MD_done()):
 *   void engine_done(MD_Front *);
 *
 * Main responsibility is for engine to free memory that it allocated.
 * The only exception is with MD_Engdata created by MD_engdata_manage(),
 * which might have been initially allocated by engine but will be freed
 * by MDAPI.  Typical behavior is for MDAPI layer to first call engine
 * cleanup routine, then to free all of its data allocations.
 * (See below how to override this behavior.)
 *
 * The last thing (probably) that the cleanup routine does is free the
 * engine's own private data structure, essentially wiping its state.
 *****************************************************************************/

  void MD_free_data(MD_Front *);
  /*
   * Calling this (from the engine_done() routine) will force MDAPI to
   * immediately free the MD_Engdata array buffer memory allocations
   * that it is managing.  Otherwise, the MD_Engdata buffers are freed
   * after the engine_done() routine has completed.
   *
   * Example:  Suppose that memory management (say for the "realloc()"
   * routine passed to MD_engdata_manage()) depends on engine state.
   * Any such data buffers would need to be freed before the cleanup
   * routine wipes the engine state.
   */


#ifdef __cplusplus
}
#endif

#endif  /* MDENGINE_H */
