/*
 * Copyright (C) 2003-2004 by David J. Hardy.  All rights reserved.
 *
 * demo_mdapi.c - lightly test MDAPI for debugging single threaded version
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "mdapi/mdfront.h"
#include "mdapi/mdengine.h"
#include "debug/debug.h"


enum { FALSE, TRUE };


/******************************************************************************
 *
 * engine
 *
 ******************************************************************************/

#define NELEMS(a)  (sizeof(a)/sizeof(a[0]))

/* new type */
struct snake {
  char m;
  char a[3];
  int32 n;
  MD_Atom adam;
  double apple[2];
};

/* member list for snake type */
const MD_Member snake_members[] = {
  { MD_CHAR, 1, "m" },
  { MD_CHAR, 3, "a" },
  { MD_INT32, 1, "n" },
  { MD_ATOM, 1, "adam" },
  { MD_DOUBLE, 2, "apple" },
};

/* engine provided realloc function */
void *my_realloc(void *p, size_t sz)
{
  printf("*** realloc size of %u bytes ***\n", (unsigned int) sz);
  return realloc(p, sz);
}

/* engine data */
enum { N = 5 };
typedef struct TestEng_tag {
  int32 err_bad;
  int32 err_fatal;
  int32 snake_type;
  double banana_boat[N];
  MD_Engdata *apple, *banana, *grape, *orange, *pear;
  int32 init;
} TestEng;

/* prototype */
int32 eng_run(MD_Front *s, int32 numsteps, int32 flags);


int32 eng_init(MD_Front *s, int32 flags)
{
  TestEng *e;
  MD_Attrib attrib;
  int32 access;

  /* allocate memory for engine */
  e = (TestEng *) calloc(1, sizeof(TestEng));
  if (e == NULL) {
    ERRCOND(e == NULL);
    return MD_FAIL;
  }

  /* call setup routine */
  if (MD_setup_engine(s, e)) {
    free(e);
    return MD_FAIL;
  }

  /* setup two errors */
  e->err_bad = MD_new_error(s, "not too bad", 0);
  if (e->err_bad == MD_FAIL) {
    fprintf(stderr, "MD_new_error(): %s\n", MD_errmsg(s));
    return MD_FAIL;
  }
  e->err_fatal = MD_new_error(s, "very bad", -1);
  if (e->err_fatal == MD_FAIL) {
    fprintf(stderr, "MD_new_error(): %s\n", MD_errmsg(s));
    return MD_FAIL;
  }

  /* create a new type */
  e->snake_type = MD_new_type(s, "snake", snake_members,
      NELEMS(snake_members), sizeof(struct snake));
  if (e->snake_type == MD_FAIL) {
    fprintf(stderr, "MD_new_type(): %s\n", MD_errmsg(s));
    return MD_FAIL;
  }

  /* create data arrays */
  memset(e->banana_boat, 0, N * MD_SIZEOF(MD_DOUBLE));
  access = MD_READ | MD_WRITE | MD_RESIZE | MD_DIRECT | MD_NOTIFY
    | MD_CBREAD | MD_CBWRITE | MD_CBSHARE
    | MD_FCBREAD | MD_FCBWRITE | MD_FCBSHARE;
  e->apple = MD_engdata(s, "apple", MD_INT32, access);
  if (e->apple == NULL) {
    fprintf(stderr, "MD_engdata(): %s\n", MD_errmsg(s));
    return MD_FAIL;
  }
  attrib.type = MD_DOUBLE;
  attrib.len = 0;
  attrib.max = N;
  attrib.access = access;
  e->banana = MD_engdata_buffer(s, "banana", attrib, e->banana_boat);
  if (e->banana == NULL) {
    fprintf(stderr, "MD_engdata_buffer(): %s\n", MD_errmsg(s));
    return MD_FAIL;
  }
  attrib.type = MD_ATOM;
  attrib.len = 0;
  attrib.max = 0;
  attrib.access = access;
  e->grape = MD_engdata_manage(s, "grape", attrib, NULL, my_realloc);
  if (e->grape == NULL) {
    fprintf(stderr, "MD_engdata_manage(): %s\n", MD_errmsg(s));
    return MD_FAIL;
  }
  access = MD_READ | MD_WRITE | MD_ERESIZE | MD_DIRECT;
  e->orange = MD_engdata(s, "orange", MD_DVEC, access);
  if (e->orange == NULL) {
    fprintf(stderr, "MD_engdata(): %s\n", MD_errmsg(s));
    return MD_FAIL;
  }
  access = MD_READ | MD_WRITE | MD_RESIZE | MD_DIRECT | MD_NOTIFY
    | MD_CBREAD | MD_CBWRITE | MD_CBSHARE
    | MD_FCBREAD | MD_FCBWRITE | MD_FCBSHARE;
  e->pear = MD_engdata(s, "pear", MD_INT32, access);
  if (e->pear == NULL) {
    fprintf(stderr, "MD_engdata(): %s\n", MD_errmsg(s));
    return MD_FAIL;
  }
  /* alias */
  if (MD_engdata_alias(s, "fuji", e->apple)) {
    fprintf(stderr, "MD_engdata_alias(): %s\n", MD_errmsg(s));
    return MD_FAIL;
  }
  if (MD_engdata_alias(s, "clementine", e->orange)) {
    fprintf(stderr, "MD_engdata_alias(): %s\n", MD_errmsg(s));
    return MD_FAIL;
  }
  if (MD_engdata_alias(s, "jonathan", e->apple)) {
    fprintf(stderr, "MD_engdata_alias(): %s\n", MD_errmsg(s));
    return MD_FAIL;
  }
  /*
   * attempt to reuse a type name causes abort
   *
  if (MD_engdata_alias(s, "grape", e->apple)) {
    fprintf(stderr, "MD_engdata_alias(): %s\n", MD_errmsg(s));
    return MD_FAIL;
  }
   */

  /* setup run */
  if (MD_setup_run(s, eng_run)) {
    fprintf(stderr, "MD_setup_run(): %s\n", MD_errmsg(s));
    return MD_FAIL;
  }

  return 0;
}


void eng_done(MD_Front *s)
{
  TestEng *e;

  e = (TestEng *) MD_engine_data(s);
  free(e);
}


int32 eng_run(MD_Front *s, int32 numsteps, int32 flags)
{
  int32 ismsgcb = FALSE;
  int32 isfcb = FALSE;
  TestEng *e = (TestEng *) MD_engine_data(s);
  int32 k;

  /* can we send messages to front end? */
  if (MD_ready_msgcallback(s)) {
    ismsgcb = TRUE;
  }

  /* are there any force callbacks? */
  if (MD_ready_fcallback(s)) {
    isfcb = TRUE;
  }

  /* do we need to initialize? */
  if ((flags & MD_UPDATE) || !(e->init)) {
    /* do whatever it is with each updated array, then clear its update flag */
    if (e->apple->attrib.access & MD_MODIFY) {
      MD_engdata_ackmod(s, e->apple);
      if (MD_exec_msgcallback(s, "acknowledge modified apple")
          || MD_wait_msgcallback(s)) return MD_FAIL;
    }
    if (e->banana->attrib.access & MD_MODIFY) {
      MD_engdata_ackmod(s, e->banana);
      if (MD_exec_msgcallback(s, "acknowledge modified banana")
          || MD_wait_msgcallback(s)) return MD_FAIL;
    }
    if (e->grape->attrib.access & MD_MODIFY) {
      MD_engdata_ackmod(s, e->grape);
      if (MD_exec_msgcallback(s, "acknowledge modified grape")
          || MD_wait_msgcallback(s)) return MD_FAIL;
    }
    if (e->orange->attrib.access & MD_MODIFY) {
      MD_engdata_ackmod(s, e->orange);
      if (MD_exec_msgcallback(s, "acknowledge modified orange")
          || MD_wait_msgcallback(s)) return MD_FAIL;
    }
    /* compute initial forces */
    if (isfcb) {
      if (MD_exec_fcallback(s, 0.0) || MD_wait_fcallback(s)) return MD_FAIL;
    }
    if (ismsgcb) {
      if (MD_exec_msgcallback(s, "initialized")
          || MD_wait_msgcallback(s)) return MD_FAIL;
    }
    /* set internal flag */
    e->init = TRUE;
  }

  /* see if we need to process callbacks before loop */
  if (MD_ready_callback(s)) {
    if (MD_exec_callback(s) || MD_wait_callback(s)) {
      return MD_FAIL;
    }
  }

  /* integration loop (leapfrog) */
  while (numsteps-- > 0) {
    /* half kick */
    /* drift */
    /* compute new force */
    if (isfcb) {
      if (MD_exec_fcallback(s, 1.0) || MD_wait_fcallback(s)) return MD_FAIL;
    }
    /* half kick */
    /* increment step number */
    MD_incrstep(s, 1);
    if (ismsgcb) {
      if (MD_exec_msgcallback(s, "finished step")
          || MD_wait_msgcallback(s)) return MD_FAIL;
    }
    /* see if we need to process callbacks */
    if (MD_ready_callback(s)) {
      if (MD_exec_callback(s) || MD_wait_callback(s)) {
        return MD_FAIL;
      }
    }
    /* print out pear */
    printf("pear:");
    for (k = 0;  k < e->pear->attrib.len;  k++) {
      printf("%5d", ((int32 *)(e->pear->buf))[k]);
    }
    printf("\n");
    /* if we don't acknowledge MODIFY, then since the NOTIFY flag is */
    /* also set on "pear", the global UPDATE flag is not cleared */
    if (e->pear->attrib.access & MD_MODIFY) {
      MD_engdata_ackmod(s, e->pear);
      if (MD_exec_msgcallback(s, "acknowledge modified pear")
          || MD_wait_msgcallback(s)) return MD_FAIL;
    }
  }

  /* all done message */
  if (ismsgcb) {
    if (MD_exec_msgcallback(s, "done with run")
        || MD_wait_msgcallback(s)) return MD_FAIL;
  }
  return 0;
}


/******************************************************************************
 *
 * front end
 *
 ******************************************************************************/

/* message callback routines */

int32 front_msgcb(void *info, const char *msg, int32 stepnum)
{
  printf("msg(stepnum=%d) >>> %s\n", stepnum, msg);
  return 0;
}


int32 front_alt_msgcb(void *info, const char *msg, int32 stepnum)
{
  /* for every other call, end line with two newlines */
  int32 *pn = (int32 *)info;
  printf("altmsg(stepnum=%d) >>> %s%s\n", stepnum, msg, (*pn ? "\n" : ""));
  *pn ^= 0x01;
  return 0;
}


/* force callback routines */

int32 front_fcb_a(void *info, MD_Cbdata *c, int32 clen, int32 n, double dt)
{
  double *t = (double *) info;

  ASSERT(t != NULL);
  *t = n + dt;
  ASSERT(c == NULL);
  ASSERT(clen == 0);
  printf("\n   FCB: apple:");
  return 0;
}


int32 front_fcb_b(void *info, MD_Cbdata *c, int32 clen, int32 n, double dt)
{
  double *t = (double *) info;
  int32 *p;
  int32 k;

  ASSERT(t != NULL);
  if (*t != n + dt) {
    BUG("wrong stepnum in front_fcb_b\n");
  }
  ASSERT(c != NULL);
  ASSERT(clen == 1);
  ASSERT(c[0].data != NULL);
  ASSERT(c[0].attrib.type == MD_INT32);
  p = (int32 *) c[0].data;
  for (k = 0;  k < c[0].nelems;  k++) {
    printf("%7d", p[k]);
  }
  return 0;
}


int32 front_fcb_c(void *info, MD_Cbdata *c, int32 clen, int32 n, double dt)
{
  double *t = (double *) info;
  int32 *p;
  double *q;
  int32 k;

  ASSERT(t != NULL);
  if (*t != n + dt) {
    BUG("wrong stepnum in front_fcb_c\n");
  }
  ASSERT(c != NULL);
  ASSERT(clen == 2);
  ASSERT(c[0].data != NULL);
  ASSERT(c[0].attrib.type == MD_INT32);
  p = (int32 *) c[0].data;
  for (k = 0;  k < c[0].nelems;  k++) {
    printf("%7d", p[k]);
  }
  printf("\n");
  ASSERT(c[1].data != NULL);
  ASSERT(c[1].attrib.type == MD_DOUBLE);
  ASSERT(c[1].nelems == -1);
  printf("   FCB: banana:");
  q = (double *) c[1].data;
  for (k = 0;  k < c[1].attrib.len;  k++) {
    /* increment before printing */
    q[k]++;
    printf("%7g", q[k]);
  }
  printf("\n");
  return 0;
}


/* standard callback routines */

int32 front_cb_a(void *info, MD_Cbdata *c, int32 clen, int32 n)
{
  int32 *step = (int32 *) info;
  int32 *p;
  int32 k;

  /* should be called every other step */
  ASSERT(step != NULL);
  if (*step != n) {
    BUG("wrong stepnum in front_cb_a\n");
  }
  ASSERT(c != NULL);
  ASSERT(clen == 1);
  ASSERT(c[0].data != NULL);
  ASSERT(c[0].attrib.type == MD_INT32);
  p = (int32 *) c[0].data;
  printf("   CB: apple:");
  /* pass first portion of apple array, increment and print */
  for (k = 0;  k < c[0].nelems;  k++) {
    /* increment before printing */
    p[k]++;
    printf("%7d", p[k]);
  }
  return 0;
}


int32 front_cb_b(void *info, MD_Cbdata *c, int32 clen, int32 n)
{
  int32 *step = (int32 *) info;
  int32 *p;
  double *q;
  int32 k;

  /* should be called every other step */
  ASSERT(step != NULL);
  if (*step != n) {
    BUG("wrong stepnum in front_cb_a\n");
  }
  /* expect to be called again in 2 more steps */
  (*step) += 2;
  ASSERT(c != NULL);
  ASSERT(clen == 2);
  ASSERT(c[0].data != NULL);
  ASSERT(c[0].attrib.type == MD_INT32);
  p = (int32 *) c[0].data;
  /* pass remainder of apple array, unbounded access (nelems==-1) */
  for (k = 0;  k < c[0].attrib.len - c[0].first;  k++) {
    /* just print */
    printf("%7d", p[k]);
  }
  printf("\n");
  ASSERT(c[1].data != NULL);
  ASSERT(c[1].attrib.type == MD_DOUBLE);
  q = (double *) c[1].data;
  printf("   CB: banana (middle):");
  for (k = 0;  k < c[1].nelems;  k++) {
    /* just print */
    printf("%7g", q[k]);
  }
  printf("\n");
  return 0;
}

/* other callback routines are simple, just print out message */
typedef struct FrontInfo_tag {
  int32 stepincr, cbid;
} FrontInfo;


int32 front_cb_c(void *info, MD_Cbdata *c, int32 clen, int32 n)
{
  int32 stepincr, cbid;

  ASSERT(info != NULL);
  ASSERT(c == NULL);
  ASSERT(clen == 0);
  stepincr = ((FrontInfo *) info)->stepincr;
  cbid = ((FrontInfo *) info)->cbid;
  ASSERT(n % stepincr == 0);
  printf("   CB: callback ID #%d, step increment %d\n", cbid, stepincr);
  return 0;
}


int32 front_cb_d(void *info, MD_Cbdata *c, int32 clen, int32 n)
{
  int32 stepincr, cbid;

  ASSERT(info != NULL);
  stepincr = ((FrontInfo *) info)->stepincr;
  cbid = ((FrontInfo *) info)->cbid;
  ASSERT(n % stepincr == 0);
  printf("   CB: callback ID #%d, step increment %d\n", cbid, stepincr);
  return 0;
}


/* demonstrate CBSHARE access */
typedef struct CbShareInfo_tag {
  int32 *buf;
  int32 buflen;
  int32 stepnum;
} CbShareInfo;

int32 front_cb_share(void *info, MD_Cbdata *c, int32 clen, int32 n)
{
  CbShareInfo *cbshare = (CbShareInfo *) info;
  void *v;
  int32 *p;
  int32 plen, k;

  ASSERT(cbshare != NULL);
  if (cbshare->stepnum != n) {
    BUG("incorrect stepnum in front_cb_share()");
  }
  /* callback should be called every step */
  (cbshare->stepnum)++;
  ASSERT(c != NULL);
  ASSERT(clen == 2);
  ASSERT(c[0].access & MD_CBREAD);
  ASSERT(c[0].attrib.type == MD_INT32);
  ASSERT(c[1].data == NULL);  /* this will point to our data buffer */
  ASSERT(c[1].access & MD_CBSHARE);
  ASSERT(c[1].attrib.type == MD_INT32);
  p = (int32 *) c[0].data;
  plen = c[0].attrib.len;
  v = realloc(cbshare->buf, (plen+1) * sizeof(int32));
  if (v == NULL && (plen+1) > 0) {
    ERRCOND(v == NULL && (plen+1) > 0);
    fprintf(stderr, "realloc() failed, %s line %d\n", __FILE__, __LINE__);
    return MD_FAIL;
  }
  cbshare->buf = (int32 *) v;
  cbshare->buflen = plen+1;
  for (k = 0;  k < plen;  k++) {
    /* go ahead and read from engine's data to set our buffer */
    cbshare->buf[k] = p[k];
  }
  cbshare->buf[k] = cbshare->buflen;
  c[1].data = cbshare->buf;
  c[1].nelems = cbshare->buflen;
  c[1].first = 0;
  return 0;
}


int front_end(MD_Engine *s)
{
  int32 j, k;

  /* for testing type subsystem */
  const char **namelist;
  int32 listlen;
  int32 typnum;
  const char *cs;
  const MD_Member *mem;
  int32 memlen;
  MD_Member member;
  MD_Atom atom;
  void *v;
  struct snake snk;

  /* for testing data array subsystem */
  double banana_peel[N];
  MD_Atom grapefruit[N];
  int32 appleseed[N] = { 1, 2, 3, 4, 5 };
  int32 *jonny;
  int32 apple_id, banana_id, grape_id, orange_id, pear_id;
  MD_Attrib attrib;
  int32 idnum, n;

  /* for testing run and callback subsystems */
  int32 numsteps;
  int32 toggle;
  MD_Cbdata fcb_b[1];
  MD_Cbdata fcb_c[2];
  double fcb_stepnum;
  int32 cb_step;
  MD_Cbdata cb_a[1];
  MD_Cbdata cb_b[2];
  FrontInfo f1, f2, f3, f4a, f4b, f5, f6;
  MD_Cbdata cb_share[2];
  CbShareInfo cb_share_info;
  MD_Cbdata cb_d[1];

  /*
   * construct engine
   */

  if (MD_init(s, "my_engine", 0, eng_init, eng_done) || MD_wait(s)) {
    fprintf(stderr, "MD_init(): %s\n", MD_errmsg(s));
    return MD_FAIL;
  }

  /*
   * test error subsystem
   *
   * (we would never access MD_Interface internals in practice)
   */

  /* print current error state */
  printf("-- current error state --\n");
  printf("errnum = %d\n", MD_errnum(s));
  printf("errmsg = \"%s\"\n", MD_errmsg(s));
  printf("\n");

  /* use fact that error numbers are sequential */
  printf("-- table of errors --\n\n");
  for (k = MD_ERR_NONE;  k < s->engerror.len + MD_ERROR_LISTLEN;  k++) {
    /* invoke error k */
    printf("invoke error %d (should return FAIL): %d\n", k, MD_error(s, k));
    printf("errnum = %d\n", MD_errnum(s));
    printf("errmsg = \"%s\"\n", MD_errmsg(s));
    printf("try to reset: %d\n", MD_reset(s));
    printf("\n");
  }
  /*
   * both of the following out-of-bounds attempts cause abort
   *
  printf("invoke error out-of-bounds: %d\n", MD_error(s, k));
  printf("invoke error out-of-bounds: %d\n", MD_error(s, -1));
   */

  /* manual reset */
  printf("manually reset error state: %d\n", MD_error(s, MD_ERR_NONE));
  printf("errnum = %d\n", MD_errnum(s));
  printf("errmsg = \"%s\"\n", MD_errmsg(s));
  printf("\n\n");

  /*
   * test type subsystem
   */

  /* list out all types */
  if ((namelist = MD_type_namelist(s, &listlen)) == NULL) {
    fprintf(stderr, "MD_type_namelist(): %s\n", MD_errmsg(s));
    return MD_FAIL;
  }
  printf("-- table of types --\n\n");
  printf("%15s%10s%10s%10s%10s%15s\n", "namelist", "typenum",
      "index", "flip", "size", "hashed");
  printf("\n");
  for (k = 0;  k < listlen;  k++) {
    if ((typnum = MD_type(s, namelist[k])) == MD_FAIL) {
      fprintf(stderr, "MD_type(): %s\n", MD_errmsg(s));
      return MD_FAIL;
    }
    if ((cs = MD_type_name(s, typnum)) == NULL) {
      fprintf(stderr, "MD_type_name(): %s\n", MD_errmsg(s));
      return MD_FAIL;
    }
    printf("%15s%10d%10d%10d%10d%15s\n", namelist[k], typnum,
        typnum >> MD_SHIFT_TYPE, (typnum & MD_FLIP_MASK) >> MD_SHIFT_FLIP,
        MD_SIZEOF(typnum), cs);
  }
  printf("\n\n");

  /* member list for each type */
  printf("-- table of members for each type --\n\n");
  for (k = 0;  k < listlen;  k++) {
    if ((typnum = MD_type(s, namelist[k])) == MD_FAIL) {
      fprintf(stderr, "MD_type(): %s\n", MD_errmsg(s));
      return MD_FAIL;
    }
    printf("%s:%10d (total size)\n\n", namelist[k], MD_SIZEOF(typnum));
    if ((mem = MD_type_memberlist(s, typnum, &memlen)) == NULL) {
      fprintf(stderr, "MD_type_memberlist(): %s\n", MD_errmsg(s));
      return MD_FAIL;
    }
    for (j = 0;  j < memlen;  j++) {
      if ((cs = MD_type_name(s, mem[j].type)) == NULL) {
        fprintf(stderr, "MD_type_name(): %s\n", MD_errmsg(s));
        return MD_FAIL;
      }
      printf("%20s:%15s     %5d (size)     %5d (len)\n",
          mem[j].name, cs, MD_SIZEOF(mem[j].type), mem[j].len);
    }
    printf("\n\n");
  }

  /* attempt to access MD_Atom members by name */
  printf("-- access MD_Atom members by name --\n\n");
  if ((typnum = MD_type(s, "MD_Atom")) == MD_FAIL) {
    fprintf(stderr, "MD_type(): %s\n", MD_errmsg(s));
    return MD_FAIL;
  }
  if ((v = MD_type_member(s, typnum, &atom, "m", &member)) == NULL
      || strcmp(member.name, "m") != 0) {
    fprintf(stderr, "MD_type_member(): %s\n", MD_errmsg(s));
    return MD_FAIL;
  }
  *((double *)v) = 1.25;
  if ((v = MD_type_member(s, typnum, &atom, "q", &member)) == NULL
      || strcmp(member.name, "q") != 0) {
    fprintf(stderr, "MD_type_member(): %s\n", MD_errmsg(s));
    return MD_FAIL;
  }
  *((double *)v) = 2.5;
  if ((v = MD_type_member(s, typnum, &atom, "prm", &member)) == NULL
      || strcmp(member.name, "prm") != 0) {
    fprintf(stderr, "MD_type_member(): %s\n", MD_errmsg(s));
    return MD_FAIL;
  }
  *((int32 *)v) = 5;
#if 0
  if ((v = MD_type_member(s, typnum, &atom, "notused", &member)) == NULL
      || strcmp(member.name, "notused") != 0) {
    fprintf(stderr, "MD_type_member(): %s\n", MD_errmsg(s));
    return MD_FAIL;
  }
  *((int32 *)v) = 10;
#endif
  if ((v = MD_type_member(s, typnum, &atom, "name", &member)) == NULL
      || strcmp(member.name, "name") != 0) {
    fprintf(stderr, "MD_type_member(): %s\n", MD_errmsg(s));
    return MD_FAIL;
  }
  strcpy((char *)v, "eve");
  if ((v = MD_type_member(s, typnum, &atom, "type", &member)) == NULL
      || strcmp(member.name, "type") != 0) {
    fprintf(stderr, "MD_type_member(): %s\n", MD_errmsg(s));
    return MD_FAIL;
  }
  strcpy((char *)v, "female");
  printf("atom:\n");
  printf("%15s = %g\n", "m", atom.m);
  printf("%15s = %g\n", "q", atom.q);
  printf("%15s = %d\n", "prm", atom.prm);
#if 0
  printf("%15s = %d\n", "notused", atom.notused);
#endif
  printf("%15s = %s\n", "name", atom.name);
  printf("%15s = %s\n", "type", atom.type);
  printf("\n\n");

  /* set snake type */
  printf("-- set snake and access \"adam\" by name --\n\n");
  memset(&snk, 0, sizeof(snk));
  /* (we would never access MD_Interface internals in practice) */
  if ((typnum = MD_type(s, "snake"))
      != ((TestEng *) MD_engine_data(s))->snake_type) {
    fprintf(stderr, "MD_type(): %s\n", MD_errmsg(s));
    return MD_FAIL;
  }
  snk.m = 't';
  snk.a[0] = 'e';
  snk.a[1] = 'm';
  snk.a[2] = 'p';
  snk.n = 0;
  snk.apple[0] = 666.0;
  snk.apple[1] = 777.0;
  if ((v = MD_type_member(s, typnum, &snk, "adam", &member)) == NULL
      || strcmp(member.name, "adam") != 0) {
    fprintf(stderr, "MD_type_member(): %s\n", MD_errmsg(s));
    return MD_FAIL;
  }
  *((MD_Atom *)v) = atom;
  printf("snake:\n");
  printf("%15s = %c\n", "m", snk.m);
  printf("%15s = %c%c%c\n", "a", snk.a[0], snk.a[1], snk.a[2]);
  printf("%15s = %d\n", "n", snk.n);
  printf("\n");
  printf("%15s =\n", "snake.adam");
  printf("%20s = %g\n", "m", snk.adam.m);
  printf("%20s = %g\n", "q", snk.adam.q);
  printf("%20s = %d\n", "prm", snk.adam.prm);
#if 0
  printf("%20s = %d\n", "notused", snk.adam.notused);
#endif
  printf("%20s = %s\n", "name", snk.adam.name);
  printf("%20s = %s\n", "type", snk.adam.type);
  printf("\n");
  printf("%15s =%5g%5g\n", "apple", snk.apple[0], snk.apple[1]);
  printf("\n\n");

  /*
   * test data array subsystem
   */

  if ((apple_id = MD_idnum(s, "fuji")) == MD_FAIL) {
    fprintf(stderr, "MD_idnum(): %s\n", MD_errmsg(s));
    return MD_FAIL;
  }
  if ((banana_id = MD_idnum(s, "banana")) == MD_FAIL) {
    fprintf(stderr, "MD_idnum(): %s\n", MD_errmsg(s));
    return MD_FAIL;
  }
  if ((grape_id = MD_idnum(s, "grape")) == MD_FAIL) {
    fprintf(stderr, "MD_idnum(): %s\n", MD_errmsg(s));
    return MD_FAIL;
  }
  if ((orange_id = MD_idnum(s, "orange")) == MD_FAIL) {
    fprintf(stderr, "MD_idnum(): %s\n", MD_errmsg(s));
    return MD_FAIL;
  }
  attrib = MD_attrib(s, banana_id);
  if (attrib.type == 0) {
    fprintf(stderr, "MD_attrib(): %s\n", MD_errmsg(s));
    return MD_FAIL;
  }
  else if (attrib.max != N) {
    fprintf(stderr, "attrib.max != %d\n", N);
    return MD_FAIL;
  }
  if (MD_setlen(s, banana_id, N)) {
    fprintf(stderr, "MD_setlen(): %s\n", MD_errmsg(s));
    return MD_FAIL;
  }
  if (MD_resize(s, apple_id, N, N)) {
    fprintf(stderr, "MD_resize(): %s\n", MD_errmsg(s));
    return MD_FAIL;
  }
  if (MD_setlen(s, grape_id, N)) {
    fprintf(stderr, "MD_setlen(): %s\n", MD_errmsg(s));
    return MD_FAIL;
  }
#ifndef DEBUG_SUPPORT
  /* non-fatal associated errors (when debugging is off) */
  if (MD_resize(s, orange_id, N, N)) {
    printf("error:  %s\n", MD_errmsg(s));
    MD_reset(s);
  }
  if (MD_setlen(s, banana_id, N)) {
    printf("error:  %s\n", MD_errmsg(s));
    MD_reset(s);
  }
#endif
  if (MD_setmax(s, apple_id, 0)) {
    fprintf(stderr, "MD_setmax(): %s\n", MD_errmsg(s));
    return MD_FAIL;
  }
  if (MD_share(s, apple_id, appleseed, N, N)) {
    fprintf(stderr, "MD_share(): %s\n", MD_errmsg(s));
    return MD_FAIL;
  }

  /* via engine */
  /* (we would never access MD_Interface internals in practice) */
  if (MD_engdata_setlen(s, ((TestEng *)MD_engine_data(s))->orange, N)) {
    fprintf(stderr, "MD_engdata_setlen(): %s\n", MD_errmsg(s));
    return MD_FAIL;
  }

  /* print name list and attributes */
  namelist = MD_namelist(s, &listlen);
  printf("data array name list:\n\n");
  printf("%12s%7s%12s%12s%7s%7s\n",
      "namelist", "idnum", "real name", "type", "len", "max");
  printf("%12s%7s%12s%12s%7s%7s\n",
      "--------", "-----", "---------", "----", "---", "---");
  for (k = 0;  k < listlen;  k++) {
    if ((idnum = MD_idnum(s, namelist[k])) == MD_FAIL) {
      fprintf(stderr, "MD_idnum(): %s\n", MD_errmsg(s));
      return MD_FAIL;
    }
    if ((cs = MD_name(s, idnum)) == NULL) {
      fprintf(stderr, "MD_name(): %s\n", MD_errmsg(s));
      return MD_FAIL;
    }
    printf("%12s%7d%12s", namelist[k], idnum, cs);
    if (strcmp(cs, namelist[k]) == 0) {
      attrib = MD_attrib(s, idnum);
      if (attrib.type == 0) {
        fprintf(stderr, "MD_attrib(): %s\n", MD_errmsg(s));
        return MD_FAIL;
      }
      if ((cs = MD_type_name(s, attrib.type)) == NULL) {
        fprintf(stderr, "MD_type_name(): %s\n", MD_errmsg(s));
        return MD_FAIL;
      }
      printf("%12s%7d%7d\n", cs, attrib.len, attrib.max);
    }
    else {
      printf("%26s\n", "( .... alias .... )");
    }
  }
  printf("\n\n");

#ifndef DEBUG_SUPPORT
  /* non-fatal associated errors (when debugging is off) */
  if (MD_name(s, 5) == NULL) {
    printf("error:  %s\n", MD_errmsg(s));
    MD_reset(s);
  }
  if (MD_idnum(s, "granny smith") == MD_FAIL) {
    printf("error:  %s\n", MD_errmsg(s));
    MD_reset(s);
  }
  if (MD_attrib(s, 5).type == 0) {
    printf("error:  %s\n", MD_errmsg(s));
    MD_reset(s);
  }
#endif

  /* clear grape */
  memset(grapefruit, 0, N * MD_SIZEOF(MD_ATOM));
  if (MD_write(s, grape_id, grapefruit, N) || MD_wait(s)) {
    fprintf(stderr, "MD_write(): %s\n", MD_errmsg(s));
    return MD_FAIL;
  }
  /* write to middle element */
  if (MD_writesub(s, grape_id, &(snk.adam), 1, N/2) || MD_wait(s)) {
    fprintf(stderr, "MD_writesub(): %s\n", MD_errmsg(s));
    return MD_FAIL;
  }
  /* read entire grape array */
  if (MD_read(s, grape_id, grapefruit, N) || MD_wait(s)) {
    fprintf(stderr, "MD_read(): %s\n", MD_errmsg(s));
    return MD_FAIL;
  }
  /* print grape.m for all array elements */
  printf("grape.m:");
  for (k = 0;  k < N;  k++) {
    printf("%7g", grapefruit[k].m);
  }
  printf("\n\n");

  /* direct access apple */
  if ((jonny = MD_direct(s, apple_id)) == NULL || MD_wait(s)) {
    fprintf(stderr, "MD_direct(): %s\n", MD_errmsg(s));
    return MD_FAIL;
  }
  /* set banana_peel using apple */
  n = MD_attrib(s, apple_id).len;
  if (n != MD_attrib(s, banana_id).len) {
    fprintf(stderr, "n != MD_attrib(s, banana_id).len");
    return MD_FAIL;
  }
  for (k = 0;  k < n;  k++) {
    banana_peel[k] = jonny[k] + 0.5;
  }

  /* write to banana */
  if (MD_write(s, banana_id, banana_peel, n) || MD_wait(s)) {
    fprintf(stderr, "MD_write(): %s\n", MD_errmsg(s));
    return MD_FAIL;
  }
  memset(banana_peel, 0, sizeof(banana_peel));

  /* read it back */
  if (MD_read(s, banana_id, banana_peel, n) || MD_wait(s)) {
    fprintf(stderr, "MD_read(): %s\n", MD_errmsg(s));
    return MD_FAIL;
  }
  /* print it */
  printf("banana:");
  for (k = 0;  k < n;  k++) {
    printf("%7g", banana_peel[k]);
  }
  printf("\n\n");

  /* change subarray */
  banana_peel[0] += 3.0;
  banana_peel[1] += 4.0;
  banana_peel[2] += 5.0;
  banana_peel[3] += 6.0;
  banana_peel[4] += 7.0;
  if (MD_writesub(s, banana_id, &banana_peel[2], 2, 2) || MD_wait(s)) {
    fprintf(stderr, "MD_writesub(): %s\n", MD_errmsg(s));
    return MD_FAIL;
  }
  /* read back all but first element */
  if (MD_readsub(s, banana_id, &banana_peel[1], 4, 1) || MD_wait(s)) {
    fprintf(stderr, "MD_readsub(): %s\n", MD_errmsg(s));
    return MD_FAIL;
  }
  /* print it */
  printf("banana_peel:");
  for (k = 0;  k < n;  k++) {
    printf("%7g", banana_peel[k]);
  }
  printf("\n\n");

  /* set modify */
  jonny[2] = 0;
  if (MD_setmod(s, apple_id)) {
    fprintf(stderr, "MD_setmod(): %s\n", MD_errmsg(s));
    return MD_FAIL;
  }
  /* print it */
  printf("apple:");
  for (k = 0;  k < n;  k++) {
    printf("%7d", jonny[k]);
  }
  printf("\n\n");

  /* unshare appleseed */
  if ((jonny = MD_unshare(s, apple_id)) != appleseed) {
    fprintf(stderr, "MD_unshare(): %s\n", MD_errmsg(s));
    return MD_FAIL;
  }

  /* set length of apple to 4 */
  if (MD_setlen(s, apple_id, 4)) {
    fprintf(stderr, "MD_setlen(): %s\n", MD_errmsg(s));
    return MD_FAIL;
  }
  /* write to it */
  appleseed[1] = 4;
  appleseed[2] = 3;
  appleseed[3] = 2;
  appleseed[4] = 1;
  if (MD_write(s, apple_id, appleseed + 1, 4) || MD_wait(s)) {
    fprintf(stderr, "MD_write(): %s\n", MD_errmsg(s));
    return MD_FAIL;
  }

  /* directly access it for reading */
  if ((jonny = MD_direct(s, apple_id)) == NULL || MD_wait(s)) {
    fprintf(stderr, "MD_direct(): %s\n", MD_errmsg(s));
    return MD_FAIL;
  }
  printf("apple:");
  attrib = MD_attrib(s, apple_id);
  for (k = 0;  k < attrib.len;  k++) {
    printf("%7d", jonny[k]);
  }
  printf("\n\n");

  /* reset apple and banana arrays */
  if (MD_resize(s, apple_id, N, N)) {
    fprintf(stderr, "MD_setlen(): %s\n", MD_errmsg(s));
    return MD_FAIL;
  }
  if ((jonny = MD_direct(s, apple_id)) == NULL || MD_wait(s)) {
    fprintf(stderr, "MD_direct(): %s\n", MD_errmsg(s));
    return MD_FAIL;
  }
  for (k = 0;  k < MD_attrib(s, apple_id).len;  k++) {
    jonny[k] = k + 1;
    banana_peel[k] = 0.5 + jonny[k];
  }
  if (MD_write(s, banana_id, banana_peel, N) || MD_wait(s)) {
    fprintf(stderr, "MD_write(): %s\n", MD_errmsg(s));
    return MD_FAIL;
  }
  if (MD_read(s, banana_id, banana_peel, N) || MD_wait(s)) {
    fprintf(stderr, "MD_read(): %s\n", MD_errmsg(s));
    return MD_FAIL;
  }

  /* print them both */
  printf("apple:");
  for (k = 0;  k < MD_attrib(s, apple_id).len;  k++) {
    printf("%7d", jonny[k]);
  }
  printf("\n\n");
  printf("banana:");
  for (k = 0;  k < MD_attrib(s, banana_id).len;  k++) {
    printf("%7g", banana_peel[k]);
  }
  printf("\n\n");

  /* get handle to pear, array should be empty array of int32 */
  pear_id = MD_idnum(s, "pear");
  if (pear_id == MD_FAIL) {
    fprintf(stderr, "MD_idnum(): %s\n", MD_errmsg(s));
    return MD_FAIL;
  }
  ASSERT(MD_attrib(s, pear_id).type == MD_INT32);
  ASSERT(MD_attrib(s, pear_id).len == 0);

  /*
   * test run and callback subsystems
   */

  /* setup three message callbacks */
  toggle = 0;
  if (MD_msgcallback(s, front_alt_msgcb, &toggle)) {
    fprintf(stderr, "MD_msgcallback(): %s\n", MD_errmsg(s));
    return MD_FAIL;
  }
  if (MD_msgcallback(s, front_msgcb, NULL)) {
    fprintf(stderr, "MD_msgcallback(): %s\n", MD_errmsg(s));
    return MD_FAIL;
  }
  if (MD_msgcallback(s, front_alt_msgcb, &toggle)) {
    fprintf(stderr, "MD_msgcallback(): %s\n", MD_errmsg(s));
    return MD_FAIL;
  }

  /* setup three force callbacks */

  if (MD_fcallback(s, front_fcb_a, &fcb_stepnum, NULL, 0)) {
    fprintf(stderr, "MD_fcallback(): %s\n", MD_errmsg(s));
    return MD_FAIL;
  }

  ASSERT(NELEMS(fcb_b) == 1);
  /* read first 3 elements of apple */
  fcb_b[0].idnum = apple_id;
  fcb_b[0].access = MD_FCBREAD;
  fcb_b[0].nelems = 3;
  fcb_b[0].first = 0;
  if (MD_fcallback(s, front_fcb_b, &fcb_stepnum, fcb_b, NELEMS(fcb_b))) {
    fprintf(stderr, "MD_fcallback(): %s\n", MD_errmsg(s));
    return MD_FAIL;
  }

  ASSERT(NELEMS(fcb_c) == 2);
  /* read last 2 elements of apple */
  fcb_c[0].idnum = apple_id;
  fcb_c[0].access = MD_FCBREAD;
  fcb_c[0].nelems = 2;
  fcb_c[0].first = 3;
  /* read and write all (5) elements of banana */
  fcb_c[1].idnum = banana_id;
  fcb_c[1].access = MD_FCBREAD | MD_FCBWRITE;
  fcb_c[1].nelems = -1;
  fcb_c[1].first = 0;
  if (MD_fcallback(s, front_fcb_c, &fcb_stepnum, fcb_c, NELEMS(fcb_c))) {
    fprintf(stderr, "MD_fcallback(): %s\n", MD_errmsg(s));
    return MD_FAIL;
  }

  /* setup standard callbacks */

  cb_step = MD_stepnum(s);
  ASSERT(cb_step == 0);
  ASSERT(NELEMS(cb_a) == 1);
  /* read and write first element of apple */
  cb_a[0].idnum = apple_id;
  cb_a[0].access = MD_CBREAD | MD_CBWRITE;
  cb_a[0].nelems = 1;
  cb_a[0].first = 0;
  /* setup to be called every 2 steps */
  if (MD_callback(s, front_cb_a, &cb_step, cb_a, NELEMS(cb_a), 2)) {
    fprintf(stderr, "MD_callback(): %s\n", MD_errmsg(s));
    return MD_FAIL;
  }

  ASSERT(NELEMS(cb_b) == 2);
  /* read remainder of apple array */
  cb_b[0].idnum = apple_id;
  cb_b[0].access = MD_CBREAD;
  cb_b[0].nelems = -1;
  cb_b[0].first = cb_a[0].nelems;
  /* read 3 middle elements of banana */
  cb_b[1].idnum = banana_id;
  cb_b[1].access = MD_CBREAD;
  cb_b[1].nelems = 3;
  cb_b[1].first = 1;
  /* setup to be called every 2 steps */
  if (MD_callback(s, front_cb_b, &cb_step, cb_b, NELEMS(cb_b), 2)) {
    fprintf(stderr, "MD_callback(): %s\n", MD_errmsg(s));
    return MD_FAIL;
  }

  /* additional callbacks set at different step increments */
  /* notice how calling order is maintained up to step incr */

  ASSERT(NELEMS(cb_d) == 1);
  /* we don't actually do anything with data passed to front_cb_d */
  cb_d[0].idnum = apple_id;
  cb_d[0].access = MD_CBREAD;
  cb_d[0].nelems = -1;
  cb_d[0].first = 0;

  f5.stepincr = 5;
  f5.cbid = 101;
  if (MD_callback(s, front_cb_c, &f5, NULL, 0, f5.stepincr)) {
    fprintf(stderr, "MD_callback(): %s\n", MD_errmsg(s));
    return MD_FAIL;
  }

  f4a.stepincr = 4;
  f4a.cbid = 202;
  if (MD_callback(s, front_cb_d, &f4a, cb_d, NELEMS(cb_d), f4a.stepincr)) {
    fprintf(stderr, "MD_callback(): %s\n", MD_errmsg(s));
    return MD_FAIL;
  }

  f1.stepincr = 1;
  f1.cbid = 303;
  if (MD_callback(s, front_cb_c, &f1, NULL, 0, f1.stepincr)) {
    fprintf(stderr, "MD_callback(): %s\n", MD_errmsg(s));
    return MD_FAIL;
  }

  f3.stepincr = 3;
  f3.cbid = 404;
  if (MD_callback(s, front_cb_d, &f3, cb_d, NELEMS(cb_d), f3.stepincr)) {
    fprintf(stderr, "MD_callback(): %s\n", MD_errmsg(s));
    return MD_FAIL;
  }

  f6.stepincr = 6;
  f6.cbid = 505;
  if (MD_callback(s, front_cb_c, &f6, NULL, 0, f6.stepincr)) {
    fprintf(stderr, "MD_callback(): %s\n", MD_errmsg(s));
    return MD_FAIL;
  }

  f4b.stepincr = 4;
  f4b.cbid = 606;
  if (MD_callback(s, front_cb_d, &f4b, cb_d, NELEMS(cb_d), f4b.stepincr)) {
    fprintf(stderr, "MD_callback(): %s\n", MD_errmsg(s));
    return MD_FAIL;
  }

  f2.stepincr = 2;
  f2.cbid = 707;
  if (MD_callback(s, front_cb_c, &f2, NULL, 0, f2.stepincr)) {
    fprintf(stderr, "MD_callback(): %s\n", MD_errmsg(s));
    return MD_FAIL;
  }

  /* callback demonstrating CBSHARE access */
  ASSERT(NELEMS(cb_share) == 2);
  /* read all of array */
  cb_share[0].idnum = pear_id;
  cb_share[0].access = MD_CBREAD;
  cb_share[0].nelems = -1;
  cb_share[0].first = 0;
  /* now share it back to engine (don't need to set nelems and first) */
  cb_share[1].idnum = pear_id;
  cb_share[1].access = MD_CBSHARE;
  /* init info */
  cb_share_info.stepnum = MD_stepnum(s);
  cb_share_info.buf = NULL;
  cb_share_info.buflen = 0;
  /* setup callback for every step */
  if (MD_callback(s, front_cb_share, &cb_share_info, cb_share, 2, 1)) {
    fprintf(stderr, "MD_callback(): %s\n", MD_errmsg(s));
    return MD_FAIL;
  }

  /* run some */
  numsteps = 5;
  printf("\nRunning engine for %d steps...\n", numsteps);
  /* pass CBFIRST flag to get standard callbacks processed on step 0 */
  if (MD_run(s, numsteps, MD_CBFIRST) || MD_wait(s)) {
    fprintf(stderr, "MD_run(): %s\n", MD_errmsg(s));
    return MD_FAIL;
  }

  /* remove alternate message callback */
  if (MD_msgcallback_undo(s, front_alt_msgcb)) {
    fprintf(stderr, "MD_msgcallback_undo(): %s\n", MD_errmsg(s));
    return MD_FAIL;
  }

  /* remove one of the standard callback routines */
  if (MD_callback_undo(s, front_cb_d)) {
    fprintf(stderr, "MD_callback_undo(): %s\n", MD_errmsg(s));
    return MD_FAIL;
  }

  /* run some more */
  numsteps = 5;
  printf("\nRunning engine for %d more steps...\n", numsteps);
  if (MD_run(s, numsteps, 0) || MD_wait(s)) {
    fprintf(stderr, "MD_run(): %s\n", MD_errmsg(s));
    return MD_FAIL;
  }
  printf("\nFinished running engine!\n\n");

  /* free memory allocated by callback share */
  free(cb_share_info.buf);

  /*
   * destroy engine
   */

  MD_done(s);
  return 0;
}


/******************************************************************************
 *
 * main
 *
 ******************************************************************************/

int main()
{
  MD_Interface sim;

  /* test front end (which also tests engine) */
  if (front_end(&sim)) {
    fprintf(stderr, "Error: %d  Message: %s\n",
       MD_errnum(&sim), MD_errmsg(&sim));
    exit(1);
  }
  printf("Success!\n");
  return 0;
}
