/*
 * Copyright (C) 2005-2006 by David J. Hardy.  All rights reserved.
 *
 * fselect.c
 *
 * setup and cleanup of atom and bond selection data structures
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "force/force.h"
#include "debug/debug.h"


ForceSelect *force_select_create(ForceParam *fprm,
    int32 *bond_sel, int32 bond_sel_len,
    int32 *angle_sel, int32 angle_sel_len,
    int32 *dihed_sel, int32 dihed_sel_len,
    int32 *impr_sel, int32 impr_sel_len,
    int32 *aset_sel, int32 aset_sel_len,
    int32 *bset_sel, int32 bset_sel_len,
    int32 *bres_sel, int32 bres_sel_len)
{
  ForceSelect *fs;

  fs = (ForceSelect *) calloc(1, sizeof(ForceSelect));
  if (fs == NULL) {
    return NULL;
  }
  else if (force_select_initialize(fs, fprm,
        bond_sel, bond_sel_len,
        angle_sel, angle_sel_len,
        dihed_sel, dihed_sel_len,
        impr_sel, impr_sel_len,
        aset_sel, aset_sel_len,
        bset_sel, bset_sel_len,
        bres_sel, bres_sel_len)) {
    force_select_cleanup(fs);
    free(fs);
    return NULL;
  }
  return fs;
}


void force_select_destroy(ForceSelect *fs)
{
  force_select_cleanup(fs);
  free(fs);
}


void force_select_cleanup(ForceSelect *s)
{
  if (s->self_alloc_sel & FORCE_SELECT_BOND) free(s->bond_sel);
  if (s->self_alloc_sel & FORCE_SELECT_ANGLE) free(s->angle_sel);
  if (s->self_alloc_sel & FORCE_SELECT_DIHED) free(s->dihed_sel);
  if (s->self_alloc_sel & FORCE_SELECT_IMPR) free(s->impr_sel);
  if (s->self_alloc_sel & FORCE_SELECT_ASET) free(s->aset_sel);
  if (s->self_alloc_sel & FORCE_SELECT_BSET) free(s->bset_sel);
  if (s->self_alloc_sel & FORCE_SELECT_BRES) free(s->bres_sel);
  memset(s, 0, sizeof(ForceSelect));
}


void force_select_reset_force(MD_Dvec f[], const int32 sel[], int32 len)
{
  int32 k, n;
  for (n = 0;  n < len;  n++) {
    k = sel[n];
    f[k].x = 0.0;
    f[k].y = 0.0;
    f[k].z = 0.0;
  }
}


void force_select_reset_potential(double e[], const int32 sel[], int32 len)
{
  int32 k, n;
  for (n = 0;  n < len;  n++) {
    k = sel[n];
    e[k] = 0.0;
  }
}


int force_select_initialize(ForceSelect *s, ForceParam *p,
    int32 *bond_sel, int32 bond_sel_len,
    int32 *angle_sel, int32 angle_sel_len,
    int32 *dihed_sel, int32 dihed_sel_len,
    int32 *impr_sel, int32 impr_sel_len,
    int32 *aset_sel, int32 aset_sel_len,
    int32 *bset_sel, int32 bset_sel_len,
    int32 *bres_sel, int32 bres_sel_len)
{
  int32 *sel;
  int32 i, j, k, len;

  /* assumption:  topology accessed through p is correct */

  /*
   * check validity of user parameters
   */
  if (s == NULL || p == NULL) return FORCE_FAIL;

  /* make sure bond_sel array is sorted */
  if (bond_sel != NULL) {
    sel = bond_sel;
    len = bond_sel_len;
    if (len <= 0 || len > p->bond_len) return FORCE_FAIL;
    if (sel[0] < 0) return FORCE_FAIL;
    for (i = 1;  i < len;  i++) {
      if (sel[i-1] >= sel[i]) return FORCE_FAIL;
    }
    if (sel[i-1] >= p->bond_len) return FORCE_FAIL;
  }
  else if (bond_sel_len != FORCE_SELECT_ALL
      && bond_sel_len != FORCE_SELECT_NONE) return FORCE_FAIL;

  /* make sure angle_sel array is sorted */
  if (angle_sel != NULL) {
    sel = angle_sel;
    len = angle_sel_len;
    if (len <= 0 || len > p->angle_len) return FORCE_FAIL;
    if (sel[0] < 0) return FORCE_FAIL;
    for (i = 1;  i < len;  i++) {
      if (sel[i-1] >= sel[i]) return FORCE_FAIL;
    }
    if (sel[i-1] >= p->angle_len) return FORCE_FAIL;
  }
  else if (angle_sel_len != FORCE_SELECT_ALL
      && angle_sel_len != FORCE_SELECT_NONE) return FORCE_FAIL;

  /* make sure dihed_sel array is sorted */
  if (dihed_sel != NULL) {
    sel = dihed_sel;
    len = dihed_sel_len;
    if (len <= 0 || len > p->dihed_len) return FORCE_FAIL;
    if (sel[0] < 0) return FORCE_FAIL;
    for (i = 1;  i < len;  i++) {
      if (sel[i-1] >= sel[i]) return FORCE_FAIL;
    }
    if (sel[i-1] >= p->dihed_len) return FORCE_FAIL;
  }
  else if (dihed_sel_len != FORCE_SELECT_ALL
      && dihed_sel_len != FORCE_SELECT_NONE) return FORCE_FAIL;

  /* make sure impr_sel array is sorted */
  if (impr_sel != NULL) {
    sel = impr_sel;
    len = impr_sel_len;
    if (len <= 0 || len > p->impr_len) return FORCE_FAIL;
    if (sel[0] < 0) return FORCE_FAIL;
    for (i = 1;  i < len;  i++) {
      if (sel[i-1] >= sel[i]) return FORCE_FAIL;
    }
    if (sel[i-1] >= p->impr_len) return FORCE_FAIL;
  }
  else if (impr_sel_len != FORCE_SELECT_ALL
      && impr_sel_len != FORCE_SELECT_NONE) return FORCE_FAIL;

  /* make sure aset_sel and bset_sel arrays are sorted and disjoint */
  if (aset_sel != NULL && bset_sel != NULL) {
    /* first check aset_sel */
    sel = aset_sel;
    len = aset_sel_len;
    if (len <= 0 || len >= p->atom_len) return FORCE_FAIL;
    if (sel[0] < 0) return FORCE_FAIL;
    for (i = 1;  i < len;  i++) {
      if (sel[i-1] >= sel[i]) return FORCE_FAIL;
    }
    if (sel[i-1] >= p->atom_len) return FORCE_FAIL;
    /* next check bset_sel */
    sel = bset_sel;
    len = bset_sel_len;
    if (len <= 0 || len >= p->atom_len) return FORCE_FAIL;
    if (sel[0] < 0) return FORCE_FAIL;
    for (i = 1;  i < len;  i++) {
      if (sel[i-1] >= sel[i]) return FORCE_FAIL;
    }
    if (sel[i-1] >= p->atom_len) return FORCE_FAIL;
    /* finally make sure arrays are disjoint */
    for (i = 0, j = 0;  i < aset_sel_len && j < bset_sel_len; ) {
      if (aset_sel[i] < bset_sel[j]) i++;
      else if (aset_sel[i] > bset_sel[j]) j++;
      else return FORCE_FAIL;
    }
  }
  else if (aset_sel != NULL) {
    if (bset_sel_len != FORCE_SELECT_ALL) return FORCE_FAIL;
    sel = aset_sel;
    len = aset_sel_len;
    if (len <= 0 || len >= p->atom_len) return FORCE_FAIL;
    if (sel[0] < 0) return FORCE_FAIL;
    for (i = 1;  i < len;  i++) {
      if (sel[i-1] >= sel[i]) return FORCE_FAIL;
    }
    if (sel[i-1] >= p->atom_len) return FORCE_FAIL;
  }
  else if (bset_sel != NULL) {
    if (aset_sel_len != FORCE_SELECT_ALL) return FORCE_FAIL;
    sel = bset_sel;
    len = bset_sel_len;
    if (len <= 0 || len >= p->atom_len) return FORCE_FAIL;
    if (sel[0] < 0) return FORCE_FAIL;
    for (i = 1;  i < len;  i++) {
      if (sel[i-1] >= sel[i]) return FORCE_FAIL;
    }
    if (sel[i-1] >= p->atom_len) return FORCE_FAIL;
  }
  else if (aset_sel_len != FORCE_SELECT_NONE
      || bset_sel_len != FORCE_SELECT_NONE) return FORCE_FAIL;

  /* make sure bres_sel array is sorted */
  if (bres_sel != NULL) {
    sel = bres_sel;
    len = bres_sel_len;
    if (len <= 0 || len > p->atom_len) return FORCE_FAIL;
    if (sel[0] < 0) return FORCE_FAIL;
    for (i = 1;  i < len;  i++) {
      if (sel[i-1] >= sel[i]) return FORCE_FAIL;
    }
    if (sel[i-1] >= p->atom_len) return FORCE_FAIL;
  }
  else if (bres_sel_len != FORCE_SELECT_ALL
      && bres_sel_len != FORCE_SELECT_NONE) return FORCE_FAIL;

  /*
   * parameters are valid, allocate and fill arrays if needed
   */

  memset(s, 0, sizeof(ForceSelect));

  /* copy provided bond array or allocate and fill a default array */
  if (bond_sel_len != FORCE_SELECT_ALL) {
    s->bond_sel = bond_sel;
    s->bond_sel_len = bond_sel_len;
  }
  else if (p->bond_len > 0) {
    s->self_alloc_sel |= FORCE_SELECT_BOND;
    s->bond_sel = (int32 *) malloc(p->bond_len * sizeof(int32));
    if (s->bond_sel == NULL) return FORCE_FAIL;
    for (i = 0;  i < p->bond_len;  i++) {
      s->bond_sel[i] = i;
    }
    s->bond_sel_len = p->bond_len;
  }

  /* copy provided angle array or allocate and fill a default array */
  if (angle_sel_len != FORCE_SELECT_ALL) {
    s->angle_sel = angle_sel;
    s->angle_sel_len = angle_sel_len;
  }
  else if (p->angle_len > 0) {
    s->self_alloc_sel |= FORCE_SELECT_ANGLE;
    s->angle_sel = (int32 *) malloc(p->angle_len * sizeof(int32));
    if (s->angle_sel == NULL) return FORCE_FAIL;
    for (i = 0;  i < p->angle_len;  i++) {
      s->angle_sel[i] = i;
    }
    s->angle_sel_len = p->angle_len;
  }

  /* copy provided dihed array or allocate and fill a default array */
  if (dihed_sel_len != FORCE_SELECT_ALL) {
    s->dihed_sel = dihed_sel;
    s->dihed_sel_len = dihed_sel_len;
  }
  else if (p->dihed_len > 0) {
    s->self_alloc_sel |= FORCE_SELECT_DIHED;
    s->dihed_sel = (int32 *) malloc(p->dihed_len * sizeof(int32));
    if (s->dihed_sel == NULL) return FORCE_FAIL;
    for (i = 0;  i < p->dihed_len;  i++) {
      s->dihed_sel[i] = i;
    }
    s->dihed_sel_len = p->dihed_len;
  }

  /* copy provided impr array or allocate and fill a default array */
  if (impr_sel_len != FORCE_SELECT_ALL) {
    s->impr_sel = impr_sel;
    s->impr_sel_len = impr_sel_len;
  }
  else if (p->impr_len > 0) {
    s->self_alloc_sel |= FORCE_SELECT_IMPR;
    s->impr_sel = (int32 *) malloc(p->impr_len * sizeof(int32));
    if (s->impr_sel == NULL) return FORCE_FAIL;
    for (i = 0;  i < p->impr_len;  i++) {
      s->impr_sel[i] = i;
    }
    s->impr_sel_len = p->impr_len;
  }

  /* copy provided aset array or allocate and fill a default array */
  if (aset_sel_len != FORCE_SELECT_ALL) {
    s->aset_sel = aset_sel;
    s->aset_sel_len = aset_sel_len;
  }
  else if (p->atom_len - bset_sel_len > 0) {
    s->self_alloc_sel |= FORCE_SELECT_ASET;
    s->aset_sel_len = p->atom_len - bset_sel_len;
    s->aset_sel = (int32 *) malloc(s->aset_sel_len * sizeof(int32));
    if (s->aset_sel == NULL) return FORCE_FAIL;
    for (i = 0, j = 0, k = 0;  i < s->aset_sel_len || j < bset_sel_len; ) {
      if (k == bset_sel[j]) {
        k++;
        j++;
      }
      else {
        s->aset_sel[i] = k;
        k++;
        i++;
      }
    }
    for ( ;  i < s->aset_sel_len;  i++, k++) {
      s->aset_sel[i] = k;
    }
    ASSERT(k == p->atom_len);
  }

  /* copy provided bset array or allocate and fill a default array */
  if (bset_sel_len != FORCE_SELECT_ALL) {
    s->bset_sel = bset_sel;
    s->bset_sel_len = bset_sel_len;
  }
  else if (p->atom_len - aset_sel_len > 0) {
    s->self_alloc_sel |= FORCE_SELECT_BSET;
    s->bset_sel_len = p->atom_len - aset_sel_len;
    s->bset_sel = (int32 *) malloc(s->bset_sel_len * sizeof(int32));
    if (s->bset_sel == NULL) return FORCE_FAIL;
    for (i = 0, j = 0, k = 0;  i < aset_sel_len || j < s->bset_sel_len; ) {
      if (k == aset_sel[i]) {
        k++;
        i++;
      }
      else {
        s->bset_sel[j] = k;
        k++;
        j++;
      }
    }
    for ( ;  j < s->bset_sel_len;  j++, k++) {
      s->bset_sel[j] = k;
    }
    ASSERT(k == p->atom_len);
  }

  /* copy provided bres array or allocate and fill a default array */
  if (bres_sel_len != FORCE_SELECT_ALL) {
    s->bres_sel = bres_sel;
    s->bres_sel_len = bres_sel_len;
  }
  else if (p->atom_len > 0) {
    s->self_alloc_sel |= FORCE_SELECT_BRES;
    s->bres_sel = (int32 *) malloc(p->atom_len * sizeof(int32));
    if (s->bres_sel == NULL) return FORCE_FAIL;
    for (i = 0;  i < p->atom_len;  i++) {
      s->bres_sel[i] = i;
    }
    s->bres_sel_len = p->atom_len;
  }

  return 0;
}
