/*
 * Copyright (C) 2005-2006 by David J. Hardy.  All rights reserved.
 */

/**@file    fresult.h
 * @brief   Force result container class.
 * @author  David J. Hardy
 * @date    2005-2006
 *
 * The @c ForceResult_t container class holds the arrays for the
 * computed forces and energies.
 */

#ifndef FRESULT_H
#define FRESULT_H

#include "force/fparam.h"

#ifdef __cplusplus
extern "C" {
#endif


  /**@brief Holds the arrays for computed forces and energies.
   *
   * The memory buffer space is allocated by the constructor based
   * on the bitwise ORing of @c ForceResultFlags_t, with the array lengths
   * determined by topology data in @c ForceParam_t object.
   * The arrays are intended to be read after each @c force_compute().
   * The memory buffer space is freed by the destructor.
   *
   * The computation of the potential energy reductions
   * (@c ForceResult_t::u_total and other potential energies)
   * and the total atomic forces @c ForceResult_t::f_total
   * are determined by the value of @c ForceParam_t::forcetypes.
   * The separated force arrays and interaction energy arrays
   * are filled during computation of @c ForceResult_t::f_total.
   */
  typedef struct ForceResult_t {

    /* potential energies */
    double u_total;    /**< Total potential energy, sum of indicated
                         potentials. */
    double u_bond;     /**< Potential energy from spring bonds. */
    double u_angle;    /**< Potential energy from angle bonds. */
    double u_dihed;    /**< Potential energy from dihedrals. */
    double u_impr;     /**< Potential energy from impropers. */
    double u_elec;     /**< Potential energy from electrostatics. */
    double u_vdw;      /**< Potential energy from van der Waals. */
    double u_bres;     /**< Potential energy from boundary restraints. */

    /* pressure virial */
    double virial[9];  /**< Summed force contribution to pressure virial. */

    /* atomic forces */
    MD_Dvec *f_total;  /**< Array of total force, sum of indicated forces.
                         Length is @c ForceParam_t::atom_len. */
    MD_Dvec *f_bond;   /**< Array of force from spring bond interactions.
                         If requested, length is @c ForceParam_t::atom_len. */
    MD_Dvec *f_angle;  /**< Array of force from angle interactions.
                         If requested, length is @c ForceParam_t::atom_len. */
    MD_Dvec *f_dihed;  /**< Array of force from dihedral interactions.
                         If requested, length is @c ForceParam_t::atom_len. */
    MD_Dvec *f_impr;   /**< Array of force from improper interactions.
                         If requested, length is @c ForceParam_t::atom_len. */
    MD_Dvec *f_elec;   /**< Array of force from electrostatic interactions.
                         If requested, length is @c ForceParam_t::atom_len. */
    MD_Dvec *f_vdw;    /**< Array of force from van der Waals interactions.
                         If requested, length is @c ForceParam_t::atom_len. */
    MD_Dvec *f_bres;   /**< Array of force from boundary restraints.
                         If requested, length is @c ForceParam_t::atom_len. */

    /* interaction energies */
    double *e_bond;    /**< Array of spring bond interaction energies.
                         If requested, length is @c ForceParam_t::bond_len. */
    double *e_angle;   /**< Array of angle interaction energies.
                         If requested, length is @c ForceParam_t::angle_len. */
    double *e_dihed;   /**< Array of dihedral interaction energies.
                         If requested, length is @c ForceParam_t::dihed_len. */
    double *e_impr;    /**< Array of improper interaction energies.
                         If requested, length is @c ForceParam_t::impr_len. */
    double *e_elec;    /**< Array of electrostatic interaction energies.
                         If requested, length is @c ForceParam_t::atom_len. */
    double *e_vdw;     /**< Array of van der Waals interaction energies.
                         If requested, length is @c ForceParam_t::atom_len. */
    double *e_bres;    /**< Array of boundary restraint energies.
                         If requested, length is @c ForceParam_t::atom_len. */
    double *e_epot;    /**< Array of electrostatic potentials.
                         If requested, length is @c ForceParam_t::atom_len. */

    int32 self_alloc;  /**< <i> (internal use) </i> Bit mask indicating
                         which arrays are allocated by constructor. */
  } ForceResult;


  enum ForceResultFlags_t {
    FORCE_RESULT_F_BOND  = 0x0001,
    /**< Flag for bond forces array. */
    FORCE_RESULT_F_ANGLE = 0x0002,
    /**< Flag for angle forces array. */
    FORCE_RESULT_F_DIHED = 0x0004,
    /**< Flag for dihedral forces array. */
    FORCE_RESULT_F_IMPR  = 0x0008,
    /**< Flag for improper forces array. */
    FORCE_RESULT_F_ELEC  = 0x0010,
    /**< Flag for electrostatic forces array. */
    FORCE_RESULT_F_VDW   = 0x0020,
    /**< Flag for van der Waals forces array. */
    FORCE_RESULT_F_BRES  = 0x0040,
    /**< Flag for boundary restraint forces array. */

    FORCE_RESULT_E_BOND  = 0x0100,
    /**< Flag for bond interaction energies array. */
    FORCE_RESULT_E_ANGLE = 0x0200,
    /**< Flag for angle interaction energies array. */
    FORCE_RESULT_E_DIHED = 0x0400,
    /**< Flag for dihedral interaction energies array. */
    FORCE_RESULT_E_IMPR  = 0x0800,
    /**< Flag for improper interaction energies array. */
    FORCE_RESULT_E_ELEC  = 0x1000,
    /**< Flag for electrostatic interaction energies array. */
    FORCE_RESULT_E_VDW   = 0x2000,
    /**< Flag for van der Waals interaction energies array. */
    FORCE_RESULT_E_BRES  = 0x4000,
    /**< Flag for boundary restraint interaction energies array. */
    FORCE_RESULT_E_EPOT  = 0x8000,
    /**< Flag for electrostatic potentials array. */

    FORCE_RESULT_ALL_F   = 0x007f,
    FORCE_RESULT_ALL_E   = 0xff00,
    FORCE_RESULT_ALL     = FORCE_RESULT_ALL_F | FORCE_RESULT_ALL_E
  };


  /**@brief Constructor.
   *
   * @param[in] fprm         Array lengths provided by topology data.
   * @param[in] alloc_flags  Flags tell which arrays to allocate.
   *
   * Creates dynamically allocated @c ForceResult_t container object
   * that holds the arrays for the computed forces and energies.
   * The memory buffer space is allocated based on the value of
   * @c alloc_flags, set by bitwise ORing of @c ForceResultFlags_t, with the
   * array lengths determined by topology data in @c ForceParam_t object.
   *
   * @return Pointer to @c ForceResult_t object or @c NULL if
   * memory allocation fails.
   */
  ForceResult *force_result_create(const ForceParam *fprm, int32 alloc_flags);

  /**@brief Destructor.
   *
   * Frees memory buffer arrays and destroys the dynamically allocated
   * @c ForceResult_t object.  Call when finished using object.
   */
  void force_result_destroy(ForceResult *);


  /**@brief Alternative constructor.
   *
   * @param[in] fprm         Array lengths provided by topology data.
   * @param[in] alloc_flags  Flags tell which arrays to allocate.
   *
   * Use this to construct a preallocated @c ForceResult_t object.
   * The internal memory buffer space is allocated based on the value of
   * @c alloc_flags, set by bitwise ORing of @c ForceResultFlags_t, with the
   * array lengths determined by topology data in @c ForceParam_t object.
   *
   * @return 0 for success or @c FORCE_FAIL on failure.
   */
  int force_result_initialize(ForceResult *, const ForceParam *fprm,
      int32 alloc_flags);

  /**@brief Alternative destructor.
   *
   * Use this to destroy a preallocated @c ForceResult_t object.
   * Frees internal memory buffer arrays.
   */
  void force_result_cleanup(ForceResult *);


#ifdef __cplusplus
}
#endif

#endif /* FRESULT_H */
