/*
 * Copyright (C) 2004-2006 by David J. Hardy.  All rights reserved.
 */

/**@file    force.h
 * @brief   Force evaluation library.
 * @author  David J. Hardy
 * @date    2004-2006
 *
 * The @c Force_t class evaluates the basic force field for biomolecules.
 * Boundary conditions may be nonperiodic or periodic (or semi-periodic)
 * cells.  There are four helper container classes:  @c ForceParam_t to
 * define the force field parameters, @c ForceDomain_t to define the cell
 * geometry, @c ForceResult_t which holds the results of the force and
 * energy computation, and the optional class @c ForceSelect_t which can
 * be used to evaluate some subset of bonds or to compute interaction
 * energies between two disjoint subsets of atoms.
 *
 * The force evaluation can computes the following:
 * @li bonds
 * @li angles
 * @li dihedrals
 * @li impropers
 * @li electrostatics - either direct all pairs or with a cutoff distance,
 *   with interaction potential either shifted (for smooth cutoff) or standard
 * @li van der Waals - either direct all pairs or with a cutoff distance,
 *   with interaction potential either switched using the same switching
 *   function as used by NAMD 2.5 (for smooth cutoff) or standard
 * @li boundary restraints - spherical and cylindrical restraints as
 *   computed by NAMD, intended for nonperiodic (or possibly semi-periodic)
 *   simulation
 *
 * Other possiblities for computation of nonbonded interactions include:
 * @li nonbonded exclusions - subtracts out excluded interactions
 *   (instead of accumulating the nonbonded pairs to be included)
 *   for either or both electrostatics and van der Waals.
 *   This can be used in conjunction with an externally called fast
 *   method for long range electrostatic interactions.
 * @li full direct pairwise interactions - this is the @f$ O(N^2) @f$
 *   approach; for periodic systems it uses nearest image convention.
 *   This is intended only for very small systems or for testing.
 *
 * The @c Force_t class is used with four helper classes:
 * @li @c ForceParam_t is configured at the beginning of the simulation
 *   to describe which aspects of the force field are to be evaluated,
 *   along with force field parameters and system topology data.
 * @li @c ForceDomain_t defines the domain cell and boundary conditions.
 * @li @c ForceResult_t provides containers for force and energy computation,
 *   providing buffer space for the accumulation of the different potential
 *   energies, atomic forces, and interaction energies.
 * @li @c ForceSelect_t (optional) select for evaluation the energies from
 *   subsets of bonds or interaction energies and potentials between two
 *   disjoint subsets of atoms.
 *
 * The atomic position coordinates for periodic simulation are
 * maintained using an array of the true atomic positions that are
 * permitted to be outside of the periodic cell.  An array of offsets
 * are maintained internally that, when added to the true position, give
 * the corresponding position coordinates located within the cell.
 * The array is recomputed with the assumption that any given position
 * never changes more than the length of the @c ForceDomain_t cell vectors
 * between calls to @c force_compute().
 *
 * The units are as outlined by the MDAPI documentation:
 * @li distance in Angstroms
 * @li time in femtoseconds
 * @li energy in kcal/mol
 * @li force in kcal/mol/Angstrom
 *
 * The internals of the force field parameter data types and topology
 * data types are discussed in detail by the MDAPI documentation.
 *
 * Caveats:
 * @li Full electrostatics are not computed (at least not with a fast method).
 *   To use with an external electrostatics library, you can either turn
 *   @e off the computation of the electrostatics forces, or instead
 *   turn @e on the computation of electrostatics forces with the option
 *   to subtract out the nonbonded exclusions for electrostatics.
 */
#ifndef FORCE_H
#define FORCE_H

 /* provides MD type definitions and data structures */
#include "mdapi/mdtypes.h"

 /* helper classes */
#include "force/fparam.h"
#include "force/fdomain.h"
#include "force/fresult.h"
#include "force/fselect.h"

#ifdef __cplusplus
extern "C" {
#endif

#define FORCE_FAIL (-1)  /**< Return value from failed function call. */


  /**@brief Force evaluation class.
   *
   * For the top-level interface, the @c Force_t object need be
   * accessed only through the provided methods.
   */
  struct Force_t;
  typedef struct Force_t Force;



/******************************************************************************
 *
 * top-level interface routines
 *
 *****************************************************************************/


  /**@brief Constructor.
   *
   * @param[in] fprm  Contains force field parameters and topology and
   *   describes which types of forces are to be evaluated.
   * @param[in] fdom  Defines domain cell and boundary conditions.
   * @param[in] fsel  (optional) Selects subsets of bonds and/or
   *   disjoint subsets of atoms for force and energy evaluation.
   * @param[in] initpos  Gives initial atomic position coordinates.
   *
   * Creates dynamically allocated @c Force_t force evaluation object and
   * initializes it.  Must call this routine first.
   * The user should not change the @c fprm after calling this routine.
   * Any changes to @c fdom should be immediately followed by a call to
   * @c force_update_domain().  If no selections desired, then @c fsel
   * can be set to @c NULL; otherwise, no changes should be made to @c fsel.
   * This routine does not compute the force, however, it is assumed that
   * the initial atomic position coordinates @c initpos will not change
   * more than the length of the @c ForceDomain_t cell vectors between
   * calling this routine and all subsequent calls to @c force_compute().
   *
   * @return Pointer to @c Force_t object or @c NULL if memory allocation
   * fails, if the @c fprm and @fsel objects are inconsistent, or if the
   * @c fdom describes a periodic domain that is too small for the
   * indicated cutoff distance.
   */
  Force *force_create(ForceParam *fprm, ForceDomain *fdom, ForceSelect *fsel,
      const MD_Dvec initpos[]);


  /**@brief Destructor.
   *
   * Frees memory allocations made internally during setup and destroys the
   * dynamically allocated @c Force_t object.  Call when finished using it.
   */
  void force_destroy(Force *);


  /**@brief Perform force evaluation.
   *
   * @param[out] fres  Receives force and potential results from computation.
   * @param[in] pos    Gives atomic position coordinates.
   *
   * Evaluate the force.  This routine is to be called for each time step
   * of a molecular dynamics simulation.
   *
   * The computed force and potential results are returned in the @c fres
   * container.  The provided @c pos coordinates do not have to be within
   * the periodic cell, but it is assumed that they do not change more than
   * the length of the @c ForceDomain_t cell vectors between subsequent calls.
   * The position offsets that "wrap" the atomic coordinates back into the
   * periodic cell are maintained internally.
   *
   * @return 0 for success or @c FORCE_FAIL on failure.
   */
  int force_compute(Force *, ForceResult *fres, const MD_Dvec pos[]);


  /**@brief Update the simulation cell domain.
   *
   * @param[in] fdom  Define a modified simulation cell domain dimensions.
   *
   * This needs to be called for changes to the initial @c ForceDomain_t
   * @c fdom object provided to @c force_create().
   * Resets the transformation matrix and internal data storage needed
   * for evaluation of nonbonded interactions.
   *
   * @return 0 for success or @c FORCE_FAIL if memory allocation fails
   * or if @c fdom describes a periodic domain that is too small for the
   * previously indicated cutoff distance.
   */
  int force_update_domain(Force *, ForceDomain *fdom);



/******************************************************************************
 *
 * alternative constructor/destructor
 *
 *****************************************************************************/


  /**@brief Alternative constructor.
   *
   * Use this to construct a preallocated @c Force_t object.
   * Initializes use of @c Force_t object, allocating memory as needed.
   * Same notes apply as for @c force_create().
   *
   * @return 0 for success or @c FORCE_FAIL on failure.
   */
  int force_initialize(Force *, ForceParam *fprm, ForceDomain *fdom,
      ForceSelect *fsel, const MD_Dvec initpos[]);


  /**@brief Alternative destructor.
   *
   * Use this to destroy a preallocated @c Force_t object.
   * Frees any memory allocations during setup and resets object memory.
   */
  void force_cleanup(Force *);



/******************************************************************************
 *
 * get some force data arrays
 *
 *****************************************************************************/


  /**@brief Return atomic coordinate offset array.
   *
   * This returns an array of position offsets that "wrap" the atomic
   * coordinates used in the most recent call to @c force_compute()
   * (or @c force_create() ) back into the periodic cell.
   */
  const MD_Dvec *force_get_poswrap(const Force *);


  /**@brief Return scaled atomic coordinate array.
   */
  const MD_Dvec *force_get_scaled_coords(const Force *);


  /**@brief Return exclusion lists.
   *
   * This returns the exclusion lists created during @c force_create().
   * The format is a jagged array of arrays.  The first array is indexed
   * for each atom, and the second array is an ordered list of atom numbers
   * indicating which pairs are to be excluded.  Each of these second
   * arrays ends with a sentinel value greater than or equal to the number
   * of atoms.  The atom numbering is C-style beginning with 0.
   */
  int32 **force_get_excl_list(const Force *);


  /**@brief Return scaled 1-4 exclusion lists.
   *
   * The same format as returned by @c force_excl_list(), except that this
   * represents only the atomic pairs participating in scaled 1-4
   * interactions.  This list is not meaningful unless the
   * @c ForceParam_t::flags has indicated @c FORCE_EXCL_SCAL14.
   */
  int32 **force_get_scaled14_list(const Force *);



/******************************************************************************
 *
 * get domain cell values
 *
 *****************************************************************************/

  double force_get_volume(const Force *);
  const MD_Dvec *force_get_cell_center(const Force *);
  const MD_Dvec *force_get_cell_vectors(const Force *);
  const double *force_get_cell_lengths(const Force *);
  const MD_Dvec *force_get_row_transform(const Force *);
  int32 force_get_cell_boundary(const Force *);
  int32 force_get_domain_update(const Force *);



/******************************************************************************
 *
 * request and retrieve min and max scaled coordinates
 * (useful for nonperiodic systems)
 *
 *****************************************************************************/

  int force_setup_scaled_minmax(Force *);
  const MD_Dvec *force_get_scaled_minmax(const Force *);



/******************************************************************************
 *
 * request a fixed lattice and retrieve lattice spacings
 * (useful for particle pair potential interpolation of nonperiodic systems)
 *
 *****************************************************************************/

  int force_setup_lattice(Force *, int32 k1, int32 k2, int32 k3);
  const MD_Dvec *force_get_lattice_spacings(const Force *);


#ifdef __cplusplus
}
#endif

/* force object internals */
#include "force/defn.h"

/* lower-level routines */
#include "force/setup.h"
#include "force/compute.h"

#endif /* FORCE_H */
