/*
 * Copyright (C) 2005 by David J. Hardy.  All rights reserved.
 */

/**@file    fdomain.h
 * @brief   Force domain cell container.
 * @author  David J. Hardy
 * @date    2005
 *
 * The @c ForceDomain_t container class specifies the domain
 * and boundary conditions of the molecular system.
 * It is separated from the @c ForceParam_t container class
 * because, unlike @c ForceParam_t,  it might change
 * dynamically depending on the simulation method.
 */

#ifndef FDOMAIN_H
#define FDOMAIN_H

#include "mdapi/mdtypes.h"

#ifdef __cplusplus
extern "C" {
#endif

  /**@brief Used to define domain and boundary conditions
   * of the molecular system.
   *
   * User sets the internal fields of this class in order to
   * specify the domain and boundary conditions.
   * This needs to be done for initialization of @c Force_t class.
   *
   * Domains can be fully periodic, nonperiodic, or semi-periodic.
   * The first three vectors specify the side lengths and orientation
   * of a parallelepiped.  The ForceDomain_t::center vector provides
   * the center of this parallelepiped.  Set a vector to zero in order
   * to indicate a nonperiodic boundary along that "dimension."
   *
   * The length of any nonzero vector must be at least as long as
   * the ForceParam_t::cutoff distance, and the parallelepiped defined
   * must have opposite sides that are at least as far apart as the
   * ForceParam_t::cutoff distance.
   */
  typedef struct ForceDomain_t {
    /* (semi-)periodic orthogonal cell size */
    MD_Dvec v1;            /**< Domain cell along v1 direction. */
    MD_Dvec v2;            /**< Domain cell along v2 direction. */
    MD_Dvec v3;            /**< Domain cell along v3 direction. */
    MD_Dvec center;        /**< Center of cell. */
  } ForceDomain;


  /**@brief Constructor.
   *
   * Creates dynamically allocated @c ForceDomain_t container object
   * used to configure the @c Force_t class.
   * The memory is cleared, which implies a nonperiodic system
   * centered at the origin, by default.
   *
   * @return Pointer to @c ForceDomain_t object or @c NULL
   * if memory allocation fails.
   */
  ForceDomain *force_domain_create(void);

  /**@brief Destructor.
   *
   * Clears memory and destroys the dynamically allocated @c ForceDomain_t
   * object.  Call when finished using @c ForceDomain_t object.
   */
  void force_domain_destroy(ForceDomain *);


  /**@brief Alternative constructor.
   *
   * Use this to construct a preallocated @c ForceDomain_t object.
   * Clears the memory of the object.
   *
   * @return 0 for success (does not fail)
   */
  int force_domain_initialize(ForceDomain *);

  /**@brief Alternative destructor.
   *
   * Use this to destroy a preallocated @c ForceDomain_t object.
   * Resets the memory of the object.
   */
  void force_domain_cleanup(ForceDomain *);


#ifdef __cplusplus
}
#endif

#endif /* FDOMAIN_H */
