/*
 * Copyright (C) 2004-2006 by David J. Hardy.  All rights reserved.
 */

/**@file    defn.h
 * @brief   Definitions for force class.
 * @author  David J. Hardy
 * @date    2004-2006
 */

#ifndef DEFN_H
#define DEFN_H

#ifdef __cplusplus
extern "C" {
#endif

  /*
   * cells for geometric hashing, implement as cursor link list
   *
   * parameterize shell thickness and neighborlist length
   * keep only half-shell of neighbors, including self
   */

  /* maximum shell thickness */
#define FORCE_SHELLMAX  3

  /* number of neighbors */
#define FORCE_NBRLISTLEN \
  ( (2*FORCE_SHELLMAX + 1)*(2*FORCE_SHELLMAX + 1)*(2*FORCE_SHELLMAX + 1) )

  typedef struct ForceCell_t {
    int32 head;         /* index of first atom/hgroup in this cell */
    int32 cnt;          /* count number of atoms/hgroups in this cell */
    int32 nnbrs;        /* number of neighbor cells in list */

    int32 nbr[FORCE_NBRLISTLEN/2 + 1];
    /* up to half-shell of neighbor cells, include self */

    char offset[FORCE_NBRLISTLEN/2 + 1];
    /* index into offset table for periodic image neighbor cells */
  } ForceCell;


  typedef struct ForcePairlist_t {
    int32 *index;       /* array of atom indices giving interactions */
    char *offset;       /* index into table of periodic image offsets */
    int32 len;          /* used length of array */
    int32 max;          /* maximum array elements allocated */
  } ForcePairlist;


  typedef struct ForceVdwparam_t {
    double a;
    double b;
    double a14;
    double b14;
    double rmin2;
  } ForceVdwparam;


  /* constants for safe extension of Buckingham */
  typedef struct ForceBucksafe_t {
    double a;
    double b;
    double rinner2;
  } ForceBucksafe;


  enum ForceBoundary_t {
    FORCE_NONPERIODIC = 0x000,  /**< indicates system is nonperiodic */
    FORCE_X_PERIODIC  = 0x001,  /**< periodic in (transformed) x-direction */
    FORCE_Y_PERIODIC  = 0x002,  /**< periodic in (transformed) y-direction */
    FORCE_Z_PERIODIC  = 0x004,  /**< periodic in (transformed) z-direction */
    FORCE_PERIODIC = (FORCE_X_PERIODIC | FORCE_Y_PERIODIC | FORCE_Z_PERIODIC)
      /**< indicates system is fully periodic */
  };


  enum ForceIndex_t {
    FORCE_INDEX_ASET  = 0x001,  /**< atom is involved in set A computation */
    FORCE_INDEX_BSET  = 0x002,  /**< atom is involved in set B computation */
    FORCE_INDEX_BOND  = 0x004,  /**< atom is involved in bond computation,
                                  bond index array is allocated */
    FORCE_INDEX_ANGLE = 0x008,  /**< atom is involved in angle computation,
                                  angle index array is allocated */
    FORCE_INDEX_DIHED = 0x010,  /**< atom is involved in dihedral computation,
                                  dihedral index array is allocated */
    FORCE_INDEX_IMPR  = 0x020,  /**< atom is involved in improper computation,
                                  improper index array is allocated */
    FORCE_INDEX_BRES  = 0x040,  /**< atom is involved in bres computation */
    FORCE_INDEX_TOTAL = 0x080   /**< total index array is allocated */
  };


  enum ForceVirial_t {
    FORCE_VIRIAL_XX = 0,  /**< index into virial array */
    FORCE_VIRIAL_XY,      /**< index into virial array */
    FORCE_VIRIAL_XZ,      /**< index into virial array */
    FORCE_VIRIAL_YX,      /**< index into virial array */
    FORCE_VIRIAL_YY,      /**< index into virial array */
    FORCE_VIRIAL_YZ,      /**< index into virial array */
    FORCE_VIRIAL_ZX,      /**< index into virial array */
    FORCE_VIRIAL_ZY,      /**< index into virial array */
    FORCE_VIRIAL_ZZ       /**< index into virial array */
  };


  /**@brief Definition of @c Force_t object.
   */
  struct Force_t {
    ForceParam *param;        /**< points to user fundamental parameters */
    ForceSelect *select;      /**< points to user atom/bond selection */
    ForceDomain *domain;      /**< points to user domain definition */

    /* needed to support efficient atom selection */
    int32 *idmap;             /**< identity mapping for atom/bond selection,
                                might need to be as long as max{ atom_len,
                                bond_len, angle_len, dihed_len, impr_len } */
    int32 *mapnb;             /**< pairwise set membership for each atom,
                                FORCE_INDEX_ASET indicates atom in set A,
                                FORCE_INDEX_BSET indicates atom in set B,
                                both indicates there is no atom selection */

    int32 *atom_bond_sel;     /**< list of atoms involved in selected bonds */
    int32 *atom_angle_sel;    /**< list of atoms involved in selected angles */
    int32 *atom_dihed_sel;    /**< list of atoms involved in selected diheds */
    int32 *atom_impr_sel;     /**< list of atoms involved in selected imprs */
    int32 *abset_sel;         /**< combined sets A and B of selected pairs */
    int32 *total_sel;         /**< list of atoms to accumulate into total */

    int32 atom_bond_sel_len;  /**< length of @c atom_bond_sel array */
    int32 atom_angle_sel_len; /**< length of @c atom_angle_sel array */
    int32 atom_dihed_sel_len; /**< length of @c atom_dihed_sel array */
    int32 atom_impr_sel_len;  /**< length of @c atom_impr_sel array */
    int32 abset_sel_len;      /**< length of @c abset_sel array */
    int32 total_sel_len;      /**< length of @c total_sel array */

    int32 is_alloc_sel;       /**< bit mask showing which arrays allocated */

    /* alternate buffer space */
    double u_buffer;          /**< alternate buffer for potential energy */
    double *e_buffer;         /**< alternate buffer for atom/bond potentials,
                                might need to be as long as max{ atom_len,
                                bond_len, angle_len, dihed_len, impr_len } */
    MD_Dvec *f_buffer;        /**< alternate buffer for atomic forces */

    /* arrays for transformed positions and periodic image wrapping */
    MD_Dvec *trpos;           /**< transformed positions into the unit
                                periodic cell [0,1)^3;
                                nonperiodic directions might extend beyond
                                the unit cell */
    MD_Dvec *poswrap;         /**< offset for periodic wrapping of atoms
                                back into cell domain;
                                each element is an integer linear combination
                                of v1, v2, v3 */

    /* constants derived from ForceParam object */
    double elec_const;        /**< electrostatic constant */
    double elec_cutoff2;      /**< square of electrostatic cutoff distance */
    double inv_elec_cutoff2;  /**< multiplicative inverse of @c elec_cutoff2 */
    double ewald_grad_coef;   /**< coefficient for Ewald real space gradient */
    double vdw_cutoff2;       /**< square of van der Waals cutoff distance */
    double switch_dist2;      /**< square of van der Waals switch distance */
    double inv_denom_switch;  /**< inverse of denominator for switching */

    ForceVdwparam *vdwparam;  /**< square matrix of van der Waals parameters
                                accessed by atom type numbers */

    ForceBucksafe *bucksafe;  /**< square matrix of extra coefficients
                                needed for safe extension to Buckingham,
                                accessed by atom type numbers */

    /* define domain for system */
    MD_Dvec v1, v2, v3;       /**< cell basis vectors */
    double lv1, lv2, lv3;     /**< lengths of cell basis vectors */
    MD_Dvec nv1, nv2, nv3;    /**< normalized basis vectors */

    MD_Dvec center;           /**< center of cell */
    MD_Dvec lowerc;           /**< lower corner of cell */

    MD_Dvec ta1, ta2, ta3;    /**< rows of transformation defined by:
                                v1 -> (1,0,0); v2 -> (0,1,0); v3 -> (0,0,1) */

    MD_Dvec offset_table[27]; /**< gives quick lookup of 1-away periodic
                                offsets that map back into domain */

    double volume;            /**< computed volume of domain */

    int32 is_periodic;        /**< bit-flag indicating periodicity of cell */
    int32 is_orthogonal;      /**< Boolean indicating that:
                                v1=(|v1|,0,0); v2=(0,|v2|,0); v3=(0,0,|v3|) */

    /* algorithmically motivated parameters */
    double domain_len_frac;   /**< Where do we place "walls" for nonperiodic
                                domain?  The walled region is mapped into the
                                unit cube and used to determine gridcell hash.
                                This parameter is fraction of length between
                                furthest atoms.  (try 0.9 for gridcell?) */
    int32 max_steps;          /**< for nonperiodic systems, maximum number of
                                @c force_compute() calls before domain update */
    int32 cnt_steps;          /**< count number of @c force_compute() calls,
                                range 0..@c max_steps */
    int32 is_domain_update;   /**< flag set to "true" (nonzero) to indicate
                                that the domain needs to be updated
                                using @c force_setup_domain() */

    /* compute min/max for scaled coordinates */
    int32 is_minmax;
    MD_Dvec smin, smax;

    /* using a lattice fixed in space */
    int32 is_fixed_lattice;   /**< flag set to indicate use of fixed lattice */
    int32 k1, k2, k3;         /**< positive integer subdivisions of cell
                                basis vectors */
    MD_Dvec dv1, dv2, dv3;    /**< lattice vector "deltas,"
                                i.e. v1=k1*dv1, v2=k2*dv2, v3=k3*dv3 */
    MD_Dvec tb1, tb2, tb3;    /**< rows of transformation defined by:
                                dv1->(1,0,0); dv2->(0,1,0); dv3->(0,0,1) */

    /* for geometric hashing to grid cells */
    double ucellsize;         /* grid cell length along u vector */
    double vcellsize;         /* grid cell length along v vector */
    double wcellsize;         /* grid cell length along w vector */
    double inv_ucellsize;     /* 1/ucellsize */
    double inv_vcellsize;     /* 1/vcellsize */
    double inv_wcellsize;     /* 1/wcellsize */

    MD_Dvec lowcell;   /**< lower corner of grid cells
                         (for nonperiodic systems, grid cells might
                         cover larger area than domain) */
    MD_Dvec delta;     /**< displacement of transformed lower corner
                         from origin */
    MD_Dvec gscale;    /**< grid cell hashing scale factor for
                         transformed coordinates */

    ForceCell *cell;   /* array of grid cells */
    int32 maxcells;    /* maximum allocation for optimal resizing */
    int32 ncells;      /* number of cells in array: nucells*nvcells*nwcells */
    int32 nucells;     /* number of cells in u direction */
    int32 nvcells;     /* number of cells in v direction */
    int32 nwcells;     /* number of cells in w direction */

    int32 *next;       /* array of next atoms/hgroups for cursor link list */

    /* for using pairlists */
    double outer_cutoff2;
    double delta_dis2;
    MD_Dvec *pos_init;
    MD_Dvec *wrap_save;
    ForcePairlist *pairlist;
    int32 pairlist_set_id;
    int32 need_pairlist_regen;

    /* for nonbonded pairwise interaction exclusions */
    int32 **excl_list;        /**< nonbonded exclusion list */
    int32 **scaled14_list;    /**< list of scaled 1-4 interactions */

    /* helper arrays for building exclusion lists */
    int32 **exclx;     /**< explicit exclusions (from @c MD_Excl) */
    int32 **excl12;    /**< lists for 1-2 exclusions (from @c MD_Bond) */
    int32 **excl13;    /**< lists for 1-3 exclusions */
    int32 **excl14;    /**< lists for 1-4 exclusions */
    int32 **scaled14;  /**< lists for scaled 1-4 exclusions */
    int32 *lenx;       /**< counters for exclx lists */
    int32 *len12;      /**< counters for excl12 lists */
    int32 *len13;      /**< counters for excl13 lists */
    int32 *accum;      /**< for merging lists */
    int32 *dest;       /**< for merging lists */

    /* for boundary restraints */
    int32 is_bres_term1;  /**< first term for boundary restraint is active */
    int32 is_bres_term2;  /**< second term for boundary restraint is active */
    double sq_minradius;  /**< square of minimum radius */
    double minlength;     /**< minimum length */

  };


#ifdef __cplusplus
}
#endif

#endif /* DEFN_H */
