/*
 * Copyright (C) 2004-2006 by David J. Hardy.  All rights reserved.
 *
 * bres.c
 *
 * compute potential and forces due to boundary restraints
 */

#include <stdlib.h>
#include <math.h>
#include "force/intdefn.h"
#include "debug/debug.h"


void force_cleanup_bres(Force *f)
{
  /* nothing to do!  no memory was allocated */
}


int force_setup_bres(Force *f)
{
  ForceParam *p = f->param;
  const int32 bresopts = p->bresopts;

  /* make sure options are set */
  if (bresopts <= FORCE_NONE || bresopts >= FORCE_MARKER_BRES) {
    ERROR("illegal value for \"bresopts\"");
    return FORCE_FAIL;
  }

  f->is_bres_term1 = (p->konst1 != 0.0);
  f->is_bres_term2 = (p->konst2 != 0.0);

  /* make sure that term1 constants are positive */
  if (f->is_bres_term1
      && (p->radius1 <= 0.0 || p->exp1 <= 0
        || (bresopts != FORCE_BRES_SPHERE && p->length1 <= 0.0))) {
    ERROR("illegal value(s) for \"bres term1\"");
    return FORCE_FAIL;
  }

  /* make sure that term2 constants are positive */
  if (f->is_bres_term2
      && (p->radius2 <= 0.0 || p->exp2 <= 0
        || (bresopts != FORCE_BRES_SPHERE && p->length2 <= 0.0))) {
    ERROR("illegal value(s) for \"bres term2\"");
    return FORCE_FAIL;
  }

  if (f->is_bres_term1 && f->is_bres_term2) {

    /* determine square of smallest radius */
    if (p->radius1 < p->radius2) {
      f->sq_minradius = p->radius1 * p->radius1;
    }
    else {
      f->sq_minradius = p->radius2 * p->radius2;
    }

    /* determine smallest length */
    if (p->length1 < p->length2) {
      f->minlength = p->length1;
    }
    else {
      f->minlength = p->length2;
    }
  }
  else if (f->is_bres_term1) {
    f->sq_minradius = p->radius1 * p->radius1;
    f->minlength = p->length1;
  }
  else if (f->is_bres_term2) {
    f->sq_minradius = p->radius2 * p->radius2;
    f->minlength = p->length2;
  }
  else {
    ERROR("neither \"bres term1\" nor \"bres term2\" are active");
    /* neither term1 nor term2 would be active */
    return FORCE_FAIL;
  }

  return 0;
}


/*
 * compute spherical boundary restraints
 * (sphere about domain center)
 */
int force_compute_bres_sphere(Force *fobj, double *u_bres,
    MD_Dvec f_bres[], double e_bres[], const MD_Dvec *pos,
    const int32 bres_sel[], int32 bres_sel_len)
{
  const MD_Dvec center = fobj->center;
  const double radius1 = fobj->param->radius1;
  const double radius2 = fobj->param->radius2;
  const double konst1 = fobj->param->konst1;
  const double konst2 = fobj->param->konst2;
  const int32 exp1 = fobj->param->exp1;
  const int32 exp2 = fobj->param->exp2;
  const double sq_minradius = fobj->sq_minradius;
  double dx, dy, dz, dis, sq_dis, diff, factor, fc;
  double e_sum = 0.0;
  double e;
  MD_Dvec *wrap = fobj->poswrap;
  const int32 is_bres_term1 = fobj->is_bres_term1;
  const int32 is_bres_term2 = fobj->is_bres_term2;
  int32 nn, n, k;

  ASSERT(radius1 >= 0.0);
  ASSERT(radius2 >= 0.0);
  ASSERT(is_bres_term1 || is_bres_term2);

  for (nn = 0;  nn < bres_sel_len;  nn++) {
    n = bres_sel[nn];

    e = 0.0;
    dx = center.x - (pos[n].x + wrap[n].x);
    dy = center.y - (pos[n].y + wrap[n].y);
    dz = center.z - (pos[n].z + wrap[n].z);
    sq_dis = dx * dx + dy * dy + dz * dz;
    if (sq_dis <= sq_minradius) continue;
    dis = sqrt(sq_dis);
    fc = 0.0;
    if (is_bres_term1 && dis > radius1) {
      diff = dis - radius1;
      factor = konst1;
      for (k = 1;  k < exp1;  k++)  factor *= diff;
      e += factor * diff;
      fc += exp1 * factor;
    }
    if (is_bres_term2 && dis > radius2) {
      diff = dis - radius2;
      factor = konst2;
      for (k = 1;  k < exp2;  k++)  factor *= diff;
      e += factor * diff;
      fc += exp2 * factor;
    }
    fc /= dis;
    f_bres[n].x += fc * dx;
    f_bres[n].y += fc * dy;
    f_bres[n].z += fc * dz;
    e_bres[n] = e;
    e_sum += e;
  }
  *u_bres = e_sum;
  return 0;
}


/*
 * compute cylindrical boundary restraints
 * (cylinder about domain center, parallel to either x, y, or z axis)
 */
int force_compute_bres_cylinder(Force *fobj, double *u_bres,
    MD_Dvec f_bres[], double e_bres[], const MD_Dvec *pos,
    const int32 *bres_sel, int32 bres_sel_len, int32 bresopts)
{
  const MD_Dvec center = fobj->center;
  const double radius1 = fobj->param->radius1;
  const double radius2 = fobj->param->radius2;
  const double length1 = fobj->param->length1;
  const double length2 = fobj->param->length2;
  const double konst1 = fobj->param->konst1;
  const double konst2 = fobj->param->konst2;
  const int32 exp1 = fobj->param->exp1;
  const int32 exp2 = fobj->param->exp2;
  const double sq_minradius = fobj->sq_minradius;
  const double minlength = fobj->minlength;
  double dx, dy, dz, dis, sq_dis, diff, factor, fr, fl, len;
  double e_sum = 0.0;
  double e;
  MD_Dvec *wrap = fobj->poswrap;
  const int32 is_bres_term1 = fobj->is_bres_term1;
  const int32 is_bres_term2 = fobj->is_bres_term2;
  int32 nn, n, k, is_neg;

  ASSERT(radius1 >= 0.0);
  ASSERT(radius2 >= 0.0);
  ASSERT(is_bres_term1 || is_bres_term2);

  for (nn = 0;  nn < bres_sel_len;  nn++) {
    n = bres_sel[nn];

    e = 0.0;
    dx = center.x - (pos[n].x + wrap[n].x);
    dy = center.y - (pos[n].y + wrap[n].y);
    dz = center.z - (pos[n].z + wrap[n].z);
    if (bresopts == FORCE_BRES_X_CYLINDER) {
      is_neg = (dx < 0.0);
      len = (is_neg ? -dx : dx);
      sq_dis = dy * dy + dz * dz;
    }
    else if (bresopts == FORCE_BRES_Y_CYLINDER) {
      is_neg = (dy < 0.0);
      len = (is_neg ? -dy : dy);
      sq_dis = dx * dx + dz * dz;
    }
    else {
      is_neg = (dz < 0.0);
      len = (is_neg ? -dz : dz);
      sq_dis = dx * dx + dy * dy;
    }
    fl = 0.0;
    if (len > minlength) { 
      if (is_bres_term1 && len > length1) {
        diff = len - length1;
        factor = konst1;
        for (k = 1;  k < exp1;  k++)  factor *= diff;
        e += factor * diff;
        fl += exp1 * factor;
      }
      if (is_bres_term2 && len > length2) {
        diff = len - length2;
        factor = konst2;
        for (k = 1;  k < exp2;  k++)  factor *= diff;
        e += factor * diff;
        fl += exp2 * factor;
      }
      if (is_neg) fl = -fl;
    }
    fr = 0.0;
    if (sq_dis > sq_minradius) {
      dis = sqrt(sq_dis);
      if (is_bres_term1 && dis > radius1) {
        diff = dis - radius1;
        factor = konst1;
        for (k = 1;  k < exp1;  k++)  factor *= diff;
        e += factor * diff;
        fr += exp1 * factor;
      }
      if (is_bres_term2 && dis > radius2) {
        diff = dis - radius2;
        factor = konst2;
        for (k = 1;  k < exp2;  k++)  factor *= diff;
        e += factor * diff;
        fr += exp2 * factor;
      }
      fr /= dis;
    }
    if (bresopts == FORCE_BRES_X_CYLINDER) {
      f_bres[n].x += fl;
      f_bres[n].y += fr * dy;
      f_bres[n].z += fr * dz;
    }
    else if (bresopts == FORCE_BRES_Y_CYLINDER) {
      f_bres[n].x += fr * dx;
      f_bres[n].y += fl;
      f_bres[n].z += fr * dz;
    }
    else {
      f_bres[n].x += fr * dx;
      f_bres[n].y += fr * dy;
      f_bres[n].z += fl;
    }
    e_bres[n] = e;
    e_sum += e;
  }
  *u_bres = e_sum;
  return 0;
}
