/*
 * Copyright (C) 2004-2006 by Wei Wang.  All rights reserved.
 */


#include <stdio.h>
#include <assert.h>
#include <string.h>
#include <stdlib.h>
#include "vecbuffer.h"
#include "helper.h"
#include "utilities.h"


MD_Int vec_buffer_init(struct VecBuffer_Tag *vb, MD_Int nvec, MD_Int veclen, 
                       const MD_Char* filename)
{
  assert(veclen > 0);
  assert(nvec >= 0);

  vb->ntotalvec = nvec;
  vb->veclen = veclen;

  vb->buffer = (0 == vb->ntotalvec) ? NULL :
    my_calloc((size_t)nvec*veclen, sizeof(MD_Double), "buffer");
    
  vb->last_offset = -1;
  if (NULL != filename) {
    printf("  read vectors from %s\n", filename);
    if (vec_buffer_binread(vb, filename)) {
      fprintf(stderr, "cannot read in previous vector values\n");
      return MD_FAIL;
    }
  }

/* purely hack */
/*
  {
  assert(vb->last_offset >= 1);
  memcpy(vb->buffer, vb->buffer + (vb->last_offset - 1) * vb->veclen,
         vb->veclen * sizeof(MD_Double));
  memset((void*)(vb->buffer+vb->veclen), 0, (vb->nvec-1)*vb->veclen*sizeof(MD_Double));
  vb->last_offset=0;
  }
*/
  assert(NULL == filename || vb->last_offset >= 0);
  printf("vector buffer has %d vectors available out of %d vectors of length %d, offset = %d\n",
	 vb->nvec, vb->ntotalvec, vb->veclen, vb->last_offset);

  return OK;
}


MD_Int vec_buffer_destroy(struct VecBuffer_Tag *vb)
{
  if (NULL != vb->buffer) free(vb->buffer);
  memset(vb, 0, sizeof(struct VecBuffer_Tag));
  return OK;
}


MD_Int vec_buffer_update(struct VecBuffer_Tag *vb, const MD_Double *vec)
{
  assert(NULL != vb);
  if (NULL == vb->buffer) return OK; /* no need to update */
  assert(NULL != vec);
  vb->last_offset =  (vb->last_offset + 1) % vb->ntotalvec;
  memcpy(vb->buffer + vb->last_offset * vb->veclen, vec, 
         vb->veclen * sizeof(MD_Double));
  vb->nvec ++;
  if (vb->nvec > vb->ntotalvec) vb->nvec = vb->ntotalvec;
  return OK;
}


MD_Int vec_buffer_get_nvec(struct VecBuffer_Tag *vb)
{
  return vb->nvec;
}


MD_Int vec_buffer_get_veclen(struct VecBuffer_Tag *vb)
{
  return vb->veclen;
}


MD_Int vec_buffer_get_vec_array(struct VecBuffer_Tag *vb, 
				MD_Double **vecarray, MD_Int nvec)
{
  MD_Int i;
  MD_Int n = nvec;

  if (nvec > vb->nvec) {
    fprintf(stderr, "Error: asking for more vectors stored in buffer\n");
    return FAILURE;
  }

  for (i = 0; i < n; i++) {
    vecarray[i] = vb->buffer + ((vb->last_offset - i + vb->ntotalvec) % 
				vb->ntotalvec) * vb->veclen;
  }

  return OK;
}


/* assume the buffer is full */
MD_Int vec_buffer_bindump(struct VecBuffer_Tag *vb, const MD_Char *filename)
{
  /* the flag "b" (binary) is for compatibility with ANSI C */
  size_t noutput;
  FILE *fd;

  if (NULL == vb) {
    printf("vector buffer is not set up, \n");
    return OK;
  }

  if (NULL == vb->buffer) {
    printf("do not need to dump\n");
    return OK;
  }

  fd = fopen(filename, "wb");
  if (NULL == fd) {
    fprintf(stderr, "cannot open file %s\n", filename);
    return MD_FAIL;
  } 

  noutput  = fwrite(&(vb->nvec), sizeof(MD_Int), (size_t) 1, fd);
  noutput += fwrite(&(vb->veclen), sizeof(MD_Int), (size_t) 1, fd);
  noutput += fwrite(&(vb->last_offset), sizeof(MD_Int), (size_t)1, fd);
  noutput += fwrite(vb->buffer, sizeof(MD_Double), 
		    (size_t)(vb->nvec * vb->veclen), fd);

  if ((size_t) (3 + vb->veclen * vb->nvec) != noutput || ferror(fd)) {
    perror("error in writing vector file");
    return MD_FAIL;
  }
  if (fclose(fd)) {
    perror("cannot close file");
    return MD_FAIL;
  }

  return OK;
}


MD_Int vec_buffer_binread(struct VecBuffer_Tag *vb, const MD_Char *filename)
{
  MD_Int veclen, nvec, last, nread;
  const size_t unit = sizeof(MD_Double);
  size_t ninput;
  FILE *fd = fopen(filename, "rb");
  MD_Double* buffer = vb->buffer;  /* probably not good */

  if (NULL == fd) {
    fprintf(stderr, "cannot open file %s\n", filename);
    return MD_FAIL;
  }

  ninput  = fread((void *)&nvec,   sizeof(MD_Int), (size_t) 1, fd);
  ninput += fread((void *)&veclen, sizeof(MD_Int), (size_t) 1, fd);
  ninput += fread((void *)&last,   sizeof(MD_Int), (size_t) 1, fd); 
  assert(0 == veclen - vb->veclen); 

  if (nvec <=  vb->ntotalvec) {
    MD_Int seg1 = last+1;
    MD_Int seg2 = nvec - seg1;
    nread = nvec;
    ninput+=fread((void *)(buffer+seg2*veclen),unit,(size_t)(seg1*veclen),fd);
    ninput+=fread((void *)buffer, unit, (size_t) (seg2*veclen), fd);
  } else {  /* smaller buffer, only read in the latest vector */
    nread = vb->ntotalvec;
    if (last+1 >= nread) { /* one read is enough */
      if (fseek(fd, (long)((last+1-nread)*veclen*unit), SEEK_CUR)) {
        perror("file size wrong");
        return MD_FAIL;
      }
      ninput+=fread((void *)buffer,unit,(size_t)(nread*veclen),fd);
    } else { /* read 2 segments */
      MD_Int seg1 = last+1;
      MD_Int seg2 = nread - seg1;
      ninput+=fread((void*)(buffer+seg2*veclen),unit,(size_t)(seg1*veclen),fd);
      if (fseek(fd, (long)((nvec-nread)*veclen*unit), SEEK_CUR)) {
        perror("file size wrong");
        return MD_FAIL;
      }
      ninput+=fread((void *)buffer, unit, (size_t)(seg2*veclen), fd);
    };
  };
  vb->last_offset = nread - 1;
  vb->nvec = nread;

  if ((size_t)(3+veclen*nread) != ninput || ferror(fd)) { /* protocol check */
    perror("error when reading vector file");
    return MD_FAIL;
  }

  if (fclose(fd)) {
    perror("cannot close vector file");
    return MD_FAIL;
  }

  printf("read in %d vectors\n", nread);

  return OK;
}

