/*
 * Copyright (C) 2004-2006 by Wei Wang.  All rights reserved.
 */

#ifndef UNIT_H
#define UNIT_H

/*
 * define unit values
 *
 *  cosntant value drawn from GROMACS manual version 3.1.1
 */

/*
 * -----------------------
 * internal energy units:
 * -----------------------
 *     mass: Atomic Mass Unit (AMU)
 *   length: Angstrom 
 *     time: femto-second 
 */


/* -------------------- Basic Units ----------------------- */

#define METER (1e10)   /* 1e10 angstrom */
#define SECOND (1e15)  /* 1e15 femto-second */
#define ATOMIC_MASS (1.6605402e-27)  /* kilogram */
#define KILOGRAM (1.0/ATOMIC_MASS)  
#define AVOGADRO (6.0221367e23)   /* 1 mol = 6.0221367e23 */

/* --------------------- Energy Units --------------------- */
/*
 * 1 Kcal/Mol = KCAL_PER_MOL  AMU*A^2/fs^2 
 * definition: 1 cal = 4.184 Joule
 */
#define KJOULE_PER_MOL (1e3*KILOGRAM*METER*METER/(SECOND*SECOND) / AVOGADRO)
#define CALORIE 4.184  /* Joule */
#define KCAL_PER_MOL (CALORIE * KJOULE_PER_MOL)

/* --------------------- Electrostatics Units --------------------- */

#define ELECTRON_CHARGE (1.60217733e-19) /* COULOMB */
/* 
 * 1 Debye = 3.33564e-30 Coulumb * Meter
 *         \approx 0.208194295641824 (electron charge) * (angstrom)
 */
#define DEBYE (3.33564e-30/ELECTRON_CHARGE*METER)

/*
 * Coulomb constant converted for use with internal energy units
 * COULOMB_SQR * (electro charge)^2 / angstrom = 1 AMU * angstrom^2 / fs^2
 * (e is the electron charge)
 */
#define ELECTRO_CONVERSION_FACTOR 1389.35485  /* KJ/mol/e^2 * Angstrom */
#define COULOMB_SQR (ELECTRO_CONVERSION_FACTOR * KJOULE_PER_MOL)

/* ------------------ Thermodynamics Units -------------------------*/

/*
 * 1 Kelvin \approx 8.314472135196751e-07 AMU * A^2 / fs^2
 * example: KELVIN * 300.0 ==> internal energy unit representation of 300K
 */
#define GAS_CONSTANT 8.314510  /* Joule per mole per Kelvin */
#define KELVIN (GAS_CONSTANT * 1e-3 * KJOULE_PER_MOL)

/* Nf = number of degrees of freedom */
#define KCPM_TO_KEL (KCAL_PER_MOL / KELVIN)
#define KCALPERMOL_TO_KELVIN (Nf) (KCPM_TO_KEL / (MD_Double) Nf)

#endif /* CONSTANT_H */
