/*
 * Copyright (C) 2004-2006 by Wei Wang.  All rights reserved.
 */


/* a 2-particle system testing the correctness of the PME code.
 * the system is 4x4x4, two atoms are located at (1,0,0) and (-1,0,0)
 * notice a few things about setting up the parameters:
 * (1) gridsize should be large enough (4 is too small), in general,
 *     it should be twice the number of the standard Ewald k-space cutoff
 * (2) PME is exact if the charge is located at integer points.
 *     however, the truncation error still exists. Also, if the gridsize
 *     is not compatible, the particles may not be at integer grid points,
 *     and the interpolation error comes in.
 *
*/

#include <stdlib.h>
#include <string.h>
#include "pme.h"
#include "standEwald.h"
#include "helper.h"


#define N 2

MD_Atom atoms[N] = {
  {1.0,  1.0, 0, 0, "1", "1"}, 
  {1.0, -1.0, 0, 0, "2", "2"}
};

MD_Double polar[N*3] = {0.1, 0.1, 0.1, 0.1, 0.1, 0.1};

MD_Dvec pos[N] = {
  { 0.0, 0.0, 0.0},
  {-1.0, 1.0, 0.0}
};
MD_Dvec systemsize = {4.0, 4.0, 4.0};

struct Dsolver_Parameters_Type dparams = {0, 0, 0, 0, 100, 1e-12};

#define Tol 1e-6

void test_stdEw(void);
void test_pme(void);

int main(void)
{
  test_pme();
  test_stdEw();
  return 0;
}


void test_stdEw(void)
{
  struct Electro_init_Tag params = {
    polar, N, atoms, pos, pos, NULL, systemsize, Tol, 0, 1, dparams};
  struct standEwald_Tag *se = my_calloc((size_t)1, sizeof *se, "ele");

  stdEw_init(se, &params);
  stdEw_compute(se);
  stdEw_destroy(se);
  free(se);
}


void test_pme(void)
{
  struct PmeParams_Tag params;
  struct Pme_Tag *pme = my_calloc((size_t)1, sizeof *pme, "pme");

  memset(&params, 0, sizeof params);
  params.PMETolerance = Tol;
  params.PMEGridSizeX = params.PMEGridSizeY = params.PMEGridSizeZ = 16;
  params.PMEInterpOrder = 4;
  params.cellBasisVector1.x = systemsize.x;
  params.cellBasisVector2.y = systemsize.y;
  params.cellBasisVector3.z = systemsize.z;
  params.cutoff = 2.0;
  params.pexcllist = NULL;
  params.patom = atoms;
  params.polarizability = polar;
  params.ppos = pos;
  params.natoms = N;
  params.restart = 0;
  params.has_induced_dipole = 1;
  params.dsolver_param = dparams;
 
  pme_init(pme, &params);
  pme_compute(pme);
  pme_destroy(pme);
  free(pme);

}
