/*
 * Copyright (C) 2004-2006 by Wei Wang.  All rights reserved.
 */

/*************************************************************************
 * 
 * standEwald.h  (standEwald.c)
 *
 * Compute the Ewald sum using standard Ewald method, the code
 * work only on cubic system. It scales as O(N^{3/2}), where N is the number 
 * of charged particles
 *
 * The module can handle induced dipole and partially permanent dipoles. 
 * The permanent dipole is assumed constant, in the sense that its
 * direction and magnitude does not change. This is not true in general,
 * since the "permanent" dipole direction depends on the orientation
 * of the molecule, only its magnitude does not change. 
 * the "permanent" feature helps to debug the code, and prepares the code
 * for further development.
 *
 * To use it, need to prepare your own stdEw_init function, then
 * call stdEw_compute to compute the energy/forces, and use
 * stdEw_get_xxx functions to get the result.
 * the module also provides 
 *   1. stdEw_compute_mol_sys_dipole:  to compute water molecule dipoles
 *   2. stdEw_compute_G2v: to compute matrix-vector product of G2 * v,
 *      where v is an arbitrary vector. This is intended to be used
 *      by dsolver module in solving the dipole equation.
 * the module members should not be accessed through other ways.
 *
 *************************************************************************/


#ifndef STANDEWALD_H
#define STANDEWALD_H

#ifdef __cplusplus
extern "C" {
#endif

#include "mdtypes.h"
#include "dsolvers.h"
#include "data.h"
#include "linkcell.h"
 
  struct standEwald_init_Tag {
    MD_Double *polarizability;
    MD_Int natoms;
    MD_Atom* patom;
    MD_Dvec *wrapped_pos;
    MD_Dvec *realpos;
    MD_Int** pexcllist;
    MD_Dvec systemsize;
    MD_Double errTol;          /* relative error tolerance */
    MD_Int restart;
    MD_Int has_induced_dipole;
    struct Dsolver_Parameters_Type dsolver_param;
  };

  struct standEwald_Tag 
  {
    /* physical system */
    MD_Dvec a1, a2, a3;        /* direct lattice base vector */
    MD_Dvec b1, b2, b3;        /* reciprocal lattice base vector */
    MD_Dvec systemsize;
    MD_Dvec* ppos;             /* points to position array */
    MD_Dvec* prealpos;         /* for comuting total magnetization */
    MD_Dvec* force;            /* self-maintained force array */
    MD_Double* palpha;         /* points to polarizability array */
    MD_Double* charge;         /* privately stored charge value */
    MD_Double volume;          /* volume of the simulation box */
    MD_Double energy;          /* total electrostatic energy */
    MD_Int natoms;             /* # of atoms */

    /* Ewald sum parameters */
    MD_Double errTol;          /* relative error tolerance */
    MD_Double beta;            /* */
    MD_Double rcut, rcut2;     /* cut off radius for the direct sum */
    MD_Double kcut;            /* (cut off radius) for the reciprocal sum */
  
    /* intermediate quantities */
    MD_Double qq;              /* q^T q */
    MD_Double self_energy_qq;  /* charge--charge self-energy, constant */

    /* direct lattice */
    MD_Dvec* dirlatt;          /* direct lattice sites */
    MD_Dvec *dirforce;
    MD_Double dirEnergy;
    MD_Double *neg_dirG1q;     /* debug usage */
    MD_Int ndirlatt;           /* # of direct lattice sites */
    MD_Double** dirG2;         /* linked list, dirG2[i][] nonzero elements */
    MD_Double** dirG3;         /* linked list */
    MD_Int** neibrlist;        /* neibrlist[i]: i's neighor's atom ID */
    MD_Int* numneibrs;         /* numneibrs[i] = # of neighbors for atom i */
    MD_Int max_neibrs;         /* control needed for memory allocation */
    MD_Int nmax[3];            /* max lattice component, debug usage */
    struct LinkCell_Tag *linkcell;
    MD_Double diagdirG0;       /*diagonal elements of G0, independent of i */
    MD_Double diagdirG2xx, diagdirG2yy, diagdirG2zz;  
    MD_Int* *pexcllist;        /* points to exclusion list */

    /* reciprocal lattice */
    MD_Dvec* reclatt;          /* reciprocal lattice sites */
    MD_Dvec* recforce;
    MD_Double *neg_recG1q;
    MD_Double recEnergy;
    MD_Double* sinkr;          /* sine and cosine array */
    MD_Double* coskr; 
    MD_Double* reS;
    MD_Double* imS;            /* structure factor (real and imaginary) */
    MD_Double* recU;           /* (4*pi/v)*exp(-k^2/(4*beta^2))/k^2 */
    MD_Double* recg2d;
    MD_Double* recg2q;         /* G2*d, G2*q */
    MD_Double* work;           /* temporary memory */
    MD_Int kmax[3];            /* max lattice component, debug usage */
    MD_Int nreclatt;           /* # of reciprocal lattice sites */

    /* dipoles */
    MD_Int has_induced_dipole;
    struct Dsolver_Tag* dsolver; /* solve dipole equation */
  };


  MD_Errcode stdEw_init(struct standEwald_Tag* se,
			struct standEwald_init_Tag* init_data);
  MD_Errcode stdEw_destroy(struct standEwald_Tag* se);
  MD_Errcode stdEw_compute(struct standEwald_Tag* se);

  /* set up the dsolver submodule */
  MD_Errcode stdEw_setup_dsolver(struct standEwald_Tag *se, 
				 struct Dsolver_Parameters_Type ds_param);

  MD_Double stdEw_get_energy(const struct standEwald_Tag* se);
  const MD_Dvec* stdEw_get_force(const struct standEwald_Tag* se);
  MD_Double* stdEw_get_dipole(const struct standEwald_Tag* se);
  void stdEw_fill_diagonal(const struct standEwald_Tag* se, MD_Double *d);

  /* flag = 0, compute -G2d, result array length is 3*natoms 
   * flag = 1, compute -G1q - G2d */
  void stdEw_compute_pseudores(struct standEwald_Tag* se,
			       const MD_Double *d, const MD_Int flag, 
			       MD_Double *pseudores);
  
  MD_Errcode stdEw_dump_dipole(const struct standEwald_Tag *se, 
			       const char* filename);

#ifdef __cplusplus
}
#endif


#endif
