/*
 * Copyright (C) 2004-2006 by Wei Wang.  All rights reserved.
 */


/* 
 * define the linkcell structure used in nonbonded force computation
 * (specifically, for Lennard-Johns/van der Waals, and the direct part
 * of Ewald sum, which has a cutoff radius in their potential implementation.
 *
 * The link cell structure divide the whole system (cubic) into small cells
 * of same size. If two cells' distance is less than the potential cutoff
 * then they are neighbors. Each cell has a neighborlist. If cell i and j
 * are neighbors, then either j is the neighborlist of i, or the other way
 * around, but not both. 
 *
 * The functionality of linkcell is to speed up the computation by neglecting
 * particles whose distance is larger than the cutoff radius. Its functionality
 * includes:
 *  -- setup neighbor list for each cell 
 *     for constant volume simulation, this is done only once in the 
 *     initialization phase. If volumne is a variable, it is needed every
 *     time there is a rescaling of positions and volume.
 *  note that cell size is small and not necessarily equal to cutoff radius.
 *     it leads to faster computation since smaller cell can screen out 
 *     many atoms which would have been computed if large cell size is used. 
 *     a 42% improvement is seen for a 864 argon system.
 *  -- hash atoms into each cell.
 *     this is an O(N) computation, and is used as a precursor for tracing
 *     through each atom and its neighbors.
 *
 * Linkcell structure is enough if the pair interaction is computated only
 * once at each force evaluation, e.g. in van der Waals evaluation and
 * the charge-only Ewald sum. If we need to store some computation result and
 * use them later, as is the case for the induced dipole computations,  
 * it needs to be combined with other structures such as neighborlist.
 *
 * A general scheme for tracing through the atom interactions is like:
 *
 *    for each cell (atombox) c
 *       for each atom (i) in this cell
 *          for each other atoms in this cell whose order is later than i
 *             compute interactions. (avoid double counting)
 *          end
 *       end
 *       for each c's neighbor cells c'
 *          for each atom (i) in cell c'
 *            compute interactions 
 *          end
 *       end
 *    end
 *
 * Limitations:
 *   -- can only deal with cubic (maybe orthonormal) box
 *   -- cell size is constant, otherwise, the hashing is inefficient.
 */

#ifndef LINKCELL_H
#define LINKCELL_H


#ifdef __cplusplus
extern "C" {
#endif

#include "mdtypes.h" 


struct Cell_Tag 
{
  MD_Dvec min;           /* extent of box */
  MD_Int nx, ny, nz;     /* ID of box */ 
  MD_Int *nbr;           /* list of adjacent neighbors of box */
  MD_Int numnbrs;        /* number of neighbors in nbr array */
};

struct LinkCell_init_Tag
{
  MD_Int nxcell, nycell, nzcell;
  MD_Dvec systemsize;
  MD_Dvec min;
  MD_Double cutoff;
  MD_Int natoms;
};

struct LinkCell_Tag 
{
  struct Cell_Tag *cell;        /* cell */
  MD_Int nxcell, nycell, nzcell;    
  MD_Int numcell;               /* total # of celles */
  MD_Dvec min;                  /* min coordinates of the simulation box */
  MD_Dvec cellsize;
  MD_Dvec systemsize;
  MD_Double cutoff;             /* from potential */
  MD_Int *head;
  MD_Int *list;
  MD_Int natoms;
}; 

MD_Errcode linkcell_init(struct LinkCell_Tag *linkcell, 
			 struct LinkCell_init_Tag *init_data);

MD_Errcode linkcell_destroy(struct LinkCell_Tag *linkcell);

MD_Errcode linkcell_hash_atoms(struct LinkCell_Tag *linkcell, 
			       const MD_Dvec *pos);


#ifdef __cplusplus
}
#endif

#endif
