/*
 * Copyright (C) 2004-2006 by Wei Wang.  All rights reserved.
 */

/*
 * fnonbond.c
 *
 */

#include <assert.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "fnonbond.h"
#include "electro.h"
#include "force.h"
#include "utilities.h"
#include "timer.h"
#include "dipole_poly.h"
#include "dpoly.h"
#include "helper.h"
#include "unit.h"
#include "pme.h"

static MD_Errcode build_exclusions(struct Fnonbond_Tag *fnonbond,
				   const struct Data_Tag* data);

/*
 * initialize the nonbond force object
 *
 * assumes that force is already allocated and memory cleared
 * data must already be initialized
 *
 */
MD_Errcode fnonbond_init(struct Fnonbond_Tag *fnonbond, 
			 struct Fnonbond_init_Tag *init_data)
{
  struct VDW_init_Tag init_vdw;
  /* void *eparams; */

  fnonbond->force = init_data->force;   /* set pointer back to engine */

  fnonbond->systemsize = fnonbond->force->systemsize;   /* cubic box */

  if (build_exclusions(fnonbond, init_data->data)) {
    fprintf(stderr, "cannot build up exclusion list\n");
    return MD_FAIL;
  }

  /* init vdw module */
  fnonbond->vdw = my_calloc((size_t)1, sizeof(struct VDW_Tag), "vdw");
  memset(&init_vdw, 0, sizeof(struct VDW_init_Tag));
  init_vdw.data         = init_data->data;
  init_vdw.ppos         = fnonbond->force->wrapped_pos;
  init_vdw.patom        = init_data->data->atom;
  init_vdw.pexcllist    = fnonbond->excllist;
  init_vdw.pscaled14    = fnonbond->scaled14;
  init_vdw.natomprms    = init_data->data->natomprms;
  init_vdw.natoms       = init_data->data->natoms;
  init_vdw.systemsize   = fnonbond->systemsize;
  init_vdw.cutoff       = init_data->data->cutoff_vdw;
  init_vdw.is_switching = init_data->data->is_switching;
  init_vdw.switch_dist  = init_data->data->switchdist_vdw;
  if (vdw_init(fnonbond->vdw, &init_vdw)) {
    fprintf(stderr, "cannot init vdw structure\n");
    return MD_FAIL;
  }

  /* init electro module */
  fnonbond->ewaldmethod = init_data->ewaldmethod;
  if (SPC == fnonbond->force->model_id) {
    if (ES_StandardEwald == init_data->ewaldmethod) {
      struct standEwald_init_Tag eparams;
      memset(&eparams, 0, sizeof eparams);
      eparams.polarizability     = NULL;
      eparams.natoms             = init_data->data->natoms;
      eparams.patom              = init_data->data->atom;
      eparams.wrapped_pos        = fnonbond->force->wrapped_pos;
      eparams.realpos            = init_data->data->pos;
      eparams.pexcllist          = fnonbond->excllist;
      eparams.systemsize         = fnonbond->systemsize;
#ifdef EXACT_EWALD
      eparams.errTol             = 1e-20;
#else
      eparams.errTol             = 1e-6; 
#endif
      eparams.restart            = fnonbond->force->restart;
      eparams.has_induced_dipole = 0;
      fnonbond->electro = my_calloc((size_t)1, sizeof(struct standEwald_Tag),
				    "standard Ewald");
      if (stdEw_init((struct standEwald_Tag *)fnonbond->electro, &eparams)) {
	fprintf(stderr, "cannot init standard Ewald module\n");
	return MD_FAIL;
      };
    } else if (ES_SPME == init_data->ewaldmethod) {
      struct PmeParams_Tag eparams;
      memset(&eparams, 0, sizeof(struct PmeParams_Tag));
      /* other components are zero */
      eparams.cellBasisVector1.x = fnonbond->systemsize.x;
      eparams.cellBasisVector2.y = fnonbond->systemsize.y;
      eparams.cellBasisVector3.z = fnonbond->systemsize.z;
# ifdef EXACT_EWALD
      eparams.PMETolerance       = 1e-20;
# else
      eparams.PMETolerance       = 1e-6; /* 1e-6; */
# endif     
#if 1
      eparams.PMEGridSizeX       = 18;  /* ad hoc */
      eparams.PMEGridSizeY       = 18;
      eparams.PMEGridSizeZ       = 18;
      eparams.cutoff             = 8.0; 
#else 
      eparams.PMEGridSizeX       = 30;  /* ad hoc */
      eparams.PMEGridSizeY       = 30;
      eparams.PMEGridSizeZ       = 30;
      eparams.cutoff             = 0.5 * eparams.cellBasisVector1.x;
#endif
      eparams.PMEInterpOrder     = 6;
      /* eparams.cellOrigin; is (0,0,0) */
      eparams.pexcllist          = fnonbond->excllist;
      eparams.patom              = init_data->data->atom;
      eparams.polarizability     = NULL;
      eparams.ppos               = fnonbond->force->wrapped_pos;
      eparams.natoms             = init_data->data->natoms;
      eparams.restart            = fnonbond->force->restart;
      eparams.has_induced_dipole = 0;
      fnonbond->electro = my_calloc((size_t)1, sizeof(struct Pme_Tag), "pme");
      if (pme_init((struct Pme_Tag*)fnonbond->electro, &eparams)) {
	fprintf(stderr, "cannot init PME\n");
	return MD_FAIL;
      };
    } else {
      fprintf(stderr, "wrong ewaldmethod: %d\n", init_data->ewaldmethod);
      return MD_FAIL;
    };
  } else if (POL1 == fnonbond->force->model_id ||
	     POL3 == fnonbond->force->model_id ||
	     RPOL == fnonbond->force->model_id) {
#ifndef DIPOLE_POLY
    if (ES_StandardEwald == init_data->ewaldmethod) {
      struct standEwald_init_Tag eparams;
      memset(&eparams, 0, sizeof(struct standEwald_init_Tag));
      eparams.polarizability    = init_data->data->polarizability;
      eparams.natoms            = init_data->data->natoms;
      eparams.patom             = init_data->data->atom;
      eparams.wrapped_pos       = fnonbond->force->wrapped_pos;
      eparams.realpos           = fnonbond->force->data->pos;
      eparams.pexcllist         = fnonbond->excllist;
      eparams.systemsize        = fnonbond->systemsize;
# ifdef EXACT_EWALD
      eparams.errTol             = 1e-20;
# else
      eparams.errTol             = 1e-6; /* 1e-6; */
# endif
      eparams.restart           = fnonbond->force->restart;
      eparams.has_induced_dipole= 1;
      eparams.dsolver_param     = init_data->dsolver_param;
      fnonbond->electro = my_calloc((size_t)1, sizeof(struct standEwald_Tag),
				    "standard Ewald");
      if (stdEw_init((struct standEwald_Tag *)fnonbond->electro, &eparams)) {
	fprintf(stderr, "cannot init standard Ewald module\n");
	return MD_FAIL;
      };
    } else if (ES_SPME == init_data->ewaldmethod) {
      struct PmeParams_Tag eparams;
      memset(&eparams, 0, sizeof(struct PmeParams_Tag));
# ifdef EXACT_EWALD
      eparams.PMETolerance       = 1e-20;
# else
      eparams.PMETolerance       = 1e-6; /* 1e-6; */
# endif     
# if 1
      eparams.PMEGridSizeX       = 18;  /* ad hoc */
      eparams.PMEGridSizeY       = 18;
      eparams.PMEGridSizeZ       = 18;
      eparams.cutoff             = 8.0;
# else
      eparams.PMEGridSizeX       = 24;  /* ad hoc */
      eparams.PMEGridSizeY       = 24;
      eparams.PMEGridSizeZ       = 24;
      eparams.cutoff             = 9.344167702602123; /* 6.0; */
      printf("PME dirsum cutoff is modified !!\n");
      /*
      eparams.PMEGridSizeX       = 40; 
      eparams.PMEGridSizeY       = 40;
      eparams.PMEGridSizeZ       = 40;
      eparams.cutoff             = 4.0;
      eparams.PMEGridSizeX       = 32; 
      eparams.PMEGridSizeY       = 32;
      eparams.PMEGridSizeZ       = 32;
      eparams.cutoff             = 5.0;
      */
# endif
      eparams.PMEInterpOrder     = 6;
      eparams.cellBasisVector1.x = fnonbond->systemsize.x;
      eparams.cellBasisVector2.y = fnonbond->systemsize.y;
      eparams.cellBasisVector3.z = fnonbond->systemsize.z;
      /* eparams.cellOrigin; is (0,0,0) */
      eparams.pexcllist          = fnonbond->excllist;
      eparams.patom              = init_data->data->atom;
      eparams.polarizability     = init_data->data->polarizability;
      eparams.ppos               = fnonbond->force->wrapped_pos;
      eparams.natoms             = init_data->data->natoms;
      eparams.restart            = fnonbond->force->restart;
      eparams.has_induced_dipole = 1;
      eparams.dsolver_param      = init_data->dsolver_param;
      fnonbond->electro = my_calloc((size_t)1, sizeof(struct Pme_Tag), "pme");
/* fprintf(stderr, "HERE!\n"); */
      if (pme_init((struct Pme_Tag*)fnonbond->electro, &eparams)) {
	fprintf(stderr, "cannot init PME structure\n");
	return MD_FAIL;
      };      
    } else {
      fprintf(stderr, "wrong ewaldmethod: %d\n", init_data->ewaldmethod);
      return MD_FAIL;
    };
#else /* DIPOLE_POLY */
# if 0
    struct dipole_poly_init init_dp;
    struct StandEwald_init_Tag* e_init = &init_dp.e_init;
    memset(&init_dp, 0, sizeof(struct dipole_poly_init));
    fnonbond->electro = my_calloc((size_t)1, sizeof(struct Dipole_Poly_Tag),
				  "electro, dipole_poly");
    {
      MD_Double emin = 0.73;
      MD_Double emax = 1.33;
      init_dp.a = -1.0/(emin*emax);
      init_dp.b = ((emin+emax)/2.0-1.0+sqrt(emin*emax))/(emin*emax);
    }
    e_init->polarizability    = init_data->data->polarizability;
    e_init->natoms            = init_data->data->natoms;
    e_init->patom             = init_data->data->atom;
    e_init->wrapped_pos       = fnonbond->force->wrapped_pos;
    e_init->realpos           = fnonbond->force->data->pos;
    e_init->pexcllist         = fnonbond->excllist;
    e_init->systemsize        = fnonbond->systemsize;
#  ifdef EXACT_EWALD
    e_init->errTol            = 1e-20;
#  else
    e_init->errTol            = 1e-6; /* 1e-6; */
#  endif
    e_init->restart           = 0;
    e_init->has_induced_dipole= 1;
    fnonbond->electro = my_calloc((size_t)1, sizeof(struct Dipole_Poly_Tag),
				  "electro (dipole_poly)");
    if (dipole_poly_init((struct Dipole_Poly_Tag *)fnonbond->electro, 
			 &init_dp)) {
      fprintf(stderr, "cannot init dipole_poly structure\n");
      return MD_FAIL;
    }
# endif
      struct PmeParams_Tag eparams;
      memset(&eparams, 0, sizeof(struct PmeParams_Tag));
      eparams.PMETolerance       = 1e-6; /* 1e-6; */
      eparams.PMEGridSizeX       = 18;  /* ad hoc */
      eparams.PMEGridSizeY       = 18;
      eparams.PMEGridSizeZ       = 18;
      eparams.cutoff             = 8.0;
      eparams.PMEInterpOrder     = 6;
      eparams.cellBasisVector1.x = fnonbond->systemsize.x;
      eparams.cellBasisVector2.y = fnonbond->systemsize.y;
      eparams.cellBasisVector3.z = fnonbond->systemsize.z;
      /* eparams.cellOrigin; is (0,0,0) */
      eparams.pexcllist          = fnonbond->excllist;
      eparams.patom              = init_data->data->atom;
      eparams.polarizability     = init_data->data->polarizability;
      eparams.ppos               = fnonbond->force->wrapped_pos;
      eparams.natoms             = init_data->data->natoms;
      eparams.restart            = fnonbond->force->restart;
      eparams.has_induced_dipole = 1;
      eparams.dsolver_param      = init_data->dsolver_param;
      fnonbond->electro = my_calloc((size_t)1, sizeof(struct DPoly_Tag),
				  "electro (dpoly)");
      if (dpoly_init((struct DPoly_Tag*)fnonbond->electro, &eparams, 2)) {
	fprintf(stderr, "cannot init dpoly structure\n");
	return MD_FAIL;
      };      
#endif
  } else {
    printf("the model does not have electrostatic potential\n");
  }
      
  return OK;
}


/*
 * destroy the fnonbond object
 *
 * free all memory allocated and clear memory used by fnonbond object
 */
MD_Errcode fnonbond_destroy(struct Fnonbond_Tag *fnonbond)
{
  if (NULL != fnonbond->vdw) {
    if (vdw_destroy(fnonbond->vdw)) {
      fprintf(stderr, "failed to destroy vdw module\n");
      return MD_FAIL;
    }
    free(fnonbond->vdw); 
  } 
  if (NULL != fnonbond->electro) {
#ifndef DIPOLE_POLY
    if (ES_StandardEwald == fnonbond->ewaldmethod) {
      if ( stdEw_destroy((struct standEwald_Tag *) fnonbond->electro) ) {
	fprintf(stderr, "failed to destory standard Ewald module\n");
	return MD_FAIL;
      }
    } else if (ES_SPME == fnonbond->ewaldmethod) {
      if ( pme_destroy((struct Pme_Tag*) fnonbond->electro) ) {
	fprintf(stderr, "failed to destory PME module\n");
	return MD_FAIL;
      }
    } else {
      fprintf(stderr, "wrong method: %d\n", fnonbond->ewaldmethod);
      return MD_FAIL;
    }
#else
    dpoly_destroy((struct DPoly_Tag *)fnonbond->electro);
#endif
    free(fnonbond->electro);  
  }


  {
    MD_Int natoms = 0;
    MD_Int i;

    if (fnonbond->force) natoms = fnonbond->force->natoms;

    if (fnonbond->excllist) {
      for (i=0;  i<natoms;  i++)  free(fnonbond->excllist[i]);
    }
    if (fnonbond->scaled14) {
      for (i=0;  i<natoms;  i++)  free(fnonbond->scaled14[i]);
    }
  }
  free(fnonbond->excllist); 
  free(fnonbond->scaled14); 

  memset(fnonbond, 0, sizeof(struct Fnonbond_Tag)); /* prevent double free */

  return OK;
}



/*
 * compute the fnonbond and the potential
 */
MD_Errcode fnonbond_compute(struct Fnonbond_Tag *fnonbond)
{
#ifdef TIMING
  static MD_Double cost_vdw = 0.0;
  static MD_Double cost_ele = 0.0;
  MD_Double tstart = time_of_day();
#endif
  /* printf("to compute vdw\n"); */
  if (vdw_compute(fnonbond->vdw)) {
    fprintf(stderr, "error in computing vdw force\n");
    return MD_FAIL;
  } 

  /* add long-range-correction, HACK !!!! */
  if (POL1 == fnonbond->force->model_id || 
      POL3 == fnonbond->force->model_id ||
      RPOL == fnonbond->force->model_id) {
    assert(216*3 == fnonbond->vdw->natoms && 2 == fnonbond->vdw->natomprms &&
	   8.0 == fnonbond->vdw->cutoff);
    if (! fnonbond->vdw->is_switching)  {  /* added June 8, 2004 */
      fnonbond->vdw->energy += -19.91663528006115 * KCAL_PER_MOL; 
   }
  }

#ifdef TIMING
  cost_vdw += time_of_day() - tstart;
  /*
  printf("vdw cost: %f second\n", cost_vdw);
  */
#endif

  if (NULL != fnonbond->electro) {
#ifdef TIMING
    tstart = time_of_day(); 
#endif

#ifndef DIPOLE_POLY
    if (ES_StandardEwald == fnonbond->ewaldmethod) {
      struct standEwald_Tag *electro = (struct standEwald_Tag *) 
	fnonbond->electro;
      if ( stdEw_compute(electro) ) { 
	fprintf(stderr, "error in computing electrostatic force\n");
	return MD_FAIL;
      }
      fnonbond->force->data->pdipole = stdEw_get_dipole(electro);
    } else if (ES_SPME == fnonbond->ewaldmethod) {
      struct Pme_Tag* electro = (struct Pme_Tag*) fnonbond->electro;
      if ( pme_compute(electro) ) {
	fprintf(stderr, "error in computing PME force\n");
	return MD_FAIL;
      }
      fnonbond->force->data->pdipole = pme_get_dipole(electro);
    };
#else 
# if 0
    if (dipole_poly_compute((struct Dipole_Poly_Tag *) fnonbond->electro)) { 
      fprintf(stderr, "error in computing dipole_poly force\n");
      return MD_FAIL;
    }
#endif
    if (dpoly_compute((struct DPoly_Tag *) fnonbond->electro)) { 
      fprintf(stderr, "error in computing dpoly force\n");
      return MD_FAIL;
    }
    fnonbond->force->data->pdipole = dpoly_get_dipole((struct DPoly_Tag *) fnonbond->electro);
#endif
#ifdef TIMING
    cost_ele += time_of_day() - tstart;
    /*
    printf("ele cost: %f second\n", cost_ele);
    */
#endif
  }
  return OK;
}



MD_Errcode fnonbond_dump(struct Fnonbond_Tag *fnonbond, const char* file)
{
  switch(fnonbond->ewaldmethod) {
  case ES_NotUse:
    break;
  case ES_StandardEwald:
    return stdEw_dump_dipole((const struct standEwald_Tag *)fnonbond->electro,
			     file);
    break;
  case ES_SPME:
    return pme_dump_dipole((const struct Pme_Tag *)fnonbond->electro, 
			   file);
    break;
  case DIPOLE_POLY_StandEwald: /* break through */
  case DIPOLE_POLY_PME: /* break through */
  default:
    printf("should not be here\n");
    exit(1);
  };

  /* should not be here */
  return OK;
}


static void build_exclusions_freemem(MD_Int natoms,
    MD_Int **exclx, MD_Int *lenx, MD_Int **excl12, MD_Int *len12,
    MD_Int **excl13, MD_Int *len13, MD_Int **excl14, MD_Int **scaled14,
    MD_Int *accum, MD_Int *dest);
static void sort(MD_Int *list, MD_Int len);
static MD_Int merge(MD_Int *dest, const MD_Int *src1, const MD_Int *src2,
    MD_Int n);


/*
 * build the exclusion lists
 *
 * lists are built from MD_Excl and MD_Bond
 *
 * algorithm (using set notation):
 * exclx[i] = { j : there is an explicit exclusion (i,j) }
 * excl12[i] = { j : there is a bond (i,j) }
 * excl13[i] = (excl12[i] U ( U_{ j \in excl12[i] } excl12[j] )) \ {i}
 * excl14[i] = (excl13[i] U ( U_{ j \in excl13[i] } excl12[j] )) \ {i}
 * scaled14[i] = excl14[i] \ excl13[i]
 *
 * fnonbond->excllist[i]  = exclx[i],              if excl_policy == EXCL_NONE
 *                     = exclx[i] U excl12[i],  if excl_policy == EXCL_12
 *                     = exclx[i] U excl13[i],  if excl_policy == EXCL_13
 *                     = exclx[i] U excl14[i],  if excl_policy == EXCL_14
 *
 * fnonbond->excllist[i]  = exclx[i] U excl13[i]
 *     AND
 * fnonbond->scaled14[i]  = scaled14[i],           if excl_policy == SCALED_14
 *
 * allocate little extra memory
 * implement by merging sorted exclusion lists
 * each atom's exclusion array is terminated by INT_MAX sentinel
 */
MD_Errcode build_exclusions(struct Fnonbond_Tag *fnonbond,
			    const struct Data_Tag* data)
{
  MD_Int **exclx, *lenx = NULL;
  MD_Int **excl12 = NULL, *len12 = NULL;
  MD_Int **excl13 = NULL, *len13 = NULL;
  MD_Int **excl14 = NULL;
  MD_Int **scaled14 = NULL;
  MD_Int *accum = NULL, *dest = NULL;
  MD_Int *list;
  MD_Int len, i, j, k, ii, jj, kk, atom1, atom2;
  MD_Int size;     /* allocated length of accum and dest arrays */
  MD_Int maxsize;  /* largest length needed to be held by accum or dest */
  MD_Int accumlen; /* used length of accum (accumlen <= maxsize <= size) */
  const MD_Int natoms = data->natoms;
  const MD_Int nexcls = data->nexcls;
  const MD_Int nbonds = data->nbonds;
  const MD_Excl *excl = data->excl;
  const MD_Bond *bond = data->bond;
  const MD_Int excl_policy = data->excl_policy;

  fnonbond->excllist = NULL;
  fnonbond->scaled14 = NULL;

  if (natoms == 0)  return OK;

  /* allocate memory for explicit exclusions list */
  if ((exclx = calloc((size_t)natoms, sizeof(MD_Int *))) == NULL) {
    build_exclusions_freemem(natoms, exclx, lenx, excl12, len12,
                             excl13, len13, excl14, scaled14, accum, dest);
    printf("build_exclusions call to calloc for exclx\n");
    return MD_FAIL;
  }
  if ((lenx = calloc((size_t)natoms, sizeof(MD_Int))) == NULL) {
    build_exclusions_freemem(natoms, exclx, lenx, excl12, len12,
                             excl13, len13, excl14, scaled14, accum, dest);
    printf("build_exclusions call to calloc for lenx\n");
    return MD_FAIL;
  }

  /* count number of explicit exclusions for each atom */
  for (i = 0;  i < nexcls;  i++) {
    if (excl[i].atom[0] != excl[i].atom[1]) {
      lenx[ excl[i].atom[0] ]++;
      lenx[ excl[i].atom[1] ]++;
    }
  }

  /* allocate memory for each row of exclx, leave space for sentinel */
  for (i = 0;  i < natoms;  i++) {
    exclx[i] = malloc((lenx[i] + 1) * sizeof(MD_Int));
    if (exclx[i] == NULL) {
      build_exclusions_freemem(natoms, exclx, lenx, excl12, len12,
                               excl13, len13, excl14, scaled14, accum, dest);
      printf("build_exclusions call to malloc");
      return MD_FAIL;
    }
    lenx[i] = 0;  /* zero this to be length counter */
  }

  /* loop over explicit exclusions to fill in the rows of exclx */
  for (i = 0;  i < nexcls;  i++) {  /* water specific */
    atom1 = excl[i].atom[0];
    atom2 = excl[i].atom[1];
    exclx[atom1][ lenx[atom1]++ ] = atom2;
    exclx[atom2][ lenx[atom2]++ ] = atom1;
  }

  /* place sentinel at end of each row */
  for (i = 0;  i < natoms;  i++) {
    exclx[i][ lenx[i] ] = INT_MAX;
  }

  /* sort each exclx row */
  for (i = 0;  i < natoms;  i++) {
    sort(exclx[i], lenx[i]);
  }

  /* if we're doing no bond exclusions, we're done */
  if (excl_policy == EXCL_NONE) {
    fnonbond->excllist = exclx;
    build_exclusions_freemem(natoms, NULL, lenx, excl12, len12,
                             excl13, len13, excl14, scaled14, accum, dest);
    return OK;
  }

  /* allocate memory for 1-2 exclusions list */
  if ((excl12 = calloc((size_t)natoms, sizeof(MD_Int *))) == NULL) {
    build_exclusions_freemem(natoms, exclx, lenx, excl12, len12,
                             excl13, len13, excl14, scaled14, accum, dest);
    printf("build_exclusions call to calloc");
    return MD_FAIL;
  }
  if ((len12 = calloc((size_t)natoms, sizeof(MD_Int))) == NULL) {
    build_exclusions_freemem(natoms, exclx, lenx, excl12, len12,
                             excl13, len13, excl14, scaled14, accum, dest);
    printf("build_exclusions call to calloc");
    return MD_FAIL;
  }

  /* find the length of each row of excl12 */
  for (i = 0;  i < nbonds;  i++) {
    len12[ bond[i].atom[0] ]++;
    len12[ bond[i].atom[1] ]++;
  }

  /* allocate memory for each row of excl12 */
  /* leave space for explicit exclusion list and sentinel */
  /* also determine maxsize */
  maxsize = 0;
  for (i = 0;  i < natoms;  i++) {
    if (maxsize < len12[i] + lenx[i] + 1) {
      maxsize = len12[i] + lenx[i] + 1;
    }
    excl12[i] = malloc((len12[i] + lenx[i] + 1) * sizeof(MD_Int));
    if (excl12[i] == NULL) {
      build_exclusions_freemem(natoms, exclx, lenx, excl12, len12,
                               excl13, len13, excl14, scaled14, accum, dest);
      printf( "build_exclusions call to malloc");
      return MD_FAIL;
    }
    len12[i] = 0;   /* zero this to be length counter */
  }

  /* loop over bonds to fill in the rows of excl12 */
  for (i = 0;  i < nbonds;  i++) {
    atom1 = bond[i].atom[0];
    atom2 = bond[i].atom[1];
    excl12[atom1][ len12[atom1]++ ] = atom2;
    excl12[atom2][ len12[atom2]++ ] = atom1;
  }

  /* place sentinel at end of each row */
  for (i = 0;  i < natoms;  i++) {
    excl12[i][ len12[i] ] = INT_MAX;
  }

  /* initialize accum and dest arrays for merge and swap */
  size = 10;
  while (size < maxsize)  size *= 2;
  if ((accum = malloc(size * sizeof(MD_Int))) == NULL) {
    build_exclusions_freemem(natoms, exclx, lenx, excl12, len12,
                             excl13, len13, excl14, scaled14, accum, dest);
    printf("build_exclusions call to malloc");
    return MD_FAIL;
  }
  if ((dest = malloc(size * sizeof(MD_Int))) == NULL) {
    build_exclusions_freemem(natoms, exclx, lenx, excl12, len12,
                             excl13, len13, excl14, scaled14, accum, dest);
    printf("build_exclusions call to malloc");
    return MD_FAIL;
  }

  /* sort each excl12 row */
  for (i = 0;  i < natoms;  i++) {
    sort(excl12[i], len12[i]);
  }

  /* if we're excluding only 1-2 interactions, we're done */
  if (excl_policy == EXCL_12) {
    /* merge each excl12 row with exclx row */
    for (i = 0;  i < natoms;  i++) {
      len = merge(dest, exclx[i], excl12[i], i);
      memcpy(excl12[i], dest, (len + 1) * sizeof(MD_Int));
    }
    fnonbond->excllist = excl12;
    build_exclusions_freemem(natoms, exclx, lenx, NULL, len12,
                             excl13, len13, excl14, scaled14, accum, dest);
    return OK;
  }

  /* allocate memory for 1-3 exclusions list */
  if ((excl13 = calloc((size_t)natoms, sizeof(MD_Int *))) == NULL) {
    build_exclusions_freemem(natoms, exclx, lenx, excl12, len12,
                             excl13, len13, excl14, scaled14, accum, dest);
    printf("build_exclusions call to calloc");
    return MD_FAIL;
  }
  if ((len13 = malloc(natoms * sizeof(MD_Int))) == NULL) {
    build_exclusions_freemem(natoms, exclx, lenx, excl12, len12,
                             excl13, len13, excl14, scaled14, accum, dest);
    printf("build_exclusions call to malloc");
    return MD_FAIL;
  }

  /* merge the excl12 lists into excl13 lists */
  for (i = 0;  i < natoms;  i++) {
    memcpy(accum, excl12[i], (len12[i] + 1) * sizeof(MD_Int));
    accumlen = len12[i];
    for (j = 0;  excl12[i][j] < INT_MAX;  j++) {
      k = excl12[i][j];
      if (k == i) continue;
      maxsize = accumlen + len12[k];
      if (size <= maxsize + lenx[i]) {
        do { size *= 2; } while (size <= maxsize + lenx[i]);
        list = (MD_Int *) realloc(accum, size * sizeof(MD_Int));
        if (list == NULL) {
          build_exclusions_freemem(natoms, exclx, lenx, excl12, len12,
                                 excl13, len13, excl14, scaled14, NULL, dest);
          printf("build_exclusions call to realloc");
          return MD_FAIL;
        }
        accum = list;
        list = (MD_Int *) realloc(dest, size * sizeof(MD_Int));
        if (list == NULL) {
          build_exclusions_freemem(natoms, exclx, lenx, excl12, len12,
                                 excl13, len13, excl14, scaled14, accum, NULL);
          printf("build_exclusions call to realloc");
          return MD_FAIL;
        }
        dest = list;
      }
      accumlen = merge(dest, accum, excl12[k], i);
      list = accum;
      accum = dest;
      dest = list;
    }
    excl13[i] = malloc((accumlen + lenx[i] + 1) * sizeof(MD_Int));
    if (excl13[i] == NULL) {
      build_exclusions_freemem(natoms, exclx, lenx, excl12, len12,
                               excl13, len13, excl14, scaled14, accum, dest);
     printf( "build_exclusions call to malloc");
      return MD_FAIL;
    }
    memcpy(excl13[i], accum, (accumlen + 1) * sizeof(MD_Int));
    len13[i] = accumlen;
  }

  /* if we're excluding 1-2 and 1-3 interactions, we're done */
  if (excl_policy == EXCL_13) {
    /* merge each excl13 row with exclx row */
    for (i = 0;  i < natoms;  i++) {
      len = merge(dest, exclx[i], excl13[i], i);
      /* modified by wei */
      memcpy(excl13[i], dest, (len + 1) * sizeof(MD_Int));
    }
    fnonbond->excllist = excl13;
    build_exclusions_freemem(natoms, exclx, lenx, excl12, len12,
                             NULL, len13, excl14, scaled14, accum, dest);
    return OK;
  }

  /* allocate memory for 1-4 exclusions list */
  if ((excl14 = calloc((size_t)natoms, sizeof(MD_Int *))) == NULL) {
    build_exclusions_freemem(natoms, exclx, lenx, excl12, len12,
                             excl13, len13, excl14, scaled14, accum, dest);
    printf("build_exclusions call to calloc");
    return MD_FAIL;
  }

  /* merge the excl13 lists into excl14 lists */
  for (i = 0;  i < natoms;  i++) {
    memcpy(accum, excl13[i], (len13[i] + 1) * sizeof(MD_Int));
    accumlen = len13[i];
    for (j = 0;  excl13[i][j] < INT_MAX;  j++) {
      k = excl13[i][j];
      if (k == i) continue;
      maxsize = accumlen + len12[k];
      if (size <= maxsize + lenx[i]) {
        do { size *= 2; } while (size <= maxsize + lenx[i]);
        list = (MD_Int *) realloc(accum, size * sizeof(MD_Int));
        if (list == NULL) {
          build_exclusions_freemem(natoms, exclx, lenx, excl12, len12,
                                 excl13, len13, excl14, scaled14, NULL, dest);
          printf("build_exclusions call to realloc");
          return MD_FAIL;
        }
        accum = list;
        list = (MD_Int *) realloc(dest, size * sizeof(MD_Int));
        if (list == NULL) {
          build_exclusions_freemem(natoms, exclx, lenx, excl12, len12,
                                 excl13, len13, excl14, scaled14, accum, NULL);
          printf("build_exclusions call to realloc");
          return MD_FAIL;
        }
        dest = list;
      }
      accumlen = merge(dest, accum, excl12[k], i);
      list = accum;
      accum = dest;
      dest = list;
    }
    excl14[i] = malloc((accumlen + lenx[i] + 1) * sizeof(MD_Int));
    if (excl14[i] == NULL) {
      build_exclusions_freemem(natoms, exclx, lenx, excl12, len12,
                               excl13, len13, excl14, scaled14, accum, dest);
      printf("build_exclusions call to malloc");
      return MD_FAIL;
    }
    memcpy(excl14[i], accum, (accumlen + 1) * sizeof(MD_Int));
  }

  /* if we're excluding 1-2, 1-3, and 1-4 interactions, we're done */
  if (excl_policy == EXCL_14) {
    /* merge each excl14 row with exclx row */
    for (i = 0;  i < natoms;  i++) {
      len = merge(dest, exclx[i], excl14[i], i);
      memcpy(excl14[i], dest, (len + 1) * sizeof(MD_Int));
    }
    fnonbond->excllist = excl14;
    build_exclusions_freemem(natoms, exclx, lenx, excl12, len12,
                             excl13, len13, NULL, scaled14, accum, dest);
    return OK;
  }

  /* allocate memory for scaled 1-4 list */
  if ((scaled14 = calloc((size_t)natoms, sizeof(MD_Int *))) == NULL) {
    build_exclusions_freemem(natoms, exclx, lenx, excl12, len12,
                             excl13, len13, excl14, scaled14, accum, dest);
    printf( "build_exclusions call to calloc");
    return MD_FAIL;
  }

  /* scaled14 list includes everything in excl14 that is not in excl13 */
  for (i = 0;  i < natoms;  i++) {
    ii = jj = kk = 0;
    while (excl14[i][ii] < INT_MAX) {
      if (excl14[i][ii] != excl13[i][jj]) dest[kk++] = excl14[i][ii++];
      else { ii++;  jj++; }
    }
    dest[kk] = INT_MAX;
    scaled14[i] = malloc((kk + 1) * sizeof(MD_Int));
    if (scaled14[i] == NULL) {
      build_exclusions_freemem(natoms, exclx, lenx, excl12, len12,
                               excl13, len13, excl14, scaled14, accum, dest);
      printf( "build_exclusions call to malloc");
      return MD_FAIL;
    }
    memcpy(scaled14[i], dest, (kk + 1) * sizeof(MD_Int));
  }

  /* set pointers to the lists we need to keep */
  /* merge each excl13 row with exclx row */
  for (i = 0;  i < natoms;  i++) {
    len = merge(dest, exclx[i], excl13[i], i);
    memcpy(excl13[i], dest, (len + 1) * sizeof(MD_Int));
  }
  fnonbond->excllist = excl13;
  fnonbond->scaled14 = scaled14;

  /* free the memory that is no longer needed */
  build_exclusions_freemem(natoms, exclx, lenx, excl12, len12,
                           NULL, len13, excl14, NULL, accum, dest);
  return OK;
}



/*
 * reclaim extra memory allocated in build_exclusions
 */
void build_exclusions_freemem(MD_Int natoms, MD_Int **exclx, MD_Int *lenx,
    MD_Int **excl12, MD_Int *len12, MD_Int **excl13, MD_Int *len13,
    MD_Int **excl14, MD_Int **scaled14, MD_Int *accum, MD_Int *dest)
{
  MD_Int k;

  if (exclx != NULL) {
    for (k = 0;  k < natoms;  k++)  free(exclx[k]);
    free(exclx);  exclx = NULL;
  }
  if (excl12 != NULL) {
    for (k = 0;  k < natoms;  k++)  free(excl12[k]);
    free(excl12); excl12 = NULL;
  }
  if (excl13 != NULL) {
    for (k = 0;  k < natoms;  k++)  free(excl13[k]);
    free(excl13);  excl13 = NULL; 
  }
  if (excl14 != NULL) {
    for (k = 0;  k < natoms;  k++)  free(excl14[k]);
    free(excl14);  excl14 = NULL;
  }
  if (scaled14 != NULL) {
    for (k = 0;  k < natoms;  k++)  free(scaled14[k]);
    free(scaled14);  excl14 = NULL;
  }
  if (NULL != lenx) {
    free(lenx);  lenx = NULL;
  }
  if (NULL != len12) {
    free(len12); lenx = NULL;
  }
  if (NULL != len13) {
    free(len13); len13 = NULL;
  }
  if (NULL != accum) {
    free(accum);  accum = NULL;
  }
  if (NULL != dest) {
    free(dest); dest = NULL;
  }
}



/*
 * sort an array of integers
 * (use insertion sort: optimal for short arrays)
 *
 * assume INT_MAX sentinel is at end of array
 */
static void sort(MD_Int *list, MD_Int len)
{
  MD_Int i, j, tmp;

  for (i = len - 2;  i >= 0;  i--) {
    tmp = list[i];
    j = i;
    while (tmp > list[j+1]) {
      list[j] = list[j+1];
      j++;
    }
    list[j] = tmp;
  }
}



/*
 * merge two sorted source arrays into a destination array,
 * keeping destination sorted and deleting duplicate entries
 * and excluding n from being merged (used for the self entry)
 *
 * assume destination array has enough space
 * assume each source array is terminated by sentinel INT_MAX
 * add terminating sentinel INT_MAX to destination array
 *
 * return length of destination (not including sentinel)
 */
MD_Int merge(MD_Int *dest, const MD_Int *src1, const MD_Int *src2, MD_Int n)
{
  MD_Int i = 0, j = 0, k = 0;

  while (src1[i] < INT_MAX || src2[j] < INT_MAX) {
    if      (src1[i] == n) i++;
    else if (src2[j] == n) j++;
    else if (src1[i] < src2[j]) dest[k++] = src1[i++];
    else if (src1[i] > src2[j]) dest[k++] = src2[j++];
    else    { dest[k++] = src1[i++];  j++; }
  }
  dest[k] = INT_MAX;
  return k;
}


