/*
 * Copyright (C) 2004-2006 by Wei Wang.  All rights reserved.
 */


/* 
 * dsolvers.h
 * 
 * dsolver modules uses iterative method to solve the linear system
 *
 *      (A+D)x = b
 * 
 * where D is a diagonal matrix, diag(A) != 0 in general.
 * D and b are given through pointers.
 * A is given implicitly: a function is given for doing 
 * the matrix-vector product of "A * v", where "v" is an
 * arbitrary vector. 
 * 
 * The solver also maintains a (circular) buffer containing the
 * the previous solutions, and use the polynomial
 * extrapolation to predict the solution of the current one.
 * This is considered useful in molecular dynamics simulation.
 * 
 * Picard Method (Successive Substitution) is:
 *        x(n+1) = D^{-1} * (b - A*x(n))
 *
 */


#ifndef DSOLVERS_H
#define DSOLVERS_H

#ifdef __cplusplus
extern "C" {
#endif

#include "mdtypes.h"
#include "data.h"
#include "predictor.h"
#include "preconditioner.h"
#include "utilities.h"


  /* convergence criteria (cc):
   *
   *  (X)   1/n * |x(n+1)-x(n)|^2 < errTol2 
   *
   *  (R)   1/n * residue^2 < errTol2
   */

  typedef enum Dsolver_method {
    Picard=0,  /* Picard method */
    damPicard, /* damped Picard method */
    CG,        /* Conjugate Gradient (CG), cc=conv. crit.=(X) */
    JCG_X,     /* Jacobi-CG, cc=(X) */
    JCG_R,     /* Jacobi-CG, cc=(R) */
    QCG_X,     /* Quickstart CG, cc=(X) */
    PCG_X,     /* preconditioned CG, cc=(X) */
    PCG_R,     /* preconditioned CG, cc=(R) */
    MCG_X,     /* modified CG, peek next x by one Picard iteration */
    Chebyshev, /* preconditioned Chebyshev method, cc=(X) */
    NDmethods
  } Dsolver_method;

  struct Electro_Tag;
    
  /* the abstract function providing matrix-vector multiplication */
  typedef void (*compute_pseudores_type)(const void *electro, 
					 const MD_Double *v, 
					 const MD_Int flag, 
					 MD_Double *res);

  struct Dsolver_Parameters_Type {
    Prediction_Type pred_type; 
    MD_Int pred_degree; 
    Dsolver_method method; 
    MD_Int restart;       
    MD_Int maxiter;
    MD_Double errTol2;
  };

  struct Dsolver_Init_Tag { /* data needed for initializing dsolver */
    void *electro;
    enum ElectroStatic_Solver_Tag ewaldmethod;
    void *mat_vec_mul_mod;
    compute_pseudores_type compute_pseudores;    
    MD_Int matrixsize;
    struct Dsolver_Parameters_Type specified_param;
    MD_Double density;
    MD_Int** neibrlist;       
    MD_Int* numneibrs;       
  };

  struct Dsolver_Tag
  {
    void *electro; /* needed for matrix analysis */
    enum ElectroStatic_Solver_Tag ewaldmethod;
    /* pointers, no memory allocation */
    MD_Double *diag;       /* self-maintained diagonal matrix D */
    MD_Int matrixsize;     /* size of the matrix, vector */

    /* providing the matrix-vector multiplication (A*v) functionality */
    compute_pseudores_type compute_pseudores; 
    void* mat_vec_mul_mod; /* abstract matrix-vector multiplication module */

    /* parameters */
    MD_Double errTol2;
    Dsolver_method method;         /* values in Dsolver_method */

    /* method related */
    MD_Double *workspace;  /* extra workspace needed for iteration methods */ 
    MD_Int maxiter;        /* maximum # of iterations */
    
    struct Preconditioner_Tag* precond;
    struct Predictor_Tag* predictor;

    MD_Double *dipole;     /* dipole solution vector */

    /* used when checking the average number of iterations */
    MD_Int output_freq;    /* frequency for output average mat-vec muls */
  };

  MD_Errcode dsolver_init(struct Dsolver_Tag *dsolver, 
			  struct Dsolver_Init_Tag *init_data);
  MD_Errcode dsolver_destroy(struct Dsolver_Tag *dsolver);

  MD_Errcode dsolver_solve(struct Dsolver_Tag *dsolver);

  /* get the solution pointer */
  MD_Double *dsolver_get_dipole(const struct Dsolver_Tag *dsolver);

  MD_Errcode dsolver_dump_dipole(const struct Dsolver_Tag *dsolver,
				 const char* filename);

#ifdef __cplusplus
}
#endif

#endif
