/*
 * Copyright (C) 2004-2006 by Wei Wang.  All rights reserved.
 *
 * Modified by David Hardy to read in a config file.
 */

/*
 * main program. 
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <assert.h>
#include "mdio/mdio.h"
#include "data.h"
#include "force.h"
#include "utilities.h"
#include "timer.h"
#include "constEnergy.h"
#include "nose_hoover_dynamics.h"
#include "Berendsen_wc.h"
#include "unit.h"
#include "helper.h"


enum MD_CLASS { 
  CONSTANT_ENERGY = 0, NOSE_HOOVER, BERENDSEN_WC, N_MD_CLASS 
}; 


typedef struct DpsimConfig_t {
  enum MD_CLASS mdtype;
  Model_Types model;
  enum ElectroStatic_Solver_Tag elecsolver;
  MD_Int nthermalsteps;
  MD_Int nsteps;
  struct Dsolver_Parameters_Type dsolver_param;

  MD_Int nmolPerLine;
  char *fromRestartFile;
  double temperature;
  double cutoff;
  double switchdist;
  double timestep;
  double dielectric;
  double scaling14;       /* not needed */
  char *exclude;          /* not needed */
  MD_Int firsttimestep;
  MD_Int outputEnergies;
  MD_Int restartfreq;
  MD_Int restartsave;
  char *restartname;
  char *forcedumpname;
  MD_Int dcdfreq;
  char *dcdfile;
  char *dcdthermalfile;
} DpsimConfig;



static void initDpsimConfig(DpsimConfig *dp);
static void cleanupDpsimConfig(DpsimConfig *dp);
static int parse_config_data(DpsimConfig *,
    const mdio_ConfigData *data, MD_Int len);



int main(int argc, char**argv)
{
  struct Data_Tag* data = NULL;
  MD_Int inited = 0;
  /* MD_Int nthermalsteps = -1; */
  /* MD_Int nsteps = -1; */

  /* enum MD_CLASS md_class = CONSTANT_ENERGY; */
  /* Model_Types model = SPC; */  /* ARGON, SPC, RPOL, POL3, POL1 */
  /* enum ElectroStatic_Solver_Tag emodule; */  /* ES_StandardEwald, ES_SPME */
  /* MD_Int restart = 1; */

  DpsimConfig dp;  /* initialize from a config file */
  MD_Double tstart = time_of_day();

  /* parameters specify dsolver module */
  /* method = Picard, damPicard, GaussSeidel, CG, JCG_X, JCG_R, QCG_X, 
             Chebyshev, dampCheby */
  /* struct Dsolver_Parameters_Type dsolver_param; */

  mdio_Config *cfgobj;
  const mdio_ConfigData *cfgdata;
  MD_Int cfgdatalen;

  /* name config file on command line */
  if (argc != 2) {
    fprintf(stderr, "usage:  %s configfile\n", argv[0]);
    exit(0);
  }

  /* read and parse config file */
  if ((cfgobj = mdio_createConfig()) == NULL) {
    exit(1);
  }
  if (mdio_readConfig(cfgobj, argv[1])) {
    exit(1);
  }
  cfgdata = mdio_getDataConfig(cfgobj, &cfgdatalen);
  initDpsimConfig(&dp);
  if (parse_config_data(&dp, cfgdata, cfgdatalen)) {
    exit(1);
  }

  /* we no longer need config object data structure */
  mdio_destroyConfig(cfgobj);


#if 0
  switch (argc) {
  case 7:  /* argon, or spc model */
      md_class = atoi(argv[paramsindex++]);
      model    = atoi(argv[paramsindex++]);
      emodule = atoi(argv[paramsindex++]);
      restart  = atoi(argv[paramsindex++]);
      nthermalsteps = atoi(argv[paramsindex++]);
      nsteps   = atoi(argv[paramsindex++]);
      assert(md_class <  N_MD_CLASS);
      assert(model < NMODELS);
      assert( (ARGON==model && ES_NotUse==emodule) ||
	      (SPC==model && ES_NotUse != emodule && ES_Nsolvers > emodule));
      assert(nthermalsteps >= 0 && nsteps >= 0);
      break;
  case 12:  /* rpol model */
      md_class = atoi(argv[paramsindex++]);
      model = atoi(argv[paramsindex++]);
      emodule = atoi(argv[paramsindex++]);
      restart = dsolver_param.restart = atoi(argv[paramsindex++]);
      nthermalsteps = atoi(argv[paramsindex++]);
      nsteps = atoi(argv[paramsindex++]);
      assert(md_class <  N_MD_CLASS);
      assert(emodule <   ES_Nsolvers);
      assert(RPOL == model || POL3 == model || POL1 == model);
      assert(nthermalsteps >= 0 && nsteps >= 0);
      dsolver_param.method      = atoi(argv[paramsindex++]);  
      dsolver_param.pred_type   = atoi(argv[paramsindex++]);
      dsolver_param.pred_degree = atoi(argv[paramsindex++]);
      dsolver_param.maxiter     = atoi(argv[paramsindex++]);    
      dsolver_param.errTol2     = atof(argv[paramsindex++]);   
      assert(dsolver_param.method < NDmethods);
      assert(dsolver_param.pred_type < NPtypes);
      assert(dsolver_param.pred_degree >= 0);
      assert(dsolver_param.maxiter >= 0);
      assert(dsolver_param.errTol2 >= 0);
      break;
  default:  /* wrong input */
    printf("usage:driver mdtype model esolver restart nthermalsteps nsteps\n");
    printf("   or driver mdtype model esolver restart nthermalsteps nsteps\n"
           "       iteration_method pred_type pred_degree maxiter errTol2\n");
    printf("mdtype: \n");
    printf("  Constant Energy: %d, Nose-Hoover: %d, Berendsen Rescaling: %d\n",
           CONSTANT_ENERGY, NOSE_HOOVER, BERENDSEN_WC);
    printf("Model: \n");
    printf("   ARGON: %d, SPC: %d, RPOL: %d, POL3: %d, POL1: %d\n", 
	   ARGON, SPC, RPOL, POL3, POL1);
    printf("Electrostatic solver: \n");
    printf("   Standard Ewald Sum: %d,   Smooth Particle-Mesh-Ewald: %d\n",
	   ES_StandardEwald, ES_SPME);
    printf("Iteration_methods:\n");
    printf("  Picard:%d, dampPicard:%d, JCG_X:%d, QCG_X:%d, PCG_X:%d, PCG_R:%d, MCG_X:%d, Chebyshev:%d\n", 
           Picard, damPicard, JCG_X, QCG_X, PCG_X, PCG_R, MCG_X, Chebyshev);
    printf("Prediction Type:\n");
    printf("  Polynomial: %d,   Polynomial using difference: %d,\n", 
           Polynomial, dPolynomial);
    printf("  Time-Reversible: %d,   high-order TR: %d,   Zero: %d,\n", 
           Time_Reversible, TR_HO, Zero);
    printf("  Least Square: %d,   Constraint Least Square: %d,\n",
            LeastSquare, ConstraintLeastSquare);
    printf("  Least Square (using difference): %d,\n", dLeastSquare);
    printf("  Constraint Least Square (using difference): %d\n", 
             dConstraintLeastSquare);
    printf("  Time-Reversible Least Square: %d\n", TimeReversibleLeastSquare);
    return MD_FAIL;
  }
#endif
  

  /* check mandatory config file params */
  if (dp.nthermalsteps < 0 || dp.nsteps < 0) {
    fprintf(stderr, "need to specify nThermalSteps and nSteps\n");
    exit(1);
  }
  if (dp.nmolPerLine == 0) {
    fprintf(stderr, "need to specify \"nmolPerLine\"\n");
    exit(1);
  }


  printf("#################################################\n");
  printf("# machine: "); fflush(stdout);
  system("uname -n"); 
  printf("# directory: "); fflush(stdout);
  system("pwd"); 
  printf("# purpose: "); fflush(stdout);
  system("cat note.txt");
  printf("# compile time: %s, %s\n", __TIME__, __DATE__);
  printf("# start running time: "); fflush(stdout);
  system("date");
  printf("# thermalize %d steps, then run %d steps\n", 
	 dp.nthermalsteps, dp.nsteps);
  printf("#################################################\n");
  printf("\n");  fflush(stdout);


  /* allocate and initialize simulation data */
  data = my_calloc((size_t)1, sizeof(struct Data_Tag), "data module");

  /* fill in parts of the data object from dpsim config object */
  data->nmolPerLine = dp.nmolPerLine;
  data->fromRestartFile = dp.fromRestartFile;
  data->temperature = dp.temperature;
  data->cutoff_vdw = dp.cutoff;
  data->switchdist_vdw = dp.switchdist;
  data->timestep = dp.timestep;
  data->dielectric = dp.dielectric;
  data->scaling14 = dp.scaling14;
  data->exclude = dp.exclude;
  data->firststepnum = dp.firsttimestep;
  data->outputEnergies = dp.outputEnergies;
  data->restartfreq = dp.restartfreq;
  data->restartsave = dp.restartsave;
  data->restartname = dp.restartname;
  data->forcedumpname = dp.forcedumpname;
  data->dcdfreq = dp.dcdfreq;
  data->dcdfile = dp.dcdfile;
  data->dcdthermalfile = dp.dcdthermalfile;

  /* setup defaults that work for all models */
  if (NULL == data->exclude) data->exclude = DEFAULT_EXCLUDE;
  if (-1 == data->firststepnum) data->firststepnum = DEFAULT_FIRSTSTEPNUM;
  if (-1 == data->dielectric) data->dielectric = DEFAULT_DIELECTRIC;
  if (-1 == data->scaling14) data->scaling14 = DEFAULT_SCALING14;
  if (-1 == data->outputEnergies) data->outputEnergies=DEFAULT_OUTPUT_ENERGIES;
  if (-1 == data->restartfreq) data->restartfreq = DEFAULT_RESTART_FREQ;
  if (-1 == data->restartsave) data->restartsave = DEFAULT_RESTART_SAVE;
  if (NULL == data->restartname) data->restartname = DEFAULT_RESTART_NAME;
  if (NULL == data->forcedumpname) data->forcedumpname = DEFAULT_FORCEDUMP_NAME;
  if (-1 == data->dcdfreq) data->dcdfreq = DEFAULT_DCD_FREQ;

  /* call initialization routines, based on chosen model */
  printf("to initialize data structure ... \n");
  if (data_routines[dp.model].init_function(data)) {
    fprintf(stderr, "failed to init data\n");
    return FAILURE;
  }

#ifdef DEBUG
  fprintf(stderr, "to output data\n");
  output_data(data); /* for check */
#endif

  /* initialize force */
  {
    struct Force_init_Tag fparams;
    fparams.data = data;
    fparams.emodule = dp.elecsolver;
    fparams.dsolver_param = dp.dsolver_param;
    fprintf(stderr, "to initialize force/MD parameters ....\n");
    if (force_init(data->force, &fparams)) {
      fprintf(stderr, "*** failed to init force\n");
      return FAILURE;   
    }
  }

#ifdef DEBUG
  output_exclusion(data->force);
  fflush(NULL);
#endif

  fprintf(stderr, "to run MD ...\n");

  switch(dp.mdtype) {
  case (CONSTANT_ENERGY):  
    if (constEnergy_MD_run(data->force, dp.nthermalsteps, THERMALIZATION)) { 
      fprintf(stderr, "failed to thermalize with constant energy MD\n"); 
    } 
    printf("thermalized \n");
    if (NULL == data->fromRestartFile
        && data_bindump_image(data, data->firststepnum)) {
      fprintf(stderr, "failed to dump image after thermalization\n");
    }    
    printf("production run \n");
    if (constEnergy_MD_run(data->force, dp.nsteps, PRODUCTION)) { 
      fprintf(stderr, "failed to run constant energy MD\n"); 
    } 
    
    /*
    if (leapfrog_MD_run(data->force,  nthermalsteps, THERMALIZATION)) {
      fprintf(stderr, "failed to thermalize with constant energy MD\n");
    }
    if (leapfrog_MD_run(data->force,  nsteps, PRODUCTION)) {
      fprintf(stderr, "failed to thermalize with constant energy MD\n");
    } 
    */   
    break;

  case (NOSE_HOOVER):
    {
      struct NoseHoover_Tag* nh = my_calloc((size_t)1, 
					    sizeof(struct NoseHoover_Tag),
					    "Nose-Hoover module");
      inited = nosehoover_init(nh, data->force, data, data->temperature);
      assert(OK==inited);

      if (nosehoover_run(nh, dp.nthermalsteps, THERMALIZATION)) {
	fprintf(stderr, "failed to thermalize\n");
      }
      if (NULL == data->fromRestartFile
          && data_bindump_image(data, data->firststepnum)) {
	fprintf(stderr, "failed to dump image after thermalization\n");
      }

      printf("to run MD for %d steps.\n", dp.nsteps);
      if (nosehoover_run(nh, dp.nsteps, PRODUCTION)) {
	fprintf(stderr, "*** failed to run MD completely\n");
	return MD_FAIL;
      }

      printf("to finish\n");
      if (nosehoover_destroy(nh)) {
	fprintf(stderr, "*** failed to destroy nose-hoover module\n");
      }
      free(nh);
      break;
    }
  case (BERENDSEN_WC):
    {
      struct Berendsen_wc_Tag* ber = my_calloc((size_t)1, 
					sizeof(struct Berendsen_wc_Tag),
					"berendsen module");
      inited = ber_init(ber, data->force, data, data->temperature);
      assert(OK == inited);

      if (ber_run(ber, dp.nthermalsteps, THERMALIZATION)) {
	fprintf(stderr, "failed to thermalize\n");
	return FAILURE;
      }
      if (NULL == data->fromRestartFile
          && data_bindump_image(data, data->firststepnum)) {
	fprintf(stderr, "failed to dump image after thermalization\n");
      }

      if (ber_run(ber, dp.nsteps, PRODUCTION)) {
	fprintf(stderr, "*** failed to run MD completely\n");
	return MD_FAIL;
      }

      fprintf(stderr, "to finish\n");
      if (ber_destroy(ber)) {
	fprintf(stderr, "failed to destroy ber module\n");
      }
      free(ber);
      break;
    }
  default:
    fprintf(stderr, "invalid mdtype: %d\n", dp.mdtype);
    exit(1);
  }

  data_routines[dp.model].destroy_function(data);
  cleanupDpsimConfig(&dp);
  printf("done!\n\n");

  printf("time cost is %f seconds\n", time_of_day() - tstart);

  return 0;
}


/******************************************************************************/

void initDpsimConfig(DpsimConfig *dp)
{
  memset(dp, 0, sizeof(DpsimConfig));

  /* for numeric values, use -1 for uninitialized */
  dp->temperature = -1;
  dp->cutoff = -1;
  dp->switchdist = -1;
  dp->timestep = -1;
  dp->dielectric = -1;
  dp->scaling14 = -1;
  dp->firsttimestep = -1;
  dp->outputEnergies = -1;
  dp->restartfreq = -1;
  dp->restartsave = -1;
  dp->dcdfreq = -1;
}


void cleanupDpsimConfig(DpsimConfig *dp)
{
  free(dp->fromRestartFile);
  free(dp->exclude);
  free(dp->restartname);
  free(dp->forcedumpname);
  free(dp->dcdfile);
}


int parse_config_data(DpsimConfig *dp,
    const mdio_ConfigData *data, MD_Int datalen)
{
  MD_Int i;

  /* default values (non-polar, rigid water) */
  dp->mdtype = CONSTANT_ENERGY;
  dp->model = SPC;
  dp->elecsolver = ES_NotUse;
  dp->nthermalsteps = -1;
  dp->nsteps = -1;

  /* recommended default dipole solving params */
  dp->dsolver_param.method = MCG_X;  /* "peek" CG */
  dp->dsolver_param.pred_type = LeastSquare;
  dp->dsolver_param.pred_degree = 6;
  dp->dsolver_param.maxiter = 10;
  dp->dsolver_param.errTol2 = 1e-12;

  dp->temperature = -1;

  /* loop over config file data */
  for (i = 0;  i < datalen;  i++) {

    if (my_strcasecmp(data[i].keyword, "mdtype") == 0) {
      if (my_strcasecmp(data[i].value, "constEnergy") == 0) {
        dp->mdtype = CONSTANT_ENERGY;
      }
      else if (my_strcasecmp(data[i].value, "NoseHoover") == 0) {
        dp->mdtype = NOSE_HOOVER;
      }
      else if (my_strcasecmp(data[i].value, "Berendsen") == 0) {
        dp->mdtype = BERENDSEN_WC;
      }
      else {
        fprintf(stderr, "\"%s = %s\" is invalid\n",
            data[i].keyword, data[i].value);
        return MD_FAIL;
      }
      printf("setting: \"%s = %s\"\n", data[i].keyword, data[i].value);
    }

    else if (my_strcasecmp(data[i].keyword, "model") == 0) {
      if (my_strcasecmp(data[i].value, "ARGON") == 0) {
        dp->model = ARGON;
      }
      else if (my_strcasecmp(data[i].value, "SPC") == 0) {
        dp->model = SPC;
      }
      else if (my_strcasecmp(data[i].value, "RPOL") == 0) {
        dp->model = RPOL;
      }
      else if (my_strcasecmp(data[i].value, "POL3") == 0) {
        dp->model = POL3;
      }
      else if (my_strcasecmp(data[i].value, "POL1") == 0) {
        dp->model = POL1;
      }
      else {
        fprintf(stderr, "\"%s = %s\" is invalid\n",
            data[i].keyword, data[i].value);
        return MD_FAIL;
      }
    }

    else if (my_strcasecmp(data[i].keyword, "elecSolver") == 0) {
      if (my_strcasecmp(data[i].value, "Ewald") == 0) {
        dp->elecsolver = ES_StandardEwald;
      }
      else if (my_strcasecmp(data[i].value, "PME") == 0) {
        dp->elecsolver = ES_SPME;
      }
      else {
        fprintf(stderr, "\"%s = %s\" is invalid\n",
            data[i].keyword, data[i].value);
        return MD_FAIL;
      }
    }

    else if (my_strcasecmp(data[i].keyword, "nThermalSteps") == 0) {
      char c;
      if (sscanf(data[i].value, "%d%c", &(dp->nthermalsteps), &c) != 1) {
        fprintf(stderr, "\"%s = %s\" is invalid\n",
            data[i].keyword, data[i].value);
        return MD_FAIL;
      }
    }

    else if (my_strcasecmp(data[i].keyword, "nSteps") == 0) {
      char c;
      if (sscanf(data[i].value, "%d%c", &(dp->nsteps), &c) != 1) {
        fprintf(stderr, "\"%s = %s\" is invalid\n",
            data[i].keyword, data[i].value);
        return MD_FAIL;
      }
    }

    else if (my_strcasecmp(data[i].keyword, "dipoleSolver") == 0) {
      if (my_strcasecmp(data[i].value, "Picard") == 0) {
        dp->dsolver_param.method = Picard;
      }
      else if (my_strcasecmp(data[i].value, "dampedPicard") == 0) {
        dp->dsolver_param.method = damPicard;
      }
      else if (my_strcasecmp(data[i].value, "CG") == 0) {
        dp->dsolver_param.method = CG;
      }
      else if (my_strcasecmp(data[i].value, "JCG_X") == 0) {
        dp->dsolver_param.method = JCG_X;
      }
      else if (my_strcasecmp(data[i].value, "JCG_R") == 0) {
        dp->dsolver_param.method = JCG_R;
      }
      else if (my_strcasecmp(data[i].value, "QCG_X") == 0) {
        dp->dsolver_param.method = QCG_X;
      }
      else if (my_strcasecmp(data[i].value, "PCG_X") == 0) {
        dp->dsolver_param.method = PCG_X;
      }
      else if (my_strcasecmp(data[i].value, "PCG_R") == 0) {
        dp->dsolver_param.method = PCG_R;
      }
      else if (my_strcasecmp(data[i].value, "MCG_X") == 0) {
        dp->dsolver_param.method = MCG_X;
      }
      else if (my_strcasecmp(data[i].value, "Chebyshev") == 0) {
        dp->dsolver_param.method = Chebyshev;
      }
      else {
        fprintf(stderr, "\"%s = %s\" is invalid\n",
            data[i].keyword, data[i].value);
        return MD_FAIL;
      }
    }

    else if (my_strcasecmp(data[i].keyword, "predictType") == 0) {
      if (my_strcasecmp(data[i].value, "polynomial") == 0) {
        dp->dsolver_param.pred_type = Polynomial;
      }
      else if (my_strcasecmp(data[i].value, "diffPolynomial") == 0) {
        dp->dsolver_param.pred_type = dPolynomial;
      }
      else if (my_strcasecmp(data[i].value, "timeReversible") == 0) {
        dp->dsolver_param.pred_type = Time_Reversible;
      }
      else if (my_strcasecmp(data[i].value, "timeReversible_HighOrder") == 0) {
        dp->dsolver_param.pred_type = Time_Reversible;
      }
      else if (my_strcasecmp(data[i].value, "zero") == 0) {
        dp->dsolver_param.pred_type = Zero;
      }
      else if (my_strcasecmp(data[i].value, "leastSquares") == 0) {
        dp->dsolver_param.pred_type = LeastSquare;
      }
      else if (my_strcasecmp(data[i].value, "constrainLeastSquares") == 0) {
        dp->dsolver_param.pred_type = ConstraintLeastSquare;
      }
      else if (my_strcasecmp(data[i].value, "diffLeastSquares") == 0) {
        dp->dsolver_param.pred_type = dLeastSquare;
      }
      else if (my_strcasecmp(data[i].value, "diffConstrainLeastSquares") == 0) {
        dp->dsolver_param.pred_type = dConstraintLeastSquare;
      }
      else if (my_strcasecmp(data[i].value, "timeReversibleLeastSquares")== 0) {
        dp->dsolver_param.pred_type = TimeReversibleLeastSquare;
      }
      else {
        fprintf(stderr, "\"%s = %s\" is invalid\n",
            data[i].keyword, data[i].value);
        return MD_FAIL;
      }
    }

    else if (my_strcasecmp(data[i].keyword, "predictDegree") == 0) {
      char c;
      if (sscanf(data[i].value, "%d%c", &(dp->dsolver_param.pred_degree), &c)
          != 1) {
        fprintf(stderr, "\"%s = %s\" is invalid\n",
            data[i].keyword, data[i].value);
        return MD_FAIL;
      }
    }

    else if (my_strcasecmp(data[i].keyword, "maxIterations") == 0) {
      char c;
      if (sscanf(data[i].value, "%d%c", &(dp->dsolver_param.maxiter), &c)
          != 1) {
        fprintf(stderr, "\"%s = %s\" is invalid\n",
            data[i].keyword, data[i].value);
        return MD_FAIL;
      }
    }

    else if (my_strcasecmp(data[i].keyword, "errorTolerance") == 0) {
      char c;
      double errtol;
      if (sscanf(data[i].value, "%lf%c", &errtol, &c) != 1) {
        fprintf(stderr, "\"%s = %s\" is invalid\n",
            data[i].keyword, data[i].value);
        return MD_FAIL;
      }
      dp->dsolver_param.errTol2 = errtol * errtol;
    }

    else if (my_strcasecmp(data[i].keyword, "nmolPerLine") == 0) {
      char c;
      if (sscanf(data[i].value, "%d%c", &(dp->nmolPerLine), &c) != 1) {
        fprintf(stderr, "\"%s = %s\" is invalid\n",
            data[i].keyword, data[i].value);
        return MD_FAIL;
      }
    }

    else if (my_strcasecmp(data[i].keyword, "fromRestartFile") == 0) {
      if ((dp->fromRestartFile = strdup(data[i].value)) == NULL) {
        fprintf(stderr, "strdup() out-of-memory\n");
        exit(1);
      }
    }

    else if (my_strcasecmp(data[i].keyword, "temperature") == 0) {
      char c;
      if (sscanf(data[i].value, "%lf%c", &(dp->temperature), &c) != 1) {
        fprintf(stderr, "\"%s = %s\" is invalid\n",
            data[i].keyword, data[i].value);
        return MD_FAIL;
      }
    }

    else if (my_strcasecmp(data[i].keyword, "cutoff") == 0) {
      char c;
      if (sscanf(data[i].value, "%lf%c", &(dp->cutoff), &c) != 1) {
        fprintf(stderr, "\"%s = %s\" is invalid\n",
            data[i].keyword, data[i].value);
        return MD_FAIL;
      }
    }

    else if (my_strcasecmp(data[i].keyword, "switchdist") == 0) {
      char c;
      if (sscanf(data[i].value, "%lf%c", &(dp->switchdist), &c) != 1) {
        fprintf(stderr, "\"%s = %s\" is invalid\n",
            data[i].keyword, data[i].value);
        return MD_FAIL;
      }
    }

    else if (my_strcasecmp(data[i].keyword, "timestep") == 0) {
      char c;
      if (sscanf(data[i].value, "%lf%c", &(dp->timestep), &c) != 1) {
        fprintf(stderr, "\"%s = %s\" is invalid\n",
            data[i].keyword, data[i].value);
        return MD_FAIL;
      }
    }

    else if (my_strcasecmp(data[i].keyword, "dielectric") == 0) {
      char c;
      if (sscanf(data[i].value, "%lf%c", &(dp->dielectric), &c) != 1) {
        fprintf(stderr, "\"%s = %s\" is invalid\n",
            data[i].keyword, data[i].value);
        return MD_FAIL;
      }
    }

    else if (my_strcasecmp(data[i].keyword, "scaling14") == 0) {
      char c;
      if (sscanf(data[i].value, "%lf%c", &(dp->scaling14), &c) != 1) {
        fprintf(stderr, "\"%s = %s\" is invalid\n",
            data[i].keyword, data[i].value);
        return MD_FAIL;
      }
    }

    else if (my_strcasecmp(data[i].keyword, "exclude") == 0) {
      if ((dp->exclude = strdup(data[i].value)) == NULL) {
        fprintf(stderr, "strdup() out-of-memory\n");
        exit(1);
      }
    }

    else if (my_strcasecmp(data[i].keyword, "outputEnergies") == 0) {
      char c;
      if (sscanf(data[i].value, "%d%c", &(dp->outputEnergies), &c) != 1) {
        fprintf(stderr, "\"%s = %s\" is invalid\n",
            data[i].keyword, data[i].value);
        return MD_FAIL;
      }
    }

    else if (my_strcasecmp(data[i].keyword, "firsttimestep") == 0) {
      char c;
      if (sscanf(data[i].value, "%d%c", &(dp->firsttimestep), &c) != 1) {
        fprintf(stderr, "\"%s = %s\" is invalid\n",
            data[i].keyword, data[i].value);
        return MD_FAIL;
      }
    }

    else if (my_strcasecmp(data[i].keyword, "restartfreq") == 0) {
      char c;
      if (sscanf(data[i].value, "%d%c", &(dp->restartfreq), &c) != 1) {
        fprintf(stderr, "\"%s = %s\" is invalid\n",
            data[i].keyword, data[i].value);
        return MD_FAIL;
      }
    }

    else if (my_strcasecmp(data[i].keyword, "restartsave") == 0) {
      if (my_strcasecmp(data[i].value, "yes") == 0
          || my_strcasecmp(data[i].value, "on") == 0) {
        dp->restartsave = 1;  /* Boolean TRUE */
      }
      else if (my_strcasecmp(data[i].value, "no") != 0
          && my_strcasecmp(data[i].value, "off") != 0) {
        fprintf(stderr, "\"%s = %s\" is invalide\n",
            data[i].keyword, data[i].value);
        return MD_FAIL;
      }
      else {
        dp->restartsave = 0;  /* Boolean FALSE */
      }
    }

    else if (my_strcasecmp(data[i].keyword, "restartname") == 0) {
      if ((dp->restartname = strdup(data[i].value)) == NULL) {
        fprintf(stderr, "strdup() out-of-memory\n");
        exit(1);
      }
    }

    else if (my_strcasecmp(data[i].keyword, "forcedumpname") == 0) {
      if ((dp->forcedumpname = strdup(data[i].value)) == NULL) {
        fprintf(stderr, "strdup() out-of-memory\n");
        exit(1);
      }
    }

    else if (my_strcasecmp(data[i].keyword, "dcdfreq") == 0) {
      char c;
      if (sscanf(data[i].value, "%d%c", &(dp->dcdfreq), &c) != 1) {
        fprintf(stderr, "\"%s = %s\" is invalid\n",
            data[i].keyword, data[i].value);
        return MD_FAIL;
      }
    }

    else if (my_strcasecmp(data[i].keyword, "dcdfile") == 0) {
      if ((dp->dcdfile = strdup(data[i].value)) == NULL) {
        fprintf(stderr, "strdup() out-of-memory\n");
        exit(1);
      }
    }

    else if (my_strcasecmp(data[i].keyword, "dcdthermalfile") == 0) {
      if ((dp->dcdthermalfile = strdup(data[i].value)) == NULL) {
        fprintf(stderr, "strdup() out-of-memory\n");
        exit(1);
      }
    }

    else {
      fprintf(stderr, "unrecognized keyword:  \"%s\"\n", data[i].keyword);
      return MD_FAIL;
    }

  }

  /* print out parameters? */

  return 0;
}
