/*
 * Copyright (C) 2004-2006 by Wei Wang.  All rights reserved.
 *
 * Modified by David Hardy to incorporate global config params.
 */

/*
 * simple vdw liquid:
 *
 * 
 */

#include <math.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <assert.h>
#include "data.h"
#include "force.h"
#include "helper.h"
#include "random.h"
#include "unit.h"

/* model parameters */
static const MD_Double EPS = 120.0 * KELVIN;  /* internal unit */
static const MD_Double SIGMA = 3.405;  /* Angstrom */
static const MD_String ATOM_NAME = "Argon";
static const MD_Double ARGON_MASS = 39.95;   /* unit: AMU */
static const MD_Double SizeFactor = 10.229; /* length = SIGMA * SizeFactor */

/*
 * Exclusion strings
 */
static const char *EXCL_STRING[] = {
  "none",
  "1-2",
  "1-3",
  "1-4",
  "scaled1-4"
};

/* 
 * local functions 
 */

/* initialize data structures, return the amount of allocated 
 * memory in bytes */
static MD_Errcode argon_data_init_value(struct Data_Tag * data);
static MD_Int init_atomprm(MD_Atom_Param **atomprm, const MD_Int natomprms);
static MD_Int init_atom(MD_Atom **patom, const MD_Int Nmols);
static MD_Int init_position(MD_Dvec **ppos,
			    const MD_Double Len, const MD_Int Ncube_Line, 
			    const MD_Int Natoms) ;
static MD_Int init_velocity(MD_Dvec **pvel, const MD_Int Natoms, 
			    const MD_Double temp);
static MD_Int init_excllist(MD_Excl **pexcl);


MD_Errcode argon_data_init(struct Data_Tag *data)
{

  assert(NULL != data);
  data->model_id = ARGON;

  printf("ARGON GAS\n");
  /* allocate memory for "force" */
  data->force = my_calloc((size_t)1, sizeof(struct Force_Tag), "force module");
  printf("  argon_data_init allocates %d bytes for Force\n",
         (MD_Int) sizeof(struct Force_Tag));

  /* initialize simulation parameters */
  if (-1 == data->temperature) data->temperature = DEFAULT_ARGON_TEMPERATURE;
  if (-1 == data->timestep) data->timestep = DEFAULT_ARGON_TIMESTEP;
  if (-1 == data->dielectric) data->dielectric = DEFAULT_DIELECTRIC;
  if (-1 == data->scaling14) data->scaling14 = DEFAULT_SCALING14;

  data->is_switching = 1;

  return argon_data_init_value(data);
}



/* 
 * note that all values are 0 initially 
 * assume called after argon_data_init
 */
MD_Errcode argon_data_init_value(struct Data_Tag * data)
{
  const MD_Double Len = SizeFactor * SIGMA;   /* angstrom */
  const MD_Int NCube_Line = data->nmolPerLine;
  const MD_Int natoms = 4 * NCube_Line * NCube_Line * NCube_Line ;
  struct Force_Tag *force = data->force;
  const MD_Double Temperature = data->temperature;  /* Kelvin , not used yet */

  MD_Int alloc_size = 0;

  /* 
   * init force's lattice vector 
   */
  assert(NULL != force);
  data->systemsize.x = data->systemsize.y = data->systemsize.z = Len;

  /* 
   * init atom parameter array for vdw computation 
   */
  data->natomprms = 1;  /* only argon */
  alloc_size += init_atomprm(&(data->atomprm), data->natomprms);

  /* 
   * init atom charge, mass, type, name (data->atom)
   */
  data->natoms = natoms;
  alloc_size += init_atom(&(data->atom), data->natoms);
  printf("  system has %d atoms.\n", data->natoms);

  /* 
   * init position and velocity, the overall linear momentum is 0
   */
  if (data->fromRestartFile) {
    MD_Int size;
    printf("Restart, read position and velocity from file \"%s\"\n",
        data->fromRestartFile);
    size = binread_pos_vel(&(data->pos), &(data->vel),
                        natoms, data->fromRestartFile, &(data->firststepnum));
    if (size < 0) {
      return MD_FAIL;
    } else {
      alloc_size += size;
    }
  } else {
    printf("to initilize position velocity (not reading from files) \n");
    alloc_size += init_position(&(data->pos),
			      Len, NCube_Line, natoms);
    alloc_size += init_velocity(&(data->vel), natoms, Temperature);
  } 


  /* 
   * init explicit direct exclusion list
   * in each molecule, O-H1, O-H2, H1-H2 are exclusion pairs.
   */
  data->nexcls = 0;
  alloc_size += init_excllist(&(data->excl));

  /* nearest image convention */
  if (-1 == data->cutoff_vdw) data->cutoff_vdw = Len;
  if (-1 == data->switchdist_vdw) data->switchdist_vdw = data->cutoff_vdw;

  if (argon_data_validate_simparam(data)) {
    fprintf(stderr,"*** invalid parameter \n");
    return MD_FAIL;
  }

  printf("  data_init_value allocate memory: %d bytes\n", alloc_size);

  /* the total linear momentum is 0 */
  data->degree_freedom = data->natoms * 3 - 3;               

  return OK; 
}



MD_Int init_atomprm(MD_Atom_Param **atomprm, const MD_Int natomprms) 
{
  assert(1 == natomprms);
  *atomprm = my_calloc((size_t)natomprms, sizeof(MD_Atom_Param), "atomprm");
  /* only one type of atom: Argon */
  (*atomprm)[0].emin = -EPS;
  (*atomprm)[0].rmin =  pow(2.0, 1.0/6.0) * SIGMA;  
  strncpy((*atomprm)[0].type, ATOM_NAME, sizeof(MD_Short_String)-1);

  return natomprms * sizeof(MD_Atom_Param);
}


MD_Int init_atom(MD_Atom **patom, const MD_Int natoms)
{
  MD_Atom *atom;
  MD_Int i;

  atom = *patom = my_calloc((size_t)natoms, sizeof(MD_Atom), "MD_Atom");

  for (i = 0; i < natoms; i++) {
    atom[i].m = ARGON_MASS;  /* amu */
    atom[i].prm = 0;    /* paramter type */
    strncpy(atom[i].name, ATOM_NAME, sizeof(MD_Short_String) - 1);
    strncpy(atom[i].type, ATOM_NAME, sizeof(MD_Short_String) - 1);
  }

  return natoms * sizeof(MD_Atom);
}


MD_Int init_position(MD_Dvec **ppos, const MD_Double Len, 
                     const MD_Int Ncube_Line, const MD_Int natoms) 
{
  MD_Dvec *pos = NULL;
  MD_Int ix, iy, iz, i;
  const MD_Double Neg_half_len = - 0.5 * Len;
  const MD_Double dl = Len / (MD_Double) (Ncube_Line+1);
  MD_Dvec corner, fac1, fac2, fac3, delta, dis;

  assert(4 * Ncube_Line * Ncube_Line * Ncube_Line == natoms);

  pos = *ppos = my_calloc((size_t)natoms, sizeof(MD_Dvec),"pos" );

  delta.x = delta.y = delta.z = dl; /* length of each cube */
  dis.x = Neg_half_len + 0.5 * delta.x;
  dis.y = Neg_half_len + 0.5 * delta.y;
  dis.z = Neg_half_len + 0.5 * delta.z;
  /* displacement of the face center atoms relative to the corner atom */
  fac1.x = 0.5 * delta.x;
  fac1.y = 0.5 * delta.y;
  fac1.z = 0.0;
  fac2.x = 0.5 * delta.x;
  fac2.y = 0.0;
  fac2.z = 0.5 * delta.z;
  fac3.x = 0.0;
  fac3.y = 0.5 * delta.y;
  fac3.z = 0.5 * delta.z;
  i = 0;
  for (ix = 0; ix < Ncube_Line; ix++) {
    for (iy = 0; iy < Ncube_Line; iy++) {
      for (iz = 0; iz < Ncube_Line; iz++) {
	/* for each cubic, there are 4 atoms, 3 face center, one corner */
	corner.x = dis.x + delta.x * (MD_Double) ix;
	corner.y = dis.y + delta.y * (MD_Double) iy;
	corner.z = dis.z + delta.z * (MD_Double) iz;
	pos[i] = corner;
	i++;
	MD_vec_add(corner, fac1, pos[i]);
	i++;
	MD_vec_add(corner, fac2, pos[i]);
	i++;
	MD_vec_add(corner, fac3, pos[i]);
	i++;
      }
    }
  }

  assert(natoms == i);

  return natoms * sizeof(MD_Dvec);
}


MD_Int init_velocity(MD_Dvec **pvel, const MD_Int natoms, const MD_Double temp)
{
  MD_Double v_sigma = sqrt(KELVIN * temp / ARGON_MASS);
  MD_Dvec *vel = NULL;
  MD_Dvec velsum = {0.0, 0.0, 0.0};
  MD_Double inv_natom = 1.0 / (MD_Double) natoms;
  MD_Int i;

  vel = *pvel = my_calloc((size_t)natoms, sizeof(MD_Dvec), "velocity");

  for (i = 0; i < natoms; i++) {
    velsum.x += vel[i].x = v_sigma * rannyu_normal();
    velsum.y += vel[i].y = v_sigma * rannyu_normal();
    velsum.z += vel[i].z = v_sigma * rannyu_normal();
  }

  /* make the total velocity 0 */
  MD_vec_mul(velsum, inv_natom, velsum);
  for (i = 0; i < natoms; i++) {
    vel[i].x -= velsum.x;
    vel[i].y -= velsum.y;
    vel[i].z -= velsum.z;
  }

  return natoms * sizeof(MD_Dvec); /* initial velocity is 0 */
}


MD_Int init_excllist(MD_Excl **pexcl)
{
  *pexcl = NULL;  /* no exclusion for argon liquid */
  return 0;
}



/*
 * destroy Data.
 */
void argon_data_destroy(struct Data_Tag *data)
{
  fprintf(stderr, "to free data->force\n");
  if (data->force) {
    if (force_destroy(data->force)) {
      fprintf(stderr, "failed to destroy force\n");
      return;
    }
  }
  free(data->force);
  fprintf(stderr, "to free memory allocated for data\n");
  if (data->atomprm) free(data->atomprm);
  if (data->bondprm) free(data->bondprm);
  if (data->angleprm) free(data->angleprm);
  if (data->dihedprm) free(data->dihedprm);
  if (data->imprprm) free(data->imprprm);
  if (data->nbfixprm) free(data->nbfixprm);
  if (data->atom) free(data->atom);
  if (data->bond) free(data->bond);
  if (data->angle) free(data->angle);
  if (data->dihed) free(data->dihed);
  if (data->impr) free(data->impr);
  if (data->excl) free(data->excl);
  if (data->pos) free(data->pos);
  if (data->vel) free(data->vel);
  if (data->polarizability) free(data->polarizability);

  return;
}


/*
 * validate the simulation parameters
 */
MD_Errcode argon_data_validate_simparam(struct Data_Tag *data)
{
  MD_Int k;

  if (data->dielectric < 1.0) {
    printf("argon_data_validate_simparam: "
           "dielectric constant should be >= 1");
    return MD_FAIL;
  }
  if (data->scaling14 > 1.0 || data->scaling14 < 0.0) {
    printf("argon_data_validate_simparam: "
           "1-4 scaling should be between 0 and 1 inclusive");
    return MD_FAIL;
  }
  if (data->switchdist_vdw <= 0.0) {
    printf("argon_data_validate_simparam: "
           "switchdist should be > 0");
    return MD_FAIL;
  }
  if (data->cutoff_vdw <= 0.0) {
    printf("argon_data_validate_simparam: "
           "cutoff should be > 0");
    return MD_FAIL;
  }
  if (data->switchdist_vdw > data->cutoff_vdw) {
    printf("argon_data_validate_simparam: "
           "switchdist should be <= cutoff");
    return MD_FAIL;
  }

  /* set exclusion policy */
  data->excl_policy = -1;
  for (k = EXCL_NONE;  k < NEXCLS;  k++) {
    if (my_strcasecmp(data->exclude, EXCL_STRING[k]) == 0) {
      data->excl_policy = k;
    }
  }
  if (data->excl_policy < 0) {
    printf("argon_data_validate_simparam: unrecognized exclusion policy.");
    return MD_FAIL;
  }

  return OK;
}



