/*
 * Copyright (C) 2004-2006 by Wei Wang.  All rights reserved.
 *
 * Modified by David Hardy to include fundamental parameters in the
 * global data structure.
 */

/*
 * data.h
 *
 * Data object contains all data needed for simulation.
 * In particular, the data arrays shared with front end.
 */

#ifndef DATA_H
#define DATA_H

#include "mdio/mdio.h"
#include "mdtypes.h"
#include "utilities.h"

#ifdef __cplusplus
extern "C" {
#endif


  /*
   * Indices into reduction array
   */
  enum {
    PE_BOND = 0, PE_ANGLE, PE_DIHED, PE_IMPR,
    PE_ELEC, PE_VDW, 
    PE, KE, TE,
    NREDUCS
  };


  typedef enum Model_Types_Tag {
    ARGON=0, SPC, RPOL, POL3, POL1,
    NMODELS
  } Model_Types;


  /*
   * forward declaration
   */
  struct Force_Tag;


  /*
   * Data contains the state of the simulation engine
   */
  struct Data_Tag {
    /* force field computation */
    struct Force_Tag *force;

    /* force field parameter arrays */
    MD_Int natomprms;
    MD_Int nbondprms;
    MD_Int nangleprms;
    MD_Int ndihedprms;
    MD_Int nimprprms;
    MD_Int nnbfixprms;
    MD_Atom_Param  *atomprm;
    MD_Bond_Param  *bondprm;  /* unused */
    MD_Angle_Param *angleprm;
    MD_Dihed_Param *dihedprm;
    MD_Impr_Param  *imprprm;
    MD_Nbfix_Param *nbfixprm;

    /* topology arrays */
    MD_Int natoms;
    MD_Int nbonds;      /* unused */
    MD_Int nangles;
    MD_Int ndiheds;
    MD_Int nimprs;
    MD_Int nexcls;
    MD_Atom  *atom;
    MD_Bond  *bond;    /* unused */
    MD_Angle *angle;
    MD_Dihed *dihed;
    MD_Impr  *impr;
    MD_Excl  *excl;
    MD_Dvec systemsize;

    /* trajectory arrays, each of length natoms */
    MD_Dvec *pos;
    MD_Dvec *vel;
    MD_Double *polarizability;  /* polarizability, size of 3*natoms */
    const MD_Double *pdipole;   /* pointer to induced dipole */

    /* error values */
    MD_Errcode err_simparam;
    MD_Errcode err_unstable;

    /* energy reductions from force computation */
    /* use derived units:  (AMU * A^2 / fs^2)  */
    MD_Double energy[NREDUCS];

    /* simulation parameters */
    MD_Double timestep;
    MD_Double dielectric;
    MD_Double scaling14;

    MD_Double switchdist_vdw;  /* for vdw force only */
    MD_Double cutoff_vdw;

    char *exclude;
    MD_Int excl_policy;
    MD_Int is_switching;
    MD_Int firststepnum;

    /* spherical boundary conditon is dropped */
    /* stuff needed for mdapi set and get is dropped */

    /* for thermodynamics use */
    MD_Int degree_freedom;

    Model_Types model_id;
    MD_Double bond_len, bond_angle;  /* temporary, for water only */

    /* added by DJH */
    MD_Int nmolPerLine;
    char *fromRestartFile;
    double temperature;
    MD_Int outputEnergies;
    MD_Int restartfreq;
    MD_Int restartsave;
    MD_Int prevsavestep;
    char *restartname;
    char *forcedumpname;
    MD_Int dcdfreq;
    char *dcdfile;
    char *dcdthermalfile;
    mdio_Dcd *dcd;         /* DCD file writer object */
    mdio_DcdCell dcdcell;  /* need to store periodic cell info */
  };

  MD_Errcode pol1_data_init(struct Data_Tag *data);
  void pol1_data_destroy(struct Data_Tag *data);
  MD_Errcode pol1_data_validate_simparam(struct Data_Tag *data);

  MD_Errcode pol3_data_init(struct Data_Tag *data);
  void pol3_data_destroy(struct Data_Tag *data);
  MD_Errcode pol3_data_validate_simparam(struct Data_Tag *data);

  MD_Errcode rpol_data_init(struct Data_Tag *data);
  void rpol_data_destroy(struct Data_Tag *data);
  MD_Errcode rpol_data_validate_simparam(struct Data_Tag *data);

  MD_Errcode spc_data_init(struct Data_Tag *data);
  void spc_data_destroy(struct Data_Tag *data);
  MD_Errcode spc_data_validate_simparam(struct Data_Tag *data);

  MD_Errcode argon_data_init(struct Data_Tag *data);
  void argon_data_destroy(struct Data_Tag *data);
  MD_Errcode argon_data_validate_simparam(struct Data_Tag *data);

  typedef MD_Errcode (*Init_Data_Type)(struct Data_Tag *data);
  typedef void (*Destroy_Data_Type)(struct Data_Tag *data);
  typedef MD_Errcode (*Validate_Data_Simparam_Type)(struct Data_Tag *data);
  typedef struct Model_Data_Type_Tag  {
    Init_Data_Type init_function;
    Destroy_Data_Type destroy_function;
    Validate_Data_Simparam_Type validate_function;
    MD_String name;
  } Model_Data_Type;

  extern Model_Data_Type data_routines[NMODELS];

  void output_data(const struct Data_Tag *data);
  /* for isotropic system */
  MD_Double compute_pressure(const MD_Dvec *force, 
			     const MD_Dvec *wrapped_pos, 
			     const MD_Dvec *vel,
			     const MD_Atom *atom,
			     const MD_Double volume, 
			     const MD_Int natoms);
  MD_Double compute_KE(const MD_Dvec *vel, const MD_Atom *atom, 
		       const MD_Int natoms);
  void compute_KETE(struct Data_Tag *data); /* kinetic and total energy */
  void data_print_energy(struct Data_Tag *data, const MD_Int istep); 
  MD_Int conserve_linear_momentum(struct Data_Tag *data);
  /* for restart use */ 
  MD_Errcode data_bindump_image(struct Data_Tag *data, 
				const MD_Int istep);

  /* in output_pdb.c */
  int output_pdb(const struct Data_Tag *data, const char *fname);

  /* in output_dcd.c */
  int output_dcd_header(struct Data_Tag *data, const enum RUN_TYPE run_type);
  int output_dcd_frame(struct Data_Tag *data, MD_Int istep);
  int output_dcd_done(struct Data_Tag *data);


  /*
   * default simparam values
   */
#if 0
#define  DEFAULT_TIMESTEP              1.0          /* femtoseconds */
#define  DEFAULT_DIELECTRIC            1.0          /* unitless */
#define  DEFAULT_SCALING14             1.0          /* unitless */
#define  DEFAULT_SWITCHDIST            10.0         /* angstroms */
#define  DEFAULT_CUTOFF                12.0         /* angstroms */
#define  DEFAULT_EXCLUDE               "none"
#define  DEFAULT_IS_SWITCHING          1
#define  DEFAULT_STEPNUM               0
#endif

#define  DEFAULT_EXCLUDE                "1-3"
#define  DEFAULT_FIRSTSTEPNUM           0
#define  DEFAULT_DIELECTRIC             1.0         /* unitless */
#define  DEFAULT_SCALING14              1.0         /* unitless */
#define  DEFAULT_OUTPUT_ENERGIES        10
#define  DEFAULT_RESTART_FREQ           1000
#define  DEFAULT_RESTART_SAVE           0
#define  DEFAULT_RESTART_NAME           "pos_vel"
#define  DEFAULT_FORCEDUMP_NAME         "dipoles"
#define  DEFAULT_DCD_FREQ               0

#define  DEFAULT_POL_TEMPERATURE        300.0       /* Kelvin */
#define  DEFAULT_POL_TIMESTEP           1.0         /* femtoseconds */
#define  DEFAULT_POL_SWITCHDIST         6.0         /* Angstroms */
#define  DEFAULT_POL_CUTOFF             8.0         /* Angstroms */

#define  DEFAULT_SPC_TEMPERATURE        300.0       /* Kelvin */
#define  DEFAULT_SPC_TIMESTEP           1.0         /* femtoseconds */
#define  DEFAULT_SPC_SWITCHDIST         6.0         /* Angstroms */
#define  DEFAULT_SPC_CUTOFF             8.5         /* Angstroms */

#define  DEFAULT_ARGON_TEMPERATURE      94.4        /* Kelvin */
#define  DEFAULT_ARGON_TIMESTEP         10.0        /* femtoseconds */

  /*
   * internal constants
   */
#define  MAX_FILENAME_LEN               512

  /*
   * Exclusion policy numbers
   */
  enum {
    EXCL_NONE = 0, EXCL_12, EXCL_13, EXCL_14, SCALED_14, NEXCLS
  };


  /*  store position and velocity. */
  void 
  dump_pos_vel(const struct Data_Tag *data, const char *filename, 
	       const MD_Int nstep);

  /* read in the data stored by dump_pos_vel */
  MD_Errcode read_pos_vel(MD_Dvec **ppos, 
			  MD_Dvec **pvel, const MD_Int natoms,
			  MD_Char *filename, MD_Int *nstep);

  /* store position and velocity in binary, used for exactly resuming MD */
  MD_Errcode bindump_pos_vel(const struct Data_Tag *data, 
			     const char *filename, const MD_Int nstep);

  /* read in data store by bindump_pos_vel */
  MD_Errcode binread_pos_vel(MD_Dvec **ppos, 
			     MD_Dvec **pvel, const MD_Int natoms,
			     const MD_Char *filename, MD_Int *pnstep);

  /* compute the magnitude of the permanent, induced and total dipole of a 
     molecule averaged over all molecules, and the overall dipole moment 
     return value unit: Debye */
  void compute_mol_sys_dipole(const MD_Atom atom[],
			      const MD_Dvec realpos[],
			      const MD_Double idipole[],  /* induced dipole */
			      const MD_Int natoms,
			      MD_Double *mol_perm_dipole,
			      MD_Double *mol_induc_dipole, 
			      MD_Double *mol_total_dipole,
			      MD_Dvec *system_total_dipole);
#ifdef __cplusplus
}
#endif

#endif /* DATA_H */
