/*
 * Copyright (C) 2004-2005 by David J. Hardy.  All rights reserved.
 *
 * update.c
 */

#include <stdlib.h>
#include <string.h>
#include <strings.h>
#include <math.h>
#include "mdapi/mdengine.h"
#include "force/force.h"
#include "mgrid/mgrid.h"
#include "deven/engine.h"
#undef DEBUG_WATCH
#include "debug/debug.h"


enum { FALSE = 0, TRUE = 1 };


int32 deven_update(MD_Front *front)
{
  Engine *eng = MD_engine_data(front);
  Param *param = &(eng->param);
  ForceParam *p = &(eng->f_param);
  ForceDomain *d = &(eng->f_domain);
  ForceResult *r = &(eng->f_result);
  Mgrid *mg = &(eng->mgrid);
  MgridParam *mgprm = &(eng->mgrid_param);
  MgridSystem *mgsys = &(eng->mgrid_system);
  PmetestParams *ptprm = &(eng->pt_param);
  PmetestSystem *ptsys = &(eng->pt_system);
  Step *step = eng->step;
  StepParam *step_param = &(eng->step_param);
  int32 is_init_steplib = FALSE;
  int32 is_init_temperature = FALSE;
#if 0
  int32 do_remove_com_motion = FALSE;
#endif

  /* make sure number of atoms is consistent */
  eng->natoms = eng->atom->attrib.len;
  if (eng->natoms <= 0
      || eng->natoms != eng->pos->attrib.len
      || eng->natoms != eng->vel->attrib.len) {
    return MD_error(front, eng->err_param);
  }

  /* handle fixed atoms */
  if (eng->fixedatomslist->attrib.access & MD_MODIFY) {
    eng->nfixedatoms = eng->fixedatomslist->attrib.len;
    eng->fixedatom = (int32 *) (eng->fixedatomslist->buf);
    if (MD_engdata_ackmod(front, eng->fixedatomslist)) {
      return MD_error(front, eng->err_force_init);
    }
    INT(eng->nfixedatoms);
  }

  /* handle harmonic constraints */
  if (eng->constraintlist->attrib.access & MD_MODIFY) {
    eng->nconstraints = eng->constraintlist->attrib.len;
    eng->constraint = (DevenConstraint *) (eng->constraintlist->buf);
    if (MD_engdata_ackmod(front, eng->constraintlist)) {
      return MD_error(front, eng->err_force_init);
    }
    INT(eng->nconstraints);
  }

  /* set flag if mgrid is "on" */
  eng->ismgrid = FALSE;
  if (strcasecmp(param->mgrid, "on") == 0
      || strcasecmp(param->mgrid, "yes") == 0) {
    eng->ismgrid = TRUE;
  }
  else if (strcasecmp(param->mgrid, "off") != 0
      && strcasecmp(param->mgrid, "no") != 0) {
    return MD_error(front, eng->err_force_init);
  }

  /* set flag if pmetest is "on" */
  eng->ispmetest = FALSE;
  if (strcasecmp(param->pmetest, "on") == 0
      || strcasecmp(param->pmetest, "yes") == 0) {
    eng->ispmetest = TRUE;
  }
  else if (strcasecmp(param->pmetest, "off") != 0
      && strcasecmp(param->pmetest, "no") != 0) {
    return MD_error(front, eng->err_force_init);
  }

  /* can use either mgrid or pmetest, but not both */
  if (eng->ismgrid && eng->ispmetest) {
    return MD_error(front, eng->err_force_init);
  }

  /* make sure force array is correct length */
  /*** lock several arrays to this length, requires resizing all of them ***/
  if (eng->natoms != eng->force_engdata->attrib.len) {

    /* resize force array to correct length */
    if (MD_engdata_setlen(front, eng->force_engdata, eng->natoms)) {
      return MD_FAIL;
    }

    /* make sure force result uses force array as its buffer space */
    memset(r, 0, sizeof(ForceResult));
    r->f_total = (MD_Dvec *) (eng->force_engdata->buf);

    /* make sure step uses this force array */
    is_init_steplib = TRUE;
#if 0
    step_system->force = r->f_total;
    step_system->is_force_valid = STEP_FALSE;
#endif

#if 0
  /*** for debugging electrostatics ***/
    /* resize force elec array to correct length */
    if (MD_engdata_setlen(front, eng->force_elec, eng->natoms)) {
      return MD_FAIL;
    }
    r->f_elec = (MD_Dvec *) (eng->force_elec->buf);
  /*** end for debugging electrostatics ***/
#endif

    if (eng->ismgrid) {
      /* resize mgrid system arrays to correct length */
      if (MD_engdata_setlen(front, eng->poswrap, eng->natoms)
          || MD_engdata_setlen(front, eng->f_elec, eng->natoms)
          || MD_engdata_setlen(front, eng->charge, eng->natoms)) {
        return MD_FAIL;
      }

      /* make sure mgrid system uses these arrays */
      memset(mgsys, 0, sizeof(MgridSystem));
      mgsys->f_elec = (MD_Dvec *) (eng->f_elec->buf);
      mgsys->pos = (MD_Dvec *) (eng->poswrap->buf);
      mgsys->charge = (double *) (eng->charge->buf);
    }

    if (eng->ispmetest) {
      /* resize pmetest system arrays to correct length */
      if (MD_engdata_setlen(front, eng->poswrap, eng->natoms)
          || MD_engdata_setlen(front, eng->f_elec, eng->natoms)
          || MD_engdata_setlen(front, eng->charge, eng->natoms)) {
        return MD_FAIL;
      }

      /* make sure pmetest system uses these arrays */
      memset(ptsys, 0, sizeof(PmetestSystem));
      ptsys->f_elec = (MD_Dvec *) (eng->f_elec->buf);
      ptsys->pos = (MD_Dvec *) (eng->poswrap->buf);
      ptsys->charge = (double *) (eng->charge->buf);
    }
  }

  /* verify that potential buffers have correct length, attach to ForceTraj */
  if ((eng->u_bond->attrib.access & MD_MODIFY)
      || (eng->u_angle->attrib.access & MD_MODIFY)
      || (eng->u_dihed->attrib.access & MD_MODIFY)
      || (eng->u_impr->attrib.access & MD_MODIFY)
      || (eng->u_elec->attrib.access & MD_MODIFY)
      || (eng->u_vdw->attrib.access & MD_MODIFY)
      || (eng->u_bound->attrib.access & MD_MODIFY)
      || (eng->pot_elec->attrib.access & MD_MODIFY)) {

    /* check for correct lengths of arrays */
    /* arrays are either length zero or same length as corresponding array */
    if ((eng->u_bond->attrib.len != 0 &&
          eng->u_bond->attrib.len != eng->bond->attrib.len)
        || (eng->u_angle->attrib.len != 0 && 
          eng->u_angle->attrib.len != eng->angle->attrib.len)
        || (eng->u_dihed->attrib.len != 0 && 
          eng->u_dihed->attrib.len != eng->dihed->attrib.len)
        || (eng->u_impr->attrib.len != 0 && 
          eng->u_impr->attrib.len != eng->impr->attrib.len)
        || (eng->u_elec->attrib.len != 0 && 
          eng->u_elec->attrib.len != eng->natoms)
        || (eng->u_vdw->attrib.len != 0 && 
          eng->u_vdw->attrib.len != eng->natoms)
        || (eng->u_bound->attrib.len != 0 && 
          eng->u_bound->attrib.len != eng->natoms)
        || (eng->pot_elec->attrib.len != 0 &&
          eng->pot_elec->attrib.len != eng->natoms)) {
      return MD_error(front, eng->err_force_init);
    }

    /* attach ForceResult array pointers to provided buffer space */
    r->e_bond = (double *) (eng->u_bond->buf);
    r->e_angle = (double *) (eng->u_angle->buf);
    r->e_dihed = (double *) (eng->u_dihed->buf);
    r->e_impr = (double *) (eng->u_impr->buf);
    r->e_elec = (double *) (eng->u_elec->buf);
    r->e_vdw = (double *) (eng->u_vdw->buf);
    r->e_bres = (double *) (eng->u_bound->buf);
    r->e_epot = (double *) (eng->pot_elec->buf);

    /* make sure that length zero arrays are NULL */
    if ((eng->u_bond->attrib.len == 0 && r->e_bond != NULL)
        || (eng->u_angle->attrib.len == 0 && r->e_angle != NULL)
        || (eng->u_dihed->attrib.len == 0 && r->e_dihed != NULL)
        || (eng->u_impr->attrib.len == 0 && r->e_impr != NULL)
        || (eng->u_elec->attrib.len == 0 && r->e_elec != NULL)
        || (eng->u_vdw->attrib.len == 0 && r->e_vdw != NULL)
        || (eng->u_bound->attrib.len == 0 && r->e_bres != NULL)
        || (eng->pot_elec->attrib.len == 0 && r->e_epot != NULL)) {
      return MD_error(front, eng->err_force_init);
    }

    /* acknowledge engdata modification to interface */
    if (((eng->u_bond->attrib.access & MD_MODIFY)
          && MD_engdata_ackmod(front, eng->u_bond))
        || ((eng->u_angle->attrib.access & MD_MODIFY)
          && MD_engdata_ackmod(front, eng->u_angle))
        || ((eng->u_dihed->attrib.access & MD_MODIFY)
          && MD_engdata_ackmod(front, eng->u_dihed))
        || ((eng->u_impr->attrib.access & MD_MODIFY)
          && MD_engdata_ackmod(front, eng->u_impr))
        || ((eng->u_elec->attrib.access & MD_MODIFY)
          && MD_engdata_ackmod(front, eng->u_elec))
        || ((eng->u_vdw->attrib.access & MD_MODIFY)
          && MD_engdata_ackmod(front, eng->u_vdw))
        || ((eng->u_bound->attrib.access & MD_MODIFY)
          && MD_engdata_ackmod(front, eng->u_bound))
        || ((eng->pot_elec->attrib.access & MD_MODIFY)
          && MD_engdata_ackmod(front, eng->pot_elec))) {
      return MD_FAIL;
    }
  }

  /* deal with velocity update */
  if (eng->vel->attrib.access & MD_MODIFY) {

    /* make sure step uses this velocity array */
    is_init_steplib = TRUE;
#if 0
    /* make sure step system uses this velocity array */
    step_system->vel = (MD_Dvec *) (eng->vel->buf);
#endif

    /* acknowledge velocity modification to interface */
    if (MD_engdata_ackmod(front, eng->vel)) {
      return MD_FAIL;
    }
  }

  /* deal with position update */
  if (eng->pos->attrib.access & MD_MODIFY) {

    /* make sure step uses this position array */
    is_init_steplib = TRUE;
#if 0
    /* make sure step system uses this position array */
    step_system->pos = (MD_Dvec *) (eng->pos->buf);
    step_system->is_force_valid = STEP_FALSE;
#endif

    /* acknowledge position modification to interface */
    if (MD_engdata_ackmod(front, eng->pos)) {
      return MD_FAIL;
    }
  }

  /* deal with temperature update */
  if (eng->init_temp_engdata->attrib.access & MD_MODIFY) {

    /* make sure step uses this temperature */
    is_init_steplib = TRUE;
    is_init_temperature = TRUE;

    /* acknowledge temperature modification to interface */
    if (MD_engdata_ackmod(front, eng->init_temp_engdata)) {
      return MD_FAIL;
    }
  }

  /* see if we need to (re)initialize step library */
  if (is_init_steplib || eng->atom->attrib.access & MD_MODIFY) {
    MD_Atom *atom = (MD_Atom *) (eng->atom->buf);
    int32 i;

    /*** XXX set atom status flags ***/
    for (i = 0;  i < eng->natoms;  i++) {
      /* Drude particle is also "hydrogen" for efficient force eval */
      atom[i].is_hydrogen = (atom[i].m < 1.5);
      atom[i].is_other = (atom[i].m < 1.);
    }
    for (i = 0;  i < eng->natoms-2;  i++) {
      /*
       * assume topology layout:  water is always O followed by two H;
       * mass is checked to distinguish H from Drude particle
       */
      atom[i].is_water = (atom[i].m > 15. && atom[i].m < 16.5
          && atom[i+1].is_hydrogen && atom[i+1].m >= 1.
          && atom[i+2].is_hydrogen && atom[i+2].m >= 1.);
    }
    for (i = 0;  i < eng->nfixedatoms;  i++) {
      atom[ eng->fixedatom[i] ].is_fixed = TRUE;
    }

#if 0
    /*** REDUNDANT - steplib maintains this ***/
    /* determine number of degrees of freedom for system */
    eng->ndegfreedom = 3 * eng->natoms;

    /*** FIXED ATOMS - adjust number of degrees of freedom ***/
    if (eng->nfixedatoms > 0) {
      eng->ndegfreedom -= 3 * eng->nfixedatoms;
      if (eng->ndegfreedom <= 0) {
        printf("\nERROR: too many fixed atoms, "
            "no degrees of freedom left in system!\n");
        return MD_error(front, eng->err_force_init);
      }
    }
#endif

    /*** HARMONIC RESTRAINTS - check for coordinate component selection ***/
    eng->is_select_components = FALSE;
    eng->is_select_xcoord = FALSE;
    eng->is_select_ycoord = FALSE;
    eng->is_select_zcoord = FALSE;
    if (strcasecmp(param->selectConstraints, "on") == 0
        || strcasecmp(param->selectConstraints, "yes") == 0) {
      eng->is_select_components = TRUE;
      if (strcasecmp(param->selectConstrX, "on") == 0
          || strcasecmp(param->selectConstrX, "yes") == 0) {
        eng->is_select_xcoord = TRUE;
      }
      else if (strcasecmp(param->selectConstrX, "off") != 0
          && strcasecmp(param->selectConstrX, "no") != 0) {
        return MD_error(front, eng->err_force_init);
      }
      if (strcasecmp(param->selectConstrY, "on") == 0
          || strcasecmp(param->selectConstrY, "yes") == 0) {
        eng->is_select_ycoord = TRUE;
      }
      else if (strcasecmp(param->selectConstrY, "off") != 0
          && strcasecmp(param->selectConstrY, "no") != 0) {
        return MD_error(front, eng->err_force_init);
      }
      if (strcasecmp(param->selectConstrZ, "on") == 0
          || strcasecmp(param->selectConstrZ, "yes") == 0) {
        eng->is_select_zcoord = TRUE;
      }
      else if (strcasecmp(param->selectConstrZ, "off") != 0
          && strcasecmp(param->selectConstrZ, "no") != 0) {
        return MD_error(front, eng->err_force_init);
      }
    }
    else if (strcasecmp(param->selectConstraints, "off") != 0
        && strcasecmp(param->selectConstraints, "no") != 0) {
      return MD_error(front, eng->err_force_init);
    }

    /* is step lib already initialized? */
    if (eng->is_step_initialized) {
      printf("# WARNING: reinitializing step library\n");
      step_cleanup(step);
      step_param->options |= STEP_RESTARTVEL;
#if 0
      /* since these are NOT initial velocities, don't remove com motion */
      do_remove_com_motion = FALSE;
#endif
    }

    step_param->options = 0;

    /* have to compute momentum and virial */
    /* (no way to tell if they are desired or not) */
    step_param->options |= STEP_MOMENTUM | STEP_VIRIAL;

    if (eng->nfixedatoms > 0) {
      step_param->options |= STEP_FIXEDATOMS;
    }

    if (eng->nconstraints > 0) {
      step_param->options |= STEP_CONSTRAINTS;
    }

    /* does step initialize velocities? */
    if (is_init_temperature) {
      step_param->options |= STEP_INITVEL;
    }

    /* allow center-of-mass motion? */
#if 0
    do_remove_com_motion = FALSE;
#endif
    if (strcasecmp(eng->com_motion, "no") == 0
        || strcasecmp(eng->com_motion, "off") == 0) {

      /* remove center of mass motion from initial velocities */
      step_param->options |= STEP_REMOVECOM;
#if 0
      do_remove_com_motion = TRUE;

      /* this reduces number of degrees of freedom (by one particle) */
      if (eng->nfixedatoms == 0 && eng->nconstraints == 0) {
        eng->ndegfreedom -= 3;
      }
#endif
    }
    else if (strcasecmp(eng->com_motion, "re") == 0
        || strcasecmp(eng->com_motion, "res") == 0) {
      step_param->options |= (STEP_REMOVECOM | STEP_RESTARTVEL);
#if 0
      /* restarting from previous simulation */
      /* don't remove center of mass motion because that changes vel */
      /* but do adjust number of degrees of freedom (by one particle) */
      if (eng->nfixedatoms == 0 && eng->nconstraints == 0) {
        eng->ndegfreedom -= 3;
      }
#endif
    }
    else if (strcasecmp(eng->com_motion, "yes") != 0
        && strcasecmp(eng->com_motion, "on") != 0) {
      return MD_error(front, eng->err_force_init);
    }

    /* callbacks from step lib */
    step_param->messageObjPtr = (void *) front;
    step_param->output_message = deven_output_from_step;
    step_param->forceObjPtr = (void *) front;
    step_param->compute_force = deven_force;
    step_param->resultsObjPtr = (void *) front;
    step_param->submit_results = deven_results_from_step;
    step_param->resultsFreq = 1;  /* grr! */
    step_param->init_stepnum = MD_stepnum(front);

    /* set step params */
    step_param->random_seed = eng->seed;
    step_param->timestep = eng->timestep;
    step_param->initial_temperature = eng->init_temp;
    step_param->atom = (MD_Atom *) (eng->atom->buf);
    step_param->natoms = eng->natoms;

#if 0
    step_param->bond = (MD_Bond *) (eng->bond->buf);  /* needed for Drude */
    step_param->nbonds = eng->bond->attrib.len;
    step_param->angle = (MD_Angle *) (eng->angle->buf);  /* for SETTLE */
    step_param->nangles = eng->angle->attrib.len;
    step_param->bondprm = (MD_BondPrm *) (eng->bondprm->buf);
    step_param->nbondprms = eng->bondprm->attrib.len;
    step_param->angleprm = (MD_AnglePrm *) (eng->angleprm->buf);
    step_param->nangleprms = eng->angleprm->attrib.len;
#endif

#if 0
    step_param->ndegfreedom = eng->ndegfreedom;
#endif
    step_param->bath_temperature = param->targetTemperature;
    step_param->relaxation_time = param->relaxationTime;
    step_param->nh_temperature = param->nhTemperature;
    step_param->nh_timescale = param->nhTimescale;
    step_param->drude_com_temperature = param->druComTemp;
    step_param->drude_bond_temperature = param->druBondTemp;
    step_param->drude_com_timescale = param->druComTimescale;
    step_param->drude_bond_timescale = param->druBondTimescale;
    step_param->drude_multisteps = param->druMultisteps;
    step_param->cgmin_dis = param->cgmindis;
    step_param->cgmin_tol = param->cgmintol;
    step_param->cgmin_eval = param->cgmineval;

    /* check for rigid waters */
#if 0
    step_param->rigid_water = 0;
#endif
    if (strcasecmp(param->rigidBonds, "water") == 0) {
      /*
       * set bond and angle equilibrium lengths,
       * assume all water molecules have same parameterization
       */
      int32 i, j;
      int32 nbonds = eng->bond->attrib.len;
      int32 nangles = eng->angle->attrib.len;
      MD_Atom *atom = (MD_Atom *) (eng->atom->buf);
      MD_Bond *bond = (MD_Bond *) (eng->bond->buf);
      MD_Angle *angle = (MD_Angle *) (eng->angle->buf);
      MD_BondPrm *bondprm = (MD_BondPrm *) (eng->bondprm->buf);
      MD_AnglePrm *angleprm = (MD_AnglePrm *) (eng->angleprm->buf);

      /* find first water */
      for (i = 0;  i < eng->natoms-2;  i++) {
        if (atom[i].is_water) break;
      }

      if (i < eng->natoms-2) {
        /*
         * find bond from this water
         * assume atom index order (i,i+1) as should be read from PSF
         */
        for (j = 0;  j < nbonds;  j++) {
          if (bond[j].atom[0] == i && bond[j].atom[1] == i+1) {
            step_param->settle_oh_dist = bondprm[ bond[j].prm ].r0;
            break;
          }
        }
        if (j == nbonds) {
          printf("#\n###ERROR: can\'t find bond equilibrium length for "
              "rigid water\n#\n");
          return MD_error(front, eng->err_force_init);
        }

        /*
         * find angle from this water
         */
        for (j = 0;  j < nangles;  j++) {
          if (angle[j].atom[1] == i
              && angle[j].atom[0] == i+1 && angle[j].atom[2] == i+2) {
            step_param->settle_hoh_angle = angleprm[ angle[j].prm ].theta0;
            break;
          }
        }
        if (j == nangles) {
          printf("#\n###ERROR: can\'t find angle equilibrium size for "
              "rigid water\n#\n");
          return MD_error(front, eng->err_force_init);
        }
      }
      step_param->options |= STEP_SETTLE;
#if 0
      step_param->rigid_water = 1;
#endif
    }
    else if (strcasecmp(param->rigidBonds, "all") == 0) {
      printf("#\n###ERROR: rigidBonds \"all\" is not generally supported\n#\n");
      return MD_error(front, eng->err_force_init);
    }
    else if (strcasecmp(param->rigidBonds, "none") != 0) {
      return MD_error(front, eng->err_force_init);
    }

    /* check for SETTLE */
    if (strcasecmp(param->useSettle, "on") == 0
        || strcasecmp(param->useSettle, "yes") == 0) {
      /* SETTLE is always used for rigid water */
    }
    else if (strcasecmp(param->useSettle, "off") != 0
        && strcasecmp(param->useSettle, "no") != 0) {
      return MD_error(front, eng->err_force_init);
    }

    /* set integration method */
    step_param->method = STEP_VERLET;  /* default */

    /* check for shadow Hamiltonian */
    if (strcasecmp(param->shadow, "on") == 0
        || strcasecmp(param->shadow, "yes") == 0) {
      step_param->method = STEP_SHADOW;
    }
    else if (strcasecmp(param->shadow, "off") != 0
        && strcasecmp(param->shadow, "no") != 0) {
      return MD_error(front, eng->err_force_init);
    }

    /* check for coupling to temperature bath */
    if (strcasecmp(param->temperatureBath, "on") == 0
        || strcasecmp(param->temperatureBath, "yes") == 0) {
      step_param->method = STEP_TEMPBATH;
    }
    else if (strcasecmp(param->temperatureBath, "off") != 0
        && strcasecmp(param->temperatureBath, "no") != 0) {
      return MD_error(front, eng->err_force_init);
    }

    /* check for explicit Nose-Hoover */
    if (strcasecmp(param->noseHoover, "on") == 0
	|| strcasecmp(param->noseHoover, "yes") == 0) {
      step_param->method = STEP_NHEXP;
    }
    else if (strcasecmp(param->noseHoover, "off") != 0
	&& strcasecmp(param->noseHoover, "no") != 0) {
      return MD_error(front, eng->err_force_init);
    }

    /* check for thermalized Drude oscillator */
    if (strcasecmp(param->drude, "on") == 0
        || strcasecmp(param->drude, "yes") == 0) {

      /* default method with dual-thermostat Nose-Hoover */
      step_param->method = STEP_DRUDE_NH;

      /* Roux's integration method for Drude oscillators */
      if (strcasecmp(param->useRoux, "on") == 0
          || strcasecmp(param->useRoux, "yes") == 0) {
        step_param->method = STEP_DRUDE_ROUX;
      }
      else if (strcasecmp(param->useRoux, "off") != 0
          && strcasecmp(param->useRoux, "no") != 0) {
        return MD_error(front, eng->err_force_init);
      }

      /* Chen's alternative method for Drude oscillators */
      if (strcasecmp(param->useChen, "on") == 0
          || strcasecmp(param->useChen, "yes") == 0) {
        step_param->method = STEP_DRUDE_CHEN;
      }
      else if (strcasecmp(param->useChen, "off") != 0
          && strcasecmp(param->useChen, "no") != 0) {
        return MD_error(front, eng->err_force_init);
      }

    }
    else if (strcasecmp(param->drude, "off") != 0
        && strcasecmp(param->drude, "no") != 0) {
      return MD_error(front, eng->err_force_init);
    }

    /* check for energy minimization */
    if (strcasecmp(param->cgmin, "on") == 0
        || strcasecmp(param->cgmin, "yes") == 0) {
      step_param->method = STEP_CGMIN;
    }
    else if (strcasecmp(param->cgmin, "off") != 0
        && strcasecmp(param->cgmin, "no") != 0) {
      return MD_error(front, eng->err_force_init);
    }

    if (STEP_FAILURE==step_setup(step, step_param,
          (MD_Dvec *)(eng->pos->buf), (MD_Dvec *)(eng->vel->buf),
          r->f_total)) {
      printf("#STEP ERROR: %s\n", step_errmsg(step));
      return MD_error(front, eng->err_force_init);
    }
    eng->is_step_initialized = TRUE;

    /*** MD_MODIFY flag is cleared from atom later ***/

  } /* done with step library initialization */

  /* see if we need to (re)initialize force library */
  if ((eng->atomprm->attrib.access & MD_MODIFY)
      || (eng->bondprm->attrib.access & MD_MODIFY)
      || (eng->angleprm->attrib.access & MD_MODIFY)
      || (eng->dihedprm->attrib.access & MD_MODIFY)
      || (eng->imprprm->attrib.access & MD_MODIFY)
      || (eng->nbfixprm->attrib.access & MD_MODIFY)
      || (eng->atom->attrib.access & MD_MODIFY)
      || (eng->bond->attrib.access & MD_MODIFY)
      || (eng->angle->attrib.access & MD_MODIFY)
      || (eng->dihed->attrib.access & MD_MODIFY)
      || (eng->impr->attrib.access & MD_MODIFY)
      || (eng->excl->attrib.access & MD_MODIFY)
      || (eng->param_engdata->attrib.access & MD_MODIFY)) {

#if 0
    int32 alg_cutoff_elec = FORCE_ELEC_PAIRLISTS;
    int32 alg_cutoff_vdw = FORCE_VDW_PAIRLISTS;
#endif
    int32 alg_cutoff_elec = FORCE_ELEC_GRIDCELLS;
    int32 alg_cutoff_vdw = FORCE_VDW_GRIDCELLS;

    if (eng->force) {
      /* must first destroy previous force */
      force_destroy(eng->force);
      /* cleanup force param object */
      force_param_cleanup(p);
    }

    /* reset force param to default values */
    force_param_initialize(p);

    if (eng->ismgrid) {
      /* must first destroy previous mgrid */
      mgrid_done(mg);
      if (mgrid_init(mg)) {
        return MD_error(front, eng->err_force_init);
      }
      /* zero mgrid param */
      memset(mgprm, 0, sizeof(MgridParam));
    }

    /* set force options */
    p->forcetypes = FORCE_ALL;
    p->exclpolicy = FORCE_NONE;  /* not yet initialized */
    p->elecopts = alg_cutoff_elec | FORCE_ELEC_STANDARD;
    p->vdwopts = alg_cutoff_vdw | FORCE_VDW_STANDARD;
    p->bresopts = FORCE_NONE;    /* no boundary restraints by default */

#if 0
    /* set energy types based on presence of energy arrays */
    p->energytypes = FORCE_NONE;
    p->energytypes |= (t->u_bond ? FORCE_BOND : 0);
    p->energytypes |= (t->u_angle ? FORCE_ANGLE : 0);
    p->energytypes |= (t->u_dihed ? FORCE_DIHED : 0);
    p->energytypes |= (t->u_impr ? FORCE_IMPR : 0);
    p->energytypes |= (t->u_elec ? FORCE_ELEC : 0);
    p->energytypes |= (t->u_vdw ? FORCE_VDW : 0);
    p->energytypes |= (t->u_bdres ? FORCE_BDRES : 0);
    p->energytypes |= (t->pot_elec ? FORCE_EPOT : 0);
#endif

    /* setup force params */
    p->atomprm = (MD_AtomPrm *) (eng->atomprm->buf);
    p->bondprm = (MD_BondPrm *) (eng->bondprm->buf);
    p->angleprm = (MD_AnglePrm *) (eng->angleprm->buf);
    p->dihedprm = (MD_TorsPrm *) (eng->dihedprm->buf);
    p->imprprm = (MD_TorsPrm *) (eng->imprprm->buf);
    p->nbfixprm = (MD_NbfixPrm *) (eng->nbfixprm->buf);
    p->atomprm_len = eng->atomprm->attrib.len;
    p->bondprm_len = eng->bondprm->attrib.len;
    p->angleprm_len = eng->angleprm->attrib.len;
    p->dihedprm_len = eng->dihedprm->attrib.len;
    p->imprprm_len = eng->imprprm->attrib.len;
    p->nbfixprm_len = eng->nbfixprm->attrib.len;

    p->atom = (MD_Atom *) (eng->atom->buf);
    p->bond = (MD_Bond *) (eng->bond->buf);
    p->angle = (MD_Angle *) (eng->angle->buf);
    p->dihed = (MD_Tors *) (eng->dihed->buf);
    p->impr = (MD_Tors *) (eng->impr->buf);
    p->excl = (MD_Excl *) (eng->excl->buf);
    p->atom_len = eng->atom->attrib.len;
    p->bond_len = eng->bond->attrib.len;
    p->angle_len = eng->angle->attrib.len;
    p->dihed_len = eng->dihed->attrib.len;
    p->impr_len = eng->impr->attrib.len;
    p->excl_len = eng->excl->attrib.len;

    /* periodicity in each "direction" determined by nonzero basis vector */
    d->center = param->cellOrigin;
    d->v1 = param->cellBasisVector1;
    d->v2 = param->cellBasisVector2;
    d->v3 = param->cellBasisVector3;

    /* setup spherical boundary conditions */
    if (strcasecmp(param->sphericalBC, "on") == 0
        || strcasecmp(param->sphericalBC, "yes") == 0) {
      /* turn spherical boundary conditions on */
      p->bresopts = FORCE_BRES_SPHERE;
      /*
       * in this case, xlen, ylen, zlen are still used by grid cell
       * algorithm to determine spatial region of molecule
       * (make sure they correspond to extent of sphere)
       */
      d->center = param->sphericalBCCenter;
      p->radius1 = param->sphericalBCr1;
      p->radius2 = param->sphericalBCr2;
      p->konst1 = param->sphericalBCk1;
      p->konst2 = param->sphericalBCk2;
      p->exp1 = param->sphericalBCexp1;
      p->exp2 = param->sphericalBCexp2;
    }
    else if (strcasecmp(param->sphericalBC, "off") != 0
        && strcasecmp(param->sphericalBC, "no") != 0) {
      return MD_error(front, eng->err_force_init);
    }

    /* setup cylindrical boundary conditions */
    if (strcasecmp(param->cylindricalBC, "on") == 0
        || strcasecmp(param->cylindricalBC, "yes") == 0) {
      /* make sure spherical boundaries were not also set */
      if (p->bresopts == FORCE_BRES_SPHERE) {
        return MD_error(front, eng->err_force_init);
      }
      /* flags depend on axis value */
      if (strcasecmp(param->cylindricalBCAxis, "x") == 0) {
        p->bresopts = FORCE_BRES_X_CYLINDER;
      }
      else if (strcasecmp(param->cylindricalBCAxis, "y") == 0) {
        p->bresopts = FORCE_BRES_Y_CYLINDER;
      }
      else if (strcasecmp(param->cylindricalBCAxis, "z") == 0) {
        p->bresopts = FORCE_BRES_Z_CYLINDER;
      }
      else {
        return MD_error(front, eng->err_force_init);
      }
      /*
       * in this case, xlen, ylen, zlen are still used by grid cell
       * algorithm to determine spatial region of molecule
       * (make sure they correspond to extent of cylinder)
       */
      d->center = param->cylindricalBCCenter;
      p->radius1 = param->cylindricalBCr1;
      p->radius2 = param->cylindricalBCr2;
      p->length1 = param->cylindricalBCl1;
      p->length2 = param->cylindricalBCl2;
      p->konst1 = param->cylindricalBCk1;
      p->konst2 = param->cylindricalBCk2;
      p->exp1 = param->cylindricalBCexp1;
      p->exp2 = param->cylindricalBCexp2;
    }
    else if (strcasecmp(param->cylindricalBC, "off") != 0
        && strcasecmp(param->cylindricalBC, "no") != 0) {
      return MD_error(front, eng->err_force_init);
    }

    /* turn off boundary restraints if not requested */
    if (p->bresopts == FORCE_NONE) {
      p->forcetypes &= ~FORCE_BRES;
    }

    /* full direct electrostatics */
    if (strcasecmp(param->fullDirect, "on") == 0
        || strcasecmp(param->fullDirect, "yes") == 0) {
      p->elecopts = FORCE_ELEC_DIRECT | FORCE_ELEC_STANDARD;
      /* overrides mgrid and pmetest */
      eng->ismgrid = FALSE;
      eng->ispmetest = FALSE;
    }
    else if (strcasecmp(param->fullDirect, "off") != 0
        && strcasecmp(param->fullDirect, "no") != 0) {
      return MD_error(front, eng->err_force_init);
    }

    /* set exclusion policy */
    if (strcasecmp(param->exclude, "none") == 0) {
      p->exclpolicy = FORCE_EXCL_NONE;
    }
    else if (strcasecmp(param->exclude, "1-2") == 0) {
      p->exclpolicy = FORCE_EXCL_12;
    }
    else if (strcasecmp(param->exclude, "1-3") == 0) {
      p->exclpolicy = FORCE_EXCL_13;
    }
    else if (strcasecmp(param->exclude, "1-4") == 0) {
      p->exclpolicy = FORCE_EXCL_14;
    }
    else if (strcasecmp(param->exclude, "scaled1-4") == 0) {
      p->exclpolicy = FORCE_EXCL_SCAL14;
    }
    else {
      printf("#\n###ERROR: must set \"exclude\"\n#\n");
      return MD_error(front, eng->err_force_init);
    }

    /* switching and smoothing */
    if (strcasecmp(param->switching, "on") == 0
        || strcasecmp(param->switching, "yes") == 0) {
      p->vdwopts = alg_cutoff_vdw | FORCE_VDW_SWITCHED;
      if ((p->elecopts & FORCE_ELEC_DIRECT) == 0) {
        p->elecopts = alg_cutoff_elec | FORCE_ELEC_SHIFTED;
      }
      p->switchdist = param->switchDist;
    }
    else if (strcasecmp(param->switching, "off") != 0
        && strcasecmp(param->switching, "no") != 0) {
      return MD_error(front, eng->err_force_init);
    }

    /* Buckingham potential */
    if (strcasecmp(param->buckingham, "on") == 0
        || strcasecmp(param->buckingham, "yes") == 0) {
      if (p->vdwopts & FORCE_VDW_SWITCHED) {
        p->vdwopts = alg_cutoff_vdw | FORCE_VDW_SWITCHBUCK;
      }
      else {
        p->vdwopts = alg_cutoff_vdw | FORCE_VDW_BUCK;
      }
    }
    else if (strcasecmp(param->buckingham, "off") != 0
        && strcasecmp(param->buckingham, "no") != 0) {
      return MD_error(front, eng->err_force_init);
    }

    /* Buckingham potential with no dispersion */
    if (strcasecmp(param->bucknodispersion, "on") == 0
        || strcasecmp(param->bucknodispersion, "yes") == 0) {
      if (p->vdwopts & FORCE_VDW_SWITCHBUCK) {
        p->vdwopts = alg_cutoff_vdw | FORCE_VDW_SWITCHBUCKND;
      }
      else if (p->vdwopts & FORCE_VDW_BUCK) {
        p->vdwopts = alg_cutoff_vdw | FORCE_VDW_BUCKND;
      }
      else {
        return MD_error(front, eng->err_force_init);
      }
    }
    else if (strcasecmp(param->bucknodispersion, "off") != 0
        && strcasecmp(param->bucknodispersion, "no") != 0) {
      return MD_error(front, eng->err_force_init);
    }

    /* Buckingham parameterization */
    if (strcasecmp(param->buckparam, "fb") == 0) {
      if (p->vdwopts & FORCE_MASK_VDW_BUCK) {
        p->vdwopts |= FORCE_VDW_BUCKPRM_FB;
      }
    }
    else if (strcasecmp(param->buckparam, "ttam") == 0) {
      if (p->vdwopts & FORCE_MASK_VDW_BUCK) {
        p->vdwopts |= FORCE_VDW_BUCKPRM_TTAM;
      }
    }
    else if (strcasecmp(param->buckparam, "bks") == 0) {
      if (p->vdwopts & FORCE_MASK_VDW_BUCK) {
        p->vdwopts |= FORCE_VDW_BUCKPRM_BKS;
      }
    }
    else if (strcasecmp(param->buckparam, "glass") == 0) {
      if (p->vdwopts & FORCE_MASK_VDW_BUCK) {
        p->vdwopts = alg_cutoff_vdw | FORCE_VDW_GLASS | FORCE_VDW_GLASSPRM;
        p->elecopts = alg_cutoff_elec | FORCE_ELEC_GLASS;
      }
    }
    else {
      return MD_error(front, eng->err_force_init);
    }

    /*
     * use safe form of Buckingham potential, with smooth extension
     * that gets rid of attractive energy well for close interactions
     */
    if (strcasecmp(param->bucksafe, "on") == 0
        || strcasecmp(param->bucksafe, "yes") == 0) {
      if (p->vdwopts & FORCE_VDW_BUCK) {
        p->vdwopts &= ~FORCE_VDW_BUCK;
        p->vdwopts |= FORCE_VDW_BUCKSAFE;
      }
      else if (p->vdwopts & FORCE_VDW_SWITCHBUCK) {
        p->vdwopts &= ~FORCE_VDW_SWITCHBUCK;
        p->vdwopts |= FORCE_VDW_SWITCHBUCKSAFE;
      }
    }
    else if (strcasecmp(param->bucksafe, "off") != 0
        && strcasecmp(param->bucksafe, "no") != 0) {
      return MD_error(front, eng->err_force_init);
    }

    /* nonbonded params */
    p->cutoff = param->cutoff;
    p->elec_const = MD_COULOMB;
    p->dielectric = param->dielectric;
    p->scaling14 = param->scaling;  /* value of "1-4scaling" */

    /* see if mgrid is to be setup (override by full direct elec on) */
    if (eng->ismgrid) {
      double c;
      int32 n;

      /* does use of boundary conditions permit mgrid for cubic domain? */
      if (d->v1.x < 0.0 || d->v2.y < 0.0 || d->v3.z < 0.0
          || d->v1.y != 0.0 || d->v1.z != 0.0
          || d->v2.x != 0.0 || d->v2.z != 0.0
          || d->v3.x != 0.0 || d->v3.y != 0.0) {
        /* choice of boundary conditions does not work */
        return MD_error(front, eng->err_force_init);
      }
      else if (d->v1.x == 0.0 && d->v2.y == 0.0 && d->v3.z == 0.0) {
        mgprm->center = d->center;
        mgprm->boundary = MGRID_NONPERIODIC;
      }
      else if (d->v1.x == d->v2.y && d->v2.y == d->v3.z
          && d->v3.z == param->mgridLength) {
        mgprm->center = d->center;
        mgprm->boundary = MGRID_PERIODIC;
      }
      else {
        /* choice of boundary conditions does not work */
        return MD_error(front, eng->err_force_init);
      }

      /* set remaining params */
      mgprm->length = param->mgridLength;
      mgprm->cutoff = ( param->mgridCutoff > 0 ?
          param->mgridCutoff : param->cutoff );
      mgprm->spacing = param->mgridSpacing;
      mgprm->nspacings = param->mgridNspacings;
      mgprm->nlevels = param->mgridNlevels;
      mgprm->natoms = eng->natoms;
      mgprm->approx = mgrid_string_to_approx(param->mgridApprox);
      mgprm->split = mgrid_string_to_split(param->mgridSplit);

      /* setup charge array, scale by sqrt of multiplicative const */
      c = sqrt(p->elec_const / p->dielectric);
      for (n = 0;  n < eng->natoms;  n++) {
        mgsys->charge[n] = c * p->atom[n].q;
      }

      if (mgrid_param_config(mgprm)) {
        printf("# mgrid failed to configure parameters\n");
        return MD_error(front, eng->err_force_init);
      }

      if (mgrid_setup(mg, mgsys, mgprm)) {
        printf("# mgrid setup failed\n");
        return MD_error(front, eng->err_force_init);
      }
      else {
        printf("# mgrid setup successful with parameters:\n");
        printf("#   center = %g %g %g\n", mgprm->center.x,
            mgprm->center.y, mgprm->center.z);
        printf("#   length = %g\n", mgprm->length);
        printf("#   cutoff = %g\n", mgprm->cutoff);
        printf("#   spacing = %g\n", mgprm->spacing);
        printf("#   nspacings = %d\n", mgprm->nspacings);
        printf("#   nlevels = %d\n", mgprm->nlevels);
        printf("#   boundary = %s\n",
            mgrid_boundary_to_string(mgprm->boundary));
        printf("#   natoms = %d\n", mgprm->natoms);
        printf("#   approx = %s\n", mgrid_approx_to_string(mgprm->approx));
        printf("#   approx = %s\n", mgrid_split_to_string(mgprm->split));
      }

#if 0
      /* turn off all force library elec */
      p->flags &= ~(FORCE_SMOOTH | FORCE_ELEC | FORCE_ELEC_EXCL);
#endif
      p->elecopts = FORCE_ELEC_STNDEXCL;
    }

    /* see if pmetest is to be setup (override by full direct elec on) */
    if (eng->ispmetest) {
      double c;
      int32 n;

      /* destroy old pmetest object if created and reset params */
      if (eng->pmetest) {
        pmetest_destroy(eng->pmetest);
        memset(ptprm, 0, sizeof(PmetestParams));
      }

      ptprm->center = d->center;
      ptprm->cellvec1 = d->v1;
      ptprm->cellvec2 = d->v2;
      ptprm->cellvec3 = d->v3;
      ptprm->cutoff = ( param->pmetestCutoff > 0.0 ?
          param->pmetestCutoff : param->cutoff );
      ptprm->tolerance = param->pmetestTolerance;
      ptprm->nxspacings = param->pmetestGridSizeX;
      ptprm->nyspacings = param->pmetestGridSizeY;
      ptprm->nzspacings = param->pmetestGridSizeZ;
      ptprm->interporder = param->pmetestInterpOrder;
      ptprm->natoms = eng->natoms;

      /* setup charge array, scale by sqrt of multiplicative const */
      c = sqrt(p->elec_const / p->dielectric);
      for (n = 0;  n < eng->natoms;  n++) {
        ptsys->charge[n] = c * p->atom[n].q;
      }

      /* create new pmetest object */
      if ((eng->pmetest = pmetest_create(ptprm)) == NULL) {
        printf("# pmetest creation failed\n");
        return MD_error(front, eng->err_force_init);
      }
      else {
        printf("# pmetest setup successful with parameters:\n");
        printf("#   cellOrigin = %g %g %g\n", ptprm->center.x,
            ptprm->center.y, ptprm->center.z);
        printf("#   cellBasisVector1 = %g %g %g\n", ptprm->cellvec1.x,
            ptprm->cellvec1.y, ptprm->cellvec1.z);
        printf("#   cellBasisVector2 = %g %g %g\n", ptprm->cellvec2.x,
            ptprm->cellvec2.y, ptprm->cellvec2.z);
        printf("#   cellBasisVector3 = %g %g %g\n", ptprm->cellvec3.x,
            ptprm->cellvec3.y, ptprm->cellvec3.z);
        printf("#   PMECutoff = %g\n", ptprm->cutoff);
        printf("#   PMETolerance = %g\n", ptprm->tolerance);
        printf("#   PMEInterpOrder = %d\n", ptprm->interporder);
        printf("#   PMEGridSizeX = %d\n", ptprm->nxspacings);
        printf("#   PMEGridSizeY = %d\n", ptprm->nyspacings);
        printf("#   PMEGridSizeZ = %d\n", ptprm->nzspacings);
        printf("#   natoms = %d\n", ptprm->natoms);
      }

      /* give ewaldcof over to force params */
      p->ewald_coef = ptprm->ewaldcof;

#if 0
      /* turn off all force library elec */
      p->flags &= ~(FORCE_SMOOTH | FORCE_ELEC | FORCE_ELEC_EXCL);
#endif
      p->elecopts = FORCE_ELEC_GRIDCELLS
        | FORCE_ELEC_STNDEXCL | FORCE_ELEC_EWALD;
    }

    /* compute force correction for constant linear momentum */
    if (strcasecmp(param->correctLinmo, "standard") == 0) {
      eng->forceopts = ENGINE_FIX_LINMO;
    }
    else if (strcasecmp(param->correctLinmo, "conserve") == 0
        || strcasecmp(param->correctLinmo, "on") == 0
        || strcasecmp(param->correctLinmo, "yes") == 0) {
      eng->forceopts = ENGINE_CONS_LINMO;
    }
    else if (strcasecmp(param->correctLinmo, "") != 0
        && strcasecmp(param->correctLinmo, "off") != 0
        && strcasecmp(param->correctLinmo, "no") != 0) {
      return MD_error(front, eng->err_force_init);
    }

    /* setup constants for force correction if needed */
    if (eng->forceopts) {
      double mass_sum = 0.0;
      void *tmp;
      const MD_Dvec *pos = (const MD_Dvec *) (eng->pos->buf);
      int32 i;

#if 0
      /*** FIXED ATOMS - not compatible ***/
      if (eng->nfixedatoms > 0) {
        printf("\nERROR: can't use force correction with fixed atoms\n");
        return MD_error(front, eng->err_force_init);
      }
#endif

      eng->inv_natoms = 1.0 / (double) eng->natoms;
      for (i = 0;  i < eng->natoms;  i++) {
        mass_sum += p->atom[i].m;
      }
      FLT(mass_sum);
      eng->inv_totalmass = 1.0 / mass_sum;

      /* store scaled mass values for mass-weighted averaging */
      tmp = realloc(eng->scaled_mass, eng->natoms * sizeof(double));
      if (tmp == NULL) return MD_error(front, eng->err_force_init);
      eng->scaled_mass = (double *) tmp;

      /* store initial positions from this point */
      tmp = realloc(eng->init_pos, eng->natoms * sizeof(MD_Dvec));
      if (tmp == NULL) return MD_error(front, eng->err_force_init);
      eng->init_pos = (MD_Dvec *) tmp;

      for (i = 0;  i < eng->natoms;  i++) {
        eng->scaled_mass[i] = p->atom[i].m * eng->inv_totalmass;
        eng->init_pos[i] = pos[i];
      }
    }

    /* initialize force library */
    VEC(((MD_Dvec *)(eng->pos->buf))[0]);
    if ((eng->force = force_create(p, d, NULL, (MD_Dvec *)(eng->pos->buf)))
        == NULL) {
      printf("# force library initialization failed\n");
      return MD_error(front, eng->err_force_init);
    }
    VEC(((MD_Dvec *)(eng->pos->buf))[0]);
    eng->is_domain_periodic = force_get_cell_boundary(eng->force);

#if 0
#ifdef DEBUG_SUPPORT
    if (force_setup_lattice(eng->force, 1, 1, 1)) {
      printf("# force_setup_lattice() failed\n");
      return MD_error(front, eng->err_force_init);
    }
#endif
#endif

#if 0
    if (eng->ismgrid) {
      /* have mgrid process exclusions */
      mgsys->excl_list = force->excl_list;
      mgsys->scaled14_list = force->scaled14_list;
      mgsys->scaling14 = p->scaling14;
    }
#endif

    /* acknowledge engdata modification to interface */
    if (((eng->atomprm->attrib.access & MD_MODIFY)
          && MD_engdata_ackmod(front, eng->atomprm))
        || ((eng->bondprm->attrib.access & MD_MODIFY)
          && MD_engdata_ackmod(front, eng->bondprm))
        || ((eng->angleprm->attrib.access & MD_MODIFY)
          && MD_engdata_ackmod(front, eng->angleprm))
        || ((eng->dihedprm->attrib.access & MD_MODIFY)
          && MD_engdata_ackmod(front, eng->dihedprm))
        || ((eng->imprprm->attrib.access & MD_MODIFY)
          && MD_engdata_ackmod(front, eng->imprprm))
        || ((eng->nbfixprm->attrib.access & MD_MODIFY)
          && MD_engdata_ackmod(front, eng->nbfixprm))
        || ((eng->atom->attrib.access & MD_MODIFY)
          && MD_engdata_ackmod(front, eng->atom))
        || ((eng->bond->attrib.access & MD_MODIFY)
          && MD_engdata_ackmod(front, eng->bond))
        || ((eng->angle->attrib.access & MD_MODIFY)
          && MD_engdata_ackmod(front, eng->angle))
        || ((eng->dihed->attrib.access & MD_MODIFY)
          && MD_engdata_ackmod(front, eng->dihed))
        || ((eng->impr->attrib.access & MD_MODIFY)
          && MD_engdata_ackmod(front, eng->impr))
        || ((eng->excl->attrib.access & MD_MODIFY)
          && MD_engdata_ackmod(front, eng->excl))
        || ((eng->param_engdata->attrib.access & MD_MODIFY)
          && MD_engdata_ackmod(front, eng->param_engdata))) {
      printf("# need to ack!!!\n");
      return MD_error(front, eng->err_force_init);
    }
  } /* done with force library initialization */

#if 0
  /* deal with temperature update */
  if (eng->init_temp_engdata->attrib.access & MD_MODIFY) {

    /* initialize velocities using this initial temperature */
    if (step_set_random_velocities(step, step_system, eng->init_temp)) {
      return MD_error(front, eng->err_force_init);
    }

    /* acknowledge temperature modification to interface */
    if (MD_engdata_ackmod(front, eng->init_temp_engdata)) {
      return MD_FAIL;
    }
  }

  /* if needed, remove center of mass motion from initial velocities */
  if (do_remove_com_motion) {
    if (step_remove_com_motion(step, step_system)) {
      return MD_error(front, eng->err_force_init);
    }
  }

  /*** FIXED ATOMS - reset their velocities to zero ***/
  if (eng->nfixedatoms > 0) {
    MD_Dvec *vel = step_system->vel;
    int32 *atomindex = eng->fixedatom;
    int32 i;

    for (i = 0;  i < eng->nfixedatoms;  i++) {
      vel[atomindex[i]].x = 0.0;
      vel[atomindex[i]].y = 0.0;
      vel[atomindex[i]].z = 0.0;
    }
  }
#endif

  return 0;
}
